(*    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{OrderedGroup\_ZF.thy}*}

theory OrderedGroup_ZF imports Group_ZF_1 AbelianGroup_ZF Order_ZF Finite_ZF_1 

begin

text{*This theory file defines and shows the basic properties of (partially
  or linearly) ordered groups. We define the set of nonnegative elements 
  and the absolute value function. 
  We show that in linearly ordered groups finite sets are bounded 
  and provide a sufficient condition for bounded sets to be finite. This 
  allows to show in @{text "Int_ZF_IML.thy"} that subsets of integers are 
  bounded iff they are finite. *}

section{*Ordered groups*}

text{*This section defines ordered groups.*}

text{*An ordered group is a group equipped with a partial order that is
  "translation invariant", that is if $a\leq b$ then $a\cdot g \leq b\cdot g$
  and $g\cdot a \leq g\cdot b$. We define the set of nonnegative elements
  in the obvious way as $G^+ =\{x\in G: 1 \leq x\}$. $G_+$ is a 
  similar set, but without the unit. We also define 
  the absolute value as a ZF-function that is the 
  identity on $G^+$ and the group inverse on the rest of the group.
  We also define the maximum absolute value of a set, that is the maximum
  of the set $\{|x|. x\in A\}$.
  The odd functions are defined as those having property 
  $f(a^{-1})=(f(a))^{-1}$. This looks a bit strange in the 
  multiplicative notation, I have to admit.
  For linearly oredered groups a function $f$ defined on the set of positive
  elements iniquely defines an odd function of the whole group. This function
  is called an odd extension of $f$.
 *}

definition
  "IsAnOrdGroup(G,P,r) \<equiv> 
  (IsAgroup(G,P) \<and> r\<subseteq>G\<times>G \<and> IsPartOrder(G,r) \<and> (\<forall>g\<in>G. \<forall>a b. 
  <a,b> \<in> r \<longrightarrow> <P`<a,g>,P`<b,g> > \<in> r \<and> <P`<g,a>,P`<g,b> > \<in> r ) )"

definition
  "Nonnegative(G,P,r) \<equiv> {x\<in>G. <TheNeutralElement(G,P),x> \<in> r}"

definition
  "PositiveSet(G,P,r) \<equiv> 
  {x\<in>G. <TheNeutralElement(G,P),x> \<in> r \<and> TheNeutralElement(G,P)\<noteq> x}"

definition
  "AbsoluteValue(G,P,r) \<equiv> id(Nonnegative(G,P,r)) \<union> 
  restrict(GroupInv(G,P),G - Nonnegative(G,P,r))"

definition
  "OddExtension(G,P,r,f) \<equiv> 
  (f \<union> {\<langle>a, GroupInv(G,P)`(f`(GroupInv(G,P)`(a)))\<rangle>. 
  a \<in> GroupInv(G,P)``(PositiveSet(G,P,r))} \<union> 
  {\<langle>TheNeutralElement(G,P),TheNeutralElement(G,P)\<rangle>})";

  (*"MaxAbs(G,P,r,A) \<equiv> Maximum(r,AbsoluteValue(G,P,r)``(A))"*)

text{*We will use a similar notation for ordered groups as for the generic 
  groups. @{text "G\<^sup>+"} denotes the set of nonnegative elements 
  (that satisfy $1\leq a$ and @{text "G\<^isub>+"} is the set of (strictly) positive
  elements. @{text "\<sm>A"} is the set inverses of elements from $A$. I hope 
  that using additive notation for this notion is not too shocking here. 
  The symbol @{text "f\<degree>"} denotes the odd extension of $f$. For a function
  defined on $G_+$ this is the unique odd function on $G$ that is 
  equal to $f$ on $G_+$. *}

locale group3 =

  fixes G and P and r

  assumes ordGroupAssum: "IsAnOrdGroup(G,P,r)"

  fixes unit ("\<one>")
  defines unit_def [simp]: "\<one> \<equiv> TheNeutralElement(G,P)"

  fixes groper (infixl "\<cdot>" 70)
  defines groper_def [simp]: "a \<cdot> b \<equiv> P`<a,b>"

  fixes inv ("_\<inverse> " [90] 91)
  defines inv_def [simp]: "x\<inverse> \<equiv> GroupInv(G,P)`(x)"

  fixes lesseq (infix "\<lsq>" 68)
  defines lesseq_def [simp]: "a \<lsq> b \<equiv> <a,b> \<in> r"

  fixes sless (infix "\<ls>" 68)
  defines sless_def [simp]: "a \<ls> b \<equiv> a\<lsq>b \<and> a\<noteq>b"

  fixes nonnegative ("G\<^sup>+")
  defines nonnegative_def [simp]: "G\<^sup>+ \<equiv> Nonnegative(G,P,r)"

  fixes positive ("G\<^isub>+")
  defines positive_def [simp]: "G\<^isub>+ \<equiv> PositiveSet(G,P,r)"
  
  fixes setinv :: "i\<Rightarrow>i" ("\<sm> _" 72)
  defines setninv_def [simp]: "\<sm>A \<equiv> GroupInv(G,P)``(A)"

  fixes abs ("| _ |")
  defines abs_def [simp]: "|a| \<equiv> AbsoluteValue(G,P,r)`(a)"

  fixes oddext ("_ \<degree>")
  defines oddext_def [simp]: "f\<degree> \<equiv> OddExtension(G,P,r,f)";


text{*In @{text "group3"} context we can use the theorems proven in the 
  @{text "group0"} context.*}

lemma (in group3) OrderedGroup_ZF_1_L1: shows "group0(G,P)"
  using ordGroupAssum IsAnOrdGroup_def group0_def by simp;

text{*Ordered group (carrier) is not empty. This is a property of
  monoids, but it is good to have it handy in the @{text "group3"} context.*}

lemma (in group3) OrderedGroup_ZF_1_L1A: shows "G\<noteq>0"
  using OrderedGroup_ZF_1_L1 group0.group0_2_L1 monoid0.group0_1_L3A
  by blast;
  
text{*The next lemma is just to see the definition of the nonnegative set
  in our notation.*}

lemma (in group3) OrderedGroup_ZF_1_L2: 
  shows "g\<in>G\<^sup>+ \<longleftrightarrow> \<one>\<lsq>g"
  using ordGroupAssum IsAnOrdGroup_def Nonnegative_def 
  by auto;

text{*The next lemma is just to see the definition of the positive set
  in our notation.*}

lemma (in group3) OrderedGroup_ZF_1_L2A: 
  shows "g\<in>G\<^isub>+ \<longleftrightarrow> (\<one>\<lsq>g \<and> g\<noteq>\<one>)"
  using ordGroupAssum IsAnOrdGroup_def PositiveSet_def 
  by auto;

text{*For total order if $g$ is not in $G^{+}$, then it has to be 
  less or equal the unit.*}

lemma (in group3) OrderedGroup_ZF_1_L2B: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G-G\<^sup>+"
  shows "a\<lsq>\<one>"
proof -
  from A2 have "a\<in>G"   "\<one> \<in> G"  "\<not>(\<one>\<lsq>a)" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2 OrderedGroup_ZF_1_L2 
    by auto;
  with A1 show ?thesis using IsTotal_def by auto;
qed;

text{*The group order is reflexive.*}

lemma (in group3) OrderedGroup_ZF_1_L3: assumes "g\<in>G"
  shows "g\<lsq>g"
  using ordGroupAssum prems IsAnOrdGroup_def IsPartOrder_def refl_def
  by simp;

text{*$1$ is nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L3A: shows "\<one>\<in>G\<^sup>+"
  using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L3
    OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
  
text{*In this context $a \leq b$ implies that both $a$ and $b$ belong 
  to $G$.*}

lemma (in group3) OrderedGroup_ZF_1_L4: 
  assumes "a\<lsq>b" shows "a\<in>G" "b\<in>G"
  using ordGroupAssum prems IsAnOrdGroup_def by auto;

text{*It is good to have transitivity handy.*}

lemma (in group3) Group_order_transitive:
  assumes A1: "a\<lsq>b"  "b\<lsq>c" shows "a\<lsq>c"
proof -
  from ordGroupAssum have "trans(r)"
    using IsAnOrdGroup_def IsPartOrder_def
    by simp
  moreover from A1 have "<a,b> \<in> r \<and> <b,c> \<in> r" by simp;
  ultimately have "<a,c> \<in> r" by (rule Fol1_L3);
  thus ?thesis by simp;
qed;

text{*The order in an ordered group is antisymmetric.*}

lemma (in group3) group_order_antisym:
  assumes A1: "a\<lsq>b"  "b\<lsq>a" shows "a=b"
proof -
  from ordGroupAssum A1 have 
    "antisym(r)"  "<a,b> \<in> r"  "<b,a> \<in> r"
    using IsAnOrdGroup_def IsPartOrder_def by auto;
  then show "a=b" by (rule Fol1_L4); 
qed;

text{*Transitivity for the strict order: if $a<b$ and $b\leq c$, then $a<c$. *}

lemma (in group3) OrderedGroup_ZF_1_L4A:
  assumes A1: "a\<ls>b"  and A2: "b\<lsq>c"
  shows "a\<ls>c"
proof -
  from A1 A2 have "a\<lsq>b"  "b\<lsq>c" by auto
  then have "a\<lsq>c" by (rule Group_order_transitive);
  moreover from A1 A2 have "a\<noteq>c" using group_order_antisym by auto;
  ultimately show "a\<ls>c" by simp
qed;
(*lemma (in group3) OrderedGroup_ZF_1_L4A:
  assumes A1: "a\<lsq>b"  and A2: "a\<noteq>b" and A3: "b\<lsq>c"
  shows "a\<lsq>c"  "a\<noteq>c"
proof -
  from A1 A3 show "a\<lsq>c" by (rule Group_order_transitive);
  from A1 A2 A3 show "a\<noteq>c" using group_order_antisym by auto;
qed;*)

text{*Another version of transitivity for the strict order: 
  if $a\leq b$ and $b<c$, then $a<c$. *}

lemma (in group3) group_strict_ord_transit:
  assumes A1: "a\<lsq>b" and A2: "b\<ls>c"
  shows "a\<ls>c"
proof -
  from A1 A2 have "a\<lsq>b"  "b\<lsq>c" by auto
  then have  "a\<lsq>c" by (rule Group_order_transitive)
  moreover from A1 A2 have "a\<noteq>c" using group_order_antisym by auto
  ultimately show "a\<ls>c" by simp
qed;
(*lemma (in group3) group_strict_ord_transit:
  assumes A1: "a\<lsq>b" and A2: "b\<lsq>c" and A3: "b\<noteq>c"
  shows "a\<lsq>c"  "a\<noteq>c"
proof -
  from A1 A2 show "a\<lsq>c" by (rule Group_order_transitive)
  from A1 A2 A3 show "a\<noteq>c" using group_order_antisym by auto
qed;*)

text{*Strict order is preserved by translations.*}

lemma (in group3) group_strict_ord_transl_inv: 
  assumes "a\<ls>b"and "c\<in>G"
  shows 
  "a\<cdot>c \<ls> b\<cdot>c"
  "c\<cdot>a \<ls> c\<cdot>b"
  using ordGroupAssum prems IsAnOrdGroup_def
    OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 group0.group0_2_L19
  by auto;

(*lemma (in group3) group_strict_ord_transl_inv: 
  assumes "a\<lsq>b"  "a\<noteq>b" and "c\<in>G"
  shows "a\<cdot>c \<lsq> b\<cdot>c"  "a\<cdot>c \<noteq> b\<cdot>c"
  using ordGroupAssum prems IsAnOrdGroup_def
    OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 group0.group0_2_L19
  by auto;*)

text{*If the group order is total, then the group is ordered linearly.*}

lemma (in group3) group_ord_total_is_lin:
  assumes "r {is total on} G"
  shows "IsLinOrder(G,r)"
  using prems ordGroupAssum IsAnOrdGroup_def Order_ZF_1_L3
  by simp;

text{*For linearly ordered groups elements in the nonnegative set are
  greater than those in the complement.*}

lemma (in group3) OrderedGroup_ZF_1_L4B:
  assumes "r {is total on} G" 
  and "a\<in>G\<^sup>+" and "b \<in> G-G\<^sup>+"
  shows "b\<lsq>a"
proof -;
  from prems have "b\<lsq>\<one>" "\<one>\<lsq>a"
    using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L2B by auto;
  thus ?thesis by (rule Group_order_transitive)
qed;

text{*If $a\leq 1$ and $a\neq 1$, then $a \in G\setminus G^{+}$. *}

lemma (in group3) OrderedGroup_ZF_1_L4C:
  assumes A1: "a\<lsq>\<one>" and A2: "a\<noteq>\<one>"
  shows "a \<in> G-G\<^sup>+"
proof (rule ccontr);
  assume "a \<notin> G-G\<^sup>+" 
  with ordGroupAssum A1 A2 show False 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L2
     OrderedGroup_ZF_1_L4 IsAnOrdGroup_def IsPartOrder_def antisym_def
    by auto;
qed;

text{*An element smaller than an element in $G\setminus G^+$ is in 
  $G\setminus G^+$.*}

lemma (in group3) OrderedGroup_ZF_1_L4D:
  assumes A1: "a\<in>G-G\<^sup>+" and A2: "b\<lsq>a"
  shows "b\<in>G-G\<^sup>+"
proof (rule ccontr);
  assume "b \<notin> G - G\<^sup>+"
  with A2 have "\<one>\<lsq>b" "b\<lsq>a"
     using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L2 by auto;
  then have "\<one>\<lsq>a" by (rule Group_order_transitive);
  with A1 show False using OrderedGroup_ZF_1_L2 by simp;
qed;

text{*The nonnegative set is contained in the group.*}
lemma (in group3) OrderedGroup_ZF_1_L4E: shows "G\<^sup>+ \<subseteq> G"
  using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L4 by auto;

text{*Taking the inverse on both sides reverses the inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L5:
  assumes A1: "a\<lsq>b" shows "b\<inverse>\<lsq>a\<inverse>"
proof -;
  from A1 have T1: "a\<in>G" "b\<in>G" "a\<inverse>\<in>G" "b\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.inverse_in_group by auto;
  with A1 ordGroupAssum have "a\<cdot>a\<inverse>\<lsq>b\<cdot>a\<inverse>" using IsAnOrdGroup_def
    by simp;
  with T1 ordGroupAssum have "b\<inverse>\<cdot>\<one>\<lsq>b\<inverse>\<cdot>(b\<cdot>a\<inverse>)"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L6 IsAnOrdGroup_def
    by simp;
  with T1 show ?thesis using
    OrderedGroup_ZF_1_L1 group0.group0_2_L2 group0.group_oper_assoc
    group0.group0_2_L6 by simp;
qed;

text{*If an element is smaller that the unit, then its inverse is greater.*}

lemma (in group3) OrderedGroup_ZF_1_L5A: 
  assumes A1: "a\<lsq>\<one>" shows "\<one>\<lsq>a\<inverse>"
proof -
  from A1 have "\<one>\<inverse>\<lsq>a\<inverse>" using OrderedGroup_ZF_1_L5
    by simp;
  then show ?thesis using OrderedGroup_ZF_1_L1 group0.group_inv_of_one 
    by simp;
qed;

text{*If an the inverse of an element is greater that the unit, 
  then the element is smaller.*}

lemma (in group3) OrderedGroup_ZF_1_L5AA: 
  assumes A1: "a\<in>G" and A2: "\<one>\<lsq>a\<inverse>"  
  shows "a\<lsq>\<one>"
proof -
  from A2 have "(a\<inverse>)\<inverse>\<lsq>\<one>\<inverse>" using OrderedGroup_ZF_1_L5
    by simp;
  with A1 show "a\<lsq>\<one>"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv group0.group_inv_of_one
    by simp;
qed;

text{*If an element is nonnegative, then the inverse is 
  not greater that the unit.
  Also shows that nonnegative elements cannot be negative*}

lemma (in group3) OrderedGroup_ZF_1_L5AB:
  assumes A1: "\<one>\<lsq>a" shows "a\<inverse>\<lsq>\<one>" and "\<not>(a\<lsq>\<one> \<and> a\<noteq>\<one>)"
proof -
  from A1 have "a\<inverse>\<lsq>\<one>\<inverse>"
    using OrderedGroup_ZF_1_L5 by simp;
  then show "a\<inverse>\<lsq>\<one>" using OrderedGroup_ZF_1_L1 group0.group_inv_of_one
    by simp;
  { assume "a\<lsq>\<one>" and "a\<noteq>\<one>"
    with A1 have False using group_order_antisym
      by blast;
  } then show "\<not>(a\<lsq>\<one> \<and> a\<noteq>\<one>)" by auto;
qed;

text{*If two elements are greater or equal than the unit, then the inverse
  of one is not greater than the other.*}

lemma (in group3) OrderedGroup_ZF_1_L5AC:
  assumes A1: "\<one>\<lsq>a"  "\<one>\<lsq>b"
  shows "a\<inverse> \<lsq> b"
proof -
  from A1 have "a\<inverse>\<lsq>\<one>"  "\<one>\<lsq>b"
    using OrderedGroup_ZF_1_L5AB by auto
  then show "a\<inverse> \<lsq> b" by (rule Group_order_transitive)
qed;

(* we probably should put the stuff about inequalities in a separate section
text{*This section developes some simple tools to deal with inequalities.*}*)


text{*Taking negative on both sides reverses the inequality, case with
  an inverse on one side.*}

lemma (in group3) OrderedGroup_ZF_1_L5AD:
  assumes A1: "b \<in> G" and A2: "a\<lsq>b\<inverse>"
  shows "b \<lsq> a\<inverse>"
proof -
  from A2 have "(b\<inverse>)\<inverse> \<lsq> a\<inverse>"
    using OrderedGroup_ZF_1_L5 by simp
  with A1 show "b \<lsq> a\<inverse>"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
    by simp
qed;

text{*We can cancel the same element on both sides of an inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L5AE:
  assumes A1: "a\<in>G"  "b\<in>G"  "c\<in>G" and A2: "a\<cdot>b \<lsq> a\<cdot>c"
  shows "b\<lsq>c"
proof -
  from ordGroupAssum A1 A2 have "a\<inverse>\<cdot>(a\<cdot>b) \<lsq> a\<inverse>\<cdot>(a\<cdot>c)"
    using OrderedGroup_ZF_1_L1 group0.inverse_in_group
      IsAnOrdGroup_def by simp;
  with A1 show "b\<lsq>c" 
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp;
qed;

text{*We can cancel the same element on both sides of an inequality,
  a version with an inverse on both sides.*}

lemma (in group3) OrderedGroup_ZF_1_L5AF:
  assumes A1: "a\<in>G"  "b\<in>G"  "c\<in>G" and A2: "a\<cdot>b\<inverse> \<lsq> a\<cdot>c\<inverse>"
  shows "c\<lsq>b"
proof -
  from A1 A2 have "(c\<inverse>)\<inverse> \<lsq> (b\<inverse>)\<inverse>"
     using OrderedGroup_ZF_1_L1 group0.inverse_in_group 
      OrderedGroup_ZF_1_L5AE OrderedGroup_ZF_1_L5 by simp;
  with A1 show "c\<lsq>b" 
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv by simp;
qed;

text{*Taking negative on both sides reverses the inequality, another case with
  an inverse on one side.*}

lemma (in group3) OrderedGroup_ZF_1_L5AG:
  assumes A1: "a \<in> G" and A2: "a\<inverse>\<lsq>b"
  shows "b\<inverse> \<lsq> a"
proof -
  from A2 have "b\<inverse> \<lsq> (a\<inverse>)\<inverse>"
    using OrderedGroup_ZF_1_L5 by simp
  with A1 show "b\<inverse> \<lsq> a"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
    by simp
qed;
  
text{*We can multiply the sides of two inequalities.*}

lemma (in group3) OrderedGroup_ZF_1_L5B:
  assumes A1: "a\<lsq>b" and A2: "c\<lsq>d"
  shows "a\<cdot>c \<lsq> b\<cdot>d"
proof -;
  from A1 A2 have "c\<in>G" "b\<in>G" using OrderedGroup_ZF_1_L4 by auto;
  with A1 A2 ordGroupAssum have "a\<cdot>c\<lsq> b\<cdot>c" "b\<cdot>c\<lsq>b\<cdot>d"
    using IsAnOrdGroup_def by auto;
  then show "a\<cdot>c \<lsq> b\<cdot>d" by (rule Group_order_transitive);
qed;

text{*We can replace first of the factors on one side of an inequality 
  with a greater one.*}

lemma (in group3) OrderedGroup_ZF_1_L5C: 
  assumes A1: "c\<in>G" and A2: "a\<lsq>b\<cdot>c" and A3: "b\<lsq>b\<^isub>1" 
  shows "a\<lsq>b\<^isub>1\<cdot>c"
proof -
  from A1 A3 have "b\<cdot>c \<lsq> b\<^isub>1\<cdot>c"
    using OrderedGroup_ZF_1_L3 OrderedGroup_ZF_1_L5B by simp;
  with A2 show "a\<lsq>b\<^isub>1\<cdot>c" by (rule Group_order_transitive);
qed;

text{*We can replace second of the factors on one side of an inequality 
  with a greater one.*}

lemma (in group3) OrderedGroup_ZF_1_L5D: 
  assumes A1: "b\<in>G" and A2: "a \<lsq> b\<cdot>c" and A3: "c\<lsq>b\<^isub>1" 
  shows "a \<lsq> b\<cdot>b\<^isub>1"
proof -
  from A1 A3 have "b\<cdot>c \<lsq> b\<cdot>b\<^isub>1"
    using OrderedGroup_ZF_1_L3 OrderedGroup_ZF_1_L5B by auto;
  with A2 show "a\<lsq>b\<cdot>b\<^isub>1" by (rule Group_order_transitive);
qed;

text{*We can replace factors on one side of an inequality 
  with greater ones.*}

lemma (in group3) OrderedGroup_ZF_1_L5E: 
  assumes A1: "a \<lsq> b\<cdot>c" and A2: "b\<lsq>b\<^isub>1"  "c\<lsq>c\<^isub>1"  
  shows "a \<lsq> b\<^isub>1\<cdot>c\<^isub>1"
proof -
  from A2 have "b\<cdot>c \<lsq> b\<^isub>1\<cdot>c\<^isub>1" using OrderedGroup_ZF_1_L5B 
    by simp;
  with A1 show "a\<lsq>b\<^isub>1\<cdot>c\<^isub>1" by (rule Group_order_transitive);
qed;

text{*We don't decrease an element of the group by multiplying by one that is
  nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L5F:
  assumes A1: "\<one>\<lsq>a" and A2: "b\<in>G"
  shows "b\<lsq>a\<cdot>b"  "b\<lsq>b\<cdot>a" 
proof -
  from ordGroupAssum A1 A2 have  
    "\<one>\<cdot>b\<lsq>a\<cdot>b"  "b\<cdot>\<one>\<lsq>b\<cdot>a"
    using IsAnOrdGroup_def by auto;
  with A2 show "b\<lsq>a\<cdot>b"  "b\<lsq>b\<cdot>a"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by auto;
qed;

text{*We can multiply the right hand side of an inequality by a nonnegative
  element.*}

lemma (in group3) OrderedGroup_ZF_1_L5G: assumes A1: "a\<lsq>b"
  and A2: "\<one>\<lsq>c" shows "a\<lsq>b\<cdot>c"  "a\<lsq>c\<cdot>b" 
proof -
  from A1 A2 have I: "b\<lsq>b\<cdot>c"  and II: "b\<lsq>c\<cdot>b"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L5F by auto
  from A1 I show "a\<lsq>b\<cdot>c" by (rule Group_order_transitive);
  from A1 II show "a\<lsq>c\<cdot>b" by (rule Group_order_transitive);
qed;

text{*We can put two elements on the other side of inequality, 
  changing their sign.*}

lemma (in group3) OrderedGroup_ZF_1_L5H: 
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "a\<cdot>b\<inverse> \<lsq> c"
  shows 
  "a \<lsq> c\<cdot>b"
  "c\<inverse>\<cdot>a \<lsq> b"
proof -
  from A2 have T: "c\<in>G"  "c\<inverse> \<in> G"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.inverse_in_group by auto;
  from ordGroupAssum A1 A2 have "a\<cdot>b\<inverse>\<cdot>b \<lsq> c\<cdot>b"
    using IsAnOrdGroup_def by simp;
  with A1 show "a \<lsq> c\<cdot>b" 
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp;
  with ordGroupAssum A2 T have "c\<inverse>\<cdot>a \<lsq> c\<inverse>\<cdot>(c\<cdot>b)"
    using IsAnOrdGroup_def by simp;
  with A1 T show "c\<inverse>\<cdot>a \<lsq> b"  
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp;
qed;

text{*We can multiply the sides of one inequality by inverse of another.*}

lemma (in group3) OrderedGroup_ZF_1_L5I: 
  assumes "a\<lsq>b" and "c\<lsq>d"
  shows "a\<cdot>d\<inverse> \<lsq> b\<cdot>c\<inverse>"
  using prems OrderedGroup_ZF_1_L5 OrderedGroup_ZF_1_L5B
  by simp;

text{*We can put an element on the other side of an inequality
  changing its sign, version with the inverse.*}

lemma (in group3) OrderedGroup_ZF_1_L5J:
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "c \<lsq> a\<cdot>b\<inverse>"
  shows "c\<cdot>b \<lsq> a"
proof -
  from ordGroupAssum A1 A2 have "c\<cdot>b \<lsq> a\<cdot>b\<inverse>\<cdot>b"
    using IsAnOrdGroup_def by simp;
  with A1 show "c\<cdot>b \<lsq> a" 
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp;
qed;

text{*We can put an element on the other side of an inequality
  changing its sign, version with the inverse.*}

lemma (in group3) OrderedGroup_ZF_1_L5JA:
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "c \<lsq> a\<inverse>\<cdot>b"
  shows "a\<cdot>c\<lsq> b"
proof -
  from ordGroupAssum A1 A2 have "a\<cdot>c \<lsq> a\<cdot>(a\<inverse>\<cdot>b)"
    using IsAnOrdGroup_def by simp;
  with A1 show "a\<cdot>c\<lsq> b" 
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp;
qed;
    

text{*A special case of @{text "OrderedGroup_ZF_1_L5J"} where $c=1$.*}

corollary (in group3) OrderedGroup_ZF_1_L5K: 
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "\<one> \<lsq> a\<cdot>b\<inverse>"
  shows "b \<lsq> a"
proof -
  from A1 A2 have "\<one>\<cdot>b \<lsq> a"
    using OrderedGroup_ZF_1_L5J by simp;
  with A1 show "b \<lsq> a"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by simp;
qed;

text{*A special case of @{text "OrderedGroup_ZF_1_L5JA"} where $c=1$.*}

corollary (in group3) OrderedGroup_ZF_1_L5KA: 
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "\<one> \<lsq> a\<inverse>\<cdot>b"
  shows "a \<lsq> b"
proof -
  from A1 A2 have "a\<cdot>\<one> \<lsq> b"
    using OrderedGroup_ZF_1_L5JA by simp;
  with A1 show "a \<lsq> b"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by simp;
qed;

text{*If the order is total, the elements that do not belong
  to the positive set are negative. We also show here that the group inverse
  of an element that does not belong to the nonnegative set does belong to the
  nonnegative set.*}

lemma (in group3) OrderedGroup_ZF_1_L6: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G-G\<^sup>+"
  shows "a\<lsq>\<one>"  "a\<inverse> \<in> G\<^sup>+"  "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+" 
proof -; 
  from A2 have T1: "a\<in>G" "a\<notin>G\<^sup>+" "\<one>\<in>G" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
  with A1 show "a\<lsq>\<one>" using OrderedGroup_ZF_1_L2 IsTotal_def
    by auto;
  then show "a\<inverse> \<in> G\<^sup>+" using OrderedGroup_ZF_1_L5A OrderedGroup_ZF_1_L2
    by simp;
  with A2 show "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+"
    using restrict by simp;
qed;

text{*If a property is invariant with respect to taking the inverse
  and it is true on the nonnegative set, than it is true on the whole
  group.*}

lemma (in group3)  OrderedGroup_ZF_1_L7:
  assumes A1: "r {is total on} G"
  and A2: "\<forall>a\<in>G\<^sup>+.\<forall>b\<in>G\<^sup>+. Q(a,b)"
  and A3: "\<forall>a\<in>G.\<forall>b\<in>G. Q(a,b)\<longrightarrow>Q(a\<inverse>,b)"
  and A4: "\<forall>a\<in>G.\<forall>b\<in>G. Q(a,b)\<longrightarrow>Q(a,b\<inverse>)"
  and A5: "a\<in>G" "b\<in>G"
  shows "Q(a,b)"
proof (cases "a\<in>G\<^sup>+");
  assume A6: "a\<in>G\<^sup>+" show "Q(a,b)"
  proof (cases "b\<in>G\<^sup>+")
    assume "b\<in>G\<^sup>+" 
    with A6 A2 show "Q(a,b)" by simp;
  next assume "b\<notin>G\<^sup>+"
    with A1 A2 A4 A5 A6 have "Q(a,(b\<inverse>)\<inverse>)"  
      using OrderedGroup_ZF_1_L6 OrderedGroup_ZF_1_L1 group0.inverse_in_group
      by simp;
    with A5 show "Q(a,b)" using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
      by simp;
  qed;
next assume "a\<notin>G\<^sup>+"
  with A1 A5 have T1: "a\<inverse> \<in> G\<^sup>+" using OrderedGroup_ZF_1_L6 by simp;
  show "Q(a,b)"
  proof (cases "b\<in>G\<^sup>+")
    assume "b\<in>G\<^sup>+"
    with A2 A3 A5 T1 have "Q((a\<inverse>)\<inverse>,b)" 
      using OrderedGroup_ZF_1_L1 group0.inverse_in_group by simp;
    with A5 show "Q(a,b)" using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
      by simp;
  next assume "b\<notin>G\<^sup>+"
    with A1 A2 A3 A4 A5 T1 have "Q((a\<inverse>)\<inverse>,(b\<inverse>)\<inverse>)"
       using OrderedGroup_ZF_1_L6 OrderedGroup_ZF_1_L1 group0.inverse_in_group
       by simp;
    with A5 show "Q(a,b)" using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
      by simp;
  qed;
qed;

text{*A lemma about splitting the ordered group "plane" into 6 subsets. Useful
  for proofs by cases.*}

lemma (in group3) OrdGroup_6cases: assumes A1: "r {is total on} G" 
  and A2:  "a\<in>G"  "b\<in>G"
  shows 
  "\<one>\<lsq>a \<and> \<one>\<lsq>b  \<or>  a\<lsq>\<one> \<and> b\<lsq>\<one>  \<or>  
  a\<lsq>\<one> \<and> \<one>\<lsq>b \<and> \<one> \<lsq> a\<cdot>b  \<or> a\<lsq>\<one> \<and> \<one>\<lsq>b \<and> a\<cdot>b \<lsq> \<one>  \<or>  
  \<one>\<lsq>a \<and> b\<lsq>\<one> \<and> \<one> \<lsq> a\<cdot>b  \<or>  \<one>\<lsq>a \<and> b\<lsq>\<one> \<and> a\<cdot>b \<lsq> \<one>"
proof -
  from A1 A2 have 
    "\<one>\<lsq>a \<or> a\<lsq>\<one>"   
    "\<one>\<lsq>b \<or> b\<lsq>\<one>"
    "\<one> \<lsq> a\<cdot>b \<or> a\<cdot>b \<lsq> \<one>"
    using OrderedGroup_ZF_1_L1 group0.group_op_closed group0.group0_2_L2
      IsTotal_def by auto;
  then show ?thesis by auto;
qed;

text{*The next lemma shows what happens when one element of a totally 
  ordered group is not greater or equal than another.*}

lemma (in group3) OrderedGroup_ZF_1_L8:
  assumes A1: "r {is total on} G"
  and A2: "a\<in>G"  "b\<in>G"
  and A3: "\<not>(a\<lsq>b)"
  shows "b \<lsq> a"  "a\<inverse> \<lsq> b\<inverse>"  "a\<noteq>b"  "b\<ls>a"
 
proof -
  from A1 A2 A3 show I: "b \<lsq> a" using IsTotal_def
    by auto;
  then show "a\<inverse> \<lsq> b\<inverse>" using OrderedGroup_ZF_1_L5 by simp;
  from A2 have "a \<lsq> a" using OrderedGroup_ZF_1_L3 by simp;
  with I A3 show "a\<noteq>b"  "b \<ls> a" by auto;
qed;

text{*If one element is greater or equal and not equal to another,
  then it is not smaller or equal.*}

lemma (in group3) OrderedGroup_ZF_1_L8AA: 
  assumes A1: "a\<lsq>b" and A2: "a\<noteq>b"
  shows "\<not>(b\<lsq>a)"
proof -
  { note A1
    moreover assume "b\<lsq>a"
    ultimately have "a=b" by (rule group_order_antisym)
    with A2 have False by simp;
  } thus "\<not>(b\<lsq>a)" by auto;
qed;

text{*A special case of @{text "OrderedGroup_ZF_1_L8"} when one of 
  the elements is the unit.*}

corollary (in group3) OrderedGroup_ZF_1_L8A:
  assumes A1: "r {is total on} G"
  and A2: "a\<in>G" and A3: "\<not>(\<one>\<lsq>a)"
  shows "\<one> \<lsq> a\<inverse>"  "\<one>\<noteq>a"  "a\<lsq>\<one>"
proof -
  from A1 A2 A3 have I:
    "r {is total on} G"
    "\<one>\<in>G"  "a\<in>G"
     "\<not>(\<one>\<lsq>a)"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by auto;
  then have "\<one>\<inverse> \<lsq> a\<inverse>"
    by (rule OrderedGroup_ZF_1_L8);
  then show "\<one> \<lsq> a\<inverse>"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_one by simp;
  from I show "\<one>\<noteq>a" by (rule OrderedGroup_ZF_1_L8);
  from A1 I show "a\<lsq>\<one>" using IsTotal_def
    by auto;
qed;

text{*A negative element can not be nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L8B:
  assumes A1: "a\<lsq>\<one>"  and A2: "a\<noteq>\<one>" shows "\<not>(\<one>\<lsq>a)"
proof -
  { assume  "\<one>\<lsq>a" 
    with A1 have "a=\<one>" using group_order_antisym
      by auto;
    with A2 have False by simp;
  } thus ?thesis by auto;
qed;

text{*An element is greater or equal than another iff the difference is 
  nonpositive.*}

lemma (in group3) OrderedGroup_ZF_1_L9:
  assumes A1: "a\<in>G"  "b\<in>G"
  shows "a\<lsq>b \<longleftrightarrow> a\<cdot>b\<inverse> \<lsq> \<one>"
proof;
  assume "a \<lsq> b"
  with ordGroupAssum A1 have "a\<cdot>b\<inverse> \<lsq> b\<cdot>b\<inverse>"
    using OrderedGroup_ZF_1_L1 group0.inverse_in_group
    IsAnOrdGroup_def by simp;
  with A1 show "a\<cdot>b\<inverse> \<lsq> \<one>" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L6
    by simp;
next assume A2: "a\<cdot>b\<inverse> \<lsq> \<one>"
  with ordGroupAssum A1 have "a\<cdot>b\<inverse>\<cdot>b \<lsq> \<one>\<cdot>b"
    using IsAnOrdGroup_def by simp;
  with A1 show "a \<lsq> b"
    using OrderedGroup_ZF_1_L1 
      group0.inv_cancel_two group0.group0_2_L2 
    by simp;
qed;

text{*We can move an element to the other side of an inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L9A:
  assumes A1: "a\<in>G"  "b\<in>G"  "c\<in>G"
  shows "a\<cdot>b \<lsq> c  \<longleftrightarrow> a \<lsq> c\<cdot>b\<inverse>"
proof
  assume "a\<cdot>b \<lsq> c"
  with ordGroupAssum A1 have "a\<cdot>b\<cdot>b\<inverse> \<lsq> c\<cdot>b\<inverse>"
    using OrderedGroup_ZF_1_L1 group0.inverse_in_group IsAnOrdGroup_def
    by simp;
  with A1 show "a \<lsq> c\<cdot>b\<inverse>"
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two by simp;
next assume "a \<lsq> c\<cdot>b\<inverse>"
  with ordGroupAssum A1 have "a\<cdot>b \<lsq> c\<cdot>b\<inverse>\<cdot>b"
    using OrderedGroup_ZF_1_L1 group0.inverse_in_group IsAnOrdGroup_def
    by simp;
  with A1 show "a\<cdot>b \<lsq> c"
     using OrderedGroup_ZF_1_L1 group0.inv_cancel_two by simp;
qed;

text{*A one side version of the previous lemma with weaker assuptions.*}

lemma (in group3) OrderedGroup_ZF_1_L9B:
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "a\<cdot>b\<inverse> \<lsq> c"
  shows "a \<lsq> c\<cdot>b"
proof - 
  from A1 A2 have "a\<in>G"  "b\<inverse>\<in>G"  "c\<in>G"
    using OrderedGroup_ZF_1_L1 group0.inverse_in_group 
      OrderedGroup_ZF_1_L4 by auto;
  with A1 A2 show "a \<lsq> c\<cdot>b"
    using OrderedGroup_ZF_1_L9A OrderedGroup_ZF_1_L1 
      group0.group_inv_of_inv by simp;
qed;

text{*We can put en element on the other side of inequality, 
  changing its sign.*}

lemma (in group3) OrderedGroup_ZF_1_L9C:
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "c\<lsq>a\<cdot>b"
  shows 
  "c\<cdot>b\<inverse> \<lsq> a"
  "a\<inverse>\<cdot>c \<lsq> b"
proof -
  from ordGroupAssum A1 A2 have
    "c\<cdot>b\<inverse> \<lsq> a\<cdot>b\<cdot>b\<inverse>"
    "a\<inverse>\<cdot>c \<lsq> a\<inverse>\<cdot>(a\<cdot>b)"
    using OrderedGroup_ZF_1_L1 group0.inverse_in_group IsAnOrdGroup_def
    by auto;
  with A1 show  
    "c\<cdot>b\<inverse> \<lsq> a"
    "a\<inverse>\<cdot>c \<lsq> b"
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by auto;
qed;

text{*If an element is greater or equal than another then the difference is 
  nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L9D: assumes A1: "a\<lsq>b"
  shows "\<one> \<lsq> b\<cdot>a\<inverse>"
proof -
  from A1 have T: "a\<in>G"  "b\<in>G"   "a\<inverse> \<in> G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.inverse_in_group by auto
  with ordGroupAssum A1 have "a\<cdot>a\<inverse> \<lsq> b\<cdot>a\<inverse>"
    using IsAnOrdGroup_def by simp
  with T show "\<one> \<lsq> b\<cdot>a\<inverse>" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L6
    by simp
qed;

text{*If an element is greater than another then the difference is 
  positive.*}

lemma (in group3) OrderedGroup_ZF_1_L9E: 
  assumes A1: "a\<lsq>b"  "a\<noteq>b"
  shows "\<one> \<lsq> b\<cdot>a\<inverse>"  "\<one> \<noteq> b\<cdot>a\<inverse>"  "b\<cdot>a\<inverse> \<in> G\<^isub>+"
proof -
  from A1 have T: "a\<in>G"  "b\<in>G" using OrderedGroup_ZF_1_L4
    by auto
  from A1 show I: "\<one> \<lsq> b\<cdot>a\<inverse>" using OrderedGroup_ZF_1_L9D
    by simp
  { assume "b\<cdot>a\<inverse> = \<one>" 
    with T have "a=b"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L11A
      by auto;
    with A1 have False by simp 
  } then show "\<one> \<noteq> b\<cdot>a\<inverse>" by auto;
  then have "b\<cdot>a\<inverse> \<noteq> \<one>" by auto;
  with I show "b\<cdot>a\<inverse> \<in> G\<^isub>+" using OrderedGroup_ZF_1_L2A
    by simp;
qed;

text{*If the difference is nonnegative, then $a\leq b$. *}

lemma (in group3) OrderedGroup_ZF_1_L9F: 
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "\<one> \<lsq> b\<cdot>a\<inverse>"
  shows "a\<lsq>b"
proof -
  from A1 A2 have "\<one>\<cdot>a \<lsq> b"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L9A
    by simp;
  with A1 show "a\<lsq>b" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by simp;
qed;


text{*If we increase the middle term in a product, the whole product 
  increases.*}

lemma (in group3) OrderedGroup_ZF_1_L10: 
  assumes "a\<in>G"  "b\<in>G" and "c\<lsq>d"
  shows "a\<cdot>c\<cdot>b \<lsq> a\<cdot>d\<cdot>b"
  using ordGroupAssum prems IsAnOrdGroup_def by simp;

text{*A product of (strictly) positive elements is not the unit.*}

lemma (in group3) OrderedGroup_ZF_1_L11: 
  assumes A1: "\<one>\<lsq>a"  "\<one>\<lsq>b"
  and A2: "\<one> \<noteq> a"  "\<one> \<noteq> b"
  shows "\<one> \<noteq> a\<cdot>b"
proof -
  from A1 have T1: "a\<in>G"  "b\<in>G"
    using OrderedGroup_ZF_1_L4 by auto;
  { assume "\<one> = a\<cdot>b"
    with A1 T1 have "a\<lsq>\<one>"  "\<one>\<lsq>a"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L9 OrderedGroup_ZF_1_L5AA 
      by auto;
    then have "a = \<one>" by (rule group_order_antisym);
    with A2 have False by simp;
  } then show "\<one> \<noteq> a\<cdot>b" by auto;
qed;

text{*A product of nonnegative elements is nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L12:
  assumes A1: "\<one> \<lsq> a"  "\<one> \<lsq> b"
  shows "\<one> \<lsq> a\<cdot>b"
proof -
  from A1 have "\<one>\<cdot>\<one> \<lsq> a\<cdot>b"
    using  OrderedGroup_ZF_1_L5B by simp;
  then show "\<one> \<lsq> a\<cdot>b" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2 
    by simp;
qed;

text{*If $a$ is not greater than $b$, then $1$ is not greater than
  $b\cdot a^{-1}$.*}

lemma (in group3) OrderedGroup_ZF_1_L12A:
  assumes A1: "a\<lsq>b" shows "\<one> \<lsq> b\<cdot>a\<inverse>"
proof -
  from A1 have T: "\<one> \<in> G"  "a\<in>G"  "b\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by auto;
  with A1 have "\<one>\<cdot>a \<lsq> b" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by simp;
  with T show "\<one> \<lsq> b\<cdot>a\<inverse>" using OrderedGroup_ZF_1_L9A
    by simp;
qed;
  
text{*We can move an element to the other side of a strict inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L12B:  
  assumes A1: "a\<in>G"  "b\<in>G" and  A2: "a\<cdot>b\<inverse> \<ls> c"
  shows "a \<ls> c\<cdot>b"
proof -
  from A1 A2 have "a\<cdot>b\<inverse>\<cdot>b \<ls> c\<cdot>b"
    using group_strict_ord_transl_inv by auto;
  moreover from A1 have "a\<cdot>b\<inverse>\<cdot>b = a"
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp
  ultimately show "a \<ls> c\<cdot>b"
    by auto
qed;

(*lemma (in group3) OrderedGroup_ZF_1_L12B:  
  assumes A1: "a\<in>G"  "b\<in>G" and  A2: "a\<cdot>b\<inverse> \<lsq> c"  "a\<cdot>b\<inverse> \<noteq> c"
  shows "a \<lsq> c\<cdot>b"  "a \<noteq> c\<cdot>b"
proof -
  from A1 A2 have "a\<cdot>b\<inverse>\<cdot>b \<lsq> c\<cdot>b"  "a\<cdot>b\<inverse>\<cdot>b \<noteq> c\<cdot>b"
    using group_strict_ord_transl_inv by auto
  moreover from A1 have "a\<cdot>b\<inverse>\<cdot>b = a"
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp
  ultimately show "a \<lsq> c\<cdot>b"  "a \<noteq> c\<cdot>b"
    by auto
qed;*)

text{*We can multiply the sides of two inequalities,
  first of them strict and we get a strict inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L12C:
  assumes A1: "a\<ls>b" and A2: "c\<lsq>d"
  shows "a\<cdot>c \<ls> b\<cdot>d"
proof -
  from A1 A2 have T: "a\<in>G"  "b\<in>G"  "c\<in>G"  "d\<in>G"
    using OrderedGroup_ZF_1_L4 by auto;
  with ordGroupAssum A2 have "a\<cdot>c \<lsq> a\<cdot>d"
    using IsAnOrdGroup_def by simp;
  moreover from A1 T have "a\<cdot>d \<ls> b\<cdot>d"
    using group_strict_ord_transl_inv by simp;
  ultimately show "a\<cdot>c \<ls> b\<cdot>d"
    by (rule group_strict_ord_transit);
qed;
  
text{*We can multiply the sides of two inequalities,
  second of them strict and we get a strict inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L12D:
  assumes A1: "a\<lsq>b" and A2: "c\<ls>d"
  shows "a\<cdot>c \<ls> b\<cdot>d"
proof -
  from A1 A2 have T: "a\<in>G"  "b\<in>G"  "c\<in>G"  "d\<in>G"
    using OrderedGroup_ZF_1_L4 by auto;
  with A2 have "a\<cdot>c \<ls> a\<cdot>d" 
    using group_strict_ord_transl_inv by simp;
  moreover from ordGroupAssum A1 T have "a\<cdot>d \<lsq> b\<cdot>d"
     using IsAnOrdGroup_def by simp;
   ultimately show "a\<cdot>c \<ls> b\<cdot>d"
     by (rule OrderedGroup_ZF_1_L4A);
qed;

section{*The set of positive elements*}

text{*In this section we study @{text "G\<^isub>+"} - the set of elements that are 
  (strictly) greater than the unit. The most important result is that every
  linearly ordered group can decomposed into $\{1\}$, @{text "G\<^isub>+"} and the 
  set of those elements $a\in G$ such that $a^{-1}\in$@{text "G\<^isub>+"}. 
  Another property of linearly ordered groups that we prove here is that 
  if @{text "G\<^isub>+"}$\neq \emptyset$, then it is infinite. This allows to show 
  that nontrivial linearly ordered groups are infinite. *}

text{*The positive set is closed under the group operation.*}

lemma (in group3) OrderedGroup_ZF_1_L13: "G\<^isub>+ {is closed under} P"
proof -
  { fix a b assume "a\<in>G\<^isub>+"  "b\<in>G\<^isub>+"
    then have T1: "\<one> \<lsq> a\<cdot>b"   and "\<one> \<noteq> a\<cdot>b"
      using PositiveSet_def OrderedGroup_ZF_1_L11 OrderedGroup_ZF_1_L12
      by auto;
    moreover from T1 have "a\<cdot>b \<in> G"
      using OrderedGroup_ZF_1_L4 by simp;
    ultimately have "a\<cdot>b \<in> G\<^isub>+" using PositiveSet_def by simp;
  } then show "G\<^isub>+ {is closed under} P" using IsOpClosed_def
    by simp;
qed;

text{*For totally ordered groups every nonunit element is positive or its 
  inverse is positive.*}

lemma (in group3) OrderedGroup_ZF_1_L14:
  assumes A1: "r {is total on} G" and A2: "a\<in>G" 
  shows "a=\<one> \<or> a\<in>G\<^isub>+ \<or> a\<inverse>\<in>G\<^isub>+"
proof -
  { assume A3: "a\<noteq>\<one>"
    moreover from A1 A2 have "a\<lsq>\<one> \<or> \<one>\<lsq>a"
      using IsTotal_def OrderedGroup_ZF_1_L1 group0.group0_2_L2
      by simp;
    moreover from A3 A2 have T1: "a\<inverse> \<noteq> \<one>"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L8B
      by simp
    ultimately have "a\<inverse>\<in>G\<^isub>+ \<or> a\<in>G\<^isub>+"
      using OrderedGroup_ZF_1_L5A OrderedGroup_ZF_1_L2A
      by auto;
  } thus "a=\<one> \<or> a\<in>G\<^isub>+ \<or> a\<inverse>\<in>G\<^isub>+" by auto;
qed;

text{*If an element belongs to the positive set, then it is not the unit
  and its inverse does not belong to the positive set.*}

lemma (in group3) OrderedGroup_ZF_1_L15:
   assumes A1: "a\<in>G\<^isub>+"  shows "a\<noteq>\<one>"  "a\<inverse>\<notin>G\<^isub>+"
proof -
  from A1 show T1: "a\<noteq>\<one>" using PositiveSet_def by auto;
  { assume "a\<inverse> \<in> G\<^isub>+" 
    with A1 have "a\<lsq>\<one>"  "\<one>\<lsq>a"
      using OrderedGroup_ZF_1_L5AA PositiveSet_def by auto;
    then have "a=\<one>" by (rule group_order_antisym);
    with T1 have False by simp;
  } then show "a\<inverse>\<notin>G\<^isub>+" by auto;
qed;

text{*If $a^{-1}$ is positive, then $a$ can not be positive or the unit.*}

lemma (in group3) OrderedGroup_ZF_1_L16:
  assumes A1: "a\<in>G" and A2: "a\<inverse>\<in>G\<^isub>+" shows "a\<noteq>\<one>"  "a\<notin>G\<^isub>+"
proof -
  from A2 have "a\<inverse>\<noteq>\<one>"  "(a\<inverse>)\<inverse> \<notin> G\<^isub>+"
    using OrderedGroup_ZF_1_L15 by auto;
  with A1 show "a\<noteq>\<one>"  "a\<notin>G\<^isub>+"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L8C group0.group_inv_of_inv 
    by auto;
qed;

text{*For linearly ordered groups each element is either the unit, 
  positive or its inverse is positive.*}

lemma (in group3) OrdGroup_decomp: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G" 
  shows "Exactly_1_of_3_holds (a=\<one>,a\<in>G\<^isub>+,a\<inverse>\<in>G\<^isub>+)"
proof -
  from A1 A2 have "a=\<one> \<or> a\<in>G\<^isub>+ \<or> a\<inverse>\<in>G\<^isub>+"
    using OrderedGroup_ZF_1_L14 by simp
  moreover from A2 have "a=\<one> \<longrightarrow> (a\<notin>G\<^isub>+ \<and> a\<inverse>\<notin>G\<^isub>+)"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_one
    PositiveSet_def by simp;
  moreover from A2 have "a\<in>G\<^isub>+ \<longrightarrow> (a\<noteq>\<one> \<and> a\<inverse>\<notin>G\<^isub>+)"
    using OrderedGroup_ZF_1_L15 by simp;
  moreover from A2 have "a\<inverse>\<in>G\<^isub>+ \<longrightarrow> (a\<noteq>\<one> \<and> a\<notin>G\<^isub>+)"
    using OrderedGroup_ZF_1_L16 by simp;
  ultimately show "Exactly_1_of_3_holds (a=\<one>,a\<in>G\<^isub>+,a\<inverse>\<in>G\<^isub>+)"
    by (rule Fol1_L5);
qed;

text{*A if $a$ is a nonunit element that is not positive, then $a^{-1}$ is 
  is positive. This is useful for some proofs by cases.*}

lemma (in group3) OrdGroup_cases:
  assumes A1: "r {is total on} G" and A2: "a\<in>G" 
  and A3: "a\<noteq>\<one>"  "a\<notin>G\<^isub>+"
  shows "a\<inverse> \<in> G\<^isub>+"
proof -
  from A1 A2 have "a=\<one> \<or> a\<in>G\<^isub>+ \<or> a\<inverse>\<in>G\<^isub>+"
    using OrderedGroup_ZF_1_L14 by simp;
  with A3 show "a\<inverse> \<in> G\<^isub>+" by auto;
qed;
  
text{*Elements from $G\setminus G_+$ are not greater that the unit.*}

lemma (in group3) OrderedGroup_ZF_1_L17:
  assumes A1: "r {is total on} G" and A2: "a \<in> G-G\<^isub>+"
  shows "a\<lsq>\<one>"
proof (cases "a = \<one>");
  assume "a=\<one>"
  with A2 show "a\<lsq>\<one>" using OrderedGroup_ZF_1_L3 by simp;
next assume "a\<noteq>\<one>"
  with A1 A2 show "a\<lsq>\<one>" 
    using PositiveSet_def OrderedGroup_ZF_1_L8A
    by auto;
qed;

text{*The next lemma allows to split proofs that something holds 
  for all $a\in G$ into cases $a=1$, $a\in G_+$, $-a\in G_+$.*}

lemma (in group3) OrderedGroup_ZF_1_L18: 
  assumes A1: "r {is total on} G" and A2: "b\<in>G"
  and A3: "Q(\<one>)" and A4: "\<forall>a\<in>G\<^isub>+. Q(a)" and A5: "\<forall>a\<in>G\<^isub>+. Q(a\<inverse>)"
  shows "Q(b)"
proof -
  from A1 A2 A3 A4 A5 have "Q(b) \<or> Q((b\<inverse>)\<inverse>)"
    using OrderedGroup_ZF_1_L14 by auto;
  with A2 show "Q(b)" using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
    by simp;
qed;

text{*All elements greater or equal than an element of @{text "G\<^isub>+"}
  belong to @{text "G\<^isub>+"}.*}

lemma (in group3) OrderedGroup_ZF_1_L19:
  assumes A1: "a \<in> G\<^isub>+" and A2: "a\<lsq>b"
  shows "b \<in> G\<^isub>+"
proof -
  from A1 have I: "\<one>\<lsq>a"  and II: "a\<noteq>\<one>"
    using OrderedGroup_ZF_1_L2A by auto;
  from I A2 have "\<one>\<lsq>b" by (rule Group_order_transitive);
  moreover have "b\<noteq>\<one>"
  proof -
    { assume "b=\<one>"
      with I A2 have "\<one>\<lsq>a"  "a\<lsq>\<one>"
	by auto;
      then have "\<one>=a" by (rule group_order_antisym)
      with II have False by simp;
    } then show "b\<noteq>\<one>" by auto;
  qed;
  ultimately show "b \<in> G\<^isub>+" 
    using OrderedGroup_ZF_1_L2A by simp;
qed;

text{*The inverse of an element of @{text "G\<^isub>+"} cannot be in @{text "G\<^isub>+"}.*}

lemma (in group3) OrderedGroup_ZF_1_L20:
  assumes A1: "r {is total on} G" and A2: "a \<in> G\<^isub>+" 
  shows "a\<inverse> \<notin> G\<^isub>+"
proof -
  from A2 have "a\<in>G" using PositiveSet_def
    by simp;
  with A1 have "Exactly_1_of_3_holds (a=\<one>,a\<in>G\<^isub>+,a\<inverse>\<in>G\<^isub>+)"
    using OrdGroup_decomp by simp;
  with A2 show "a\<inverse> \<notin> G\<^isub>+" by (rule Fol1_L7);
qed;

text{*The set of positive elements of a 
  nontrivial linearly ordered group is not empty.*}

lemma (in group3) OrderedGroup_ZF_1_L21:
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}"
  shows "G\<^isub>+ \<noteq> 0"
proof -
  have "\<one> \<in> G" using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by simp;
  with A2 obtain a where "a\<in>G"  "a\<noteq>\<one>" by auto;
  with A1 have "a\<in>G\<^isub>+ \<or> a\<inverse>\<in>G\<^isub>+" 
    using OrderedGroup_ZF_1_L14 by auto;
  then show "G\<^isub>+ \<noteq> 0" by auto;
qed;

text{*If $b\in$@{text "G\<^isub>+"}, then $a < a\cdot b$. 
  Multiplying $a$ by a positive elemnt increases $a$. *}

lemma (in group3) OrderedGroup_ZF_1_L22:
  assumes A1: "a\<in>G"  "b\<in>G\<^isub>+"
  shows "a\<lsq>a\<cdot>b"  "a \<noteq> a\<cdot>b"  "a\<cdot>b \<in> G"
proof -
  from ordGroupAssum A1 have "a\<cdot>\<one> \<lsq> a\<cdot>b"
    using OrderedGroup_ZF_1_L2A IsAnOrdGroup_def
    by simp;
  with A1 show "a\<lsq>a\<cdot>b" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by simp;
  then show "a\<cdot>b \<in> G"
    using OrderedGroup_ZF_1_L4 by simp;
  { from A1 have "a\<in>G"  "b\<in>G"  
      using PositiveSet_def by auto;
    moreover assume "a = a\<cdot>b"
    ultimately have "b = \<one>"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L7
      by simp;
    with A1 have False using PositiveSet_def
      by simp;
  } then show "a \<noteq> a\<cdot>b" by auto;
qed;

text{*If $G$ is a nontrivial linearly ordered hroup, 
  then for every element of $G$ we can find one in @{text "G\<^isub>+"} that is
  greater or equal.*}

lemma (in group3) OrderedGroup_ZF_1_L23: 
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}"
  and A3: "a\<in>G"
  shows "\<exists>b\<in>G\<^isub>+. a\<lsq>b"
proof (cases "a\<in>G\<^isub>+")
  assume A4: "a\<in>G\<^isub>+" then have "a\<lsq>a"
    using PositiveSet_def OrderedGroup_ZF_1_L3
    by simp;
  with A4 show "\<exists>b\<in>G\<^isub>+. a\<lsq>b" by auto;
next assume "a\<notin>G\<^isub>+"
  with A1 A3 have I: "a\<lsq>\<one>" using OrderedGroup_ZF_1_L17
    by simp;
  from A1 A2 obtain b where II: "b\<in>G\<^isub>+" 
    using OrderedGroup_ZF_1_L21 by auto;
  then have "\<one>\<lsq>b" using PositiveSet_def by simp;
  with I have "a\<lsq>b" by (rule Group_order_transitive);
  with II show "\<exists>b\<in>G\<^isub>+. a\<lsq>b" by auto;
qed;

text{*The @{text "G\<^sup>+"} is @{text "G\<^isub>+"} plus the unit.*}
lemma (in group3) OrderedGroup_ZF_1_L24: shows "G\<^sup>+ = G\<^isub>+\<union>{\<one>}"
  using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L2A OrderedGroup_ZF_1_L3A
  by auto;

text{*What is $-G_+$, really?*}

lemma (in group3) OrderedGroup_ZF_1_L25: shows 
  "(\<sm>G\<^isub>+) = {a\<inverse>. a\<in>G\<^isub>+}"
  "(\<sm>G\<^isub>+) \<subseteq> G"
proof -
  from ordGroupAssum have I: "GroupInv(G,P) : G\<rightarrow>G"
    using IsAnOrdGroup_def group0_2_T2 by simp;
  moreover have "G\<^isub>+ \<subseteq> G" using PositiveSet_def by auto;
  ultimately show 
    "(\<sm>G\<^isub>+) = {a\<inverse>. a\<in>G\<^isub>+}"
    "(\<sm>G\<^isub>+) \<subseteq> G"
    using func_imagedef func1_1_L6 by auto;
qed;

text{*If the inverse of $a$ is in @{text "G\<^isub>+"}, then $a$ is in the inverse
  of @{text "G\<^isub>+"}.*}

lemma (in group3) OrderedGroup_ZF_1_L26:
  assumes A1: "a\<in>G" and A2: "a\<inverse> \<in> G\<^isub>+"
  shows "a \<in> (\<sm>G\<^isub>+)"
proof -
  from A1 have "a\<inverse> \<in> G"  "a = (a\<inverse>)\<inverse>" using OrderedGroup_ZF_1_L1 
    group0.inverse_in_group group0.group_inv_of_inv
    by auto;
  with A2 show "a \<in> (\<sm>G\<^isub>+)" using OrderedGroup_ZF_1_L25
    by auto;
qed;

text{*If $a$ is in the inverse of  @{text "G\<^isub>+"}, 
  then its inverse is in $G_+$.*}

lemma (in group3) OrderedGroup_ZF_1_L27:
  assumes "a \<in> (\<sm>G\<^isub>+)"
  shows "a\<inverse> \<in> G\<^isub>+"
  using prems OrderedGroup_ZF_1_L25 PositiveSet_def 
    OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
  by auto;

text{*A linearly ordered group can be decomposed into $G_+$, $\{1\}$ and
  $-G$*}

lemma (in group3) OrdGroup_decomp2: 
  assumes A1: "r {is total on} G"
  shows 
  "G = G\<^isub>+ \<union> (\<sm>G\<^isub>+)\<union> {\<one>}"
  "G\<^isub>+\<inter>(\<sm>G\<^isub>+) = 0"
  "\<one> \<notin> G\<^isub>+\<union>(\<sm>G\<^isub>+)"
proof -
  { fix a assume A2: "a\<in>G"
    with A1 have "a\<in>G\<^isub>+ \<or> a\<inverse>\<in>G\<^isub>+ \<or> a=\<one>"
      using OrderedGroup_ZF_1_L14 by auto;
    with A2 have "a\<in>G\<^isub>+ \<or> a\<in>(\<sm>G\<^isub>+) \<or> a=\<one>"
      using OrderedGroup_ZF_1_L26 by auto;
    then have "a \<in> (G\<^isub>+ \<union> (\<sm>G\<^isub>+)\<union> {\<one>})"
      by auto;
  } then have "G \<subseteq> G\<^isub>+ \<union> (\<sm>G\<^isub>+)\<union> {\<one>}"
    by auto;
  moreover have "G\<^isub>+ \<union> (\<sm>G\<^isub>+)\<union> {\<one>} \<subseteq> G"
    using OrderedGroup_ZF_1_L25 PositiveSet_def
      OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by auto;
  ultimately show "G = G\<^isub>+ \<union> (\<sm>G\<^isub>+)\<union> {\<one>}" by auto
  { def DA: A \<equiv> "G\<^isub>+\<inter>(\<sm>G\<^isub>+)"
    assume "G\<^isub>+\<inter>(\<sm>G\<^isub>+) \<noteq> 0"
    with DA have "A\<noteq>0" by simp;
    then obtain a where "a\<in>A" by auto;
    with DA have False using OrderedGroup_ZF_1_L15 OrderedGroup_ZF_1_L27
      by auto;
  } then show "G\<^isub>+\<inter>(\<sm>G\<^isub>+) = 0" by auto;
  show "\<one> \<notin> G\<^isub>+\<union>(\<sm>G\<^isub>+)"
    using OrderedGroup_ZF_1_L27
      OrderedGroup_ZF_1_L1 group0.group_inv_of_one
      OrderedGroup_ZF_1_L2A by auto;
qed;

text{*If $a\cdot b^{-1}$ is nonnegative, then $b \leq a$. This maybe used to
  recover the order from the set of nonnegative elements and serve as a way
  to define order by prescibing that set (see the "Alternative definitions"
  section).*}

lemma (in group3) OrderedGroup_ZF_1_L28:
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "a\<cdot>b\<inverse> \<in> G\<^sup>+"
  shows "b\<lsq>a"
proof -
  from A2 have "\<one> \<lsq> a\<cdot>b\<inverse>" using OrderedGroup_ZF_1_L2
    by simp
  with A1 show "b\<lsq>a" using OrderedGroup_ZF_1_L5K
    by simp;
qed

text{*A special case of @{text "OrderedGroup_ZF_1_L28"} when
  $a\cdot b^{-1}$ is positive.*}

corollary (in group3) OrderedGroup_ZF_1_L29:
  assumes A1: "a\<in>G"  "b\<in>G" and A2: "a\<cdot>b\<inverse> \<in> G\<^isub>+"
  shows "b\<lsq>a"  "b\<noteq>a"
proof -
  from A2 have "\<one> \<lsq> a\<cdot>b\<inverse>" and I: "a\<cdot>b\<inverse> \<noteq> \<one>"
    using OrderedGroup_ZF_1_L2A by auto
  with A1 show "b\<lsq>a" using OrderedGroup_ZF_1_L5K
    by simp;
  from A1 I show "b\<noteq>a" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L6
    by auto
qed;

text{*A bit stronger that @{text "OrderedGroup_ZF_1_L29"}, adds
   case when two elements are equal.*}

lemma (in group3) OrderedGroup_ZF_1_L30:
  assumes "a\<in>G"  "b\<in>G" and "a=b \<or> b\<cdot>a\<inverse> \<in> G\<^isub>+"
  shows "a\<lsq>b"
  using prems OrderedGroup_ZF_1_L3 OrderedGroup_ZF_1_L29
  by auto;

text{*A different take on decomposition: we can have $a=b$ or $a<b$
  or $b<a$.*}

lemma (in group3) OrderedGroup_ZF_1_L31: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G"  "b\<in>G"
  shows "a=b \<or> (a\<lsq>b \<and> a\<noteq>b) \<or> (b\<lsq>a \<and> b\<noteq>a)"
proof -
  from A2 have "a\<cdot>b\<inverse> \<in> G" using OrderedGroup_ZF_1_L1
    group0.inverse_in_group group0.group_op_closed
    by simp
  with A1 have "a\<cdot>b\<inverse> = \<one> \<or> a\<cdot>b\<inverse> \<in> G\<^isub>+ \<or> (a\<cdot>b\<inverse>)\<inverse> \<in> G\<^isub>+"
    using OrderedGroup_ZF_1_L14 by simp;
  moreover
  { assume "a\<cdot>b\<inverse> = \<one>"
    then have "a\<cdot>b\<inverse>\<cdot>b = \<one>\<cdot>b" by simp;
    with A2 have "a=b \<or> (a\<lsq>b \<and> a\<noteq>b) \<or> (b\<lsq>a \<and> b\<noteq>a)"
      using OrderedGroup_ZF_1_L1
	group0.inv_cancel_two group0.group0_2_L2 by auto }
  moreover
  { assume "a\<cdot>b\<inverse> \<in> G\<^isub>+"
    with A2 have "a=b \<or> (a\<lsq>b \<and> a\<noteq>b) \<or> (b\<lsq>a \<and> b\<noteq>a)"
      using OrderedGroup_ZF_1_L29 by auto }
  moreover
  { assume "(a\<cdot>b\<inverse>)\<inverse> \<in> G\<^isub>+"
    with A2 have "b\<cdot>a\<inverse> \<in> G\<^isub>+" using OrderedGroup_ZF_1_L1
      group0.group0_2_L12 by simp;
    with A2 have "a=b \<or> (a\<lsq>b \<and> a\<noteq>b) \<or> (b\<lsq>a \<and> b\<noteq>a)"
      using OrderedGroup_ZF_1_L29 by auto }
  ultimately show "a=b \<or> (a\<lsq>b \<and> a\<noteq>b) \<or> (b\<lsq>a \<and> b\<noteq>a)"
    by auto;
qed;

(*text{*If $a<b$ then $b-a \in G_+$.*}

lemma (in group3) OrderedGroup_ZF_1_L32:
  assumes A1: "a\<lsq>b"  "a\<noteq>b" shows "b\<cdot>a\<inverse> \<in> G\<^isub>+"
proof -*)  

section{*Intervals and bounded sets*}

text{*A bounded set can be translated to put it in  $G^+$ and then it is 
 still bounded above. *}

lemma (in group3) OrderedGroup_ZF_2_L1: 
  assumes A1: "\<forall>g\<in>A. L\<lsq>g \<and> g\<lsq>M"
  and A2: "S = RightTranslation(G,P,L\<inverse>)" 
  and A3: "a \<in> S``(A)"
  shows "a \<lsq> M\<cdot>L\<inverse>"   "\<one>\<lsq>a"
proof -
  from A3 have "A\<noteq>0" using func1_1_L13A by fast;
  then obtain g where "g\<in>A" by auto
  with A1 have T1: "L\<in>G" "M\<in>G" "L\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
    group0.inverse_in_group by auto;
  with A2 have "S : G\<rightarrow>G" using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover from A1 have T2: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  ultimately have "S``(A) = {S`(b). b\<in>A}" using func_imagedef
    by simp;
  with A3 obtain b where T3: "b\<in>A" "a = S`(b)" by auto;
  with A1 ordGroupAssum T1 have "b\<cdot>L\<inverse>\<lsq>M\<cdot>L\<inverse>" "L\<cdot>L\<inverse>\<lsq>b\<cdot>L\<inverse>"
    using IsAnOrdGroup_def by auto;
  with T3 A2 T1 T2 show "a\<lsq>M\<cdot>L\<inverse>" "\<one>\<lsq>a"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L2 group0.group0_2_L6 
    by auto;
qed;

text{*Every bounded set is an image of a subset of an interval that starts at 
  $1$.*}

lemma (in group3) OrderedGroup_ZF_2_L2:
  assumes A1: "IsBounded(A,r)" 
  shows "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)";
proof (cases "A=0");
  assume A2: "A=0" 
  let ?B = "0"
  let ?g = "\<one>"
  let ?T = "ConstantFunction(G,\<one>)"
  have "?g\<in>G\<^sup>+" using OrderedGroup_ZF_1_L3A by simp;
  moreover have "?T : G\<rightarrow>G"
    using func1_3_L1 OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
  moreover from A2 have "A = T``(?B)" by simp;
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)" by simp;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;
next assume A3: "A\<noteq>0"
  with A1 have "\<exists>L. \<forall>x\<in>A. L\<lsq>x" and "\<exists>U. \<forall>x\<in>A. x\<lsq>U"
    using IsBounded_def IsBoundedBelow_def IsBoundedAbove_def 
    by auto;
  then obtain L U where D1: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U "
    by auto;
  with A3 have T1: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  from A3 obtain a where "a\<in>A" by auto;
  with D1 have T2: "L\<lsq>a" "a\<lsq>U" by auto;
  then have T3: "L\<in>G" "L\<inverse>\<in> G" "U\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.inverse_in_group by auto;
  let ?T = "RightTranslation(G,P,L)"
  let ?B = "RightTranslation(G,P,L\<inverse>)``(A)"
  let ?g = "U\<cdot>L\<inverse>"
  have "?g\<in>G\<^sup>+"
  proof -
    from T2 have "L\<lsq>U" using Group_order_transitive by fast;
    with ordGroupAssum T3 have "L\<cdot>L\<inverse>\<lsq>?g" 
      using IsAnOrdGroup_def by simp;
    with T3 show ?thesis using OrderedGroup_ZF_1_L1 group0.group0_2_L6
      OrderedGroup_ZF_1_L2 by simp;
  qed;
  moreover from T3 have "?T : G\<rightarrow>G"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover have "A = ?T``(?B)"
  proof -;
    from T3 T1 have "?T``(?B) = {a\<cdot>L\<inverse>\<cdot>L. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_5_L6
      by simp;
    moreover from T3 T1 have "\<forall>a\<in>A. a\<cdot>L\<inverse>\<cdot>L = a\<cdot>(L\<inverse>\<cdot>L)"
      using OrderedGroup_ZF_1_L1 group0.group_oper_assoc by auto;
    ultimately have "?T``(?B) = {a\<cdot>(L\<inverse>\<cdot>L). a\<in>A}" by simp;
    with T3 have "?T``(?B) = {a\<cdot>\<one>. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L6 by simp;
    moreover from T1 have "\<forall>a\<in>A. a\<cdot>\<one>=a"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
    ultimately show ?thesis by simp
  qed
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)"
  proof
    fix y assume A4: "y \<in> ?B"
    def D2: S \<equiv> "RightTranslation(G,P,L\<inverse>)"
    from D1 have T4: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U" by simp;
    moreover from D2 have T5: "S = RightTranslation(G,P,L\<inverse>)" 
      by simp; 
    moreover from A4 D2 have T6: "y \<in> S``(A)" by simp;
    ultimately have "y\<lsq>U\<cdot>L\<inverse>" using OrderedGroup_ZF_2_L1
      by blast;
    moreover from T4 T5 T6 have "\<one>\<lsq>y" by (rule OrderedGroup_ZF_2_L1);
    ultimately show "y \<in> Interval(r,\<one>,?g)" using Interval_def by auto;
  qed;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;  
qed;
      
text{*If every interval starting at $1$ is finite, then every bounded set is 
  finite. I find it interesting that this does not require the group to be 
  linearly ordered (the order to be total).*}

theorem (in group3) OrderedGroup_ZF_2_T1:
  assumes A1: "\<forall>g\<in>G\<^sup>+. Interval(r,\<one>,g) \<in> Fin(G)"
  and A2: "IsBounded(A,r)" 
  shows "A \<in> Fin(G)"
proof -
  from A2 have
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    using OrderedGroup_ZF_2_L2 by simp;
  then obtain B g T where D1: "g\<in>G\<^sup>+" "B \<subseteq> Interval(r,\<one>,g)" 
    and D2: "T : G\<rightarrow>G" "A = T``(B)" by auto;
  from D1 A1 have "B\<in>Fin(G)" using Fin_subset_lemma by blast;
  with D2 show ?thesis using Finite1_L6A by simp;
qed;

text{*In linearly ordered groups finite sets are bounded.*}

theorem (in group3) ord_group_fin_bounded:
  assumes "r {is total on} G" and "B\<in>Fin(G)" 
  shows "IsBounded(B,r)"
  using ordGroupAssum prems IsAnOrdGroup_def IsPartOrder_def Finite_ZF_1_T1
  by simp;

(*text{*If for every element of $G$ we can find a different one in $A$ 
  that is greater, then the $A$ can not be finite. This is a property
  of relations that are total, transitive and antisymmetric. *}

lemma (in group3) OrderedGroup_ZF_2_L2A:
  assumes A1: "r {is total on} G"
  and A2: "\<forall>a\<in>G. \<exists>b\<in>A. a\<noteq>b \<and> a\<lsq>b"
  shows "A \<notin> Fin(G)"
  using ordGroupAssum prems IsAnOrdGroup_def IsPartOrder_def 
    OrderedGroup_ZF_1_L1A Finite_ZF_1_1_L3 by simp;*)

text{*For nontrivial linearly ordered groups  if for every element $G$ 
  we can find one in $A$ 
  that is greater or equal (not necessarily strictly greater), then $A$
  can neither be finite nor bounded above.*}

lemma (in group3) OrderedGroup_ZF_2_L2A:
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}"
  and A3: "\<forall>a\<in>G. \<exists>b\<in>A. a\<lsq>b"
  shows 
  "\<forall>a\<in>G. \<exists>b\<in>A. a\<noteq>b \<and> a\<lsq>b"
  "\<not>IsBoundedAbove(A,r)"
  "A \<notin> Fin(G)"
proof -
  { fix a
    from A1 A2 obtain c where "c \<in> G\<^isub>+"
      using OrderedGroup_ZF_1_L21 by auto;
    moreover assume "a\<in>G"
    ultimately have 
      "a\<cdot>c \<in> G"  and I: "a \<ls> a\<cdot>c"
      using OrderedGroup_ZF_1_L22 by auto;
    with A3 obtain b where II: "b\<in>A"  and III: "a\<cdot>c \<lsq> b"
      by auto;
    moreover from I III have "a\<ls>b" by (rule OrderedGroup_ZF_1_L4A);
    ultimately have "\<exists>b\<in>A. a\<noteq>b \<and> a\<lsq>b" by auto;
  } thus "\<forall>a\<in>G. \<exists>b\<in>A. a\<noteq>b \<and> a\<lsq>b" by simp;
  with ordGroupAssum A1 show 
    "\<not>IsBoundedAbove(A,r)"
    "A \<notin> Fin(G)"
    using IsAnOrdGroup_def IsPartOrder_def 
    OrderedGroup_ZF_1_L1A Order_ZF_3_L14 Finite_ZF_1_1_L3
    by auto;
qed;


text{*Nontrivial linearly ordered groups are infinite. Recall 
  that @{text "Fin(A)"} is the collection of finite subsets of $A$. 
  In this lemma we show that $G\notin$ @{text "Fin(G)"}, that is that
  $G$ is not a finite subset of itself. This is a way of saying that $G$ 
  is infinite. We also show that for nontrivial linearly ordered groups 
  @{text "G\<^isub>+"} is infinite.*}

theorem (in group3) Linord_group_infinite:
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}"
  shows 
  "G\<^isub>+ \<notin> Fin(G)"
  "G \<notin> Fin(G)"
proof -
  from A1 A2 show I: "G\<^isub>+ \<notin> Fin(G)"
    using OrderedGroup_ZF_1_L23 OrderedGroup_ZF_2_L2A
    by simp;
  { assume "G \<in> Fin(G)"
    moreover have "G\<^isub>+ \<subseteq> G" using PositiveSet_def by auto;
    ultimately have "G\<^isub>+ \<in> Fin(G)" using Fin_subset_lemma
      by blast;
    with I have False by simp
  } then show "G \<notin> Fin(G)" by auto;
qed;
  
text{*A property of nonempty subsets of linearly ordered groups that don't
  have a maximum: for any element in such subset we can find one that
  is strictly greater.*}

lemma (in group3) OrderedGroup_ZF_2_L2B:
  assumes A1: "r {is total on} G" and A2: "A\<subseteq>G" and 
  A3: "\<not>HasAmaximum(r,A)" and A4: "x\<in>A"
  shows "\<exists>y\<in>A. x\<ls>y"
proof -
  from ordGroupAssum prems have
    "antisym(r)"
    "r {is total on} G"
    "A\<subseteq>G"  "\<not>HasAmaximum(r,A)"  "x\<in>A"
    using IsAnOrdGroup_def IsPartOrder_def
    by auto;
  then have "\<exists>y\<in>A. \<langle>x,y\<rangle> \<in> r \<and> y\<noteq>x"
    using Order_ZF_4_L16 by simp;
  then show "\<exists>y\<in>A. x\<ls>y" by auto;
qed;
    
text{*In linearly ordered groups $G\setminus G_+$ is bounded above. *}

lemma (in group3) OrderedGroup_ZF_2_L3:
  assumes A1: "r {is total on} G" shows "IsBoundedAbove(G-G\<^isub>+,r)"
proof -
  from A1 have "\<forall>a\<in>G-G\<^isub>+. a\<lsq>\<one>"
    using OrderedGroup_ZF_1_L17 by simp;
  then show "IsBoundedAbove(G-G\<^isub>+,r)" 
    using IsBoundedAbove_def by auto;
qed;

text{*In linearly ordered groups if $A\cap G_+$ is finite, 
  then $A$ is bounded above.*}

lemma (in group3) OrderedGroup_ZF_2_L4:
  assumes A1: "r {is total on} G" and A2: "A\<subseteq>G" 
  and A3: "A \<inter> G\<^isub>+ \<in> Fin(G)"
  shows "IsBoundedAbove(A,r)"
proof -
  have "A \<inter> (G-G\<^isub>+) \<subseteq> G-G\<^isub>+" by auto;
  with A1 have "IsBoundedAbove(A \<inter> (G-G\<^isub>+),r)"
    using OrderedGroup_ZF_2_L3 Order_ZF_3_L13
    by blast;
  moreover from A1 A3 have "IsBoundedAbove(A \<inter> G\<^isub>+,r)"
    using ord_group_fin_bounded IsBounded_def 
    by simp;
  moreover from A1 ordGroupAssum have
    "r {is total on} G"  "trans(r)"  "r\<subseteq>G\<times>G"
    using IsAnOrdGroup_def IsPartOrder_def by auto;
  ultimately have "IsBoundedAbove(A \<inter> (G-G\<^isub>+) \<union> A \<inter> G\<^isub>+,r)"
    using Order_ZF_3_L3 by simp;
  moreover from A2 have "A = A \<inter> (G-G\<^isub>+) \<union> A \<inter> G\<^isub>+"
    by auto;
  ultimately show  "IsBoundedAbove(A,r)" by simp;
qed;

text{*If a set $-A\subseteq G$ is bounded above, then $A$ is bounded below.*}

lemma (in group3) OrderedGroup_ZF_2_L5:
  assumes A1: "A\<subseteq>G" and A2: "IsBoundedAbove(\<sm>A,r)"
  shows "IsBoundedBelow(A,r)"
proof (cases "A = 0")
  assume "A = 0" show "IsBoundedBelow(A,r)"
    using IsBoundedBelow_def by auto;
next assume A3: "A\<noteq>0"
  from ordGroupAssum have I: "GroupInv(G,P) : G\<rightarrow>G"
    using IsAnOrdGroup_def group0_2_T2 by simp;
  with A1 A2 A3 obtain u where D: "\<forall>a\<in>(\<sm>A). a\<lsq>u"
    using func1_1_L15A IsBoundedAbove_def by auto;
  { fix b assume "b\<in>A"
    with A1 I D have "b\<inverse> \<lsq> u" and T: "b\<in>G"
      using func_imagedef by auto;
    then have "u\<inverse>\<lsq>(b\<inverse>)\<inverse>" using OrderedGroup_ZF_1_L5
      by simp;
    with T have "u\<inverse>\<lsq>b"
      using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
      by simp;
  } then have "\<forall>b\<in>A. \<langle>u\<inverse>,b\<rangle> \<in> r" by simp;
  then show "IsBoundedBelow(A,r)"
    using Order_ZF_3_L9 by blast;
qed;

text{*if $a\leq b$, then the image of the interval $a..b$ by any function is
  nonempty.*}

lemma (in group3) OrderedGroup_ZF_2_L6: 
  assumes "a\<lsq>b" and "f:G\<rightarrow>G"
  shows "f``(Interval(r,a,b)) \<noteq> 0"
  using ordGroupAssum prems OrderedGroup_ZF_1_L4 
    Order_ZF_2_L6 Order_ZF_2_L2A 
    IsAnOrdGroup_def IsPartOrder_def func1_1_L15A
  by auto
  
section{*Absolute value and the triangle inequality*};

text{*The goal of this section is to prove the triangle inequality for 
  ordered groups.*};

text{*Absolute value maps $G$ into $G$.*}

lemma (in group3) OrderedGroup_ZF_3_L1: 
  "AbsoluteValue(G,P,r) : G\<rightarrow>G"
proof -;
  let ?f = "id(G\<^sup>+)"
  let ?g = "restrict(GroupInv(G,P),G-G\<^sup>+)"
  have "?f : G\<^sup>+\<rightarrow>G\<^sup>+" using id_type by simp;
  then have "?f : G\<^sup>+\<rightarrow>G" using OrderedGroup_ZF_1_L4E
    by (rule fun_weaken_type);
  moreover have "?g : G-G\<^sup>+\<rightarrow>G"
  proof -;
    from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G" 
      using IsAnOrdGroup_def group0_2_T2 by simp;
    moreover have "G-G\<^sup>+ \<subseteq> G" by auto;
    ultimately show ?thesis using restrict_type2 by simp;
  qed;
  moreover have "G\<^sup>+\<inter>(G-G\<^sup>+) = 0" by blast;
  ultimately have "?f \<union> ?g : G\<^sup>+\<union>(G-G\<^sup>+)\<rightarrow>G\<union>G" 
    by (rule fun_disjoint_Un);
  moreover have "G\<^sup>+\<union>(G-G\<^sup>+) = G" using OrderedGroup_ZF_1_L4E
    by auto;
  ultimately show "AbsoluteValue(G,P,r) : G\<rightarrow>G" 
    using AbsoluteValue_def by simp;
qed;

text{*If $a\in G^+$, then $|a| = a$.*}

lemma (in group3) OrderedGroup_ZF_3_L2:
  assumes A1: "a\<in>G\<^sup>+" shows "|a| = a"
proof -
  from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G"
    using IsAnOrdGroup_def group0_2_T2 by simp;
  with A1 show ?thesis using
    func1_1_L1 OrderedGroup_ZF_1_L4E fun_disjoint_apply1
    AbsoluteValue_def id_conv by simp;
qed;

lemma (in group3) OrderedGroup_ZF_3_L2A: 
  shows "|\<one>| = \<one>" using OrderedGroup_ZF_1_L3A OrderedGroup_ZF_3_L2
  by simp;

text{*If $a$ is positive, then $|a| = a$.*}

lemma (in group3) OrderedGroup_ZF_3_L2B:
  assumes "a\<in>G\<^isub>+" shows "|a| = a"
  using prems PositiveSet_def Nonnegative_def OrderedGroup_ZF_3_L2
  by auto;

text{*If $a\in G\setminus G^+$, then $|a| = a^{-1}$.*}

lemma (in group3) OrderedGroup_ZF_3_L3:
   assumes A1: "a \<in> G-G\<^sup>+" shows "|a| = a\<inverse>"
proof -
  have "domain(id(G\<^sup>+)) = G\<^sup>+"
    using id_type func1_1_L1 by auto;
  with A1 show ?thesis using fun_disjoint_apply2 AbsoluteValue_def
    restrict by simp;
qed;

text{*For elements that not greater than the unit, the absolute value is
  the inverse.*}

lemma (in group3) OrderedGroup_ZF_3_L3A:
  assumes A1: "a\<lsq>\<one>" 
  shows "|a| = a\<inverse>"
proof (cases "a=\<one>");
  assume "a=\<one>" then show "|a| = a\<inverse>" 
    using OrderedGroup_ZF_3_L2A OrderedGroup_ZF_1_L1 group0.group_inv_of_one
    by simp;
next assume "a\<noteq>\<one>" 
  with A1 show "|a| = a\<inverse>" using OrderedGroup_ZF_1_L4C OrderedGroup_ZF_3_L3
    by simp;
qed;

text{*In linearly ordered groups the absolute value of any element 
  is in $G^+$.*}

lemma (in group3) OrderedGroup_ZF_3_L3B: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G"
  shows "|a| \<in> G\<^sup>+"
proof (cases "a\<in>G\<^sup>+");
  assume "a \<in> G\<^sup>+" then show "|a| \<in> G\<^sup>+" 
    using OrderedGroup_ZF_3_L2 by simp;
next assume "a \<notin> G\<^sup>+" 
  with A1 A2 show "|a| \<in> G\<^sup>+" using OrderedGroup_ZF_3_L3
    OrderedGroup_ZF_1_L6 by simp;
qed;
  
text{*For linearly ordered groups (where the order is total), the absolute
  value maps the group into the positive set.*}

lemma (in group3) OrderedGroup_ZF_3_L3C:
  assumes A1: "r {is total on} G"
  shows "AbsoluteValue(G,P,r) : G\<rightarrow>G\<^sup>+"
proof-;
  have "AbsoluteValue(G,P,r) : G\<rightarrow>G" using OrderedGroup_ZF_3_L1
    by simp;
  moreover from A1 have T2: 
    "\<forall>g\<in>G. AbsoluteValue(G,P,r)`(g)  \<in> G\<^sup>+" 
    using OrderedGroup_ZF_3_L3B by simp;
  ultimately show ?thesis by (rule func1_1_L1A);
qed;

text{*If the absolute value is the unit, then the elemnent is the unit.*}

lemma (in group3) OrderedGroup_ZF_3_L3D: 
  assumes A1: "a\<in>G" and A2: "|a| = \<one>"
  shows "a = \<one>"
proof (cases "a\<in>G\<^sup>+")
  assume "a \<in> G\<^sup>+" 
  with A2 show "a = \<one>" using OrderedGroup_ZF_3_L2 by simp;
next assume "a \<notin> G\<^sup>+"
  with A1 A2 show "a = \<one>" using 
    OrderedGroup_ZF_3_L3 OrderedGroup_ZF_1_L1 group0.group0_2_L8A
    by auto;
qed;

text{*In linearly ordered groups the unit is not greater than the absolute 
  value of any element.*}

lemma (in group3) OrderedGroup_ZF_3_L3E: 
  assumes "r {is total on} G" and "a\<in>G"
  shows "\<one> \<lsq> |a|"
  using prems OrderedGroup_ZF_3_L3B OrderedGroup_ZF_1_L2 by simp;

text{*If $b$ is greater than both $a$ and $a^{-1}$, then $b$ is greater than
  $|a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L4: 
  assumes A1: "a\<lsq>b" and A2: "a\<inverse>\<lsq> b" 
  shows "|a|\<lsq> b"
proof (cases "a\<in>G\<^sup>+");
  assume "a\<in>G\<^sup>+" 
  with A1 show "|a|\<lsq> b" using OrderedGroup_ZF_3_L2 by simp;
next assume "a\<notin>G\<^sup>+"
  with A1 A2 show "|a|\<lsq> b" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_3_L3 by simp;
qed;

text{*In linearly ordered groups $a\leq |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L5: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G"
  shows "a \<lsq> |a|"
proof (cases "a\<in>G\<^sup>+");
  assume "a \<in> G\<^sup>+"
  with A2 show "a \<lsq> |a|" 
    using OrderedGroup_ZF_3_L2 OrderedGroup_ZF_1_L3 by simp;
next assume "a \<notin> G\<^sup>+"
  with A1 A2 show "a \<lsq> |a|"
    using OrderedGroup_ZF_3_L3B OrderedGroup_ZF_1_L4B by simp;
qed;

text{*$a^{-1}\leq |a|$ (in additive notation it would be $-a\leq |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L6: 
  assumes A1: "a\<in>G" shows "a\<inverse> \<lsq> |a|"
proof (cases "a\<in>G\<^sup>+")
  assume "a \<in> G\<^sup>+"
  then have T1: "\<one>\<lsq>a" and T2: "|a| = a" using OrderedGroup_ZF_1_L2  
    OrderedGroup_ZF_3_L2 by auto;
  then have "a\<inverse>\<lsq>\<one>\<inverse>" using OrderedGroup_ZF_1_L5 by simp;
  then have T3: "a\<inverse>\<lsq>\<one>" 
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_one by simp;
  from T3 T1 have "a\<inverse>\<lsq>a" by (rule Group_order_transitive);
  with T2 show "a\<inverse> \<lsq> |a|" by simp;
next assume A2: "a \<notin> G\<^sup>+"
  from A1 have "|a| \<in> G" 
    using OrderedGroup_ZF_3_L1 apply_funtype by auto;
  with ordGroupAssum have "|a| \<lsq> |a|" 
    using IsAnOrdGroup_def IsPartOrder_def refl_def by simp;
  with A1 A2 show "a\<inverse> \<lsq> |a|" using OrderedGroup_ZF_3_L3 by simp;
qed;

text{*Some inequalities about the product of two elements of a linearly 
  ordered group and its absolute value.*}

lemma (in group3) OrderedGroup_ZF_3_L6A:
  assumes "r {is total on} G" and "a\<in>G"  "b\<in>G"
  shows
  "a\<cdot>b \<lsq>|a|\<cdot>|b|"
  "a\<cdot>b\<inverse> \<lsq>|a|\<cdot>|b|"
  "a\<inverse>\<cdot>b \<lsq>|a|\<cdot>|b|"
  "a\<inverse>\<cdot>b\<inverse> \<lsq>|a|\<cdot>|b|"
  using prems OrderedGroup_ZF_3_L5 OrderedGroup_ZF_3_L6
    OrderedGroup_ZF_1_L5B by auto;

text{*$|a^{-1}|\leq |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L7:
  assumes "r {is total on} G" and "a\<in>G"
  shows "|a\<inverse>|\<lsq>|a|"
  using prems OrderedGroup_ZF_3_L5 OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
    OrderedGroup_ZF_3_L6 OrderedGroup_ZF_3_L4 by simp;

text{*$|a^{-1}| = |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L7A:
  assumes A1: "r {is total on} G" and A2: "a\<in>G"
  shows "|a\<inverse>| = |a|"
proof -
  from A2 have "a\<inverse>\<in>G" using OrderedGroup_ZF_1_L1 group0.inverse_in_group
    by simp;
  with A1 have "|(a\<inverse>)\<inverse>| \<lsq> |a\<inverse>|" using OrderedGroup_ZF_3_L7 by simp;
  with A1 A2 have "|a\<inverse>| \<lsq> |a|"  "|a| \<lsq> |a\<inverse>|"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv OrderedGroup_ZF_3_L7
    by auto;
  then show ?thesis by (rule group_order_antisym);
qed;

text{*$|a\cdot b^{-1}| = |b\cdot a^{-1}|$. It doesn't look so strange in the 
  additive notation: $|a-b| = |b-a|$. *}

lemma (in group3) OrderedGroup_ZF_3_L7B:
  assumes A1: "r {is total on} G" and A2: "a\<in>G" "b\<in>G"
  shows "|a\<cdot>b\<inverse>| = |b\<cdot>a\<inverse>|"
proof -
  from A1 A2 have "|(a\<cdot>b\<inverse>)\<inverse>| = |a\<cdot>b\<inverse>|" using
    OrderedGroup_ZF_1_L1 group0.inverse_in_group group0.group0_2_L1 
    monoid0.group0_1_L1 OrderedGroup_ZF_3_L7A by simp;
  moreover from A2 have "(a\<cdot>b\<inverse>)\<inverse> =  b\<cdot>a\<inverse>" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L12 by simp;
  ultimately show ?thesis by simp;
qed;

text{*Triangle inequality for linearly ordered abelian groups. 
  It would be nice to drop commutativity or give an example that shows we
  can't do that.*}

theorem (in group3) OrdGroup_triangle_ineq:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and A3: "a\<in>G"  "b\<in>G" 
  shows "|a\<cdot>b| \<lsq> |a|\<cdot>|b|"
proof -;
  from A1 A2 A3 have 
    "a \<lsq> |a|" "b \<lsq> |b|" "a\<inverse> \<lsq> |a|" "b\<inverse> \<lsq> |b|"
    using OrderedGroup_ZF_3_L5 OrderedGroup_ZF_3_L6 by auto;
  then have "a\<cdot>b \<lsq> |a|\<cdot>|b|" "a\<inverse>\<cdot>b\<inverse> \<lsq> |a|\<cdot>|b|"
    using OrderedGroup_ZF_1_L5B by auto;
  with A1 A3 show "|a\<cdot>b| \<lsq> |a|\<cdot>|b|"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_two IsCommutative_def 
    OrderedGroup_ZF_3_L4 by simp;
qed;

text{*We can multiply the sides of an inequality with absolute value.*}

lemma (in group3) OrderedGroup_ZF_3_L7C:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and A3: "a\<in>G" "b\<in>G"
  and A4: "|a| \<lsq> c"  "|b| \<lsq> d"
  shows "|a\<cdot>b| \<lsq> c\<cdot>d"
proof -
  from A1 A2 A3 A4 have "|a\<cdot>b| \<lsq> |a|\<cdot>|b|"
    using OrderedGroup_ZF_1_L4 OrdGroup_triangle_ineq 
    by simp;
  moreover from A4 have "|a|\<cdot>|b| \<lsq> c\<cdot>d"
    using OrderedGroup_ZF_1_L5B by simp
  ultimately show ?thesis by (rule Group_order_transitive);
qed;

text{*A version of the @{text "OrderedGroup_ZF_3_L7C"} 
  but with multiplying by the inverse.*}

lemma (in group3) OrderedGroup_ZF_3_L7CA:
  assumes "P {is commutative on} G" 
  and "r {is total on} G" and "a\<in>G"  "b\<in>G"
  and "|a| \<lsq> c"  "|b| \<lsq> d"
  shows "|a\<cdot>b\<inverse>| \<lsq> c\<cdot>d"
  using prems OrderedGroup_ZF_1_L1 group0.inverse_in_group
  OrderedGroup_ZF_3_L7A OrderedGroup_ZF_3_L7C by simp;

text{*Triangle inequality with three integers.*}

lemma (in group3) OrdGroup_triangle_ineq3:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and A3: "a\<in>G"  "b\<in>G"  "c\<in>G" 
  shows "|a\<cdot>b\<cdot>c| \<lsq> |a|\<cdot>|b|\<cdot>|c|"
proof -
  from A3 have T: "a\<cdot>b \<in> G"  "|c| \<in> G"
    using OrderedGroup_ZF_1_L1 group0.group_op_closed
      OrderedGroup_ZF_3_L1 apply_funtype by auto;
  with A1 A2 A3 have "|a\<cdot>b\<cdot>c| \<lsq> |a\<cdot>b|\<cdot>|c|"
    using OrdGroup_triangle_ineq by simp;
  moreover from ordGroupAssum A1 A2 A3 T have
    "|a\<cdot>b|\<cdot>|c| \<lsq> |a|\<cdot>|b|\<cdot>|c|"
    using OrdGroup_triangle_ineq IsAnOrdGroup_def by simp;
  ultimately show "|a\<cdot>b\<cdot>c| \<lsq> |a|\<cdot>|b|\<cdot>|c|"
    by (rule Group_order_transitive);
qed;

text{*Some variants of the triangle inequality.*}

lemma (in group3) OrderedGroup_ZF_3_L7D:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and A3: "a\<in>G"  "b\<in>G"
  and A4: "|a\<cdot>b\<inverse>| \<lsq> c"
  shows 
  "|a| \<lsq> c\<cdot>|b|" 
  "|a| \<lsq> |b|\<cdot>c"
  "c\<inverse>\<cdot>a \<lsq> b"
  "a\<cdot>c\<inverse> \<lsq> b"
  "a \<lsq> b\<cdot>c"
proof -
  from A3 A4 have 
    T: "a\<cdot>b\<inverse> \<in> G"  "|b| \<in> G"  "c\<in>G"  "c\<inverse> \<in> G"
    using OrderedGroup_ZF_1_L1 
      group0.inverse_in_group group0.group0_2_L1 monoid0.group0_1_L1
      OrderedGroup_ZF_3_L1 apply_funtype  OrderedGroup_ZF_1_L4 
    by auto;
  from A3 have "|a| = |a\<cdot>b\<inverse>\<cdot>b|"
    using OrderedGroup_ZF_1_L1 group0.inv_cancel_two
    by simp;
  with A1 A2 A3 T have "|a| \<lsq> |a\<cdot>b\<inverse>|\<cdot>|b|"
    using OrdGroup_triangle_ineq by simp;
  with T A4 show "|a| \<lsq> c\<cdot>|b|" using OrderedGroup_ZF_1_L5C
    by blast
  with T A1 show "|a| \<lsq> |b|\<cdot>c"
    using IsCommutative_def by simp;
  from A2 T have "a\<cdot>b\<inverse> \<lsq> |a\<cdot>b\<inverse>|"
    using OrderedGroup_ZF_3_L5 by simp
  moreover from A4 have "|a\<cdot>b\<inverse>| \<lsq> c" .
  ultimately have I: "a\<cdot>b\<inverse> \<lsq> c"
    by (rule Group_order_transitive)
  with A3 show "c\<inverse>\<cdot>a \<lsq> b"
    using OrderedGroup_ZF_1_L5H by simp;
  with A1 A3 T show "a\<cdot>c\<inverse> \<lsq> b"
    using IsCommutative_def by simp;
  from A1 A3 T I show "a \<lsq> b\<cdot>c"
    using OrderedGroup_ZF_1_L5H IsCommutative_def
    by auto;
qed;

text{*Some more variants of the triangle inequality.*}

lemma (in group3) OrderedGroup_ZF_3_L7E:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and A3: "a\<in>G"  "b\<in>G"
  and A4: "|a\<cdot>b\<inverse>| \<lsq> c"
  shows "b\<cdot>c\<inverse> \<lsq> a"
proof -
  from A3 have "a\<cdot>b\<inverse> \<in> G"
    using OrderedGroup_ZF_1_L1 
      group0.inverse_in_group group0.group_op_closed
    by auto
  with A2 have "|(a\<cdot>b\<inverse>)\<inverse>| = |a\<cdot>b\<inverse>|"
    using OrderedGroup_ZF_3_L7A by simp
  moreover from A3 have "(a\<cdot>b\<inverse>)\<inverse> = b\<cdot>a\<inverse>"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L12
    by simp;
  ultimately have "|b\<cdot>a\<inverse>| = |a\<cdot>b\<inverse>|"
    by simp
  with A1 A2 A3 A4 show "b\<cdot>c\<inverse> \<lsq> a"
    using OrderedGroup_ZF_3_L7D by simp;
qed;

text{*An application of the triangle inequality with four group
  elements.*}

lemma (in group3) OrderedGroup_ZF_3_L7F:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and 
  A3: "a\<in>G"  "b\<in>G"  "c\<in>G"  "d\<in>G"
  shows "|a\<cdot>c\<inverse>| \<lsq> |a\<cdot>b|\<cdot>|c\<cdot>d|\<cdot>|b\<cdot>d\<inverse>|"
proof -
  from A3 have T:
    "a\<cdot>c\<inverse> \<in> G"  "a\<cdot>b \<in> G"  "c\<cdot>d \<in> G"  "b\<cdot>d\<inverse> \<in> G"
    "(c\<cdot>d)\<inverse> \<in> G"  "(b\<cdot>d\<inverse>)\<inverse> \<in> G"
    using OrderedGroup_ZF_1_L1 
      group0.inverse_in_group group0.group_op_closed
    by auto;
  with A1 A2 have "|(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse>\<cdot>(b\<cdot>d\<inverse>)\<inverse>| \<lsq> |a\<cdot>b|\<cdot>|(c\<cdot>d)\<inverse>|\<cdot>|(b\<cdot>d\<inverse>)\<inverse>|"
    using OrdGroup_triangle_ineq3 by simp;
  moreover from A2 T have "|(c\<cdot>d)\<inverse>| =|c\<cdot>d|" and "|(b\<cdot>d\<inverse>)\<inverse>| = |b\<cdot>d\<inverse>|"
    using OrderedGroup_ZF_3_L7A by auto;
  moreover from A1 A3 have "(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse>\<cdot>(b\<cdot>d\<inverse>)\<inverse> = a\<cdot>c\<inverse>"
    using OrderedGroup_ZF_1_L1 group0.group0_4_L8
    by simp;
  ultimately show "|a\<cdot>c\<inverse>| \<lsq> |a\<cdot>b|\<cdot>|c\<cdot>d|\<cdot>|b\<cdot>d\<inverse>|"
    by simp;
qed;
    
text{*$|a|\leq L$ implies $L^{-1} \leq a$
  (it would be $-L \leq a$ in the additive notation).*}

lemma (in group3) OrderedGroup_ZF_3_L8:
  assumes A1:  "a\<in>G" and A2: "|a|\<lsq>L"
   shows 
  "L\<inverse>\<lsq>a"
proof -
  from A1 have I: "a\<inverse> \<lsq> |a|" using OrderedGroup_ZF_3_L6 by simp;
  from I A2 have "a\<inverse> \<lsq> L" by (rule Group_order_transitive);
  then have "L\<inverse>\<lsq>(a\<inverse>)\<inverse>" using OrderedGroup_ZF_1_L5 by simp;
  with A1 show "L\<inverse>\<lsq>a" using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
    by simp
qed;

text{*In linearly ordered groups $|a|\leq L$ implies $a \leq L$
  (it would be $a \leq L$ in the additive notation).*}

lemma (in group3) OrderedGroup_ZF_3_L8A:
  assumes A1: "r {is total on} G" 
  and A2: "a\<in>G" and A3: "|a|\<lsq>L"
  shows 
  "a\<lsq>L"
  "\<one>\<lsq>L"
proof -;
  from A1 A2 have I: "a \<lsq> |a|" using OrderedGroup_ZF_3_L5 by simp;
  from I A3 show "a\<lsq>L" by (rule Group_order_transitive);
  from A1 A2 A3 have "\<one> \<lsq> |a|"  "|a|\<lsq>L"
     using OrderedGroup_ZF_3_L3B OrderedGroup_ZF_1_L2 by auto;
   then show "\<one>\<lsq>L" by (rule Group_order_transitive);
qed;

text{*A somewhat generalized version of the above lemma.*}

lemma (in group3) OrderedGroup_ZF_3_L8B:
  assumes A1: "a\<in>G" and A2: "|a|\<lsq>L" and A3: "\<one>\<lsq>c"
  shows "(L\<cdot>c)\<inverse> \<lsq> a"
proof -
  from A1 A2 A3 have "c\<inverse>\<cdot>L\<inverse> \<lsq> \<one>\<cdot>a"
    using OrderedGroup_ZF_3_L8 OrderedGroup_ZF_1_L5AB
    OrderedGroup_ZF_1_L5B by simp;
  with A1 A2 A3 show "(L\<cdot>c)\<inverse> \<lsq> a"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1
      group0.group_inv_of_two group0.group0_2_L2
    by simp;
qed;

text{*If $b$ is between $a$ and $a\cdot c$, then $b\cdot a^{-1}\leq c$.*}

lemma (in group3) OrderedGroup_ZF_3_L8C:
  assumes A1: "a\<lsq>b" and A2: "c\<in>G" and A3: "b\<lsq>c\<cdot>a"
  shows "|b\<cdot>a\<inverse>| \<lsq> c"
proof -
  from A1 A2 A3 have "b\<cdot>a\<inverse> \<lsq> c"
    using OrderedGroup_ZF_1_L9C OrderedGroup_ZF_1_L4
    by simp;
  moreover have "(b\<cdot>a\<inverse>)\<inverse> \<lsq> c"
  proof -
    from A1 have T: "a\<in>G"  "b\<in>G"
      using OrderedGroup_ZF_1_L4 by auto;
    with A1 have "a\<cdot>b\<inverse> \<lsq> \<one>"
      using OrderedGroup_ZF_1_L9 by blast;
    moreover
    from A1 A3 have "a\<lsq>c\<cdot>a"
      by (rule Group_order_transitive);
    with ordGroupAssum T have "a\<cdot>a\<inverse> \<lsq> c\<cdot>a\<cdot>a\<inverse>"
      using OrderedGroup_ZF_1_L1 group0.inverse_in_group
      IsAnOrdGroup_def by simp;
    with T A2 have "\<one> \<lsq> c"
      using OrderedGroup_ZF_1_L1 
	group0.group0_2_L6 group0.inv_cancel_two
      by simp;
    ultimately have "a\<cdot>b\<inverse> \<lsq> c"
      by (rule Group_order_transitive);
    with T show "(b\<cdot>a\<inverse>)\<inverse> \<lsq> c"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L12
      by simp;
  qed;
  ultimately show "|b\<cdot>a\<inverse>| \<lsq> c"
    using OrderedGroup_ZF_3_L4 by simp;
qed;
  
text{*For linearly ordered groups if the absolute values of elements in a set
  are bounded, then the set is bounded.*}

lemma (in group3) OrderedGroup_ZF_3_L9:
  assumes A1: "r {is total on} G"
  and A2: "A\<subseteq>G" and A3: "\<forall>a\<in>A. |a| \<lsq> L"
  shows "IsBounded(A,r)"
proof -;
  from A1 A2 A3 have 
    "\<forall>a\<in>A. a\<lsq>L"  "\<forall>a\<in>A. L\<inverse>\<lsq>a" 
    using OrderedGroup_ZF_3_L8 OrderedGroup_ZF_3_L8A by auto;
  then show "IsBounded(A,r)" using
    IsBoundedAbove_def IsBoundedBelow_def IsBounded_def
    by auto;
qed;

text{* A slightly more general version of the previous lemma, stating the same
  fact for a set defined by separation.*}

lemma (in group3) OrderedGroup_ZF_3_L9A:
  assumes A1: "r {is total on} G"
  and A2: "\<forall>x\<in>X. b(x)\<in>G \<and> |b(x)|\<lsq>L"
  shows "IsBounded({b(x). x\<in>X},r)"
proof -
  from A2 have "{b(x). x\<in>X} \<subseteq> G" "\<forall>a\<in>{b(x). x\<in>X}. |a| \<lsq> L" 
    by auto;
  with A1 show ?thesis using OrderedGroup_ZF_3_L9 by blast;
qed;

text{*A special form of the previous lemma stating a similar fact for an
  image of a set by a function with values in a linearly ordered group.*}

lemma (in group3) OrderedGroup_ZF_3_L9B:
  assumes A1: "r {is total on} G"
  and A2: "f:X\<rightarrow>G" and A3: "A\<subseteq>X"
  and A4: "\<forall>x\<in>A. |f`(x)| \<lsq> L"
  shows "IsBounded(f``(A),r)"
proof -
  from A2 A3 A4 have "\<forall>x\<in>A. f`(x) \<in> G \<and>  |f`(x)| \<lsq> L"
    using apply_funtype by auto;
  with A1 have  "IsBounded({f`(x). x\<in>A},r)"
    by (rule OrderedGroup_ZF_3_L9A);
  with A2 A3 show "IsBounded(f``(A),r)"
    using func_imagedef by simp;
qed;
  
text{*For linearly ordered groups if $l\leq a\leq u$ then 
  $|a|$ is smaller than the greater of $|l|,|u|$.*}

lemma (in group3) OrderedGroup_ZF_3_L10:
  assumes A1: "r {is total on} G"
  and A2: "l\<lsq>a"  "a\<lsq>u" 
  shows 
  "|a| \<lsq> GreaterOf(r,|l|,|u|)"
proof (cases "a\<in>G\<^sup>+");
  from A2 have T1: "|l| \<in> G" "|a| \<in> G" "|u| \<in> G"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_3_L1 apply_funtype
    by auto;
  assume A3: "a\<in>G\<^sup>+"
  with A2 have "\<one>\<lsq>a" "a\<lsq>u" 
    using OrderedGroup_ZF_1_L2 by auto;
  then have "\<one>\<lsq>u" by (rule Group_order_transitive)
  with A2 A3 have "|a|\<lsq>|u|" 
    using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_3_L2 by simp;
  moreover from A1 T1 have "|u| \<lsq> GreaterOf(r,|l|,|u|)"
    using Order_ZF_3_L2 by simp;
  ultimately show "|a| \<lsq> GreaterOf(r,|l|,|u|)"
    by (rule Group_order_transitive);
next assume A4: "a\<notin>G\<^sup>+"
  with A2 have T2: 
    "l\<in>G" "|l| \<in> G" "|a| \<in> G" "|u| \<in> G" "a \<in> G-G\<^sup>+"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_3_L1 apply_funtype
    by auto;
  with A2 have "l \<in> G-G\<^sup>+" using OrderedGroup_ZF_1_L4D by fast;
  with T2 A2 have "|a| \<lsq> |l|" 
    using OrderedGroup_ZF_3_L3 OrderedGroup_ZF_1_L5
    by simp;
  moreover from A1 T2 have "|l| \<lsq> GreaterOf(r,|l|,|u|)"
    using Order_ZF_3_L2 by simp; 
  ultimately show "|a| \<lsq> GreaterOf(r,|l|,|u|)"
    by (rule Group_order_transitive);
qed;

text{*For linearly ordered groups if a set is bounded then the absolute 
  values are bounded.*}

lemma (in group3) OrderedGroup_ZF_3_L10A:
  assumes A1: "r {is total on} G"
  and A2: "IsBounded(A,r)"
  shows "\<exists>L. \<forall>a\<in>A. |a| \<lsq> L"
proof (cases "A=0");
  assume "A = 0" then show ?thesis by auto;
next assume A3: "A\<noteq>0" 
  with A2 have "\<exists>u. \<forall>g\<in>A. g\<lsq>u" and "\<exists>l.\<forall>g\<in>A. l\<lsq>g"
    using IsBounded_def IsBoundedAbove_def IsBoundedBelow_def
    by auto;
  then obtain u l where "\<forall>g\<in>A. l\<lsq>g \<and>  g\<lsq>u" 
    by auto;
  with A1 have "\<forall>a\<in>A. |a| \<lsq> GreaterOf(r,|l|,|u|)"
    using OrderedGroup_ZF_3_L10 by simp;
  then show ?thesis by auto;
qed;
  
text{* A slightly more general version of the previous lemma, stating the same
  fact for a set defined by separation.*}

lemma (in group3) OrderedGroup_ZF_3_L11:
  assumes A1: "r {is total on} G"
  and A2: "IsBounded({b(x).x\<in>X},r)"
  shows "\<exists>L. \<forall>x\<in>X. |b(x)| \<lsq> L"
proof -
  from A1 A2 show ?thesis using OrderedGroup_ZF_3_L10A
    by blast;
qed;

text{*Absolute values of elements of a finite image of a nonempty set are 
  bounded by an element of the group.*}

lemma (in group3) OrderedGroup_ZF_3_L11A:
  assumes A1: "r {is total on} G" 
  and A2: "X\<noteq>0" and A3: "{b(x). x\<in>X} \<in> Fin(G)"
  shows "\<exists>L\<in>G. \<forall>x\<in>X. |b(x)| \<lsq> L"
proof -
  from A1 A3 have "\<exists>L. \<forall>x\<in>X. |b(x)| \<lsq> L"
     using  ord_group_fin_bounded OrderedGroup_ZF_3_L11
     by simp;
  then obtain L where I: "\<forall>x\<in>X. |b(x)| \<lsq> L"
    using OrderedGroup_ZF_3_L11 by auto;
  from A2 obtain x where "x\<in>X" by auto;
  with I show ?thesis using OrderedGroup_ZF_1_L4
    by blast;
qed;

text{*In totally oredered groups the absolute value of a 
  nonunit element is in @{text "G\<^isub>+"}.*}

lemma (in group3) OrderedGroup_ZF_3_L12:
  assumes A1: "r {is total on} G" 
  and A2: "a\<in>G"  and A3: "a\<noteq>\<one>"
  shows "|a| \<in> G\<^isub>+"
proof -
  from A1 A2 have "|a| \<in> G"  "\<one> \<lsq> |a|" 
    using OrderedGroup_ZF_3_L1 apply_funtype
      OrderedGroup_ZF_3_L3B OrderedGroup_ZF_1_L2 
    by auto;
  moreover from A2 A3 have "|a| \<noteq> \<one>"
    using OrderedGroup_ZF_3_L3D by auto;
  ultimately show "|a| \<in> G\<^isub>+"
    using PositiveSet_def by auto;
qed;

(*text{*If an nonnegative element is less or equal than another,
  then so is its absolute value.*}

lemma (in group3) OrderedGroup_ZF_3_L13: 
  assumes A1: "\<one>\<lsq>a" and A2: "a\<lsq>b"
  shows "|a| \<lsq> b"
proof -
  from A1 have "|a| = a" using 
    OrderedGroup_ZF_1_L2 OrderedGroup_ZF_3_L2 by simp
  with A2 show "|a| \<lsq> b" by simp
qed;*)

section{*Maximum absolute value of a set *}

text{* Quite often when considering inequalities we prefer to talk about
  the absolute values instead of raw elements of a set. This section formalizes
  some material that is useful for that. *}

text{*If a set has a maximum and minimum, then the greater of the 
  absolute value of the maximum and minimum belongs to the image of the set 
  by the absolute value function. *}

lemma (in group3) OrderedGroup_ZF_4_L1:
  assumes "A \<subseteq> G"
  and "HasAmaximum(r,A)" "HasAminimum(r,A)"
  and "M = GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)"
  shows "M \<in> AbsoluteValue(G,P,r)``(A)"
  using ordGroupAssum prems IsAnOrdGroup_def IsPartOrder_def 
    Order_ZF_4_L3 Order_ZF_4_L4 OrderedGroup_ZF_3_L1 
    func_imagedef GreaterOf_def by auto;

text{*If a set has a maximum and minimum, then the greater of the 
  absolute value of the maximum and minimum bounds absolute values of all 
  elements of the set. *}

lemma (in group3) OrderedGroup_ZF_4_L2: 
  assumes A1: "r {is total on} G"
  and A2: "HasAmaximum(r,A)"  "HasAminimum(r,A)"
  and A3: "a\<in>A"
  shows "|a|\<lsq> GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)" 
proof -;
  from ordGroupAssum A2 A3 have 
    "Minimum(r,A)\<lsq> a" "a\<lsq> Maximum(r,A)" 
    using IsAnOrdGroup_def IsPartOrder_def Order_ZF_4_L3 Order_ZF_4_L4
    by auto;
  with A1 show ?thesis by (rule OrderedGroup_ZF_3_L10);
qed;

text{*If a set has a maximum and minimum, then the greater of the 
  absolute value of the maximum and minimum bounds absolute values of all 
  elements of the set. In this lemma the absolute values of ekements of a 
  set are represented as the elements of the image of the set by the absolute
  value function.*}

lemma (in group3) OrderedGroup_ZF_4_L3: 
  assumes "r {is total on} G" and "A \<subseteq> G"
  and "HasAmaximum(r,A)" "HasAminimum(r,A)"
  and "b \<in> AbsoluteValue(G,P,r)``(A)"
  shows "b\<lsq> GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)"
  using prems OrderedGroup_ZF_3_L1 func_imagedef OrderedGroup_ZF_4_L2
  by auto;

text{*If a set has a maximum and minimum, then the set of absolute values 
  also has a maximum.*}

lemma (in group3) OrderedGroup_ZF_4_L4:
  assumes A1: "r {is total on} G" and A2: "A \<subseteq> G"
  and A3: "HasAmaximum(r,A)" "HasAminimum(r,A)"
  shows "HasAmaximum(r,AbsoluteValue(G,P,r)``(A))"
proof -;
  let ?M = "GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)"
  from A2 A3 have "?M \<in> AbsoluteValue(G,P,r)``(A)"
    using OrderedGroup_ZF_4_L1 by simp;
  moreover from A1 A2 A3 have 
    "\<forall>b \<in> AbsoluteValue(G,P,r)``(A). b \<lsq> ?M"
    using OrderedGroup_ZF_4_L3 by simp;
  ultimately show ?thesis using HasAmaximum_def by auto;
qed;

text{*If a set has a maximum and a minimum, then all absolute values are
  bounded by the maximum of the set of absolute values.*}

lemma (in group3) OrderedGroup_ZF_4_L5:
  assumes A1: "r {is total on} G" and A2: "A \<subseteq> G"
  and A3: "HasAmaximum(r,A)" "HasAminimum(r,A)"
  and A4: "a\<in>A"
  shows "|a| \<lsq> Maximum(r,AbsoluteValue(G,P,r)``(A))"
proof -;
  from A2 A4 have "|a| \<in> AbsoluteValue(G,P,r)``(A)" 
    using OrderedGroup_ZF_3_L1 func_imagedef by auto;
  with ordGroupAssum A1 A2 A3 show ?thesis using 
    IsAnOrdGroup_def IsPartOrder_def OrderedGroup_ZF_4_L4
    Order_ZF_4_L3 by simp;
qed;

section{*Alternative definitions*}

text{*Sometimes it is usful to define the order by prescibing the set
  of positive or nonnegative elements. This section deals with two such 
  definitions. One takes a subset $H$ of $G$ that is closed under the group
  operation, $1\notin H$ and for every $a\in H$ we have either $a\in H$ or 
  $a^{-1}\in H$. Then the order is defined as $a\leq b$ iff $a=b$ or 
  $a^{-1}b \in H$. For abelian groups this makes a linearly ordered group. 
  We will refer to order defined this way in the comments as the order 
  defined by a positive set. The context used in this section is the 
  @{text "group0"} context defined in @{text "Group_ZF"} theory. Recall that
  @{text "f"} in that context denotes the group operation (unlike in the 
  previous sections where the group operation was denoted @{text "P"}.*}

text{*The order defined by a positive set is the same as the order defined by
  a nonnegative set.*}

lemma (in group0) OrderedGroup_ZF_5_L1:
  assumes A1: "r = {p \<in> G\<times>G. fst(p) = snd(p) \<or> fst(p)\<inverse>\<cdot>snd(p) \<in> H}"
  shows "\<langle>a,b\<rangle> \<in> r  \<longleftrightarrow> a\<in>G \<and> b\<in>G \<and> a\<inverse>\<cdot>b \<in> H \<union> {\<one>}"
proof;
  assume "\<langle>a,b\<rangle> \<in> r"
  with A1 show "a\<in>G \<and> b\<in>G \<and> a\<inverse>\<cdot>b \<in> H \<union> {\<one>}" 
    using group0_2_L6 by auto;
next assume "a\<in>G \<and> b\<in>G \<and> a\<inverse>\<cdot>b \<in> H \<union> {\<one>}"
   then have "a\<in>G \<and> b\<in>G \<and> b=(a\<inverse>)\<inverse> \<or>  a\<in>G \<and> b\<in>G \<and> a\<inverse>\<cdot>b \<in> H"
    using  inverse_in_group group0_2_L9 by auto;
  with A1 show "\<langle>a,b\<rangle> \<in> r" using group_inv_of_inv
    by auto;
qed;
  
text{*The relation defined by a positive set is antisymmetric.*}

lemma (in group0) OrderedGroup_ZF_5_L2:
  assumes A1: "r = {p \<in> G\<times>G. fst(p) = snd(p) \<or> fst(p)\<inverse>\<cdot>snd(p) \<in> H}"
  and A2: "\<forall>a\<in>G. a\<noteq>\<one> \<longrightarrow> (a\<in>H) Xor (a\<inverse>\<in>H)"
  shows "antisym(r)"
proof -
  { fix a b assume A3: "\<langle>a,b\<rangle> \<in> r"  "\<langle>b,a\<rangle> \<in> r"
    with A1 have T: "a\<in>G"  "b\<in>G" by auto;
    { assume A4: "a\<noteq>b"
      with A1 A3 have "a\<inverse>\<cdot>b \<in> G"  "a\<inverse>\<cdot>b \<in> H"  "(a\<inverse>\<cdot>b)\<inverse> \<in> H"
	using inverse_in_group group0_2_L1 monoid0.group0_1_L1 group0_2_L12
	by auto;
      with A2 have "a\<inverse>\<cdot>b = \<one>" using Xor_def by auto;
      with T A4 have False using group0_2_L11 by auto;
    } then have "a=b" by auto;
  } then show "antisym(r)" by (rule antisymI);
qed;

text{*The relation defined by a positive set is transitive.*}

lemma (in group0) OrderedGroup_ZF_5_L3:
  assumes A1: "r = {p \<in> G\<times>G. fst(p) = snd(p) \<or> fst(p)\<inverse>\<cdot>snd(p) \<in> H}"
  and A2: "H\<subseteq>G"  "H {is closed under} P"
  shows "trans(r)"
proof -
  { fix a b c assume "\<langle>a,b\<rangle> \<in> r"  "\<langle>b,c\<rangle> \<in> r"
    with A1 have 
      "a\<in>G \<and> b\<in>G \<and> a\<inverse>\<cdot>b \<in> H \<union> {\<one>}"
      "b\<in>G \<and> c\<in>G \<and> b\<inverse>\<cdot>c \<in> H \<union> {\<one>}"
      using OrderedGroup_ZF_5_L1 by auto;
    with A2 have 
      I: "a\<in>G"  "b\<in>G"  "c\<in>G"
      and "(a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c) \<in>  H \<union> {\<one>}"
      using inverse_in_group group0_2_L17 IsOpClosed_def
      by auto
    moreover from I have "a\<inverse>\<cdot>c = (a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c)"
      by (rule group0_2_L14A);
    ultimately have "\<langle>a,c\<rangle> \<in> G\<times>G"  "a\<inverse>\<cdot>c  \<in>  H \<union> {\<one>}"
      by auto;
    with A1 have "\<langle>a,c\<rangle> \<in> r" using OrderedGroup_ZF_5_L1
      by auto;
  } then have "\<forall> a b c. \<langle>a, b\<rangle> \<in> r \<and> \<langle>b, c\<rangle> \<in> r \<longrightarrow> \<langle>a, c\<rangle> \<in> r"
    by blast;
  then show  "trans(r)" by (rule Fol1_L2);
qed;

text{*The relation defined by a positive set is translation invariant.
  With our definition this step requires the group to be abelian. *}

lemma (in group0) OrderedGroup_ZF_5_L4:
  assumes A1: "r = {p \<in> G\<times>G. fst(p) = snd(p) \<or> fst(p)\<inverse>\<cdot>snd(p) \<in> H}"
  and A2: "P {is commutative on} G"
  and A3: "\<langle>a,b\<rangle> \<in> r"  and A4: "c\<in>G"
  shows "\<langle>a\<cdot>c,b\<cdot>c\<rangle> \<in> r \<and> \<langle>c\<cdot>a,c\<cdot>b\<rangle> \<in> r"
proof
  from A1 A3 A4 have 
    I: "a\<in>G"  "b\<in>G"  "a\<cdot>c \<in> G"  "b\<cdot>c \<in> G"
    and II: "a\<inverse>\<cdot>b \<in> H \<union> {\<one>}"
    using OrderedGroup_ZF_5_L1 group_op_closed 
    by auto;
  with A2 A4 have "(a\<cdot>c)\<inverse>\<cdot>(b\<cdot>c) \<in> H \<union> {\<one>}"
    using group0_4_L6D by simp;
  with A1 I show "\<langle>a\<cdot>c,b\<cdot>c\<rangle> \<in> r" using  OrderedGroup_ZF_5_L1
    by auto;
  with A2 A4 I show "\<langle>c\<cdot>a,c\<cdot>b\<rangle> \<in> r"
    using IsCommutative_def by simp;
qed;
  
text{*If $H\subseteq G$ is closed under the group operation
  $1\notin H$ and for every $a\in H$ we have either $a\in H$ or 
  $a^{-1}\in H$, then the relation "$\leq$" defined by 
  $a\leq b \Leftrightarrow a^{-1}b \in H$ orders the group $G$.
  In such order $H$ may be the set of positive or nonnegative
  elements.*}

lemma (in group0) OrderedGroup_ZF_5_L5: 
  assumes A1: "P {is commutative on} G"
  and A2: "H\<subseteq>G"  "H {is closed under} P"
  and A3: "\<forall>a\<in>G. a\<noteq>\<one> \<longrightarrow> (a\<in>H) Xor (a\<inverse>\<in>H)"
  and A4: "r = {p \<in> G\<times>G. fst(p) = snd(p) \<or> fst(p)\<inverse>\<cdot>snd(p) \<in> H}"
  shows 
  "IsAnOrdGroup(G,P,r)"
  "r {is total on} G"
  "Nonnegative(G,P,r) = PositiveSet(G,P,r) \<union> {\<one>}"
proof -
  from groupAssum A2 A3 A4 have 
    "IsAgroup(G,P)"  "r \<subseteq> G\<times>G"  "IsPartOrder(G,r)"
    using refl_def OrderedGroup_ZF_5_L2 OrderedGroup_ZF_5_L3
      IsPartOrder_def by auto
  moreover from A1 A4 have 
    "\<forall>g\<in>G. \<forall>a b. <a,b> \<in> r \<longrightarrow> \<langle>a\<cdot>g,b\<cdot>g\<rangle> \<in> r \<and> \<langle>g\<cdot>a,g\<cdot>b\<rangle> \<in> r"
    using OrderedGroup_ZF_5_L4 by blast;
  ultimately show "IsAnOrdGroup(G,P,r)" 
    using IsAnOrdGroup_def by simp;
  then show "Nonnegative(G,P,r) = PositiveSet(G,P,r) \<union> {\<one>}"
    using group3_def group3.OrderedGroup_ZF_1_L24
    by simp;
  { fix a b 
    assume T: "a\<in>G"  "b\<in>G"
    then have T1: "a\<inverse>\<cdot>b \<in> G"
      using inverse_in_group group_op_closed by simp;
    { assume "<a,b> \<notin> r"
      with A4 T have I: "a\<noteq>b" and II: "a\<inverse>\<cdot>b \<notin> H" 
	by auto;
      from A3 T T1 I have "(a\<inverse>\<cdot>b \<in> H) Xor ((a\<inverse>\<cdot>b)\<inverse> \<in> H)"
	using group0_2_L11 by auto;
      with A4 T II have "<b,a> \<in> r"
	using Xor_def group0_2_L12 by simp;
    } then have "<a,b> \<in> r \<or> <b,a> \<in> r" by auto;
  } then show "r {is total on} G" using IsTotal_def
    by simp;
qed;

text{*If the set defined as in @{text "OrderedGroup_ZF_5_L4"} does not 
  contain the neutral element, then it is the positive set for the resulting
  order.*}

lemma (in group0) OrderedGroup_ZF_5_L6: 
  assumes "P {is commutative on} G"
  and "H\<subseteq>G" and "\<one> \<notin> H"
  and "r = {p \<in> G\<times>G. fst(p) = snd(p) \<or> fst(p)\<inverse>\<cdot>snd(p) \<in> H}"
  shows "PositiveSet(G,P,r) = H"
  using prems group_inv_of_one group0_2_L2 PositiveSet_def
  by auto;

text{*The next definition describes how we construct an order relation
  from the prescribed set of positive elements.*}

constdefs
  "OrderFromPosSet(G,P,H) \<equiv> 
  {p \<in> G\<times>G. fst(p) = snd(p) \<or> P`\<langle>GroupInv(G,P)`(fst(p)),snd(p)\<rangle> \<in> H }"

text{*The next theorem rephrases lemmas 
  @{text "OrderedGroup_ZF_5_L5"} and @{text "OrderedGroup_ZF_5_L6"}
  using the definition of the order from the positive set 
  @{text "OrderFromPosSet"}. To simmarize, this is what it says:
  Suppose that $H\subseteq G$ is a set closed under that group operation
  such that $1\notin H$ and for every nonunit group element $a$ either $a\in H$
  or $a^{-1}\in H$. Define the order as $a\leq b$ iff $a=b$ or 
  $a^{-1}\cdot b \in H$. Then this order makes $G$ into a linearly ordered 
  group such $H$ is the set of positive elements (and then of course 
  $H \cup \{1\}$ is the set of nonnegative elements).*}

theorem (in group0) Group_ord_by_positive_set: 
  assumes "P {is commutative on} G"
  and "H\<subseteq>G"   "H {is closed under} P"   "\<one> \<notin> H"
  and "\<forall>a\<in>G. a\<noteq>\<one> \<longrightarrow> (a\<in>H) Xor (a\<inverse>\<in>H)"
  shows 
  "IsAnOrdGroup(G,P,OrderFromPosSet(G,P,H))"
  "OrderFromPosSet(G,P,H) {is total on} G"
  "PositiveSet(G,P,OrderFromPosSet(G,P,H)) = H"
  "Nonnegative(G,P,OrderFromPosSet(G,P,H)) = H \<union> {\<one>}" 
  using prems OrderFromPosSet_def OrderedGroup_ZF_5_L5 OrderedGroup_ZF_5_L6
  by auto;

section{*Odd Extensions*}

text{*In this section we verify properties of odd extensions of functions
  defined on $G_+$. An odd extension of a function $f: G_+ \rightarrow G$
  is a function $f^\circ : G\rightarrow G$ defined by $f^\circ (x) = f(x)$ 
  if $x\in G_+$, $f(1) = 1$ and $f^\circ (x) = (f(x^{-1}))^{-1}$ for $x < 1$.
  Such function is the unique odd function that is equal to $f$ when
  restricted to $G_+$.*}

text{*The next lemma is just to see the definition of the odd extension
  in the notation used in the @{text "group1"} context.*}

lemma (in group3) OrderedGroup_ZF_6_L1:
  shows "f\<degree> = f \<union> {\<langle>a, (f`(a\<inverse>))\<inverse>\<rangle>. a \<in> \<sm>G\<^isub>+} \<union> {\<langle>\<one>,\<one>\<rangle>}"
  using OddExtension_def by simp;

text{*A technical lemma that states that from a function defined on 
  @{text "G\<^isub>+"} with values in $G$ we have $(f(a^{-1}))^{-1}\in G$.*}

lemma (in group3) OrderedGroup_ZF_6_L2:
  assumes "f: G\<^isub>+\<rightarrow>G" and "a\<in>\<sm>G\<^isub>+"
  shows 
  "f`(a\<inverse>) \<in> G"  
  "(f`(a\<inverse>))\<inverse> \<in> G"
  using prems OrderedGroup_ZF_1_L27 apply_funtype
    OrderedGroup_ZF_1_L1 group0.inverse_in_group
  by auto;

text{*The main theorem about odd extensions. It basically says that the odd 
  extension of a function is what we want to to be.*}

lemma (in group3) odd_ext_props: 
  assumes A1: "r {is total on} G" and A2: "f: G\<^isub>+\<rightarrow>G"
  shows 
  "f\<degree> : G \<rightarrow> G"
  "\<forall>a\<in>G\<^isub>+. (f\<degree>)`(a) = f`(a)"
  "\<forall>a\<in>(\<sm>G\<^isub>+). (f\<degree>)`(a) = (f`(a\<inverse>))\<inverse>"
  "(f\<degree>)`(\<one>) = \<one>"
proof -
  from A1 A2 have I:
    "f: G\<^isub>+\<rightarrow>G"
    "\<forall>a\<in>\<sm>G\<^isub>+. (f`(a\<inverse>))\<inverse> \<in> G"
    "G\<^isub>+\<inter>(\<sm>G\<^isub>+) = 0"  
    "\<one> \<notin> G\<^isub>+\<union>(\<sm>G\<^isub>+)"
    "f\<degree> = f \<union> {\<langle>a, (f`(a\<inverse>))\<inverse>\<rangle>. a \<in> \<sm>G\<^isub>+} \<union> {\<langle>\<one>,\<one>\<rangle>}"
    using OrderedGroup_ZF_6_L2 OrdGroup_decomp2 OrderedGroup_ZF_6_L1
    by auto;
  then have "f\<degree>: G\<^isub>+ \<union> (\<sm>G\<^isub>+) \<union> {\<one>} \<rightarrow>G\<union>G\<union>{\<one>}"
    by (rule func1_1_L11E);
  moreover from A1 have 
    "G\<^isub>+ \<union> (\<sm>G\<^isub>+) \<union> {\<one>} = G"
    "G\<union>G\<union>{\<one>} = G"
    using OrdGroup_decomp2 OrderedGroup_ZF_1_L1 group0.group0_2_L2
    by auto;
  ultimately show "f\<degree> : G \<rightarrow> G" by simp;
  from I show "\<forall>a\<in>G\<^isub>+. (f\<degree>)`(a) = f`(a)"
    by (rule func1_1_L11E);
  from I show "\<forall>a\<in>(\<sm>G\<^isub>+). (f\<degree>)`(a) = (f`(a\<inverse>))\<inverse>"
    by (rule func1_1_L11E);
  from I show "(f\<degree>)`(\<one>) = \<one>"
    by (rule func1_1_L11E);
qed;

text{*Odd extensions are odd, of course.*}

lemma (in group3) oddext_is_odd:
  assumes A1: "r {is total on} G" and A2: "f: G\<^isub>+\<rightarrow>G"
  and A3: "a\<in>G"
  shows "(f\<degree>)`(a\<inverse>) = ((f\<degree>)`(a))\<inverse>"
proof -
  from A1 A3 have "a\<in>G\<^isub>+ \<or> a \<in> (\<sm>G\<^isub>+) \<or> a=\<one>"
    using OrdGroup_decomp2 by blast;
  moreover
  { assume "a\<in>G\<^isub>+"  
    with A1 A2 have "a\<inverse> \<in> \<sm>G\<^isub>+" and  "(f\<degree>)`(a) = f`(a)"
      using OrderedGroup_ZF_1_L25 odd_ext_props by auto;
    with A1 A2 have 
      "(f\<degree>)`(a\<inverse>) = (f`((a\<inverse>)\<inverse>))\<inverse>"  and "(f`(a))\<inverse> = ((f\<degree>)`(a))\<inverse>"
      using odd_ext_props by auto;
    with A3 have "(f\<degree>)`(a\<inverse>) = ((f\<degree>)`(a))\<inverse>"
      using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
      by simp } 
  moreover
  { assume A4: "a \<in> \<sm>G\<^isub>+"
    with A1 A2  have "a\<inverse> \<in> G\<^isub>+" and  "(f\<degree>)`(a) = (f`(a\<inverse>))\<inverse>"
      using OrderedGroup_ZF_1_L27 odd_ext_props
      by auto;
    with A1 A2 A4 have "(f\<degree>)`(a\<inverse>) = ((f\<degree>)`(a))\<inverse>"
      using odd_ext_props OrderedGroup_ZF_6_L2 
	OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
      by simp; }
  moreover
  { assume "a = \<one>" 
    with A1 A2 have "(f\<degree>)`(a\<inverse>) = ((f\<degree>)`(a))\<inverse>"
      using OrderedGroup_ZF_1_L1 group0.group_inv_of_one 
	odd_ext_props by simp;
  }
  ultimately show "(f\<degree>)`(a\<inverse>) = ((f\<degree>)`(a))\<inverse>"
    by auto;
qed;

text{*Another way of saying that odd extensions are odd.*}

lemma (in group3) oddext_is_odd_alt:
  assumes A1: "r {is total on} G" and A2: "f: G\<^isub>+\<rightarrow>G"
  and A3: "a\<in>G"
  shows "((f\<degree>)`(a\<inverse>))\<inverse> = (f\<degree>)`(a)"
proof -
  from A1 A2 have 
    "f\<degree> : G \<rightarrow> G"
    "\<forall>a\<in>G. (f\<degree>)`(a\<inverse>) = ((f\<degree>)`(a))\<inverse>"
    using odd_ext_props oddext_is_odd by auto
  then have "\<forall>a\<in>G. ((f\<degree>)`(a\<inverse>))\<inverse> = (f\<degree>)`(a)"
    using OrderedGroup_ZF_1_L1 group0.group0_6_L2 by simp;
  with A3 show "((f\<degree>)`(a\<inverse>))\<inverse> = (f\<degree>)`(a)" by simp;
qed;

section{*Functions with infinite limits*}

text{*In this section we consider functions $f: G\rightarrow G$ with the 
  property that for $f(x)$ is arbitrarily large for large enough $x$.
  More precisely, for every $a\in G$ there exist $b\in G_+$ such that 
  for every $x\geq b$ we have $f(x)\geq a$. In a sense this means that
  $\lim_{x\rightarrow \infty}f(x) = \infty$, hence the title of this section.
  We also prove dual statements for functions such that 
  $\lim_{x\rightarrow -\infty}f(x) = -\infty$. 
  *}

text{*If an image of a set by a function with infinite positive limit 
  is bounded above, then the set itself is bounded above.*}

lemma (in group3) OrderedGroup_ZF_7_L1:
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}" and
  A3: "f:G\<rightarrow>G" and 
  A4: "\<forall>a\<in>G.\<exists>b\<in>G\<^isub>+.\<forall>x. b\<lsq>x \<longrightarrow> a \<lsq> f`(x)" and 
  A5: "A\<subseteq>G" and 
  A6: "IsBoundedAbove(f``(A),r)"
  shows "IsBoundedAbove(A,r)"
proof -
  { assume "\<not>IsBoundedAbove(A,r)"
    then have I: "\<forall>u. \<exists>x\<in>A. \<not>(x\<lsq>u)"
      using IsBoundedAbove_def by auto;
    have "\<forall>a\<in>G. \<exists>y\<in>f``(A). a\<lsq>y"
    proof -
      { fix a assume "a\<in>G" 
	with A4 obtain b where 
	  II: "b\<in>G\<^isub>+" and III: "\<forall>x. b\<lsq>x \<longrightarrow> a \<lsq> f`(x)"
	  by auto;
	from I obtain x where IV: "x\<in>A" and "\<not>(x\<lsq>b)"
	  by auto;
	with A1 A5 II have 
	  "r {is total on} G"
	  "x\<in>G"  "b\<in>G"  "\<not>(x\<lsq>b)"
	  using PositiveSet_def by auto;
	with III have "a \<lsq> f`(x)"
	  using OrderedGroup_ZF_1_L8 by blast;
	with A3 A5 IV have "\<exists>y\<in>f``(A). a\<lsq>y"
	  using func_imagedef by auto;
      } thus ?thesis by simp
    qed
    with A1 A2 A6 have False using OrderedGroup_ZF_2_L2A
      by simp;
  } thus ?thesis by auto;
qed;

text{*If an image of a set defined by separation 
  by a function with infinite positive limit 
  is bounded above, then the set itself is bounded above.*}

lemma (in group3) OrderedGroup_ZF_7_L2:
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}" and
  A3: "X\<noteq>0" and A4: "f:G\<rightarrow>G" and 
  A5: "\<forall>a\<in>G.\<exists>b\<in>G\<^isub>+.\<forall>y. b\<lsq>y \<longrightarrow> a \<lsq> f`(y)" and 
  A6: "\<forall>x\<in>X. b(x) \<in> G \<and> f`(b(x)) \<lsq> U"
  shows "\<exists>u.\<forall>x\<in>X. b(x) \<lsq> u"
proof -
  let ?A = "{b(x). x\<in>X}"
  from A6 have I: "?A\<subseteq>G" by auto;
  moreover note prems 
  moreover have "IsBoundedAbove(f``(?A),r)"
  proof -
    from A4 A6 I have "\<forall>z\<in>f``(?A). \<langle>z,U\<rangle> \<in> r"
      using func_imagedef by simp;
    then show "IsBoundedAbove(f``(?A),r)"
      by (rule Order_ZF_3_L10);
  qed;
  ultimately have "IsBoundedAbove(?A,r)" using OrderedGroup_ZF_7_L1
    by simp;
  with A3 have "\<exists>u.\<forall>y\<in>?A. y \<lsq> u"
    using IsBoundedAbove_def by simp;
  then show "\<exists>u.\<forall>x\<in>X. b(x) \<lsq> u" by auto;
qed;

text{*If the image of a set defined by separation 
  by a function with infinite negative limit 
  is bounded below, then the set itself is bounded above.
  This is dual to @{text "OrderedGroup_ZF_7_L2"}.*}

lemma (in group3) OrderedGroup_ZF_7_L3:
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}" and
  A3: "X\<noteq>0" and A4: "f:G\<rightarrow>G" and 
  A5: "\<forall>a\<in>G.\<exists>b\<in>G\<^isub>+.\<forall>y. b\<lsq>y \<longrightarrow> f`(y\<inverse>) \<lsq> a" and 
  A6: "\<forall>x\<in>X. b(x) \<in> G \<and> L \<lsq> f`(b(x))"
  shows "\<exists>l.\<forall>x\<in>X. l \<lsq> b(x)"
proof -
  let ?g = "GroupInv(G,P) O f O GroupInv(G,P)"
  from ordGroupAssum have I: "GroupInv(G,P) : G\<rightarrow>G"
    using IsAnOrdGroup_def group0_2_T2 by simp;
  with A4 have II: "\<forall>x\<in>G. ?g`(x) = (f`(x\<inverse>))\<inverse>"
    using func1_1_L18 by simp;
  note A1 A2 A3
  moreover from A4 I have "?g : G\<rightarrow>G"
    using comp_fun by blast;
  moreover have "\<forall>a\<in>G.\<exists>b\<in>G\<^isub>+.\<forall>y. b\<lsq>y \<longrightarrow> a \<lsq> ?g`(y)"
  proof -
  { fix a assume A7: "a\<in>G"
    then have "a\<inverse> \<in> G"
      using OrderedGroup_ZF_1_L1 group0.inverse_in_group
      by simp;
    with A5 obtain b where 
      III: "b\<in>G\<^isub>+" and "\<forall>y. b\<lsq>y \<longrightarrow> f`(y\<inverse>) \<lsq> a\<inverse>"
      by auto;
    with II A7 have "\<forall>y. b\<lsq>y \<longrightarrow> a \<lsq> ?g`(y)"
      using OrderedGroup_ZF_1_L5AD OrderedGroup_ZF_1_L4
      by simp;
    with III have "\<exists>b\<in>G\<^isub>+.\<forall>y. b\<lsq>y \<longrightarrow> a \<lsq> ?g`(y)"
      by auto;
  } then show "\<forall>a\<in>G.\<exists>b\<in>G\<^isub>+.\<forall>y. b\<lsq>y \<longrightarrow> a \<lsq> ?g`(y)"
    by simp;
  qed;
  moreover have "\<forall>x\<in>X. b(x)\<inverse> \<in> G \<and> ?g`(b(x)\<inverse>) \<lsq> L\<inverse>"
  proof-
    { fix x assume "x\<in>X"
      with A6 have 
	T: "b(x) \<in> G"  "b(x)\<inverse> \<in> G" and "L \<lsq> f`(b(x))"
	using OrderedGroup_ZF_1_L1 group0.inverse_in_group
	by auto;
      then have "(f`(b(x)))\<inverse> \<lsq> L\<inverse>"
	using OrderedGroup_ZF_1_L5 by simp;
      moreover from II T have "(f`(b(x)))\<inverse> =  ?g`(b(x)\<inverse>)"
	using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
	by simp;
      ultimately have "?g`(b(x)\<inverse>) \<lsq> L\<inverse>" by simp;
      with T have "b(x)\<inverse> \<in> G \<and> ?g`(b(x)\<inverse>) \<lsq> L\<inverse>"
	by simp;
    } then show "\<forall>x\<in>X. b(x)\<inverse> \<in> G \<and> ?g`(b(x)\<inverse>) \<lsq> L\<inverse>"
      by simp;
  qed;
  ultimately have "\<exists>u.\<forall>x\<in>X. (b(x))\<inverse> \<lsq> u"
    by (rule OrderedGroup_ZF_7_L2);
  then have "\<exists>u.\<forall>x\<in>X. u\<inverse> \<lsq> (b(x)\<inverse>)\<inverse>"
    using OrderedGroup_ZF_1_L5 by auto;
  with A6 show "\<exists>l.\<forall>x\<in>X. l \<lsq> b(x)"
    using OrderedGroup_ZF_1_L1 group0.group_inv_of_inv
    by auto;
qed;

text{*The next lemma combines @{text "OrderedGroup_ZF_7_L2"} and 
  @{text "OrderedGroup_ZF_7_L3"} to show that if an image of a set 
  defined by separation by a function with infinite limits is bounded,
  then the set itself i bounded.*}

lemma (in group3) OrderedGroup_ZF_7_L4:
  assumes A1: "r {is total on} G" and A2: "G \<noteq> {\<one>}" and
  A3: "X\<noteq>0" and A4: "f:G\<rightarrow>G" and 
  A5: "\<forall>a\<in>G.\<exists>b\<in>G\<^isub>+.\<forall>y. b\<lsq>y \<longrightarrow> a \<lsq> f`(y)" and 
  A6: "\<forall>a\<in>G.\<exists>b\<in>G\<^isub>+.\<forall>y. b\<lsq>y \<longrightarrow> f`(y\<inverse>) \<lsq> a" and 
  A7: "\<forall>x\<in>X. b(x) \<in> G \<and> L \<lsq> f`(b(x)) \<and> f`(b(x)) \<lsq> U"
shows "\<exists>M.\<forall>x\<in>X. |b(x)| \<lsq> M"
proof -
  from A7 have 
    I: "\<forall>x\<in>X. b(x) \<in> G \<and> f`(b(x)) \<lsq> U" and
    II: "\<forall>x\<in>X. b(x) \<in> G \<and> L \<lsq> f`(b(x))"
    by auto;
  from A1 A2 A3 A4 A5 I have "\<exists>u.\<forall>x\<in>X. b(x) \<lsq> u"
    by (rule OrderedGroup_ZF_7_L2);
  moreover from  A1 A2 A3 A4 A6 II have "\<exists>l.\<forall>x\<in>X. l \<lsq> b(x)"
    by (rule OrderedGroup_ZF_7_L3);
  ultimately have "\<exists>u l. \<forall>x\<in>X. l\<lsq>b(x) \<and> b(x) \<lsq> u"
    by auto;
  with A1 have "\<exists>u l.\<forall>x\<in>X. |b(x)| \<lsq> GreaterOf(r,|l|,|u|)"
    using OrderedGroup_ZF_3_L10 by blast;
  then show "\<exists>M.\<forall>x\<in>X. |b(x)| \<lsq> M"
    by auto;
qed;

end
