(*   This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006, 2007  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Order\_ZF.thy}*}

theory Order_ZF imports Fol1 

begin

text{*This theory file considers various notion related to order. We redefine
  the notions of a total order, linear order and partial order to have the 
  same terminology as wikipedia (I found it very consistent across different 
  areas of math). We also define and study the notions of intervals and bounded
  sets. We show the inclusion relations between the intervals with endpoints
  being in certain order. We also show that union of bounded sets are bounded. 
  This allows to show in @{text "Finite_ZF.thy"} that finite sets are bounded.*}

section{*Definitions*}

text{*In this section we formulate the definitions related to order 
  relations.*}

text{*We define a linear order as a binary relation that is antisymmetric, 
  transitive and total. Note that this terminology is different than the
  one used the standard Order.thy file. The sets that are bounded below and 
  above are also defined, as are bounded sets. Empty sets are defined as 
  bounded.
  The notation for the definition of an interval may be mysterious for some
  readers, see lemma @{text "Order_ZF_2_L1"} for more intuitive notation. 
  We also define the maximum (the greater of) two elemnts and the minmum 
  (the smaller of) two elements. 
  We say that a set has a maximum (minimum) if it has an element that is 
  not smaller (not greater, resp.) that any other one. We show that
  under some conditions this element of the set is unique (if exists).
  The element with this
  property is called the maximum (minimum) of the set.
  The supremum of a set $A$ is defined as the minimum of the set of
  upper bounds, i.e. the set 
  $\{u.\forall_{a\in A} \langle a,u\rangle \in r\}=\bigcap_{a\in A} r\{a\}$. 
  Infimum is defined analogously. Recall that in Isabelle/ZF
  @{text "r-``(A)"} denotes the inverse image of the set $A$ by relation $r$
  (i.e. @{text "r-``(A)"}=$\{ x: \langle x,y\rangle\in r$ for some $y\in A\}$). 
  We define a relation to be complete if every nonempty bounded
  above set has a supremum.*}

constdefs 

  IsTotal (infixl "{is total on}" 65)
  "r {is total on} X \<equiv> (\<forall>a\<in>X.\<forall>b\<in>X. <a,b> \<in> r \<or> <b,a> \<in> r)"

  "IsLinOrder(X,r) \<equiv> ( antisym(r) \<and> trans(r) \<and> (r {is total on} X))"

  "IsPartOrder(X,r) \<equiv> (refl(X,r) \<and> antisym(r) \<and> trans(r))"

  "IsBoundedAbove(A,r) \<equiv> ( A=0 \<or> (\<exists>u. \<forall>x\<in>A. <x,u> \<in> r))"

  "IsBoundedBelow(A,r) \<equiv> (A=0 \<or> (\<exists>l. \<forall>x\<in>A. <l,x> \<in> r))"

  "IsBounded(A,r) \<equiv> (IsBoundedAbove(A,r) \<and> IsBoundedBelow(A,r))"

  "Interval(r,a,b) \<equiv> r``{a} \<inter> r-``{b}"

  "GreaterOf(r,a,b) \<equiv> (if <a,b> \<in> r then b else a)"

  "SmallerOf(r,a,b) \<equiv> (if <a,b> \<in> r then a else b)"

  "HasAmaximum(r,A) \<equiv> \<exists>M\<in>A.\<forall>x\<in>A. <x,M> \<in> r"

  "HasAminimum(r,A) \<equiv> \<exists>m\<in>A.\<forall>x\<in>A. <m,x> \<in> r"

  "Maximum(r,A) \<equiv> THE M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)"

  "Minimum(r,A) \<equiv> THE m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)"

  "Supremum(r,A) \<equiv> Minimum(r,\<Inter>a\<in>A. r``{a})"

  "Infimum(r,A) \<equiv> Maximum(r,\<Inter>a\<in>A. r-``{a})"

  IsComplete ("_ {is complete}")
  "r {is complete} \<equiv> 
  \<forall>A. IsBoundedAbove(A,r) \<and> A\<noteq>0 \<longrightarrow> HasAminimum(r,\<Inter>a\<in>A. r``{a})";

text{*The essential condition to show that a total relation is reflexive.*}

lemma Order_ZF_1_L1: assumes "r {is total on} X" and "a\<in>X"
  shows "<a,a> \<in> r" using prems IsTotal_def by auto;

text{*A total relation is reflexive.*}

lemma total_is_refl:
  assumes "r {is total on} X"
  shows "refl(X,r)" using prems Order_ZF_1_L1 refl_def by simp;

text{*A linear order is partial order.*}

lemma Order_ZF_1_L2: assumes "IsLinOrder(X,r)" 
  shows "IsPartOrder(X,r)" 
  using prems IsLinOrder_def IsPartOrder_def refl_def Order_ZF_1_L1
  by auto;

text{*Partial order that is total is linear.*}

lemma Order_ZF_1_L3: 
  assumes "IsPartOrder(X,r)" and "r {is total on} X"
  shows "IsLinOrder(X,r)"
  using prems IsPartOrder_def IsLinOrder_def
  by simp;

text{*Relation that is total on a set is total on any subset.*}

lemma Order_ZF_1_L4: assumes "r {is total on} X" and "A\<subseteq>X"
  shows "r {is total on} A"
  using prems IsTotal_def by auto;

text{*If the relation is total, then every set is a union of those elements
  that are nongreater than a given one and nonsmaller than a given one.*}

lemma Order_ZF_1_L5: 
  assumes "r {is total on} X" and "A\<subseteq>X" and "a\<in>X"
  shows "A = {x\<in>A. \<langle>x,a\<rangle> \<in> r} \<union> {x\<in>A. \<langle>a,x\<rangle> \<in> r}"
  using prems IsTotal_def by auto;

section{*Intervals*}

text{*In this section we discuss intervals.*}

text{*The next lemma explains the notation of the definition of an interval.*}

lemma Order_ZF_2_L1: 
  shows "x \<in> Interval(r,a,b) \<longleftrightarrow> <a,x> \<in> r \<and> <x,b> \<in> r"
  using Interval_def by auto;

text{*Since there are some problems with applying the above lemma 
  (seems that simp and auto don't handle equivalence very well), we
  split @{text "Order_ZF_2_L1"} into two lemmas.*}

lemma Order_ZF_2_L1A: assumes "x \<in> Interval(r,a,b)"
  shows "<a,x> \<in> r"  "<x,b> \<in> r"
  using prems  Order_ZF_2_L1 by auto;

text{*@{text "Order_ZF_2_L1"}, implication from right to left.*}

lemma Order_ZF_2_L1B: assumes "<a,x> \<in> r"  "<x,b> \<in> r"
  shows "x \<in> Interval(r,a,b)"
  using prems Order_ZF_2_L1 by simp;

text{*If the relation is reflexive, the endpoints belong to the interval.*}

lemma Order_ZF_2_L2: assumes "refl(X,r)" 
  and "a\<in>X"  "b\<in>X" and "<a,b> \<in> r"
  shows 
  "a \<in> Interval(r,a,b)"  
  "b \<in> Interval(r,a,b)"  
  using prems refl_def Order_ZF_2_L1 by auto;

text{*Under the assumptions of @{text "Order_ZF_2_L2"}, the interval is 
  nonempty. *}

lemma Order_ZF_2_L2A: assumes "refl(X,r)" 
  and "a\<in>X"  "b\<in>X" and "<a,b> \<in> r"
  shows "Interval(r,a,b) \<noteq> 0"
proof -
  from prems have "a \<in> Interval(r,a,b)"
    using Order_ZF_2_L2 by simp;
  then show "Interval(r,a,b) \<noteq> 0" by auto;
qed;

text{*If $a,b,c,d$ are in this order, then $[b,c]\subseteq [a,d]$. We
  only need trasitivity for this to be true.*}

lemma Order_ZF_2_L3: 
  assumes A1: "trans(r)" and A2:"<a,b>\<in>r"  "<b,c>\<in>r"  "<c,d>\<in>r"
shows "Interval(r,b,c) \<subseteq> Interval(r,a,d)"
proof;
  fix x assume A3: "x \<in> Interval(r, b, c)"
  from A1 have "trans(r)" .
  moreover from A2 A3 have "<a,b> \<in> r \<and> <b,x> \<in> r" using Order_ZF_2_L1A
    by simp;
  ultimately have T1: "<a,x> \<in> r" by (rule Fol1_L3);
  from A1 have "trans(r)" .
  moreover from A2 A3 have "<x,c> \<in> r \<and> <c,d> \<in> r" using Order_ZF_2_L1A
    by simp;
  ultimately have "<x,d> \<in> r" by (rule Fol1_L3);
  with T1 show "x \<in> Interval(r,a,d)" using Order_ZF_2_L1B
    by simp;
qed;

text{*For reflexive and antisymmetric relations the interval with equal 
  endpoints consists only of that endpoint.*}

lemma Order_ZF_2_L4: 
  assumes A1: "refl(X,r)" and A2: "antisym(r)" and A3: "a\<in>X"
  shows "Interval(r,a,a) = {a}"
proof;
  from A1 A3 have "<a,a> \<in> r" using refl_def by simp;
  with A1 A3 show "{a} \<subseteq> Interval(r,a,a)" using Order_ZF_2_L2 by simp;
  from A2 show "Interval(r,a,a) \<subseteq> {a}" using Order_ZF_2_L1A Fol1_L4
    by fast;
qed;

text{*For transitive relations the endpoints have to be in the relation for
  the interval to be nonempty.*}

lemma Order_ZF_2_L5: assumes A1: "trans(r)" and A2: "<a,b> \<notin> r"
  shows "Interval(r,a,b) = 0"
proof (rule ccontr);
  assume "Interval(r,a,b)\<noteq>0" then obtain x where "x \<in> Interval(r,a,b)"
    by auto;
  with A1 A2 show False using Order_ZF_2_L1A Fol1_L3 by fast;
qed;

text{*If a relation is defined on a set, then intervals are subsets of that
  set.*}

lemma Order_ZF_2_L6: assumes A1: "r \<subseteq> X\<times>X"
  shows "Interval(r,a,b) \<subseteq> X"
  using prems Interval_def by auto;

section{*Bounded sets*}

text{*In this section we consider properties of bounded sets.*}

text{*For reflexive relations singletons are bounded.*}

lemma Order_ZF_3_L1: assumes "refl(X,r)" and "a\<in>X"
  shows "IsBounded({a},r)"
  using prems refl_def IsBoundedAbove_def IsBoundedBelow_def
    IsBounded_def by auto;

text{*Sets that are bounded above are contained in the domain of 
  the relation.*}

lemma Order_ZF_3_L1A: assumes "r \<subseteq> X\<times>X" 
  and "IsBoundedAbove(A,r)"
  shows "A\<subseteq>X" using prems IsBoundedAbove_def by auto;


text{*Sets that are bounded below are contained in the domain of 
  the relation.*}

lemma Order_ZF_3_L1B: assumes "r \<subseteq> X\<times>X" 
  and "IsBoundedBelow(A,r)"
  shows "A\<subseteq>X" using prems IsBoundedBelow_def by auto;

text{*For a total relation, the greater of two elements, 
  as defined above, is indeed greater of any of the two.*}

lemma Order_ZF_3_L2: assumes "r {is total on} X"
  and "x\<in>X" "y\<in>X"
  shows 
  "\<langle>x,GreaterOf(r,x,y)\<rangle> \<in> r" 
  "\<langle>y,GreaterOf(r,x,y)\<rangle> \<in> r"
  "\<langle>SmallerOf(r,x,y),x\<rangle> \<in> r" 
  "\<langle>SmallerOf(r,x,y),y\<rangle> \<in> r"
  using prems IsTotal_def Order_ZF_1_L1 GreaterOf_def SmallerOf_def
  by auto;

text{*If $A$ is bounded above by $u$, $B$ is bounded above by $w$,
  then $A\cup B$ is bounded above by the greater of $u,w$.*}

lemma Order_ZF_3_L2B:  
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "u\<in>X" "w\<in>X" 
  and A4: "\<forall>x\<in>A. <x,u> \<in> r" "\<forall>x\<in>B. <x,w> \<in> r"
  shows "\<forall>x\<in>A\<union>B. \<langle>x,GreaterOf(r,u,w)\<rangle> \<in> r"
proof;
  let ?v = "GreaterOf(r,u,w)"
  from A1 A3 have T1: "<u,?v> \<in> r" and T2: "<w,?v> \<in> r"
    using Order_ZF_3_L2 by auto;
  fix x assume A5: "x\<in>A\<union>B" show "\<langle>x,?v\<rangle> \<in> r"
  proof (cases "x\<in>A");
    assume "x\<in>A"
    with A4 T1 have "<x,u> \<in> r \<and> <u,?v> \<in> r" by simp
    with A2 show "\<langle>x,?v\<rangle> \<in> r" by (rule Fol1_L3);
  next assume "x\<notin>A" 
    with A5 A4 T2 have "<x,w> \<in> r \<and> <w,?v> \<in> r" by simp;
    with A2 show "\<langle>x,?v\<rangle> \<in> r" by (rule Fol1_L3);
  qed;
qed;

text{*For total and transitive relation the union of two sets bounded 
  above is bounded above.*}

lemma Order_ZF_3_L3: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedAbove(A,r)" "IsBoundedAbove(B,r)"
  and A4: "r \<subseteq> X\<times>X"
  shows "IsBoundedAbove(A\<union>B,r)"
proof (cases "A=0 \<or> B=0");
  assume "A=0 \<or> B=0" 
  with A3 show ?thesis by auto;
next assume "\<not> (A = 0 \<or> B = 0)"
  then have T1: "A\<noteq>0" "B\<noteq>0" by auto;
  with A3 obtain u w where D1: "\<forall>x\<in>A. <x,u> \<in> r" "\<forall>x\<in>B. <x,w> \<in> r"
    using IsBoundedAbove_def by auto;
  let ?U = "GreaterOf(r,u,w)"
  from T1 A4 D1 have "u\<in>X" "w\<in>X" by auto;
  with A1 A2 D1 have "\<forall>x\<in>A\<union>B.<x,?U> \<in> r"
    using Order_ZF_3_L2B by blast;
  then show "IsBoundedAbove(A\<union>B,r)"
    using IsBoundedAbove_def by auto;
qed;
  
text{*For total and transitive relations if a set $A$ is bounded above then 
  $A\cup \{a\}$ is bounded above.*}

lemma Order_ZF_3_L4: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedAbove(A,r)" and A4: "a\<in>X" and A5: "r \<subseteq> X\<times>X"
  shows "IsBoundedAbove(A\<union>{a},r)"
proof -
  from A1 have "refl(X,r)"
    using total_is_refl by simp;
  with prems show ?thesis using
    Order_ZF_3_L1 IsBounded_def Order_ZF_3_L3 by simp;
qed;

text{*If $A$ is bounded below by $l$, $B$ is bounded below by $m$,
  then $A\cup B$ is bounded below by the smaller of $u,w$.*}

lemma Order_ZF_3_L5B:  
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "l\<in>X" "m\<in>X" 
  and A4: "\<forall>x\<in>A. <l,x> \<in> r" "\<forall>x\<in>B. <m,x> \<in> r"
  shows "\<forall>x\<in>A\<union>B. \<langle>SmallerOf(r,l,m),x\<rangle> \<in> r"
proof;
  let ?k = "SmallerOf(r,l,m)"
  from A1 A3 have T1: "<?k,l> \<in> r" and T2: "<?k,m> \<in> r"
    using Order_ZF_3_L2 by auto;
  fix x assume A5: "x\<in>A\<union>B" show "\<langle>?k,x\<rangle> \<in> r"
  proof (cases "x\<in>A");
    assume "x\<in>A"
    with A4 T1 have "<?k,l> \<in> r \<and> <l,x> \<in> r" by simp
    with A2 show "\<langle>?k,x\<rangle> \<in> r" by (rule Fol1_L3);
  next assume "x\<notin>A" 
    with A5 A4 T2 have "<?k,m> \<in> r \<and> <m,x> \<in> r" by simp;
    with A2 show "\<langle>?k,x\<rangle> \<in> r" by (rule Fol1_L3);
  qed;
qed;

text{*For total and transitive relation the union of two sets bounded 
  below is bounded below.*}

lemma Order_ZF_3_L6: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedBelow(A,r)" "IsBoundedBelow(B,r)"
  and A4: "r \<subseteq> X\<times>X"
  shows "IsBoundedBelow(A\<union>B,r)"
proof (cases "A=0 \<or> B=0");
  assume "A=0 \<or> B=0" 
  with A3 show ?thesis by auto;
next assume "\<not> (A = 0 \<or> B = 0)"
  then have T1: "A\<noteq>0" "B\<noteq>0" by auto;
  with A3 obtain l m where D1: "\<forall>x\<in>A. <l,x> \<in> r" "\<forall>x\<in>B. <m,x> \<in> r"
    using IsBoundedBelow_def by auto;
  let ?L = "SmallerOf(r,l,m)"
  from T1 A4 D1 have T1: "l\<in>X" "m\<in>X" by auto;
  with A1 A2 D1 have "\<forall>x\<in>A\<union>B.<?L,x> \<in> r"
    using Order_ZF_3_L5B by blast;
  then show "IsBoundedBelow(A\<union>B,r)"
    using IsBoundedBelow_def by auto;
qed;

text{*For total and transitive relations if a set $A$ is bounded below then 
  $A\cup \{a\}$ is bounded below.*}

lemma Order_ZF_3_L7:
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedBelow(A,r)" and A4: "a\<in>X" and A5: "r \<subseteq> X\<times>X"
  shows "IsBoundedBelow(A\<union>{a},r)"
proof -
  from A1 have "refl(X,r)"
    using total_is_refl by simp;
  with prems show ?thesis using
    Order_ZF_3_L1 IsBounded_def Order_ZF_3_L6 by simp;
qed;

text{*For total and transitive relations unions of two bounded sets are 
  bounded.*}

theorem Order_ZF_3_T1: 
  assumes "r {is total on} X" and "trans(r)" 
  and "IsBounded(A,r)" "IsBounded(B,r)"
  and "r \<subseteq> X\<times>X"
  shows "IsBounded(A\<union>B,r)"
  using prems Order_ZF_3_L3 Order_ZF_3_L6 Order_ZF_3_L7 IsBounded_def
  by simp;

text{*For total and transitive relations if a set $A$ is bounded then 
  $A\cup \{a\}$ is bounded.*}

lemma Order_ZF_3_L8: 
  assumes "r {is total on} X" and "trans(r)"
  and "IsBounded(A,r)" and "a\<in>X" and "r \<subseteq> X\<times>X"
  shows "IsBounded(A\<union>{a},r)"
  using prems total_is_refl Order_ZF_3_L1 Order_ZF_3_T1 by blast;

text{*A sufficient condition for a set to be bounded below.*}

lemma Order_ZF_3_L9: assumes A1: "\<forall>a\<in>A. \<langle>l,a\<rangle> \<in> r"
  shows "IsBoundedBelow(A,r)"
proof -
  from A1 have "\<exists>l. \<forall>x\<in>A. \<langle>l,x\<rangle> \<in> r"
    by auto;
  then show "IsBoundedBelow(A,r)"
    using IsBoundedBelow_def by simp;
qed;

text{*A sufficient condition for a set to be bounded above.*}

lemma Order_ZF_3_L10: assumes A1: "\<forall>a\<in>A. \<langle>a,u\<rangle> \<in> r"
  shows "IsBoundedAbove(A,r)"
proof -
  from A1 have "\<exists>u. \<forall>x\<in>A. \<langle>x,u\<rangle> \<in> r"
    by auto;
  then show "IsBoundedAbove(A,r)"
    using IsBoundedAbove_def by simp;
qed;

text{*Intervals are bounded. *}
(*proof that uses Order_ZF_3_L9 and Order_ZF_3_L10 and is not shorter *)
lemma Order_ZF_3_L11: shows 
  "IsBoundedAbove(Interval(r,a,b),r)"
  "IsBoundedBelow(Interval(r,a,b),r)"
  "IsBounded(Interval(r,a,b),r)"
proof -
  { fix x assume "x \<in> Interval(r,a,b)" 
    then have "<x,b> \<in> r"  "<a,x> \<in> r" 
      using Order_ZF_2_L1A by auto
  } then have 
      "\<exists>u. \<forall>x\<in>Interval(r,a,b). <x,u> \<in> r" 
      "\<exists>l. \<forall>x\<in>Interval(r,a,b). <l,x> \<in> r"
    by auto;
  then show 
    "IsBoundedAbove(Interval(r,a,b),r)"
    "IsBoundedBelow(Interval(r,a,b),r)"
    "IsBounded(Interval(r,a,b),r)"
    using IsBoundedAbove_def IsBoundedBelow_def IsBounded_def
    by auto;
qed;

text{*A subset of a set that is bounded below is bounded below.*}

lemma Order_ZF_3_L12: assumes A1: "IsBoundedBelow(A,r)" and A2: "B\<subseteq>A"
  shows "IsBoundedBelow(B,r)"
proof -
  { assume "A = 0"
    with assms have "IsBoundedBelow(B,r)" 
      using IsBoundedBelow_def by auto }
  moreover
  { assume "A \<noteq> 0"
    with A1 have "\<exists>l. \<forall>x\<in>A. \<langle>l,x\<rangle> \<in> r"
      using IsBoundedBelow_def by simp
    with A2 have "\<exists>l.\<forall>x\<in>B. \<langle>l,x\<rangle> \<in> r" by auto;
    then have "IsBoundedBelow(B,r)" using IsBoundedBelow_def
      by auto; }
  ultimately show "IsBoundedBelow(B,r)" by auto;
qed;

text{*A subset of a set that is bounded above is bounded above.*}

lemma Order_ZF_3_L13: assumes A1: "IsBoundedAbove(A,r)" and A2: "B\<subseteq>A"
  shows "IsBoundedAbove(B,r)"
proof -
  { assume "A = 0"
    with assms have "IsBoundedAbove(B,r)" 
      using IsBoundedAbove_def by auto }
  moreover
  { assume "A \<noteq> 0"
    with A1 have "\<exists>u. \<forall>x\<in>A. \<langle>x,u\<rangle> \<in> r"
      using IsBoundedAbove_def by simp;
    with A2 have "\<exists>u.\<forall>x\<in>B. \<langle>x,u\<rangle> \<in> r" by auto;
    then have "IsBoundedAbove(B,r)" using IsBoundedAbove_def
      by auto; }
  ultimately show "IsBoundedAbove(B,r)" by auto;
qed;

text{*If for every element of $X$ we can find one in $A$ 
  that is greater, then the $A$ can not be bounded above.
  Works for relations that are total, transitive and antisymmetric.*}

lemma Order_ZF_3_L14:  
  assumes A1: "r {is total on} X" 
  and A2: "trans(r)" and A3: "antisym(r)"
  and A4: "r \<subseteq> X\<times>X" and A5: "X\<noteq>0" 
  and A6: "\<forall>x\<in>X. \<exists>a\<in>A. x\<noteq>a \<and> \<langle>x,a\<rangle> \<in> r"
  shows "\<not>IsBoundedAbove(A,r)";
proof -
  { from A5 A6 have I: "A\<noteq>0" by auto
    moreover assume "IsBoundedAbove(A,r)"
    ultimately obtain u where II: "\<forall>x\<in>A. <x,u> \<in> r"
      using IsBounded_def IsBoundedAbove_def by auto;
    with A4 I have "u\<in>X" by auto;
    with A6 obtain b where "b\<in>A" and III: "u\<noteq>b" and "\<langle>u,b\<rangle> \<in> r"
      by auto;
    with II have "\<langle>b,u\<rangle> \<in> r"  "\<langle>u,b\<rangle> \<in> r" by auto;
    with A3 have "b=u" by (rule Fol1_L4);
    with III have False by simp;
  } thus "\<not>IsBoundedAbove(A,r)" by auto;
qed;

text{*The set of elements in a set $A$ that are nongreater than 
  a given element is bounded above.*}

lemma Order_ZF_3_L15: shows "IsBoundedAbove({x\<in>A. \<langle>x,a\<rangle> \<in> r},r)"
  using IsBoundedAbove_def by auto;

text{*If $A$ is bounded below, then the set of elements in a set $A$ 
  that are nongreater than a given element is bounded. *}

lemma Order_ZF_3_L16: assumes A1: "IsBoundedBelow(A,r)"
  shows "IsBounded({x\<in>A. \<langle>x,a\<rangle> \<in> r},r)"
proof (cases "A=0")
  assume "A=0" 
  then show "IsBounded({x\<in>A. \<langle>x,a\<rangle> \<in> r},r)"
    using IsBoundedBelow_def IsBoundedAbove_def IsBounded_def
    by auto;
next assume "A\<noteq>0"
  with A1 obtain l where I: "\<forall>x\<in>A. \<langle>l,x\<rangle> \<in> r"
    using IsBoundedBelow_def by auto;
  then have "\<forall>y\<in>{x\<in>A. \<langle>x,a\<rangle> \<in> r}. \<langle>l,y\<rangle> \<in> r" by simp;
  then have "IsBoundedBelow({x\<in>A. \<langle>x,a\<rangle> \<in> r},r)"
    by (rule Order_ZF_3_L9);
  then show "IsBounded({x\<in>A. \<langle>x,a\<rangle> \<in> r},r)" 
    using Order_ZF_3_L15 IsBounded_def by simp;
qed;

section{*Maximum and minimum of a set*}

text{*In this section we show that maximum and minimum are unique if they 
  exist. We also show that union of sets that have maxima (minima) has a 
  maximum (minimum). We also show that singletons have maximum and minimum.
  All this allows to show (in Finite\_ZF.thy) that every finite set has 
  well-defined maximum and minimum. *}

text{*For antisymmetric relations maximum of a set is unique if it exists.*}

lemma Order_ZF_4_L1: assumes A1: "antisym(r)" and A2: "HasAmaximum(r,A)"
  shows "\<exists>!M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)"
proof;
  from A2 show "\<exists>M. M \<in> A \<and> (\<forall>x\<in>A. \<langle>x, M\<rangle> \<in> r)"
    using HasAmaximum_def by auto;
  fix M1 M2 assume 
    A2: "M1 \<in> A \<and> (\<forall>x\<in>A. \<langle>x, M1\<rangle> \<in> r)" "M2 \<in> A \<and> (\<forall>x\<in>A. \<langle>x, M2\<rangle> \<in> r)"
    then have "\<langle>M1,M2\<rangle> \<in> r" "\<langle>M2,M1\<rangle> \<in> r" by auto;
    with A1 show "M1=M2" by (rule Fol1_L4);
qed;

text{*For antisymmetric relations minimum of a set is unique if it exists.*}

lemma Order_ZF_4_L2: assumes A1: "antisym(r)" and A2: "HasAminimum(r,A)"
  shows "\<exists>!m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)"
proof;
  from A2 show "\<exists>m. m \<in> A \<and> (\<forall>x\<in>A. \<langle>m, x\<rangle> \<in> r)"
    using HasAminimum_def by auto;
  fix m1 m2 assume 
    A2: "m1 \<in> A \<and> (\<forall>x\<in>A. \<langle>m1, x\<rangle> \<in> r)" "m2 \<in> A \<and> (\<forall>x\<in>A. \<langle>m2, x\<rangle> \<in> r)"
    then have "\<langle>m1,m2\<rangle> \<in> r" "\<langle>m2,m1\<rangle> \<in> r" by auto;
    with A1 show "m1=m2" by (rule Fol1_L4);
qed;

text{*Maximum of a set has desired properties. *}

lemma Order_ZF_4_L3: assumes A1: "antisym(r)" and A2: "HasAmaximum(r,A)"
  shows "Maximum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>x,Maximum(r,A)\<rangle> \<in> r"
proof -; 
  let ?Max = "THE M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)" 
  from A1 A2 have "\<exists>!M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)"
    by (rule Order_ZF_4_L1);
  then have "?Max \<in> A \<and> (\<forall>x\<in>A. <x,?Max> \<in> r)"
    by (rule theI);
  then show "Maximum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>x,Maximum(r,A)\<rangle> \<in> r"
    using Maximum_def by auto;
qed;
  
text{*Minimum of a set has desired properties.*}
    
lemma Order_ZF_4_L4: assumes A1: "antisym(r)" and A2: "HasAminimum(r,A)"
  shows "Minimum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>Minimum(r,A),x\<rangle> \<in> r"
proof -; 
  let ?Min = "THE m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)" 
  from A1 A2 have "\<exists>!m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)"
    by (rule Order_ZF_4_L2);
  then have "?Min \<in> A \<and> (\<forall>x\<in>A. <?Min,x> \<in> r)"
    by (rule theI);
  then show "Minimum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>Minimum(r,A),x\<rangle> \<in> r"
    using Minimum_def by auto;
qed;

text{*For total and transitive relations a union a of two sets that have 
  maxima has a maximum.*}

lemma Order_ZF_4_L5: 
  assumes A1: "r {is total on} (A\<union>B)" and A2: "trans(r)"
  and A3: "HasAmaximum(r,A)" "HasAmaximum(r,B)"
  shows "HasAmaximum(r,A\<union>B)"
proof -;
  from A3 obtain M K where 
    D1: "M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)" "K\<in>B \<and> (\<forall>x\<in>B. <x,K> \<in> r)" 
    using HasAmaximum_def by auto
  let ?L = "GreaterOf(r,M,K)"
  from D1 have T1: "M \<in> A\<union>B" "K \<in> A\<union>B" 
    "\<forall>x\<in>A. <x,M> \<in> r" "\<forall>x\<in>B. <x,K> \<in> r"
    by auto;
  with A1 A2 have "\<forall>x\<in>A\<union>B.<x,?L> \<in> r" by (rule Order_ZF_3_L2B);
  moreover from T1 have "?L \<in> A\<union>B" using GreaterOf_def IsTotal_def 
    by simp;
  ultimately show "HasAmaximum(r,A\<union>B)" using HasAmaximum_def by auto;
qed;

text{*For total and transitive relations A union a of two sets that have 
  minima has a minimum.*}

lemma Order_ZF_4_L6: 
  assumes A1: "r {is total on} (A\<union>B)" and A2: "trans(r)"
  and A3: "HasAminimum(r,A)" "HasAminimum(r,B)"
  shows "HasAminimum(r,A\<union>B)"
proof -;
  from A3 obtain m k where 
    D1: "m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)" "k\<in>B \<and> (\<forall>x\<in>B. <k,x> \<in> r)" 
    using HasAminimum_def by auto
  let ?l = "SmallerOf(r,m,k)"
  from D1 have T1: "m \<in> A\<union>B" "k \<in> A\<union>B" 
    "\<forall>x\<in>A. <m,x> \<in> r" "\<forall>x\<in>B. <k,x> \<in> r"
    by auto;
  with A1 A2 have "\<forall>x\<in>A\<union>B.<?l,x> \<in> r" by (rule Order_ZF_3_L5B);
  moreover from T1 have "?l \<in> A\<union>B" using SmallerOf_def IsTotal_def 
    by simp;
  ultimately show "HasAminimum(r,A\<union>B)" using HasAminimum_def by auto;
qed;

text{*Set that has a maximum is bounded above.*}

lemma Order_ZF_4_L7:
  assumes "HasAmaximum(r,A)"
  shows "IsBoundedAbove(A,r)"
  using prems HasAmaximum_def IsBoundedAbove_def by auto;

text{*Set that has a minimum is bounded below.*}

lemma Order_ZF_4_L8A:
  assumes "HasAminimum(r,A)"
  shows "IsBoundedBelow(A,r)"
  using prems HasAminimum_def IsBoundedBelow_def by auto;


text{*For reflexive relations singletons have a minimum and maximum.*}

lemma Order_ZF_4_L8: assumes "refl(X,r)" and "a\<in>X"
  shows "HasAmaximum(r,{a})" "HasAminimum(r,{a})"
  using prems refl_def HasAmaximum_def HasAminimum_def by auto;

text{*For total and transitive relations if we add an element to a set 
  that has a maximum, the set still has a maximum.*}

lemma Order_ZF_4_L9: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "A\<subseteq>X" and A4: "a\<in>X" and A5: "HasAmaximum(r,A)"
  shows "HasAmaximum(r,A\<union>{a})"
proof -;
  from A3 A4 have "A\<union>{a} \<subseteq> X" by auto;
  with A1 have "r {is total on} (A\<union>{a})"
    using Order_ZF_1_L4 by blast;
  moreover from A1 A2 A4 A5 have
    "trans(r)" "HasAmaximum(r,A)" by auto;
  moreover from A1 A4 have "HasAmaximum(r,{a})"
    using total_is_refl Order_ZF_4_L8 by blast;
  ultimately show "HasAmaximum(r,A\<union>{a})" by (rule Order_ZF_4_L5);
qed;

text{*For total and transitive relations if we add an element to a set 
  that has a minimum, the set still has a minimum.*}

lemma Order_ZF_4_L10: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "A\<subseteq>X" and A4: "a\<in>X" and A5: "HasAminimum(r,A)"
  shows "HasAminimum(r,A\<union>{a})"
proof -;
  from A3 A4 have "A\<union>{a} \<subseteq> X" by auto;
  with A1 have "r {is total on} (A\<union>{a})"
    using Order_ZF_1_L4 by blast;
  moreover from A1 A2 A4 A5 have
    "trans(r)" "HasAminimum(r,A)" by auto;
  moreover from A1 A4 have "HasAminimum(r,{a})"
    using total_is_refl Order_ZF_4_L8 by blast;
  ultimately show "HasAminimum(r,A\<union>{a})" by (rule Order_ZF_4_L6);
qed;

text{*If the order relation has a property that every nonempty bounded set 
  attains a minimum (for example integers are like that), 
  then every nonempty set bounded below attains a minimum.*}

lemma Order_ZF_4_L11: 
  assumes A1: "r {is total on} X" and 
  A2: "trans(r)" and 
  A3: "r \<subseteq> X\<times>X" and
  A4: "\<forall>A. IsBounded(A,r) \<and> A\<noteq>0 \<longrightarrow> HasAminimum(r,A)" and 
  A5: "B\<noteq>0" and A6: "IsBoundedBelow(B,r)"
  shows "HasAminimum(r,B)"
proof -
  from A5 obtain b where T: "b\<in>B" by auto
  let ?L = "{x\<in>B. \<langle>x,b\<rangle> \<in> r}"
  from A3 A6 T have T1: "b\<in>X" using Order_ZF_3_L1B by blast;
  with A1 T have T2: "b \<in> ?L"
    using total_is_refl refl_def by simp;
  then have "?L \<noteq> 0" by auto;
  moreover have "IsBounded(?L,r)"
  proof -
    have "?L \<subseteq> B" by auto
    with A6 have "IsBoundedBelow(?L,r)"
      using Order_ZF_3_L12 by simp;
    moreover have "IsBoundedAbove(?L,r)"
      by (rule Order_ZF_3_L15);
    ultimately have "IsBoundedAbove(?L,r) \<and> IsBoundedBelow(?L,r)"
      by blast;
    then show "IsBounded(?L,r)" using IsBounded_def
      by simp;
  qed;
  ultimately have "IsBounded(?L,r) \<and> ?L \<noteq> 0" by blast;
  with A4 have "HasAminimum(r,?L)" by simp;
  then obtain m where I: "m\<in>?L" and II: "\<forall>x\<in>?L. <m,x> \<in> r" 
    using HasAminimum_def by auto;
  then have III: "\<langle>m,b\<rangle> \<in> r" by simp;
  from I have "m\<in>B" by simp;
  moreover have "\<forall>x\<in>B. \<langle>m,x\<rangle> \<in> r"
  proof
    fix x assume A7: "x\<in>B"
    from A3 A6 have "B\<subseteq>X" using Order_ZF_3_L1B by blast;
    with A1 A7 T1 have "x \<in>  ?L \<union> {x\<in>B. \<langle>b,x\<rangle> \<in> r}"
      using Order_ZF_1_L5 by simp;
    then have "x\<in>?L \<or> \<langle>b,x\<rangle> \<in> r" by auto;
    moreover
    { assume "x\<in>?L"
      with II have "\<langle>m,x\<rangle> \<in> r" by simp };
    moreover
    { assume "\<langle>b,x\<rangle> \<in> r"
      with A2 III have "trans(r)" and "\<langle>m,b\<rangle> \<in> r \<and> \<langle>b,x\<rangle> \<in> r"
	by auto;
      then have  "\<langle>m,x\<rangle> \<in> r" by (rule Fol1_L3) }
    ultimately show "\<langle>m,x\<rangle> \<in> r" by auto;
  qed;
  ultimately show "HasAminimum(r,B)" using HasAminimum_def
    by auto;
qed;

text{*A dual to @{text "Order_ZF_4_L11"}: 
  If the order relation has a property that every nonempty bounded set 
  attains a maximum (for example integers are like that), 
  then every nonempty set bounded above attains a maximum.*}

lemma Order_ZF_4_L11A: 
  assumes A1: "r {is total on} X" and 
  A2: "trans(r)" and 
  A3: "r \<subseteq> X\<times>X" and
  A4: "\<forall>A. IsBounded(A,r) \<and> A\<noteq>0 \<longrightarrow> HasAmaximum(r,A)" and 
  A5: "B\<noteq>0" and A6: "IsBoundedAbove(B,r)"
  shows "HasAmaximum(r,B)"
proof -
  from A5 obtain b where T: "b\<in>B" by auto
  let ?U = "{x\<in>B. \<langle>b,x\<rangle> \<in> r}"
  from A3 A6 T have T1: "b\<in>X" using Order_ZF_3_L1A by blast;
  with A1 T have T2: "b \<in> ?U"
    using total_is_refl refl_def by simp;
  then have "?U \<noteq> 0" by auto;
  moreover have "IsBounded(?U,r)"
  proof -
    have "?U \<subseteq> B" by auto
    with A6 have "IsBoundedAbove(?U,r)"
      using Order_ZF_3_L13 by blast;
    moreover have "IsBoundedBelow(?U,r)"
      using IsBoundedBelow_def by auto;
    ultimately have "IsBoundedAbove(?U,r) \<and> IsBoundedBelow(?U,r)"
      by blast;
    then show "IsBounded(?U,r)" using IsBounded_def
      by simp;
  qed;
  ultimately have "IsBounded(?U,r) \<and> ?U \<noteq> 0" by blast;
  with A4 have "HasAmaximum(r,?U)" by simp;
  then obtain m where I: "m\<in>?U" and II: "\<forall>x\<in>?U. \<langle>x,m\<rangle> \<in> r"
    using HasAmaximum_def by auto;
  then have III: "\<langle>b,m\<rangle> \<in> r" by simp;
  from I have "m\<in>B" by simp;
  moreover have "\<forall>x\<in>B. \<langle>x,m\<rangle> \<in> r"
  proof
    fix x assume A7: "x\<in>B"
    from A3 A6 have "B\<subseteq>X" using Order_ZF_3_L1A by blast;
    with A1 A7 T1 have "x \<in> {x\<in>B. \<langle>x,b\<rangle> \<in> r} \<union> ?U"
      using Order_ZF_1_L5 by simp;
    then have "x\<in>?U \<or> \<langle>x,b\<rangle> \<in> r" by auto;
    moreover
    { assume "x\<in>?U"
      with II have "\<langle>x,m\<rangle> \<in> r" by simp };
    moreover
    { assume "\<langle>x,b\<rangle> \<in> r"
      with A2 III have "trans(r)" and "\<langle>x,b\<rangle> \<in> r \<and> \<langle>b,m\<rangle> \<in> r"
	by auto;
      then have  "\<langle>x,m\<rangle> \<in> r" by (rule Fol1_L3) }
    ultimately show "\<langle>x,m\<rangle> \<in> r" by auto;
  qed
  ultimately show "HasAmaximum(r,B)" using HasAmaximum_def
    by auto;
qed;

text{*If a set has a minimum and $L$ is less or equal than 
  all elements of the set, then $L$ is less or equal than the minimum.*}

lemma Order_ZF_4_L12: 
  assumes "antisym(r)" and "HasAminimum(r,A)" and "\<forall>a\<in>A. \<langle>L,a\<rangle> \<in> r"
  shows "\<langle>L,Minimum(r,A)\<rangle> \<in> r"
  using prems Order_ZF_4_L4 by simp;


text{*If a set has a maximum and all its elements are less or equal than 
  $M$, then the maximum of the set is less or equal than $M$.*}

lemma Order_ZF_4_L13: 
  assumes "antisym(r)" and "HasAmaximum(r,A)" and "\<forall>a\<in>A. \<langle>a,M\<rangle> \<in> r"
  shows "\<langle>Maximum(r,A),M\<rangle> \<in> r"
  using prems Order_ZF_4_L3 by simp;

text{*If an element belongs to a set and is greater or equal
  than all elements of that set, then it is the maximum of that set.*}

lemma Order_ZF_4_L14: 
  assumes A1: "antisym(r)" and A2: "M \<in> A" and 
  A3: "\<forall>a\<in>A. \<langle>a,M\<rangle> \<in> r"
  shows "Maximum(r,A) = M"
proof -
  from A2 A3 have I: "HasAmaximum(r,A)" using HasAmaximum_def
    by auto;
  with A1 have "\<exists>!M. M\<in>A \<and> (\<forall>x\<in>A. \<langle>x,M\<rangle> \<in> r)"
    using Order_ZF_4_L1 by simp;
  moreover from A2 A3 have "M\<in>A \<and> (\<forall>x\<in>A. \<langle>x,M\<rangle> \<in> r)" by simp;
  moreover from A1 I have 
    "Maximum(r,A) \<in> A \<and> (\<forall>x\<in>A. \<langle>x,Maximum(r,A)\<rangle> \<in> r)"
    using Order_ZF_4_L3 by simp;
  ultimately show "Maximum(r,A) = M" by auto;
qed;

text{*If an element belongs to a set and is less or equal
  than all elements of that set, then it is the minimum of that set.*}

lemma Order_ZF_4_L15: 
  assumes A1: "antisym(r)" and A2: "m \<in> A" and 
  A3: "\<forall>a\<in>A. \<langle>m,a\<rangle> \<in> r"
  shows "Minimum(r,A) = m"
proof -
  from A2 A3 have I: "HasAminimum(r,A)" using HasAminimum_def
    by auto;
  with A1 have "\<exists>!m. m\<in>A \<and> (\<forall>x\<in>A. \<langle>m,x\<rangle> \<in> r)"
    using Order_ZF_4_L2 by simp;
  moreover from A2 A3 have "m\<in>A \<and> (\<forall>x\<in>A. \<langle>m,x\<rangle> \<in> r)" by simp;
  moreover from A1 I have 
    "Minimum(r,A) \<in> A \<and> (\<forall>x\<in>A. \<langle>Minimum(r,A),x\<rangle> \<in> r)"
    using Order_ZF_4_L4 by simp;
  ultimately show "Minimum(r,A) = m" by auto;
qed;

text{*If a set does not have a maximum, then for any its element we can
  find one that is (strictly) greater.*}

lemma Order_ZF_4_L16: 
  assumes A1: "antisym(r)" and A2: "r {is total on} X" and 
  A3: "A\<subseteq>X" and
  A4: "\<not>HasAmaximum(r,A)" and 
  A5: "x\<in>A"
  shows "\<exists>y\<in>A. \<langle>x,y\<rangle> \<in> r \<and> y\<noteq>x"
proof -
  { assume A6: "\<forall>y\<in>A. \<langle>x,y\<rangle> \<notin> r \<or> y=x"
    have "\<forall>y\<in>A. \<langle>y,x\<rangle> \<in> r"
    proof
      fix y assume A7: "y\<in>A"
      with A6 have "\<langle>x,y\<rangle> \<notin> r \<or> y=x" by simp
      with A2 A3 A5 A7 show "\<langle>y,x\<rangle> \<in> r"
	using IsTotal_def Order_ZF_1_L1 by auto;
    qed
    with A5 have "\<exists>x\<in>A.\<forall>y\<in>A. \<langle>y,x\<rangle> \<in> r"
      by auto;
    with A4 have False using HasAmaximum_def by simp;
  } then show "\<exists>y\<in>A. \<langle>x,y\<rangle> \<in> r \<and> y\<noteq>x" by auto;
qed;


section{*Supremum and Infimum*}

text{*In this section we consider the notions of supremum and infimum a set. *}

text{*Elements of the set of upper bounds are indeed upper bounds.
  Isabelle also thinks it is obvious.*}

lemma Order_ZF_5_L1: assumes "u \<in> (\<Inter>a\<in>A. r``{a})" and "a\<in>A"
  shows "\<langle>a,u\<rangle> \<in> r"
  using prems by auto;

text{*Elements of the set of lower bounds are indeed lower bounds.
  Isabelle also thinks it is obvious.*}
  
lemma Order_ZF_5_L2: assumes "l \<in> (\<Inter>a\<in>A. r-``{a})" and "a\<in>A"
  shows "\<langle>l,a\<rangle> \<in> r"
  using prems by auto;

text{*If the set of upper bounds has a minimum, then the supremum 
  is less or equal than any upper bound. We can probably do away with
  the assumption that $A$ is not empty, (ab)using the fact that 
  intersection over an empty family is defined in Isabelle to be empty.*}

lemma Order_ZF_5_L3: assumes A1: "antisym(r)" and A2: "A\<noteq>0" and
  A3: "HasAminimum(r,\<Inter>a\<in>A. r``{a})" and 
  A4: "\<forall>a\<in>A. \<langle>a,u\<rangle> \<in> r"
  shows "\<langle>Supremum(r,A),u\<rangle> \<in> r"
proof -
  let ?U = "\<Inter>a\<in>A. r``{a}"
  from A4 have "\<forall>a\<in>A. u \<in> r``{a}" using image_singleton_iff
    by simp;
  with A2 have "u\<in>?U" by auto;
  with A1 A3 show "\<langle>Supremum(r,A),u\<rangle> \<in> r"
    using Order_ZF_4_L4 Supremum_def by simp;
qed;

text{*Infimum is greater or equal than any lower bound.*}

lemma Order_ZF_5_L4: assumes A1: "antisym(r)" and A2: "A\<noteq>0" and
  A3: "HasAmaximum(r,\<Inter>a\<in>A. r-``{a})" and 
  A4: "\<forall>a\<in>A. \<langle>l,a\<rangle> \<in> r"
  shows "\<langle>l,Infimum(r,A)\<rangle> \<in> r"
proof -
  let ?L = "\<Inter>a\<in>A. r-``{a}"
  from A4 have "\<forall>a\<in>A. l \<in> r-``{a}" using vimage_singleton_iff
    by simp;
  with A2 have "l\<in>?L" by auto; 
  with A1 A3 show "\<langle>l,Infimum(r,A)\<rangle> \<in> r"
    using Order_ZF_4_L3 Infimum_def by simp;
qed;

text{*If $z$ is an upper bound for $A$ and is greater or equal than
  any other upper bound, then $z$ is the supremum of $A$.*}

lemma Order_ZF_5_L5: assumes A1: "antisym(r)" and A2: "A\<noteq>0" and
  A3: "\<forall>x\<in>A. \<langle>x,z\<rangle> \<in> r" and 
  A4: "\<forall>y. (\<forall>x\<in>A. \<langle>x,y\<rangle> \<in> r) \<longrightarrow> \<langle>z,y\<rangle> \<in> r"
  shows 
  "HasAminimum(r,\<Inter>a\<in>A. r``{a})"
  "z = Supremum(r,A)"
proof -
  let ?B = "\<Inter>a\<in>A. r``{a}"
  from A2 A3 A4 have I: "z \<in> ?B"   "\<forall>y\<in>?B. \<langle>z,y\<rangle> \<in> r"
    by auto;
  then show "HasAminimum(r,\<Inter>a\<in>A. r``{a})"
    using HasAminimum_def by auto
  from A1 I show "z = Supremum(r,A)"
    using Order_ZF_4_L15 Supremum_def by simp;
qed;

text{*If a set has a maximum, then the maximum is the supremum.*}

lemma Order_ZF_5_L6: 
  assumes A1:  "antisym(r)" and A2: "A\<noteq>0" and 
  A3: "HasAmaximum(r,A)"
  shows 
  "HasAminimum(r,\<Inter>a\<in>A. r``{a})"
  "Maximum(r,A) = Supremum(r,A)"
proof -
  let ?M = "Maximum(r,A)"
  from A1 A3 have I: "?M \<in> A" and II: "\<forall>x\<in>A. \<langle>x,?M\<rangle> \<in> r"
    using Order_ZF_4_L3 by auto;
  from I have III: "\<forall>y. (\<forall>x\<in>A. \<langle>x,y\<rangle> \<in> r) \<longrightarrow> \<langle>?M,y\<rangle> \<in> r"
    by simp;
  with A1 A2 II show "HasAminimum(r,\<Inter>a\<in>A. r``{a})"
    by (rule Order_ZF_5_L5);
  from A1 A2 II III show "?M = Supremum(r,A)"
    by (rule Order_ZF_5_L5);
qed;

text{*Properties of supremum of a set for complete relations.*}

lemma Order_ZF_5_L7: 
  assumes A1: "r \<subseteq> X\<times>X" and A2: "antisym(r)" and 
  A3: "r {is complete}" and
  A4: "A\<subseteq>X"  "A\<noteq>0" and A5: "\<exists>x\<in>X. \<forall>y\<in>A. \<langle>y,x\<rangle> \<in> r"
  shows 
  "Supremum(r,A) \<in> X"
  "\<forall>x\<in>A. \<langle>x,Supremum(r,A)\<rangle> \<in> r"
proof -
  from A5 have "IsBoundedAbove(A,r)" using IsBoundedAbove_def
    by auto;
  with A3 A4 have "HasAminimum(r,\<Inter>a\<in>A. r``{a})"
    using IsComplete_def by simp
  with A2 have "Minimum(r,\<Inter>a\<in>A. r``{a}) \<in> ( \<Inter>a\<in>A. r``{a} )"
    using Order_ZF_4_L4 by simp;
  moreover have "Minimum(r,\<Inter>a\<in>A. r``{a}) = Supremum(r,A)"
    using Supremum_def by simp;
  ultimately have I: "Supremum(r,A) \<in>  ( \<Inter>a\<in>A. r``{a} )"
    by simp;
  moreover from A4 obtain a where "a\<in>A" by auto;
  ultimately have "\<langle>a,Supremum(r,A)\<rangle> \<in> r" using Order_ZF_5_L1
    by simp;
  with A1 show "Supremum(r,A) \<in> X" by auto;
  from I show "\<forall>x\<in>A. \<langle>x,Supremum(r,A)\<rangle> \<in> r" using Order_ZF_5_L1
    by simp;
qed;

text{*If the relation is a linear order then for any 
  element $y$ smaller than the supremum of a set we can
  find one element of the set that is greater than $y$.*}

lemma Order_ZF_5_L8:
  assumes A1: "r \<subseteq> X\<times>X"  and A2: "IsLinOrder(X,r)" and 
  A3: "r {is complete}" and
  A4: "A\<subseteq>X"  "A\<noteq>0" and A5: "\<exists>x\<in>X. \<forall>y\<in>A. \<langle>y,x\<rangle> \<in> r" and
  A6: "\<langle>y,Supremum(r,A)\<rangle> \<in> r"   "y \<noteq> Supremum(r,A)"
  shows "\<exists>z\<in>A. \<langle>y,z\<rangle> \<in> r \<and> y \<noteq> z"
proof -
  from A2 have 
    I: "antisym(r)" and
    II: "trans(r)" and
    III: "r {is total on} X"
    using IsLinOrder_def by auto;
  from A1 A6 have T1: "y\<in>X" by auto;
  { assume A7: "\<forall>z \<in> A. \<langle>y,z\<rangle> \<notin> r \<or> y=z"
    from A4 I have "antisym(r)" and "A\<noteq>0" by auto;
    moreover have "\<forall>x\<in>A. \<langle>x,y\<rangle> \<in> r"  
    proof      
      fix x assume A8: "x\<in>A"
      with A4 have T2: "x\<in>X" by auto;
      from A7 A8 have "\<langle>y,x\<rangle> \<notin> r \<or> y=x" by simp;
      with III T1 T2 show "\<langle>x,y\<rangle> \<in> r"
	using IsTotal_def total_is_refl refl_def by auto
    qed;
    moreover have "\<forall>u. (\<forall>x\<in>A. \<langle>x,u\<rangle> \<in> r) \<longrightarrow> \<langle>y,u\<rangle> \<in> r"
    proof-
      { fix u assume A9: "\<forall>x\<in>A. \<langle>x,u\<rangle> \<in> r"
	from A4 A5 have "IsBoundedAbove(A,r)" and "A\<noteq>0"
	  using IsBoundedAbove_def by auto;
	with  A3 A4 A6 I A9  have 
	  "\<langle>y,Supremum(r,A)\<rangle> \<in> r \<and> \<langle>Supremum(r,A),u\<rangle> \<in> r"
	  using IsComplete_def Order_ZF_5_L3 by simp;
	with II have "\<langle>y,u\<rangle> \<in> r" by (rule Fol1_L3);
      } then show "\<forall>u. (\<forall>x\<in>A. \<langle>x,u\<rangle> \<in> r) \<longrightarrow> \<langle>y,u\<rangle> \<in> r"
	by simp;
    qed;
    ultimately have "y = Supremum(r,A)"
      by (rule Order_ZF_5_L5);
    with A6 have False by simp;
  } then show "\<exists>z\<in>A. \<langle>y,z\<rangle> \<in> r \<and> y \<noteq> z" by auto;
qed;

section{*Strict versions of order relations*}

text{*One of the problems with translating formalized mathematics from
  Metamath to IsarMathLib is that Metamath uses strict orders (of the $<$ 
  type) while in IsarMathLib we mostly use nonstrict orders (of the 
  $\leq$ type). 
  This doesn't really make any difference, but is annoying as we 
  have to prove many theorems twice. In this section we prove some theorems
  to make it easier to translate the statements about strict orders to
  statements about the corresponding non-strict order and vice versa. *}

text{*We define a strict version of a relation by removing the $y=x$ line 
  from the relation.*}

constdefs
  "StrictVersion(r) \<equiv> r - {\<langle>x,x\<rangle>. x \<in> domain(r)}";

text{*A reformulation of the definition of a strict version of an order.
  *}

lemma def_of_strict_ver: shows 
  "\<langle>x,y\<rangle> \<in> StrictVersion(r) \<longleftrightarrow> \<langle>x,y\<rangle> \<in> r \<and> x\<noteq>y"
  using StrictVersion_def domain_def by auto;

text{*The next lemma is about the strict version of an antisymmetric
  relation.*}

lemma strict_of_antisym: 
  assumes A1: "antisym(r)" and A2: "\<langle>a,b\<rangle> \<in> StrictVersion(r)"
  shows "\<langle>b,a\<rangle> \<notin> StrictVersion(r)"
proof -
  { assume A3: "\<langle>b,a\<rangle> \<in> StrictVersion(r)"
    with A2 have "\<langle>a,b\<rangle> \<in> r"  and "\<langle>b,a\<rangle> \<in> r"
      using def_of_strict_ver by auto;
    with A1 have "a=b" by (rule Fol1_L4);
    with A2 have False using def_of_strict_ver
      by simp;
  } then show "\<langle>b,a\<rangle> \<notin> StrictVersion(r)" by auto;
qed;

text{*The strict version of totality.*}

lemma strict_of_tot:
  assumes "r {is total on} X" and "a\<in>X"  "b\<in>X"  "a\<noteq>b"
  shows "\<langle>a,b\<rangle> \<in> StrictVersion(r) \<or> \<langle>b,a\<rangle> \<in> StrictVersion(r)"
  using prems IsTotal_def def_of_strict_ver by auto;

text{*A trichotomy law for the strict version of a total 
  and antisymmetric
  relation. It is kind of interesting that one does not need
  the full linear order for this.*}

lemma strict_ans_tot_trich: 
  assumes A1: "antisym(r)" and A2: "r {is total on} X"
  and A3: "a\<in>X"  "b\<in>X"
  and A4: "s = StrictVersion(r)"
  shows "Exactly_1_of_3_holds(\<langle>a,b\<rangle> \<in> s, a=b,\<langle>b,a\<rangle> \<in> s)"
proof -
  let ?p = "\<langle>a,b\<rangle> \<in> s"
  let ?q = "a=b"
  let ?r = "\<langle>b,a\<rangle> \<in> s"
  from A2 A3 A4 have "?p \<or> ?q \<or> ?r"
    using strict_of_tot by auto;
  moreover from A1 A4 have "?p \<longrightarrow> \<not>?q \<and> \<not>?r"
    using def_of_strict_ver strict_of_antisym by simp;
  moreover from A4 have "?q \<longrightarrow> \<not>?p \<and> \<not>?r"
    using def_of_strict_ver by simp;
  moreover from A1 A4 have "?r \<longrightarrow> \<not>?p \<and> \<not>?q"
    using def_of_strict_ver strict_of_antisym by auto;
  ultimately show "Exactly_1_of_3_holds(?p, ?q, ?r)"
    by (rule Fol1_L5);
qed;

text{*A trichotomy law for linear order. This is a special
  case of @{text "strict_ans_tot_trich"}. *}

corollary strict_lin_trich: assumes A1: "IsLinOrder(X,r)" and
  A2: "a\<in>X"  "b\<in>X" and 
  A3: "s = StrictVersion(r)"
  shows "Exactly_1_of_3_holds(\<langle>a,b\<rangle> \<in> s, a=b,\<langle>b,a\<rangle> \<in> s)"
  using prems IsLinOrder_def strict_ans_tot_trich by auto;

text{*For an antisymmetric relation if a pair is in relation then
  the reversed pair is not in the strict version of the relation. 
  *}

lemma geq_impl_not_less: 
  assumes A1: "antisym(r)" and A2: "\<langle>a,b\<rangle> \<in> r"
  shows "\<langle>b,a\<rangle> \<notin> StrictVersion(r)"
proof -
  { assume A3: "\<langle>b,a\<rangle> \<in>  StrictVersion(r)"
    with A2 have "\<langle>a,b\<rangle> \<in> StrictVersion(r)"
      using def_of_strict_ver by auto;
    with A1 A3 have False using strict_of_antisym
      by blast;
  } then show "\<langle>b,a\<rangle> \<notin> StrictVersion(r)" by auto;
qed;
 
text{*If an antisymmetric relation is transitive, 
  then the strict version is also transitive, an explicit
  version @{text "strict_of_transB"} below.*}

lemma strict_of_transA: 
  assumes A1: "trans(r)" and A2: "antisym(r)" and  
  A3: "s= StrictVersion(r)" and  A4: "\<langle>a,b\<rangle> \<in> s"  "\<langle>b,c\<rangle> \<in> s"
  shows "\<langle>a,c\<rangle> \<in> s"
proof -
  from A3 A4 have I: "\<langle>a,b\<rangle> \<in> r \<and> \<langle>b,c\<rangle> \<in> r"
    using def_of_strict_ver by simp;
  with A1 have "\<langle>a,c\<rangle> \<in> r" by (rule Fol1_L3);
  moreover
  { assume "a=c"
    with I have "\<langle>a,b\<rangle> \<in> r" and "\<langle>b,a\<rangle> \<in> r" by auto;
    with A2 have "a=b" by (rule Fol1_L4);
    with A3 A4 have False using def_of_strict_ver by simp;
  } then have "a\<noteq>c" by auto;
  ultimately have  "\<langle>a,c\<rangle> \<in> StrictVersion(r)"
    using def_of_strict_ver by simp;
  with A3 show ?thesis by simp;
qed;

text{*If an antisymmetric relation is transitive, 
  then the strict version is also transitive.*}

lemma strict_of_transB: 
  assumes A1: "trans(r)" and A2: "antisym(r)"
  shows "trans(StrictVersion(r))"
proof -
  let ?s = "StrictVersion(r)"
  from A1 A2 have 
    "\<forall> x y z. \<langle>x, y\<rangle> \<in> ?s \<and> \<langle>y, z\<rangle> \<in> ?s \<longrightarrow> \<langle>x, z\<rangle> \<in> ?s"
    using strict_of_transA by blast;
  then show "trans(StrictVersion(r))" by (rule Fol1_L2);
qed;

text{*The next lemma provides a condition that is satisfied by
  the strict version of a relation if the original relation 
  is a complete linear order. *}

lemma strict_of_compl: 
  assumes A1: "r \<subseteq> X\<times>X" and A2: "IsLinOrder(X,r)" and 
  A3: "r {is complete}" and 
  A4: "A\<subseteq>X"  "A\<noteq>0" and A5: "s = StrictVersion(r)" and 
  A6: "\<exists>u\<in>X. \<forall>y\<in>A. \<langle>y,u\<rangle> \<in> s"
  shows 
  "\<exists>x\<in>X. ( \<forall>y\<in>A. \<langle>x,y\<rangle> \<notin> s ) \<and> (\<forall>y\<in>X. \<langle>y,x\<rangle> \<in> s \<longrightarrow> (\<exists>z\<in>A. \<langle>y,z\<rangle> \<in> s))"
proof -
  let ?x = "Supremum(r,A)"
  from A2 have I: "antisym(r)" using IsLinOrder_def
    by simp;
  moreover from A5 A6 have "\<exists>u\<in>X. \<forall>y\<in>A. \<langle>y,u\<rangle> \<in> r"
    using def_of_strict_ver by auto;
  moreover note A1 A3 A4 
  ultimately have II: "?x \<in> X"   "\<forall>y\<in>A. \<langle>y,?x\<rangle> \<in> r"
    using Order_ZF_5_L7 by auto
  then have III: "\<exists>x\<in>X. \<forall>y\<in>A. \<langle>y,x\<rangle> \<in> r" by auto;
  from A5 I II have "?x \<in> X"   "\<forall>y\<in>A. \<langle>?x,y\<rangle> \<notin> s"
    using geq_impl_not_less by auto;
  moreover from A1 A2 A3 A4 A5 III have 
    "\<forall>y\<in>X. \<langle>y,?x\<rangle> \<in> s \<longrightarrow> (\<exists>z\<in>A. \<langle>y,z\<rangle> \<in> s)"
    using def_of_strict_ver Order_ZF_5_L8 by simp;
  ultimately show
    "\<exists>x\<in>X. ( \<forall>y\<in>A. \<langle>x,y\<rangle> \<notin> s ) \<and> (\<forall>y\<in>X. \<langle>y,x\<rangle> \<in> s \<longrightarrow> (\<exists>z\<in>A. \<langle>y,z\<rangle> \<in> s))"
    by auto;
qed;

text{*Strict version of a relation on a set is a relation on that
  set.*}

lemma strict_ver_rel: assumes A1: "r \<subseteq> A\<times>A"
  shows "StrictVersion(r) \<subseteq> A\<times>A"
  using prems StrictVersion_def by auto;


end