(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Topology\_ZF\_1.thy}*}
theory Topology_ZF_1 imports Topology_ZF Fol1 

begin

section{*Separation axioms.*}

text{*A topology on $X$ is $T_0$ if for every pair of distinct points of $X$
there is an open set that contains only one of them.
A topology is $T_1$ if for every such pair there exist an open set that 
contains the first point but not the second.
A topology is $T_2$ (Hausdorff) if for every pair of points there exist a 
pair of disjoint open sets each containing one of the points.*}

constdefs 
  "isT0(X,T) \<equiv> 
  \<forall> x y. ((x\<in>X & y\<in>X & x\<noteq>y) \<longrightarrow> (\<exists>U\<in>T. (x\<in>U & y\<notin>U)|(y\<in>U & x\<notin>U)))"

  "isT1(X,T) \<equiv> 
  \<forall> x y. ((x\<in>X & y\<in>X & x\<noteq>y) \<longrightarrow> (\<exists>U\<in>T. (x\<in>U & y\<notin>U)))"

  "isT2(X,T) \<equiv>  
  \<forall> x y. ((x\<in>X & y\<in>X & x\<noteq>y) \<longrightarrow> (\<exists>U V. (U\<in>T & V\<in>T & x\<in>U & y\<in>V & U\<inter>V=0)))";

text{*If a topology is $T_1$ then it is $T_0$. 
We don't really assume here that $T$ is a topology on $X$. 
We prove the relation between isT0 condition and isT1. *}

lemma topZF1_1_L1: assumes A:"isT1(X,T)" shows "isT0(X,T)"
proof-;
  from A have "\<forall> x y. (x\<in>X & y\<in>X & x\<noteq>y \<longrightarrow> (\<exists>U\<in>T. (x\<in>U & y\<notin>U)))"
    using isT1_def by auto
  then have 
    "\<forall>x y. (x\<in>X & y\<in>X & x\<noteq>y \<longrightarrow> (\<exists>U\<in>T. (x\<in>U & y\<notin>U)|(y\<in>U & x\<notin>U)))" 
    by auto
  then show ?thesis using isT0_def by simp
qed;

text{*If a topology is $T_2$ then it is $T_1$.*}

lemma topZF1_1_L2: assumes A:"isT2(X,T)" shows "isT1(X,T)"
proof -;
  from A have
    "\<forall> x y. ((x\<in>X & y\<in>X & x\<noteq>y) \<longrightarrow> (\<exists>U\<in>T. (x\<in>U & y\<notin>U)))" 
    using isT2_def Finite1_L2 by auto;
  then show ?thesis using isT1_def by simp;
qed;

locale top_spaceT0 = top_space1 + 
  assumes T0assum:"isT0(X,T)"
  

text{*In a $T_0$ space two points that can not be separated 
  by an open set are equal.
  The proof is by contradiction, which is indicated by rule ccontr.*}

lemma (in top_spaceT0) top_ZF_1_1_L3:
  assumes A1: "x\<in>X" "y\<in>X" and A2: "\<forall>U\<in>T. (x\<in>U \<longleftrightarrow> y\<in>U)" 
  shows "x=y"; 
proof (rule ccontr);
  assume "x\<noteq>y"
  with A1 have "\<exists>U\<in>T. (x\<in>U & y\<notin>U)|(y\<in>U & x\<notin>U)" 
    using T0assum isT0_def by simp;
  with A2 show False by simp;
qed;

locale top_spaceT2 = top_space1 + 
  assumes T2assum:"isT2(X,T)"

text{*In a $T_2$ space two points can be separated by an open set with its 
boundary. This condition is weaker than $T_2$ but stronger than $T_1$, 
so it can be called "$T_1$ and half".*}

lemma (in top_spaceT2) top_ZF_1_1_L4: 
  assumes A1: "x\<in>X & y\<in>X & x\<noteq>y"
  shows "\<exists>U\<in>T. (x\<in>U & y\<notin>U\<^sup>-)"
proof -;
  from A1 have "\<exists>U V. (U\<in>T & V\<in>T & x\<in>U & y\<in>V & U\<inter>V=0)"
    using T2assum isT2_def by simp;
  then obtain U V where "U\<in>T & V\<in>T & x\<in>U & y\<in>V & U\<inter>V=0" by auto;
  then have "U\<in> T & x\<in> U & y\<in> V & U\<^sup>- \<inter> V = 0" using topZF_3_L14 by simp;
  then show ?thesis by auto;
qed

text{*In a $T2$ space compact sets are closed. Doing a formal proof of this 
  theorem gave me an interesting insight into the role of 
  the Axiom of Choice in romantic proofs.

  A typical romantic proof of this fact goes like this: we want to show 
  that $K^c$ is open. To do this, choose an arbitrary point $y\in K^c$.
  Since $X$ is $T_2$, for every point $x\in K$ we can find an 
  open set $U_x$ such that $y\notin \overline{U_x}$. 
  Obviously $\{U_x\}_{x\in K}$ covers $K$, so select a finite subcollection
  that covers $K$, and so on. I have never realized that 
  such reasoning requires (an) Axiom of Choice. 
  Namely, suppose we have a lemma that 
  states "In $T_2$ spaces, if $x\neq y$, then there is an open set 
  $U$ such that $x\in U$ and $y\notin \overline{U}$". This only states that
  the set of such open sets $U$ is not empty. To get the collection 
  $\{U_x \}_{x\in K}$ in the above proof we have to select one such set 
  among many for every $x\in K$ and this is where we use (an) Axiom of Choice. 
  Probably in 99/100 cases when a romatic calculus proof states something like
  $\forall \varepsilon \exists \delta_\varepsilon \cdots$ the proof uses 
  Axiom of Choice. In the proof below we avoid using Axiom of Choice 
  (read it to find out how). It is an interesting question which such calculus 
  proofs can be reformulated so that the usage of AC is avoided. I remember 
  Sierpi\'{n}ski published a paper in 1919 (I think, my memory is not 
  that good any more) that claimed that one needs an Axiom of Choice to show
  the equivalence of the Heine and Cauchy definitions of limits.
 *}

theorem (in top_spaceT2) top_ZF_1_1_T1: 
  assumes A1:"K {is compact}" shows "K {is closed}"
proof -;
  have "\<forall> y\<in>X-K. \<exists>U\<in>T. (y\<in>U & U\<subseteq>X-K)"
  proof
    fix y
    assume A2:"y\<in>X-K" show "\<exists>U\<in>T. y \<in> U \<and> U \<subseteq> X - K "
    proof -;
      let ?B = "\<Union>x\<in>K.{V\<in>T. x\<in>V & y\<notin> V\<^sup>-}"
      have T1:"?B\<in> Pow(T)" by auto
      then have T2:"\<forall>V\<in>?B. V\<in> Pow(X)"
	using topSpaceAssum isAtopologyOn_def by auto;
      have "\<exists> N\<in> Fin(?B). K \<subseteq> \<Union>N" 
      proof -;
	from A1 A2 have "\<forall>x\<in>K. x\<in> X & y\<in>X & y\<noteq>x" 
	  using IsCompact_def by auto;
	hence "\<forall>x\<in>K. ({V\<in>T. x\<in>V & y\<notin> V\<^sup>-}\<noteq>0)" 
	  using top_ZF_1_1_L4 by auto;
	hence "K \<subseteq> \<Union>?B" by blast;
	with T1 A1 show "\<exists> N\<in> Fin(?B). K \<subseteq> \<Union>N" 
	  using IsCompact_def by auto;
      qed;
      then obtain N where D1:"N\<in> Fin(?B) & K \<subseteq> \<Union>N" by auto;
      have "Fin(?B)\<subseteq>Pow(?B)" using Fin.dom_subset by simp
      with D1 have "N\<in>Pow(?B)" by auto;
      then have T3:"\<forall>V\<in>N. V\<in>?B" by auto;
      let ?M = "{V\<^sup>-. V\<in>N}"
      let ?C = "{D\<in>Pow(X). D {is closed}}"
      have "X -\<Union>?M \<in> T"
      proof -;
	from D1 have T4:"N\<in> Fin(?B)" by simp;
	have T5:"\<forall>V\<in>?B. (V\<^sup>- {is closed})"
	  using topSpaceAssum isAtopologyOn_def topZF_3_L4 
	  by auto (*blast no-go*)
	then have "\<forall>V\<in>?B. (V\<^sup>- \<in> Pow(X))" using IsClosed_def by simp;
	with T5 have T6:"\<forall>V\<in>?B. (V\<^sup>- \<in> ?C)" by simp;
	from T6 T4 have "?M \<in> Fin(?C)" by (rule fin_image_fin);
	then show "X -\<Union>?M \<in> T" using topZF_3_L4B IsClosed_def by simp;
      qed;
      moreover from T3 A2 have "y \<in> X - \<Union>?M" by simp; 
      moreover have "X-\<Union>?M \<subseteq> X - K"
      proof -;
	from T3 T2 have "\<forall>V\<in>N. V\<subseteq>X" by auto;
	then have "\<forall>V\<in>N. V\<subseteq>V\<^sup>-" using topZF_3_L9 by simp;
	then have "\<Union>N \<subseteq>  \<Union>?M" by auto;
	with D1 have "K \<subseteq> \<Union>?M" by auto;
	then show "X-\<Union>?M \<subseteq> X - K" by auto;
      qed
      ultimately have "X-\<Union>?M \<in> T & y \<in> X-\<Union>?M & X-\<Union>?M \<subseteq> X - K" 
	(is "?U \<in> T & y \<in> ?U & ?U\<subseteq>X-K") by simp;
      then have "\<exists>U. U\<in> T &  y \<in> U \<and> U \<subseteq> X - K" by auto;
      then show ?thesis by auto;
    qed;
  qed;
  with A1 show "K {is closed}" 
    using topZF_2_L9 IsCompact_def IsClosed_def by simp;
qed;

section{*Bases and subbases.*}

text{*A base of topology is a collection of open sets such that every 
open set is a union of the sets from the base. A subbase is a collection 
of open sets such that finite intersection of those sets form a base.*}

constdefs

  IsAbaseFor (infixl "{is a base for}" 65)
  "B {is a base for} T \<equiv> B\<subseteq>T & T = {\<Union>A. A\<in>Pow(B)}"

  IsAsubBaseFor (infixl "{is a subbase for}" 65)
  "B {is a subbase for} T \<equiv> 
  B \<subseteq> T & {\<Inter>A. A\<in>Fin(B)} {is a base for} T"

  SatisfiesBaseCondition ("_ {satisfies the base condition}" [50] 50)  
  "B {satisfies the base condition} \<equiv> 
  \<forall>U V. ((U\<in>B & V\<in>B) \<longrightarrow> (\<forall>x\<in>(U\<inter>V). \<exists>W\<in>B. (x\<in>W & W \<subseteq> U \<inter> V)))";
  

text{*Each open set is a union of some sets from the base.*}
lemma top_ZF_1_2_L1: assumes "B {is a base for} T" "U\<in>T"
  shows "\<exists>A\<in>Pow(B). U = \<Union>A"
  using prems IsAbaseFor_def by simp;

text{*The following theorems provide necessary and sufficient conditions 
for a collection of sets to be a base for some topology.*}

text{*A necessary condition: every point in the intersection
of two sets in the base has a neighboorhood from the base contained
in the intersection.*}

lemma top_ZF_1_2_T1: 
  assumes A1:"\<exists>T. T {is a topology on} \<Union>B & B {is a base for} T"
  and A2:"V\<in>B & W\<in>B"
  shows "\<forall>x\<in>(V\<inter>W). \<exists>U\<in>B. (x\<in>U & U \<subseteq> V \<inter> W)";
proof - 
  from A1 obtain T where 
    D1:"T {is a topology on} \<Union>B & B {is a base for} T"
    by auto;
  then have "B \<subseteq> T" using IsAbaseFor_def by auto;
  with A2 have "V\<in>T & W\<in>T" using IsAbaseFor_def by auto;
  with D1 have "V \<inter> W \<in> T" using isAtopologyOn_def by simp;
  with D1 have "\<exists>A\<in>Pow(B). V \<inter> W = \<Union>A" using top_ZF_1_2_L1 by auto;
  then obtain A where D2:"A\<in>Pow(B) & V \<inter> W = \<Union>A" by auto;
  then have "\<forall>x\<in> V\<inter>W.\<exists>U\<in>A. (x\<in>U & U \<subseteq> V \<inter> W)" by auto;
  with D2 show "\<forall>x\<in> V\<inter>W.\<exists>U\<in>B. (x\<in>U & U \<subseteq> V \<inter> W)" by auto (*blast no-go*)
qed;

text{*A sufficient condition: If every point in the intersection
of two sets in the collection has a neighboorhood in the base contained
in the intersection, then the collection is a base for some topology 
(defined in the lemma). We prove part of it is a separate lemma to 
avoid having to fix to variables.*}

text{*We will define the topology as the collection of unions of (would-be)
base. First we prove that if the collection of sets satisfies the 
condition we want to show to be sufficient, the the intersection belongs
to what we will define as topology (am I clear here?).*}

lemma top_ZF_1_2_L2:
  assumes As1: "V \<in> B" "W\<in>B"
  and As2: "\<forall>x\<in>(V\<inter>W). \<exists>U\<in>B. (x\<in>U & U \<subseteq> V \<inter> W)"
  shows "V\<inter>W \<in> {\<Union>A. A\<in>Pow(B)}"
proof;
  let ?C = "\<Union>x\<in>V\<inter>W. {U\<in>B. x\<in>U & U \<subseteq> V \<inter> W}";
  show "?C\<in>Pow(B)" by auto;
  from As2 show "V\<inter>W = \<Union>?C" by blast;
qed;

text{*The next lemma is needed when proving that the would-be topology is
closed with respect to taking intersections. We show here that intersection
of two sets from this (would-be) topology can be written as union of sets 
from the topology.*}

lemma top_ZF_1_2_L3: 
  assumes As1: "U1 \<in> {\<Union>A. A\<in>Pow(B)}" and As2: "U2 \<in> {\<Union>A. A\<in>Pow(B)}"
  and As3: "B {satisfies the base condition}"
  shows "\<exists>C.(C \<subseteq> {\<Union>A. A\<in>Pow(B)} &  U1\<inter>U2 = \<Union>C)"
proof -;
  from As1 obtain A1 where D1:"A1\<in> Pow(B) & U1 = \<Union>A1" by auto
  from As2 obtain A2 where D2:"A2\<in> Pow(B) & U2 = \<Union>A2" by auto
  let ?C = "\<Union>U\<in>A1.{U\<inter>V. V\<in>A2}";
  from D1 D2 have "(\<forall>U\<in>A1. U\<in>B) & (\<forall>V\<in>A2. V\<in>B)" by auto;
  with As3 have "\<forall>W\<in>?C. W \<in> {\<Union>A. A\<in>Pow(B)}"
    using top_ZF_1_2_L2 SatisfiesBaseCondition_def by simp;
  then have "?C \<subseteq> {\<Union>A . A \<in> Pow(B)}" by auto;
  moreover from D1 D2 have "U1 \<inter> U2 = \<Union>?C" by auto
  ultimately show ?thesis by auto;
qed;

text{*Now the theorem. *}
  
lemma top_ZF_1_2_T2: 
assumes A1:"B {satisfies the base condition}"
and A2:"T = {\<Union>A. A\<in>Pow(B)}"
shows "T {is a topology on} \<Union>B" and "B {is a base for} T";
proof -;
  show "T {is a topology on} \<Union>B"  
  proof -;
    from A2 have "T \<subseteq> Pow(\<Union>B)" by auto;
    moreover from A2 have "\<Union>B \<in> T" by auto;
    moreover from A2 have "0\<in>T" by auto;
    moreover have T1:"\<forall> C. (C \<subseteq> T \<longrightarrow> ( ( \<Union> C )\<in> T ))";
    proof;
      fix C show "C \<subseteq> T \<longrightarrow> \<Union>C \<in> T";
      proof;
	assume A3:"C \<subseteq> T" show "\<Union>C \<in> T";
	proof -;
	  let ?Q = "\<Union> {\<Union>{A\<in>Pow(B). U = \<Union>A}. U\<in>C}";
	  from A3 A2 have "\<forall>U\<in>C. \<exists>A\<in>Pow(B). U = \<Union>A" by auto;
	  then have "\<Union>?Q = \<Union>C" using  Finite1_L8 by simp; 
	  moreover from A2 have "\<Union>?Q \<in> T" by auto; 
	  ultimately show "\<Union>C \<in> T" by simp;
	qed;
      qed;
    qed;
    moreover have "\<forall> U V. ( (U\<in>T & V\<in>T)  \<longrightarrow>  ( U\<inter>V \<in> T))"
    proof -
      { 
	fix U V assume  "U \<in> T \<and> V \<in> T"
	with A1 A2 have "\<exists>C.(C \<subseteq> T &  U\<inter>V = \<Union>C)"
	  using top_ZF_1_2_L3 by simp;
	then obtain C where "C \<subseteq> T &  U\<inter>V = \<Union>C"
	  by auto;
	with T1 have "U\<inter>V \<in> T" by simp;
      } then show ?thesis by simp;
    qed;
    ultimately show "T {is a topology on} \<Union>B" 
      using isAtopologyOn_def by simp;
  qed;
  show "B {is a base for} T" 
  proof -;
    from A2 have "B\<subseteq>T" by auto;
    with A2 show ?thesis
      using IsAbaseFor_def by simp;
  qed;  
qed;

section{*Product topology*}

text{*Given two topological spaces we can define a topology on the product of 
  the carriers such that the cartesian products of the sets of the topologies 
  are a base for the product topology. 
  The definition of the ProductCollection should be in ZF1.thy.*};

constdefs

  "ProductCollection(T,S) \<equiv> \<Union>U\<in>T.{U\<times>V. V\<in>S}"

  "ProductTopology(T,S) \<equiv> {\<Union>W. W \<in> Pow(ProductCollection(T,S))}"

text{*The product collection satisfies the base condition.*}

lemma top_ZF_1_3_L2: 
  assumes A1: "T {is a topology on} X"
  and A2: "S {is a topology on} Y"
  and A3: "A \<in> ProductCollection(T,S)" 
  and A4: "B \<in> ProductCollection(T,S)"
  shows 
  "\<forall>x\<in>(A\<inter>B). \<exists>W\<in>ProductCollection(T,S). (x\<in>W & W \<subseteq> A \<inter> B)"
proof;
  fix x assume A5:"x \<in> A\<inter>B" 
  from A3 obtain U1 V1 where D1:"U1\<in>T & V1\<in>S & A=U1\<times>V1"
    using ProductCollection_def by auto;
  from A4 obtain U2 V2 where D2:"U2\<in>T & V2\<in>S & B=U2\<times>V2"
    using ProductCollection_def by auto;
  let ?W = "(U1\<inter>U2)\<times>(V1\<inter>V2)";
  from A1 D1 D2 have "U1\<inter>U2 \<in> T" using isAtopologyOn_def by simp;
  moreover from A2 D1 D2 have "V1\<inter>V2 \<in> S" 
    using isAtopologyOn_def by simp;
  ultimately have "?W \<in> ProductCollection(T,S)" 
    using ProductCollection_def by auto
  moreover from A5 D1 D2 have "x\<in>?W & ?W \<subseteq> A \<inter> B" by auto;
  ultimately have "\<exists>W. (W \<in> ProductCollection(T,S) & x\<in>W & W \<subseteq> A \<inter> B)"
    by blast;
  thus "\<exists>W\<in>ProductCollection(T,S). (x\<in>W & W \<subseteq> A \<inter> B)" by auto;
qed;

text{*The product topology is indeed a topology on the product.*}

theorem top_ZF_1_3_T1: 
  assumes A1: "T {is a topology on} X" "S {is a topology on} Y"
  shows "ProductTopology(T,S) {is a topology on} X\<times>Y";
proof -;
  from A1 have 
    "ProductTopology(T,S) {is a topology on} \<Union>ProductCollection(T,S)"
    using top_ZF_1_3_L2 ProductCollection_def 
      SatisfiesBaseCondition_def top_ZF_1_2_T2 ProductTopology_def 
    by simp;
  moreover have "\<Union>ProductCollection(T,S) = X\<times>Y"
  proof;
    from A1 show "X \<times> Y \<subseteq> \<Union>ProductCollection(T, S)"
      using isAtopologyOn_def ProductCollection_def by auto;
    from A1 have "\<forall>U\<in>T. \<forall>V\<in>S. U\<times>V\<subseteq>X\<times>Y" 
      using isAtopologyOn_def by auto;
    thus "\<Union>ProductCollection(T, S) \<subseteq> X\<times>Y" 
      using ProductCollection_def by auto;
  qed;
  ultimately show ?thesis by simp;
qed;

end;