(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005 - 2007  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Nat\_ZF.thy}*}

theory Nat_ZF imports Nat 

begin

text{*This theory contains lemmas that are missing from the standard 
  Isabelle's Nat.thy file.*}

section{*Induction*}

text{*The induction lemmas in the standard Isabelle's Nat.thy file like 
  for example @{text "nat_induct"} require the induction step to 
  be a higher order 
  statement (the one that uses the $\Longrightarrow$ sign). I found it 
  difficult to apply from Isar, which is perhaps more of an indication of 
  my Isar skills than anything else. Anyway, here we provide a first order
  version that is easier to reference in Isar declarative style proofs.*}

text{*The induction step for the first order induction.*}

lemma Nat_ZF_1_L1: assumes "x\<in>nat"  "P(x)" 
  and "\<forall>k\<in>nat. P(k)\<longrightarrow>P(succ(k))"
  shows "P(succ(x))" using prems by simp;

text{*The actual first order induction on natural numbers.*}

lemma Nat_ZF_1_L2: 
  assumes A1: "n\<in>nat" and A2: "P(0)" and A3: "\<forall>k\<in>nat. P(k)\<longrightarrow>P(succ(k))"
  shows "P(n)"
proof -;
  from A1 A2 have "n\<in>nat" "P(0)" by auto
  then show "P(n)" using Nat_ZF_1_L1 by (rule nat_induct);
qed;

text{*A nonzero natural number has a predecessor.*}

lemma Nat_ZF_1_L3: assumes A1: "n \<in> nat" and A2: "n\<noteq>0"
  shows "\<exists>k\<in>nat. n = succ(k)"
proof -
  from A1 have "n \<in> {0} \<union> {succ(k). k\<in>nat}"
    using nat_unfold by simp;
  with A2 show ?thesis by simp
qed;

text{*Empty set is an element of every natural number which is not zero.*}

lemma empty_in_every_succ: assumes A1: "n \<in> nat"
  shows "0 \<in> succ(n)"
proof -
  note A1
  moreover have "0 \<in> succ(0)" by simp;
  moreover
  { fix k assume "k \<in> nat" and A2: "0 \<in> succ(k)"
    then have "succ(k) \<subseteq> succ(succ(k))" by auto;
    with A2 have "0 \<in> succ(succ(k))" by auto;
  } then have "\<forall>k \<in> nat. 0 \<in> succ(k) \<longrightarrow> 0 \<in> succ(succ(k))"
    by simp;
  ultimately show "0 \<in> succ(n)" by (rule Nat_ZF_1_L2);
qed;  

text{*If one natural number is less than another then their successors
  are in the same relation.*}
lemma succ_ineq: assumes A1: "n \<in> nat"
  shows "\<forall>i \<in> n. succ(i) \<in> succ(n)"
proof -
  note A1
  moreover have "\<forall>k \<in> 0. succ(k) \<in> succ(0)" by simp; 
  moreover
  { fix k assume A2: "\<forall>i\<in>k. succ(i) \<in> succ(k)"
    { fix i assume "i \<in> succ(k)"
      then have "i \<in> k \<or> i = k" by auto;
      moreover
      { assume "i\<in>k"
	with A2 have "succ(i) \<in> succ(k)" by simp;
	hence "succ(i) \<in> succ(succ(k))" by auto }
      moreover
      { assume "i = k"
	then have "succ(i) \<in> succ(succ(k))" by auto }
      ultimately have "succ(i) \<in> succ(succ(k))" by auto;
    } then have "\<forall>i \<in> succ(k). succ(i) \<in> succ(succ(k))"
      by simp;
  } then have "\<forall>k \<in> nat. 
      ( (\<forall>i\<in>k. succ(i) \<in> succ(k)) \<longrightarrow> (\<forall>i \<in> succ(k). succ(i) \<in> succ(succ(k))) )"
    by simp;
  ultimately show "\<forall>i \<in> n. succ(i) \<in> succ(n)" by (rule Nat_ZF_1_L2);
qed;

text{*For natural numbers if $k\subseteq n$ the similar holds for 
  their successors. *}

lemma succ_subset: assumes A1: "k \<in> nat"  "n\<in>nat" and A2: "k\<subseteq>n" 
  shows "succ(k) \<subseteq> succ(n)"
proof -
  from A1 have T: "Ord(k)" and "Ord(n)"
    using nat_into_Ord by auto;
  with A2 have "succ(k) \<le> succ(n)"
    using subset_imp_le by simp;
  then show "succ(k) \<subseteq> succ(n)" using le_imp_subset
    by simp;
qed;

text{*For any two natural numbers one of them is contained in the other.*}

lemma nat_incl_total: assumes A1: "i \<in> nat"  "j \<in> nat"
  shows "i \<subseteq> j \<or> j \<subseteq> i"
proof -
  from A1 have T: "Ord(i)"   "Ord(j)" 
    using nat_into_Ord by auto;
  then have "i\<in>j \<or> i=j \<or> j\<in>i" using Ord_linear
    by simp;
  moreover
  { assume "i\<in>j"
    with T have "i\<subseteq>j \<or> j\<subseteq>i"
      using lt_def leI le_imp_subset by simp; }
  moreover
  { assume "i=j"
    then have "i\<subseteq>j \<or> j\<subseteq>i" by simp }
  { assume "j\<in>i"
    with T have "i\<subseteq>j \<or> j\<subseteq>i" 
      using lt_def leI  le_imp_subset by simp }
  ultimately show "i \<subseteq> j \<or> j \<subseteq> i" by auto;
qed;

text{*The set of natural numbers is the union of all successors of natural
  numbers.*}

lemma nat_union_succ: shows "nat = (\<Union>n \<in> nat. succ(n))"
proof;
  show "nat \<subseteq> (\<Union>n \<in> nat. succ(n))" by auto;
next
  { fix k assume A2: "k \<in> (\<Union>n \<in> nat. succ(n))"
    then obtain n where T: "n \<in> nat" and I: "k \<in> succ(n)"
      by auto;
    with I have "k \<le> n" using nat_into_Ord lt_def
      by simp;
    with T have "k \<in> nat" using le_in_nat by simp
  } then show  "(\<Union>n \<in> nat. succ(n)) \<subseteq> nat" by auto;
qed;

text{*Successors of natural numbers are subsets of the set of natural numbers.*}

lemma succnat_subset_nat: assumes A1: "n \<in> nat" shows "succ(n) \<subseteq> nat"
proof -
  from A1 have "succ(n) \<subseteq> (\<Union>n \<in> nat. succ(n))" by auto;
  then show "succ(n) \<subseteq> nat" using nat_union_succ by simp;
qed;





end;