(* 
This file is a part of IsarMathLib - 
a library of formalized mathematics for Isabelle/Isar.

Copyright (C) 2006  Slawomir Kolodynski

This program is free software; Redistribution and use in source and binary forms, 
with or without modification, are permitted provided that the 
following conditions are met:

1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Metamath\_sampler.thy}*}

theory Metamath_sampler imports Metamath_interface MMI_Complex_ZF_1

begin

text{*This theory file contains some examples of theorems 
 translated from Metamath and formulated in the @{text "complex0"} context.*}

text{*Metamath uses the set of real numbers extended with $+\infty$ and $-\infty$. 
  The $+\infty$ and $-\infty$ symbols are defined quite arbitrarily as $\mathbb{C}$
  and $\mathbb{\{ C\} }$, respectively. The next lemma that corresponds to 
  Metamath's @{text "renfdisj"} states that $+\infty$ and $-\infty$ are not 
  elements of $\mathbb{R}$.*}

lemma (in complex0) renfdisj: shows "\<real> \<inter> {\<cpnf>,\<cmnf>} = 0"
proof -
  let ?real = "\<real>"
  let ?complex = "\<complex>"
  let ?one = "\<one>"
  let ?zero = "\<zero>"
  let ?iunit = "\<i>"
  let ?caddset = "CplxAdd(R,A)"
  let ?cmulset= "CplxMul(R,A,M)"
  let ?lessrrel = "StrictVersion(CplxROrder(R,A,r))"
  have "MMIsar0
    (?real, ?complex, ?one, ?zero, ?iunit, ?caddset, ?cmulset, ?lessrrel)"
    using MMIsar_valid by simp;
  then have "?real \<inter> {?complex, {?complex}} = 0"
    by (rule MMIsar0.MMI_renfdisj);
  thus "\<real> \<inter> {\<cpnf>,\<cmnf>} = 0" by simp;
qed;
  
text{*  The order relation used most often in Metamath is defined on 
  the set of complex reals extended with   $+\infty$ and $-\infty$. 
  The next lemma
  allows to use Metamath's @{text "xrltso"} that states that the @{text "\<ls>"}
  relations is a strict linear order on the extended set.*}

lemma (in complex0) xrltso: "\<cltrrset> Orders \<real>\<^isup>*"
proof -
  let ?real = "\<real>"
  let ?complex = "\<complex>"
  let ?one = "\<one>"
  let ?zero = "\<zero>"
  let ?iunit = "\<i>"
  let ?caddset = "CplxAdd(R,A)"
  let ?cmulset= "CplxMul(R,A,M)"
  let ?lessrrel = "StrictVersion(CplxROrder(R,A,r))"
  have "MMIsar0
    (?real, ?complex, ?one, ?zero, ?iunit, ?caddset, ?cmulset, ?lessrrel)"
    using MMIsar_valid by simp;
  then have 
    "(?lessrrel \<inter> ?real \<times> ?real \<union> 
    {\<langle>{?complex}, ?complex\<rangle>} \<union> ?real \<times> {?complex} \<union>
      {{?complex}} \<times> ?real) Orders (?real \<union> {?complex, {?complex}})"
    by (rule MMIsar0.MMI_xrltso);
  moreover have "?lessrrel \<inter> ?real \<times> ?real = ?lessrrel"
    using cplx_strict_ord_on_cplx_reals by auto
  ultimately show "\<cltrrset> Orders \<real>\<^isup>*" by simp;
qed;

text{*Metamath defines the usual $<$ and $\leq$ ordering relations for the
  extended real line, including $+\infty$ and $-\infty$.*}

lemma (in complex0) xrrebndt: assumes A1: "x \<in> \<real>\<^isup>*"
  shows "x \<in> \<real> \<longleftrightarrow> ( \<cmnf> \<ls> x \<and> x \<ls> \<cpnf> )"
proof -
  let ?real = "\<real>"
  let ?complex = "\<complex>"
  let ?one = "\<one>"
  let ?zero = "\<zero>"
  let ?iunit = "\<i>"
  let ?caddset = "CplxAdd(R,A)"
  let ?cmulset= "CplxMul(R,A,M)"
  let ?lessrrel = "StrictVersion(CplxROrder(R,A,r))"
  have "MMIsar0
    (?real, ?complex, ?one, ?zero, ?iunit, ?caddset, ?cmulset, ?lessrrel)"
    using MMIsar_valid by simp;
  then have "x \<in> \<real> \<union> {\<complex>, {\<complex>}} \<longrightarrow>
    x \<in> \<real> \<longleftrightarrow> \<langle>{\<complex>}, x\<rangle> \<in> ?lessrrel \<inter> \<real> \<times> \<real> \<union> {\<langle>{\<complex>}, \<complex>\<rangle>} \<union> 
    \<real> \<times> {\<complex>} \<union> {{\<complex>}} \<times> \<real> \<and>
    \<langle>x, \<complex>\<rangle> \<in> ?lessrrel \<inter> \<real> \<times> \<real> \<union> {\<langle>{\<complex>}, \<complex>\<rangle>} \<union> 
    \<real> \<times> {\<complex>} \<union> {{\<complex>}} \<times> \<real>"
    by (rule MMIsar0.MMI_xrrebndt);
  then have "x \<in> \<real>\<^isup>* \<longrightarrow> ( x \<in> \<real> \<longleftrightarrow> ( \<cmnf> \<ls> x \<and> x \<ls> \<cpnf> ) )"
    by simp;
  with A1 show ?thesis by simp;
qed;

text{*A quite involved inequality.*}

lemma (in complex0) lt2mul2divt: 
  assumes A1: "a \<in> \<real>"  "b \<in> \<real>"  "c \<in> \<real>"  "d \<in> \<real>" and
  A2: "\<zero> \<ls> b"  "\<zero> \<ls> d"
  shows "a\<cdot>b \<ls> c\<cdot>d \<longleftrightarrow> a\<cdiv>d \<ls> c\<cdiv>b"
proof -
  let ?real = "\<real>"
  let ?complex = "\<complex>"
  let ?one = "\<one>"
  let ?zero = "\<zero>"
  let ?iunit = "\<i>"
  let ?caddset = "CplxAdd(R,A)"
  let ?cmulset= "CplxMul(R,A,M)"
  let ?lessrrel = "StrictVersion(CplxROrder(R,A,r))"
  have "MMIsar0
    (?real, ?complex, ?one, ?zero, ?iunit, ?caddset, ?cmulset, ?lessrrel)"
    using MMIsar_valid by simp;
  then have
 "(a \<in> ?real \<and> b \<in> ?real) \<and>
 (c \<in> ?real \<and> d \<in> ?real) \<and>
 \<langle>?zero, b\<rangle> \<in> ?lessrrel \<inter> ?real \<times> ?real \<union> 
 {\<langle>{?complex}, ?complex\<rangle>} \<union> ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real \<and>
 \<langle>?zero, d\<rangle> \<in> ?lessrrel \<inter> ?real \<times> ?real \<union> 
 {\<langle>{?complex}, ?complex\<rangle>} \<union> ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real \<longrightarrow>
 \<langle>?cmulset ` \<langle>a, b\<rangle>, ?cmulset ` \<langle>c, d\<rangle>\<rangle> \<in>
 ?lessrrel \<inter> ?real \<times> ?real \<union> {\<langle>{?complex}, ?complex\<rangle>} \<union> 
   ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real \<longleftrightarrow>
 \<langle>\<Union>{x \<in> ?complex . ?cmulset ` \<langle>d, x\<rangle> = a}, 
   \<Union>{x \<in> ?complex . ?cmulset ` \<langle>b, x\<rangle> = c}\<rangle> \<in>
   ?lessrrel \<inter> ?real \<times> ?real \<union> {\<langle>{?complex}, ?complex\<rangle>} \<union> 
   ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real" 
   by (rule  MMIsar0.MMI_lt2mul2divt);
 with A1 A2 show ?thesis by simp;
qed;

text{*A real number is smaller than its half iff it is positive. *}

lemma (in complex0) halfpos: assumes A1: "a \<in> \<real>"
  shows "\<zero> \<ls> a \<longleftrightarrow> a\<cdiv>\<two> \<ls> a"
proof -
  let ?real = "\<real>"
  let ?complex = "\<complex>"
  let ?one = "\<one>"
  let ?zero = "\<zero>"
  let ?iunit = "\<i>"
  let ?caddset = "CplxAdd(R,A)"
  let ?cmulset= "CplxMul(R,A,M)"
  let ?lessrrel = "StrictVersion(CplxROrder(R,A,r))"
  from A1 have "MMIsar0
    (?real, ?complex, ?one, ?zero, ?iunit, ?caddset, ?cmulset, ?lessrrel)"
    "a \<in> ?real"
    using MMIsar_valid by simp;
  then have
    "\<langle>?zero, a\<rangle> \<in>
    ?lessrrel \<inter> ?real \<times> ?real \<union> {\<langle>{?complex}, ?complex\<rangle>} \<union> 
    ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real \<longleftrightarrow>
    \<langle>\<Union>{x \<in> ?complex . ?cmulset ` \<langle>?caddset ` \<langle>?one, ?one\<rangle>, x\<rangle> = a}, a\<rangle> \<in>
    ?lessrrel \<inter> ?real \<times> ?real \<union> 
    {\<langle>{?complex}, ?complex\<rangle>} \<union> ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real"
    by (rule MMIsar0.MMI_halfpos);
  then show ?thesis by simp;
qed;

text{*One more inequality.*}

lemma (in complex0) ledivp1t: 
  assumes A1:  "a \<in> \<real>"   "b \<in> \<real>" and
  A2: "\<zero> \<lsq> a"  "\<zero> \<lsq> b"
  shows "(a\<cdiv>(b \<ca> \<one>))\<cdot>b \<lsq> a"
proof -
  let ?real = "\<real>"
  let ?complex = "\<complex>"
  let ?one = "\<one>"
  let ?zero = "\<zero>"
  let ?iunit = "\<i>"
  let ?caddset = "CplxAdd(R,A)"
  let ?cmulset= "CplxMul(R,A,M)"
  let ?lessrrel = "StrictVersion(CplxROrder(R,A,r))"
  have "MMIsar0
    (?real, ?complex, ?one, ?zero, ?iunit, ?caddset, ?cmulset, ?lessrrel)"
    using MMIsar_valid by simp;
  then have
    "(a \<in> ?real \<and> \<langle>a, ?zero\<rangle> \<notin>
    ?lessrrel \<inter> ?real \<times> ?real \<union> {\<langle>{?complex}, ?complex\<rangle>} \<union> 
    ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real) \<and> 
    b \<in> ?real \<and> \<langle>b, ?zero\<rangle> \<notin> ?lessrrel \<inter> ?real \<times> ?real \<union> 
    {\<langle>{?complex}, ?complex\<rangle>} \<union> ?real \<times> {?complex} \<union>
    {{?complex}} \<times> ?real \<longrightarrow>
    \<langle>a,?cmulset`\<langle>\<Union>{x \<in> ?complex . ?cmulset`\<langle>?caddset`\<langle>b, ?one\<rangle>, x\<rangle> = a}, b\<rangle>\<rangle> \<notin>
    ?lessrrel \<inter> ?real \<times> ?real \<union> {\<langle>{?complex}, ?complex\<rangle>} \<union> 
    ?real \<times> {?complex} \<union> {{?complex}} \<times> ?real"
    by (rule MMIsar0.MMI_ledivp1t)
  with A1 A2 show ?thesis by simp;
qed
end