{- This source file is part of mmisar - a tool for converting
Metamath proofs to Isabelle/Isar.
Copyright 2006 Slawomir Kolodynski

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
-}


module Inner
    where

import Data.List 
import Utils

------------------------ Level 1-----------------------------------------------

-- translates a Metamath formula into an Isar formula. This will probably get very 
-- involved, for now it just deals with quantifiers and 
-- translates the symbols and surrounds with quotes
-- Examples:
-- mmForm2Isar "ph -> A e. CC" == "\<phi> --> A \<in> \<complex>"
-- mmForm2Isar "A e. RR -> A e. CC" == "A \<in> \<real> --> A \<in> \<complex>"
-- mmForm2Isar "E! x e. A ph <-> ( E. x e. A ph /\ A. x e. A A. y e. A ( ( ph /\ ps ) -> x = y ) )"
-- == "(\<exists>! x. x \<in> A \<and> \<phi> ) \<longleftrightarrow> ( ( \<exists> x \<in> A. \<phi> ) \<and> ( \<forall> x
-- \<in> A . \<forall> y \<in> A . ( ( \<phi> \<and> \<psi> ) \<longrightarrow> x = y ) ) )"


mmForm2Isar :: String -> String
--mmForm2Isar s = "\"" ++ (unwords $ (map mmSym2Isar) $ words s) ++ "\""
mmForm2Isar s = 
   "\"" ++ (unwords $ replaceAll mm2IsarTranslTable $ fixNeg $ fixQuants $ words s) ++ "\""

--------------------- Level 2 -------------------------------------------------

-- table of phrase replacements for tranlating metamath formula into Isar formula
-- the table consists of pairs (mmPrhase, isarPhrase), where phrase is a list ow words
-- Most of the phrases are single words, but some are not
mm2IsarTranslTable =
 [ (["CC"], ["\\<complex>"]),
   (["e."], ["\\<in>"]),
   (["ph"], ["\\<phi>"]),
   (["RR"], ["\\<real>"]),
   (["(_"], ["\\<subseteq>"]),
   (["->"], ["\\<longrightarrow>"]),
   (["0"],  ["\\<zero>"]),
   (["1"],  ["\\<one>"]),
   (["=/="],["\\<noteq>"]),
   (["th"], ["\\<theta>"]),
   (["ta"], ["\\<tau>"]),
   (["et"], ["\\<eta>"]),
   (["ze"], ["\\<zeta>"]),
   (["ps"], ["\\<psi>"]),
   (["<->"], ["\\<longleftrightarrow>"]),
   (["/\\"], ["\\<and>"]),
   (["-->"], ["\\<rightarrow>"]),
   (["X."], ["\\<times>"]),
   (["e."], ["\\<in>"]),
   (["i"], ["\\<i>"]),
   (["-u"], ["\\<cn>"]),
   (["-"], ["\\<cs>"]),
   (["E!"], ["\\<exists>!"]),
   (["E."], ["\\<exists>"]),
   (["A."], ["\\<forall>"]),
   (["|"],["."]),
   (["U."],["\\<Union>"]),
   (["e.","V"], ["isASet"]),
   (["x."], ["\\<cdot>"]),
   (["x.", "e."], ["\\<cmulset>", "e."]), -- sometimes \cdot means a set, sometimes notation 
   (["x.", ":"], ["\\<cmulset>", ":"]),
   (["+"], ["\\<ca>"]),
   (["+", "e."], ["\\<caddset>", "e."]), 
   (["+",":"], ["\\<caddset>",":"]) ]

-- corrects quantifiers
-- examples 
-- unwords $ fixQuants $ words "E. x e. CC ( A + x ) = 0" = "E. x e. CC . ( A + x ) == 0"
-- unwords $ fixQuants $ words "A e. CC -> E. x e. CC ( A + x ) = 0" == "A e. CC -> ( E. x e. . CC ( A + x ) = 0 )"
-- unwords $ fixQuants $ words "( A e. CC /\\ A =/= 0 ) -> E. x e. CC ( A x. x ) = 1" ==
-- "( A e. CC /\ A =/= 0 ) -> ( E. x e. CC . ( A x. x ) = 1 )"
-- unwords $ fixQuants $ words "A e. CC" == "A e. CC"
-- unwords $ fixQuants $ words "E. x e. A ph -> ps" == "( E. x e. A . ph ) -> ps"
-- unwords $ fixQuants $ words "E. x e. CC ( A + x ) = 0 -> ( ( A + B ) = ( A + C ) -> B = C )" ==
-- "( E. x e. CC ( A + x ) = 0 ) -> ( ( A + B ) = ( A + C ) -> B = C )"
-- unwords $ fixQuants $ words "E! x e. A ph <-> ( E. x e. A ph /\\ A. x e. A A. y e. A ( ( ph /\\ ps ) -> x = y ) )" ==
-- (E! x . x e. A /\ ph ) <-> ( ( E. x e. A . ph ) /\\ ( A. x e. A . y e. A . ( ( ph /\\ ps ) -> x = y ) ) )"
-- unwords $ fixQuants $ words "( x e. A /\ E! x e. A ph ) -> ( ph <-> U. { x e. A | ph } = x )"
-- == ( x e. A /\ ( E! x e. A . ph ) ) -> ( ph <-> U. { x e. A | ph } = x )"

fixQuants :: [String] -> [String]
fixQuants = quantExpr2words . words2QuantExpr

-- add parantheses around negatives
-- examples:
-- unwords $ fixNeg $ words "-u A = ( 0 - A )" = "( -u A ) = ( 0 - A )"
-- unwords $ fixNeg $ words "( A e. CC -> -u A e. CC )" == "( A e. CC -> ( -u A ) e. CC )
-- unwords $ fixNeg $ words "-u -u ( A + B ) = A + B" == "( -u ( -u ( A + B ) ) ) = A + B"
-- unwords $ fixNeg $ words "-u A = -u B <-> A = B" == "( -u A ) = ( -u B ) <-> A = B  
fixNeg :: [String] -> [String]
fixNeg = negExpr2words . words2NegExpr


--------------------------- Level 3 -------------------------------------------

data QuantExpr = 
   QuantExpr { preQ :: [String],
               quantBlock :: [String],
               postQ :: [String]
             } deriving Show


-- parses string and splits it into three parts: before a quantifier block,
-- the quantifier block and after the quantifier block
words2QuantExpr :: [String] -> QuantExpr
words2QuantExpr w = QuantExpr { preQ = fst bq,
                                quantBlock = fst eb,
                                postQ = snd eb
                              }
                              where 
                                 bq = break isQuant w
                                 eb = extrQblock (snd bq)

-- fixes the problem that quantifiers have different asociation 
-- strength in Metamath and Isar
-- using the expression split into three parts it adds parantheses around 
-- the quantifier expression if needed and inserts a dot at the right place
quantExpr2words :: QuantExpr -> [String]
quantExpr2words qe | (null $ quantBlock qe)                = preQ qe
                   | (null $ preQ qe) &&  (null $ postQ qe) = fixQblock $ quantBlock qe
                   | otherwise = (preQ qe) ++ 
                                 ["("] ++ (fixQblock $ quantBlock qe) ++ [")"] ++
                                 (fixQuants $ postQ qe)

data NegExpr = 
   NegExpr { preN :: [String],
             neqExp :: [String],
             postN :: [String]
           } deriving Show

-- parses a string and splits it into hree parts: before a negative expression,
-- the negative expression and after the neg expression
words2NegExpr :: [String] -> NegExpr
words2NegExpr w = NegExpr { preN = fst bn,
                            neqExp = fst eb,
                            postN = snd eb
                          }
                          where 
                             bn = break (== "-u") w
                             eb = extrNblock (snd bn)

-- negExpr2words adds parantheses around a negative expression
-- just a stub now, needs to be finished
negExpr2words :: NegExpr -> [String]
negExpr2words ne = if null $ neqExp ne then preN ne
                   else (preN ne) ++ ["(","-u"] ++ (fixNeg $ tail $ neqExp ne) ++ [")"] ++ (fixNeg $ postN ne)



----------------------------- Level 4 -----------------------------------------

-- a predicate that tests if given word means a quantifier
isQuant :: String -> Bool
isQuant s = any (s==) ["E.","E!","A."]


-- extrQblock extracts quantifier block from an expression. We assume here 
-- that the expression starts with a quantifier. 
-- Examples extrQblock $ words "E. x e. CC ( A + x ) = 0" == ( words "E. x e. CC ( A + x ) = 0",[] )
--  extrQblock $ words "E. x e. CC ( A + x ) = 0 -> ( ( A + B ) = ( A + C ) -> B = C )" ==
-- (words "E. x e. CC ( A + x ) = 0", words "-> ( ( A + B ) = ( A + C ) -> B = C )"
extrQblock :: [String] -> ([String], [String])
extrQblock [] = ([],[])
extrQblock w = if (not $ isQuant $ head w) then 
                  error ("extrQblock: " ++ (unwords w) ++ 
                  " does not start with a quantifier")
               else (snd3 z, thrd3 z) where
                  z = spanSameNestLev "(" ")" isLogOp (0,[],w) 

-- extrNblock extracts negative expression block from an expression. We assume
-- here that the expression starts with the string "-u"
-- examples (roughly)
-- extrNblock $ words "-u A = ( 0 - A )" == ("-u A", "= ( 0 - A )")
-- extrNblock $ words "-u ( A + B) = -u A + -u B" == ("-u ( A + B)","= -u A + -u B") 
extrNblock :: [String] -> ([String], [String])
extrNblock [] = ([],[])
extrNblock w = if (head w /= "-u") then 
                  error ("extrNblock: " ++ (unwords w) ++ 
                  " does not start with -u ")
                  else (snd3 z, thrd3 z) where
                  z = spanSameNestLev "(" ")" (\s -> (isAlgOp s || isLogOp s) ) (0,[],w) 


-- inserts a dot at the right place in a quantifier block
-- example: fixQblock ["A.","x","e.","A","A.","y","e.","A","(","(","ph","/\\","ps",")","->","x","=","y",")",")"] ==
--"["A.","x","e.","A",".","(","A.","y","e.","A",".","(","ph","/\\","ps",")","->","x","=","y",")",")"]
-- the "E!" (exist unique) gets special treatment since Isar does not have the syntactic
-- sugar of the form "\\<exists>! x \<in> A. Q", this hast to be written ""\\<exists>! x. x \<in> A. \<and> Q"

fixQblock :: [String] -> [String]
fixQblock ws | (length ws) < 3 = error ("the length of qblock " ++ (unwords ws) ++ " less than 3") 
             | head ws == "E!" && ws !! 2 == "e." = 
                  ["E!"] ++ [(ws !! 1)] ++ ["."] ++ [(ws !! 1)] ++ ["e."] ++ [(ws !! 3)] ++ ["/\\"] ++ 
                  (fixQuants $ drop 4 ws)
             | ws !! 2 == "e." = insertAtApply 4 "." fixQuants ws
             | otherwise       = insertAtApply 2 "." fixQuants ws



----------------------- Level 5 -----------------------------------------------


-- a predicate that tests if given word is a logical oparator
isLogOp :: String -> Bool
isLogOp s = any (s==) ["->", "/\\", "<->"]

-- a predicate that tests if a given word is an algebraic operator
-- An "algebraic operator" is whatever may show up in a stetement
-- about numbers
isAlgOp :: String -> Bool
isAlgOp s = any (s==) ["+", "-","x.","=","e."]
