(* 
    This file is a part of MMIsar - a translation of Metamath's set.mm to Isabelle 2005 (ZF logic).

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{MMI\_prelude.thy}*} 

theory MMI_prelude imports equalities

begin

text{*In this theory file we define the context in which theorems
  imported from Metamath are proven and prove the logic and set theory
  Metamath lemmas that the proofs of Metamath theorems about 
  real and complex numbers depend on.*}

section{*Importing from Metamath - how is it done*}

(* Metamath's set.mm features a large (over 8000) collection of theorems 
  proven in the ZFC set theory. We aim at making this collection available
  for Isabelle's ZF logic users.*)

text{* We are interested in importing the theorems about complex numbers
  that start from the "recnt" theorem on. This is done mostly automatically
  by the mmisar tool that is included in the IsarMathLib distribution.
  The tool works as follows:

  First it reads the list of (Metamath) names of theorems 
  that are already imported to IsarMathlib ("known theorems") 
  and the list of theorems that are intended to be imported in this 
  session ("new theorems"). 
  The new theorems are consecutive theorems about complex numbers
  as they appear in the Metamath database.
  Then mmisar creates a "Metamath script" that contains 
  Metamath commands that open a log file and put the stataments
  and proofs of the new theorems in that file in a readable format. 
  The tool writes this script to a disk file and executes metamath 
  with standard input redirected from that file. Then the log file is read
  and its contents converted to the Isar format. In Metamath, 
  the proofs of theorems about complex numbers 
  depend only on 28 axioms of complex numbers and some basic logic and 
  set theory theorems.
  The tool finds which of these dependencies are not known yet and repeats 
  the process of getting their statements from Metamath as with the 
  new theorems. As a result of this process mmisar creates files 
  new\_theorems.thy, new\_deps.thy and new\_known\_theorems.txt.
  The file new\_theorems.thy contains the theorems (with proofs) 
  imported from Metamath in this session. These theorems are added
  (by hand) to the MMIsar.thy file. The file new\_deps.thy contains the 
  statements of new dependencies with generic proofs "by auto".
  These are added to the @{text "MMI_prelude.thy"}.
  Most of the dependencies can be proven automatically by Isabelle.
  Then some manual work has to be done for the dependencies
  that Isabelle can not prove by itself and to correct problems related
  to the fact that Metamath uses a metalogic based on 
  distinct variable constraints (Tarski-Megill metalogic), 
  rather than an explicit notion of free and bound variables.

  The old list of known theorems is replaced by the new list and
  mmisar is ready to convert the next batch of new theorems.
  Of course this never works in practice without tweaking mmisar.hs
  every time a new batch is processed.*}

section{*The context for Metamath theorems*}

text{*We list the Metamth's axioms of complex numbers
  and define notation here.*}

text{*The next definition is what Metamath $X\in V$ is
  translated to. I am not sure why it works, probably because
  Isabelle does a type inference and the "=" sign
  indicates that both sides are sets.*}

consts
   IsASet :: "i\<Rightarrow>o" ("_ isASet" [90] 90)

defs
  set_def [simp]: "X isASet \<equiv>  X = X"

text{*The next locale sets up the context to which Metamath theorems
  about complex numbers are imported. It assumes the axioms
  of complex numbers and defines the notation used for complex numbers.
  
  One of the problems with importing theorems from Metamath is that
  Metamath allows direct infix notation for binary operations so 
  that the notation $a f b$ is allowed where $f$ is a function 
  (that is, a set of pairs). To my knowledge, 
  Isar allows only notation @{text "f`\<langle>a,b\<rangle>"} with a possibility of 
  defining a syntax say @{text "a \<ca> b"} to mean the same as @{text "f`\<langle>a,b\<rangle>"} 
  (please correct me if I am wrong here). This is why we have
  two objects for addition: one called @{text "caddset"} that represents
  the binary function, and the second one called @{text "ca"} which 
  defines the @{text "a \<ca> b"} notation for @{text "caddset`\<langle>a,b\<rangle>"}.
  The same applies to multiplication of real numbers.*}

locale MMIsar0 =
  fixes real ("\<real>")
  fixes complex ("\<complex>")
  fixes one :: i ("\<one>")
  fixes zero :: i ("\<zero>")
  fixes caddset ("\<caddset>")
  fixes cmulset ("\<cmulset>")
  fixes iunit :: i ("\<i>")

  fixes ca (infixl "\<ca>" 69)
  defines ca_def: "a \<ca> b \<equiv> \<caddset>`\<langle>a,b\<rangle>"
  fixes cm (infixl "\<cdot>" 71)
  defines cm_def: "a \<cdot> b \<equiv> \<cmulset>`\<langle>a,b\<rangle>"
  fixes sub (infixl "\<cs>" 69)
  defines sub_def: "a \<cs> b \<equiv> \<Union> { x \<in> \<complex>. b \<ca> x = a }"
  fixes cneg :: "i\<Rightarrow>i" ("\<cn>_" 95)
  defines cneg_def: "\<cn> a \<equiv> \<zero> \<cs> a"

  assumes MMI_axresscn: "\<real> \<subseteq> \<complex>"
  assumes MMI_ax1ne0: "\<one> \<noteq> \<zero>"
  assumes MMI_axcnex: "\<complex> isASet"
  assumes MMI_axaddopr: "\<caddset> : ( \<complex> \<times> \<complex> ) \<rightarrow> \<complex>"
  assumes MMI_axmulopr: "\<cmulset> : ( \<complex> \<times> \<complex> ) \<rightarrow> \<complex>"
  assumes MMI_axmulcom: "A \<in> \<complex> \<and> B \<in> \<complex> \<longrightarrow> A \<cdot> B = B \<cdot> A"
  assumes MMI_axaddcl: "A \<in> \<complex> \<and> B \<in> \<complex> \<longrightarrow> A \<ca> B \<in> \<complex>"
  assumes MMI_axmulcl: "A \<in> \<complex> \<and> B \<in> \<complex> \<longrightarrow> A \<cdot> B \<in> \<complex>"
  assumes MMI_axdistr: 
  "A \<in> \<complex> \<and> B \<in> \<complex> \<and> C \<in> \<complex> \<longrightarrow> A\<cdot>(B \<ca> C) = A\<cdot>B \<ca> A\<cdot>C" 
  assumes MMI_axaddcom: "A \<in> \<complex> \<and> B \<in> \<complex> \<longrightarrow> A \<ca> B = B \<ca> A"
  assumes MMI_axaddass: 
  "A \<in> \<complex> \<and> B \<in> \<complex> \<and> C \<in> \<complex> \<longrightarrow> A \<ca> B \<ca> C = A \<ca> (B \<ca> C)"
  assumes MMI_axmulass: 
  "A \<in> \<complex> \<and> B \<in> \<complex> \<and> C \<in> \<complex> \<longrightarrow> A \<cdot> B \<cdot> C = A \<cdot> (B \<cdot> C)"
  assumes MMI_ax1re: "\<one> \<in> \<real>"
  assumes MMI_axi2m1: "\<i> \<cdot> \<i> \<ca> \<one> = \<zero>"
  assumes MMI_ax0id: "A \<in> \<complex> \<longrightarrow> A \<ca> \<zero> = A"
  assumes MMI_axicn: "\<i> \<in> \<complex>"
  assumes MMI_axnegex: "A \<in> \<complex> \<longrightarrow> ( \<exists> x \<in> \<complex>. ( A \<ca> x ) = \<zero> )"
  assumes MMI_axrecex: "A \<in> \<complex> \<and> A \<noteq> \<zero> \<longrightarrow> ( \<exists> x \<in> \<complex>. A \<cdot> x = \<one>)"
  assumes MMI_ax1id: "A \<in> \<complex> \<longrightarrow> A \<cdot> \<one> = A"
  assumes MMI_axaddrcl: "A \<in> \<real> \<and> B \<in> \<real> \<longrightarrow> A \<ca> B \<in> \<real>"
  assumes MMI_axmulrcl: "A \<in> \<real> \<and> B \<in> \<real> \<longrightarrow> A \<cdot> B \<in> \<real>"
  assumes MMI_axrnegex: "A \<in> \<real> \<longrightarrow> ( \<exists> x \<in> \<real>. A \<ca> x = \<zero> )"
  
section{*Basic Metamath theorems*}

text{* This section contains Metamath theorems that the more advanced 
  theorems from @{text "MMIsar.thy"} depend on. Most of these theorems 
  are proven automatically by Isabelle, some have to be proven by hand 
  and some have to be modified to convert from Tarski-Megill 
  metalogic used by Metamath to one based on explicit notion of 
  free and bound variables. *}   

lemma MMI_ax_mp: assumes "\<phi>" and "\<phi> \<longrightarrow> \<psi>" shows "\<psi>"
  using prems by auto;

lemma MMI_sseli: assumes A1: "A \<subseteq> B"   
   shows "C \<in> A \<longrightarrow> C \<in> B"
   using prems by auto

lemma MMI_sselii: assumes A1: "A \<subseteq> B" and
    A2: "C \<in> A"   
   shows "C \<in> B"
   using prems by auto

lemma MMI_syl: assumes A1: "\<phi> \<longrightarrow> ps" and
    A2: "ps \<longrightarrow> ch"   
   shows "\<phi> \<longrightarrow> ch"
   using prems by auto;

lemma MMI_elimhyp: assumes A1: "A = if ( \<phi> , A , B ) \<longrightarrow> ( \<phi> \<longleftrightarrow> \<psi> )" and
    A2: "B = if ( \<phi> , A , B ) \<longrightarrow> ( ch \<longleftrightarrow> \<psi> )" and
    A3: "ch"   
   shows "\<psi>"
proof -
  { assume "\<phi>"
    with A1 have "\<psi>" by simp; }
  moreover
  { assume "\<not>\<phi>"
    with A2 A3 have "\<psi>" by simp; }
  ultimately show "\<psi>" by auto;
qed;

lemma MMI_neeq1: 
   shows "A = B \<longrightarrow> ( A \<noteq> C \<longleftrightarrow> B \<noteq> C )"
  by auto;

lemma MMI_mp2: assumes A1: "\<phi>" and
    A2: "\<psi>" and
    A3: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> chi )"   
   shows "chi"
   using prems by auto;

lemma MMI_xpex: assumes A1: "A isASet" and
    A2: "B isASet"   
   shows "( A \<times> B ) isASet"
   using prems by auto

lemma MMI_fex: 
   shows 
  "A \<in> C \<longrightarrow> ( F : A \<rightarrow> B \<longrightarrow> F isASet )"
  "A isASet \<longrightarrow> ( F : A \<rightarrow> B \<longrightarrow> F isASet )"
  by auto;

lemma MMI_3eqtr4d: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> C = A" and
    A3: "\<phi> \<longrightarrow> D = B"   
   shows "\<phi> \<longrightarrow> C = D"
   using prems by auto

lemma MMI_3coml: assumes A1: "( \<phi> \<and> \<psi> \<and> chi ) \<longrightarrow> th"   
   shows "( \<psi> \<and> chi \<and> \<phi> ) \<longrightarrow> th"
   using prems by auto;

lemma MMI_sylan: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> chi" and
    A2: "th \<longrightarrow> \<phi>"   
   shows "( th \<and> \<psi> ) \<longrightarrow> chi"
   using prems by auto;

lemma MMI_3impa: assumes A1: "( ( \<phi> \<and> \<psi> ) \<and> chi ) \<longrightarrow> th"   
   shows "( \<phi> \<and> \<psi> \<and> chi ) \<longrightarrow> th"
   using prems by auto;

lemma MMI_3adant2: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> chi"   
   shows "( \<phi> \<and> th \<and> \<psi> ) \<longrightarrow> chi"
   using prems by auto;

lemma MMI_3adant1: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> chi"   
   shows "( th \<and> \<phi> \<and> \<psi> ) \<longrightarrow> chi"
   using prems by auto;

lemma (in MMIsar0) MMI_opreq12d: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> C = D"   
   shows 
  "\<phi> \<longrightarrow> ( A \<ca> C ) = ( B \<ca> D )"
  "\<phi> \<longrightarrow> ( A \<cdot> C ) = ( B \<cdot> D )"
  "\<phi> \<longrightarrow> ( A \<cs> C ) = ( B \<cs> D )"
   using prems by auto;

lemma MMI_mp2an: assumes A1: "\<phi>" and
    A2: "\<psi>" and
    A3: "( \<phi> \<and> \<psi> ) \<longrightarrow> chi"   
   shows "chi"
   using prems by auto;

lemma MMI_mp3an: assumes A1: "\<phi>" and
    A2: "\<psi>" and
    A3: "ch" and
    A4: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "\<theta>"
   using prems by auto

lemma MMI_eqeltrr: assumes A1: "A = B" and
    A2: "A \<in> C"   
   shows "B \<in> C"
   using prems by auto

lemma MMI_eqtr: assumes A1: "A = B" and
    A2: "B = C"   
   shows "A = C"
   using prems by auto;

(*********************10-20 ******************************************)

lemma MMI_impbi: assumes A1: "\<phi> \<longrightarrow> \<psi>" and
    A2: "\<psi> \<longrightarrow> \<phi>"   
   shows "\<phi> \<longleftrightarrow> \<psi>"
proof;
  assume "\<phi>" with A1 show "\<psi>" by simp;
next
  assume "\<psi>" with A2 show "\<phi>" by simp;
qed;

lemma MMI_mp3an3: assumes A1: "ch" and
    A2: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "( \<phi> \<and> \<psi> ) \<longrightarrow> \<theta>"
   using prems by auto;

lemma MMI_eqeq12d: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> C = D"   
   shows "\<phi> \<longrightarrow> ( A = C \<longleftrightarrow> B = D )"
   using prems by auto

lemma MMI_mpan2: assumes A1: "\<psi>" and
    A2: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "\<phi> \<longrightarrow> ch"
   using prems by auto;

lemma (in MMIsar0) MMI_opreq2: 
   shows 
  "A = B \<longrightarrow> ( C \<ca> A ) = ( C \<ca> B )"
  "A = B \<longrightarrow> ( C \<cdot> A ) = ( C \<cdot> B )"
  "A = B \<longrightarrow> ( C \<cs> A ) = ( C \<cs> B )"
  by auto;

lemma MMI_syl5bir: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "\<theta> \<longrightarrow> ch"   
   shows "\<phi> \<longrightarrow> ( \<theta> \<longrightarrow> \<psi> )"
   using prems by auto

lemma MMI_adantr: assumes A1: "\<phi> \<longrightarrow> \<psi>"   
   shows "( \<phi> \<and> ch ) \<longrightarrow> \<psi>"
   using prems by auto

lemma MMI_mpan: assumes A1: "\<phi>" and
    A2: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "\<psi> \<longrightarrow> ch"
   using prems by auto;

lemma MMI_eqeq1d: assumes A1: "\<phi> \<longrightarrow> A = B"   
   shows "\<phi> \<longrightarrow> ( A = C \<longleftrightarrow> B = C )"
   using prems by auto

lemma (in MMIsar0) MMI_opreq1: 
   shows 
  "A = B \<longrightarrow> ( A \<cdot> C ) = ( B \<cdot> C )"
  "A = B \<longrightarrow> ( A \<ca> C ) = ( B \<ca> C )"
  "A = B \<longrightarrow> ( A \<cs> C ) = ( B \<cs> C )"
  by auto;

lemma MMI_syl6eq: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "B = C"   
   shows "\<phi> \<longrightarrow> A = C"
   using prems by auto

lemma MMI_syl6bi: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "ch \<longrightarrow> \<theta>"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> \<theta> )"
   using prems by auto

lemma MMI_imp: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ch )"   
   shows "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_sylibd: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ch )" and
    A2: "\<phi> \<longrightarrow> ( ch \<longleftrightarrow> \<theta> )"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> \<theta> )"
   using prems by auto

lemma MMI_ex: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ch )"
   using prems by auto

lemma MMI_r19_23aiv: assumes A1: "\<forall>x.  (x \<in> A  \<longrightarrow> (\<phi>(x) \<longrightarrow> \<psi> ))"   
   shows "( \<exists> x \<in> A . \<phi>(x) ) \<longrightarrow> \<psi>"
  using prems by auto;

lemma MMI_bitr: assumes A1: "\<phi> \<longleftrightarrow> \<psi>" and
    A2: "\<psi> \<longleftrightarrow> ch"   
   shows "\<phi> \<longleftrightarrow> ch"
   using prems by auto

lemma MMI_eqeq12i: assumes A1: "A = B" and
    A2: "C = D"   
   shows "A = C \<longleftrightarrow> B = D"
   using prems by auto

lemma MMI_dedth3h: 
  assumes A1: "A = if ( \<phi> , A , D ) \<longrightarrow> ( \<theta> \<longleftrightarrow> ta )" and
    A2: "B = if ( \<psi> , B , R ) \<longrightarrow> ( ta \<longleftrightarrow> et )" and
    A3: "C = if ( ch , C , S ) \<longrightarrow> ( et \<longleftrightarrow> ze )" and
    A4: "ze"   
   shows "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"
   using prems by auto

lemma MMI_bibi1d: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )"   
   shows "\<phi> \<longrightarrow> ( ( \<psi> \<longleftrightarrow> \<theta> ) \<longleftrightarrow> ( ch \<longleftrightarrow> \<theta> ) )"
   using prems by auto

lemma MMI_eqeq1: 
   shows "A = B \<longrightarrow> ( A = C \<longleftrightarrow> B = C )"
  by auto

lemma MMI_bibi12d: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "\<phi> \<longrightarrow> ( \<theta> \<longleftrightarrow> ta )"   
   shows "\<phi> \<longrightarrow> ( ( \<psi> \<longleftrightarrow> \<theta> ) \<longleftrightarrow> ( ch \<longleftrightarrow> ta ) )"
   using prems by auto

lemma MMI_eqeq2d: assumes A1: "\<phi> \<longrightarrow> A = B"   
   shows "\<phi> \<longrightarrow> ( C = A \<longleftrightarrow> C = B )"
   using prems by auto

lemma MMI_eqeq2: 
   shows "A = B \<longrightarrow> ( C = A \<longleftrightarrow> C = B )"
  by auto

lemma MMI_elimel: assumes A1: "B \<in> C"   
   shows "if ( A \<in> C , A , B ) \<in> C"
   using prems by auto

lemma MMI_3adant3: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( \<phi> \<and> \<psi> \<and> \<theta> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_bitr3d: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> \<theta> )"   
   shows "\<phi> \<longrightarrow> ( ch \<longleftrightarrow> \<theta> )"
   using prems by auto;

(****************** 20-30 add12t - peano2cn *************)

lemma MMI_3eqtr3d: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> A = C" and
    A3: "\<phi> \<longrightarrow> B = D"   
   shows "\<phi> \<longrightarrow> C = D"
   using prems by auto;

lemma (in MMIsar0) MMI_opreq1d: assumes A1: "\<phi> \<longrightarrow> A = B"   
   shows 
  "\<phi> \<longrightarrow> ( A \<ca> C ) = ( B \<ca> C )"
  "\<phi> \<longrightarrow> ( A \<cs> C ) = ( B \<cs> C )"
  "\<phi> \<longrightarrow> ( A \<cdot> C ) = ( B \<cdot> C )"
   using prems by auto;

lemma MMI_3com12: assumes A1: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "( \<psi> \<and> \<phi> \<and> ch ) \<longrightarrow> \<theta>"
   using prems by auto;

lemma (in MMIsar0) MMI_opreq2d: assumes A1: "\<phi> \<longrightarrow> A = B"   
   shows 
  "\<phi> \<longrightarrow> ( C \<ca> A ) = ( C \<ca> B )"
  "\<phi> \<longrightarrow> ( C \<cs> A ) = ( C \<cs> B )"
  "\<phi> \<longrightarrow> ( C \<cdot> A ) = ( C \<cdot> B )"
   using prems by auto;

lemma MMI_3com23: assumes A1: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "( \<phi> \<and> ch \<and> \<psi> ) \<longrightarrow> \<theta>"
   using prems by auto;

lemma MMI_3expa: assumes A1: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "( ( \<phi> \<and> \<psi> ) \<and> ch ) \<longrightarrow> \<theta>"
   using prems by auto;

lemma MMI_adantrr: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( \<phi> \<and> ( \<psi> \<and> \<theta> ) ) \<longrightarrow> ch"
   using prems by auto;

lemma MMI_3expb: assumes A1: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "( \<phi> \<and> ( \<psi> \<and> ch ) ) \<longrightarrow> \<theta>"
   using prems by auto;

lemma MMI_an4s: assumes A1: "( ( \<phi> \<and> \<psi> ) \<and> ( ch \<and> \<theta> ) ) \<longrightarrow> \<tau>"   
   shows "( ( \<phi> \<and> ch ) \<and> ( \<psi> \<and> \<theta> ) ) \<longrightarrow> \<tau>"
   using prems by auto;

lemma MMI_eqtrd: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> B = C"   
   shows "\<phi> \<longrightarrow> A = C"
   using prems by auto;

lemma MMI_ad2ant2l: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( ( \<theta> \<and> \<phi> ) \<and> ( \<tau> \<and> \<psi> ) ) \<longrightarrow> ch"
   using prems by auto;

lemma MMI_pm3_2i: assumes A1: "\<phi>" and
    A2: "\<psi>"   
   shows "\<phi> \<and> \<psi>"
   using prems by auto;

lemma (in MMIsar0) MMI_opreq2i: assumes A1: "A = B"   
   shows "( C \<ca> A ) = ( C \<ca> B )"
   using prems by auto;

(************31,33 peano2re, negeu, subval ******************************)

lemma MMI_mpbir2an: assumes A1: "\<phi> \<longleftrightarrow> ( \<psi> \<and> ch )" and
    A2: "\<psi>" and
    A3: "ch"   
   shows "\<phi>"
   using prems by auto;

lemma MMI_reu4: assumes A1: "\<forall>x y. x = y \<longrightarrow> ( \<phi>(x) \<longleftrightarrow> \<psi>(y) )"   
   shows "( \<exists>! x . x \<in> A \<and> \<phi>(x) ) \<longleftrightarrow> 
  ( ( \<exists> x \<in> A . \<phi>(x) ) \<and> ( \<forall> x \<in> A . \<forall> y \<in> A . 
  ( ( \<phi>(x) \<and> \<psi>(y) ) \<longrightarrow> x = y ) ) )"
   using prems by auto;

lemma MMI_risset: 
   shows "A \<in> B \<longleftrightarrow> ( \<exists> x \<in> B . x = A )"
  by auto;

lemma MMI_sylib: assumes A1: "\<phi> \<longrightarrow> \<psi>" and
    A2: "\<psi> \<longleftrightarrow> ch"   
   shows "\<phi> \<longrightarrow> ch"
   using prems by auto;

lemma MMI_mp3an13: assumes A1: "\<phi>" and
    A2: "ch" and
    A3: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "\<psi> \<longrightarrow> \<theta>"
   using prems by auto;

lemma MMI_eqcomd: assumes A1: "\<phi> \<longrightarrow> A = B"   
   shows "\<phi> \<longrightarrow> B = A"
   using prems by auto;

lemma MMI_sylan9eqr: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<psi> \<longrightarrow> B = C"   
   shows "( \<psi> \<and> \<phi> ) \<longrightarrow> A = C"
   using prems by auto;

lemma MMI_exp32: assumes A1: "( \<phi> \<and> ( \<psi> \<and> ch ) ) \<longrightarrow> \<theta>"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ( ch \<longrightarrow> \<theta> ) )"
   using prems by auto;

lemma MMI_impcom: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ch )"   
   shows "( \<psi> \<and> \<phi> ) \<longrightarrow> ch"
   using prems by auto;

lemma MMI_a1d: assumes A1: "\<phi> \<longrightarrow> \<psi>"   
   shows "\<phi> \<longrightarrow> ( ch \<longrightarrow> \<psi> )"
   using prems by auto;

lemma MMI_r19_21aiv: assumes A1: "\<forall>x. \<phi> \<longrightarrow> ( x \<in> A \<longrightarrow> \<psi>(x) )"   
   shows "\<phi> \<longrightarrow> ( \<forall> x \<in> A . \<psi>(x) )"
   using prems by auto;

lemma MMI_r19_22: 
   shows "( \<forall> x \<in> A . ( \<phi>(x) \<longrightarrow> \<psi>(x) ) ) \<longrightarrow> 
  ( ( \<exists> x \<in> A . \<phi>(x) ) \<longrightarrow> ( \<exists> x \<in> A . \<psi>(x) ) )"
  by auto;

lemma MMI_syl6: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ch )" and
    A2: "ch \<longrightarrow> \<theta>"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> \<theta> )"
   using prems by auto;

lemma MMI_mpid: assumes A1: "\<phi> \<longrightarrow> ch" and
    A2: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ( ch \<longrightarrow> \<theta> ) )"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> \<theta> )"
   using prems by auto;

lemma MMI_eqtr3t: 
   shows "( A = C \<and> B = C ) \<longrightarrow> A = B"
  by auto;

lemma MMI_syl5bi: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "\<theta> \<longrightarrow> \<psi>"   
   shows "\<phi> \<longrightarrow> ( \<theta> \<longrightarrow> ch )"
   using prems by auto;

lemma MMI_mp3an1: assumes A1: "\<phi>" and
    A2: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "( \<psi> \<and> ch ) \<longrightarrow> \<theta>"
   using prems by auto;

lemma MMI_rgen2: assumes A1: "\<forall>x y. ( x \<in> A \<and> y \<in> A ) \<longrightarrow> \<phi>(x,y)"   
   shows "\<forall> x \<in> A . \<forall> y \<in> A . \<phi>(x,y)"
   using prems by auto;

(*************** 35-37 negeq-negeqd **************************)

lemma MMI_ax_17: shows "\<phi> \<longrightarrow> (\<forall>x. \<phi>)" by simp;


lemma MMI_3eqtr4g: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "C = A" and
    A3: "D = B"   
   shows "\<phi> \<longrightarrow> C = D"
   using prems by auto;

(*** hbneq ***************************************************)

lemma MMI_3imtr4: assumes A1: "\<phi> \<longrightarrow> \<psi>" and
    A2: "ch \<longleftrightarrow> \<phi>" and
    A3: "\<theta> \<longleftrightarrow> \<psi>"   
   shows "ch \<longrightarrow> \<theta>"
   using prems by auto

(*lemma MMI_hbopr: assumes A1: "y \<in> A \<longrightarrow> ( \<forall> x . y \<in> A )" and
    A2: "y \<in> F \<longrightarrow> ( \<forall> x . y \<in> F )" and
    A3: "y \<in> B \<longrightarrow> ( \<forall> x . y \<in> B )"   
   shows "y \<in> ( A F B ) \<longrightarrow> ( \<forall> x . y \<in> ( A F B ) )"
   using prems by auto 
  no way to translate hopefuly we will manage to avoid using this*)

lemma MMI_eleq2i: assumes A1: "A = B"   
   shows "C \<in> A \<longleftrightarrow> C \<in> B"
   using prems by auto

lemma MMI_albii: assumes A1: "\<phi> \<longleftrightarrow> \<psi>"   
   shows "( \<forall> x . \<phi> ) \<longleftrightarrow> ( \<forall> x . \<psi> )"
   using prems by auto;

(*************subcl-subadd **********************************)
lemma MMI_reucl: 
   shows "( \<exists>! x . x \<in> A \<and> \<phi>(x) ) \<longrightarrow> \<Union> { x \<in> A . \<phi>(x) } \<in> A"
proof;
  assume A1: "\<exists>! x . x \<in> A \<and> \<phi>(x)"
  then obtain a where I: "a\<in>A"  and "\<phi>(a)" by auto;
  with A1 have "{ x \<in> A . \<phi>(x) } = {a}" by blast;
  with I show " \<Union> { x \<in> A . \<phi>(x) } \<in> A" by simp;
qed;

lemma MMI_dedth2h: assumes A1: "A = if ( \<phi> , A , C ) \<longrightarrow> ( ch \<longleftrightarrow> \<theta> )" and
    A2: "B = if ( \<psi> , B , D ) \<longrightarrow> ( \<theta> \<longleftrightarrow> \<tau> )" and
    A3: "\<tau>"   
   shows "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_eleq1d: assumes A1: "\<phi> \<longrightarrow> A = B"   
   shows "\<phi> \<longrightarrow> ( A \<in> C \<longleftrightarrow> B \<in> C )"
   using prems by auto

lemma MMI_syl5eqel: assumes A1: "\<phi> \<longrightarrow> A \<in> B" and
    A2: "C = A"   
   shows "\<phi> \<longrightarrow> C \<in> B"
   using prems by auto

(** a lemma in ZF that roughly corresponds to Mematamath euuni **)

lemma IML_eeuni: assumes A1: "x \<in> A" and A2: "\<exists>! t . t \<in> A \<and> \<phi>(t)"
  shows "\<phi>(x) \<longleftrightarrow> \<Union> { x \<in> A . \<phi>(x) } = x"
proof;
  assume "\<phi>(x)" 
  with A1 A2 show "\<Union> { x \<in> A . \<phi>(x) } = x" by auto;
next assume A3: "\<Union> { x \<in> A . \<phi>(x) } = x"
  from A2 obtain y where "y\<in>A" and I: "\<phi>(y)" by auto;
  with A2 A3 have "x = y" by auto;
  with I show "\<phi>(x)" by simp;
qed;
    
lemma MMI_reuuni1: 
   shows "( x \<in> A \<and> ( \<exists>! x . x \<in> A \<and> \<phi>(x) ) ) \<longrightarrow> 
  ( \<phi>(x) \<longleftrightarrow> \<Union> { x \<in> A . \<phi>(x) } = x )"
  using IML_eeuni by simp;

lemma MMI_eqeq1i: assumes A1: "A = B"   
   shows "A = C \<longleftrightarrow> B = C"
   using prems by auto

lemma MMI_syl6rbbr: assumes A1: "\<forall>x. \<phi>(x) \<longrightarrow> ( \<psi>(x) \<longleftrightarrow> ch(x) )" and
    A2: "\<forall>x. \<theta>(x) \<longleftrightarrow> ch(x)"   
   shows "\<forall> x. \<phi>(x) \<longrightarrow> ( \<theta>(x) \<longleftrightarrow> \<psi>(x) )"
   using prems by auto;

(*** the original version of MMI_syl6rbbr without quantifiers **********)

lemma MMI_syl6rbbrA: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "\<theta> \<longleftrightarrow> ch"   
   shows "\<phi> \<longrightarrow> ( \<theta> \<longleftrightarrow> \<psi> )"
   using prems by auto;

lemma MMI_vtoclga: assumes A1: "\<forall>x. x = A \<longrightarrow> ( \<phi>(x) \<longleftrightarrow> \<psi>)" and
    A2: "\<forall>x. x \<in> B \<longrightarrow> \<phi>(x)"
   shows "A \<in> B \<longrightarrow> \<psi>"
   using prems by auto;

(************  subsub23 - addsubt ******************)

lemma MMI_3bitr4: assumes A1: "\<phi> \<longleftrightarrow> \<psi>" and
    A2: "ch \<longleftrightarrow> \<phi>" and
    A3: "\<theta> \<longleftrightarrow> \<psi>"   
   shows "ch \<longleftrightarrow> \<theta>"
   using prems by auto

lemma MMI_mpbii: assumes Amin: "\<psi>" and
    Amaj: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )"   
   shows "\<phi> \<longrightarrow> ch"
   using prems by auto

lemma MMI_eqid: 
   shows "A = A"
  by auto

lemma MMI_pm3_27: 
   shows "( \<phi> \<and> \<psi> ) \<longrightarrow> \<psi>"
  by auto

lemma MMI_pm3_26: 
   shows "( \<phi> \<and> \<psi> ) \<longrightarrow> \<phi>"
  by auto

lemma MMI_ancoms: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( \<psi> \<and> \<phi> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_syl3anc: assumes A1: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>" and
    A2: "\<tau> \<longrightarrow> \<phi>" and
    A3: "\<tau> \<longrightarrow> \<psi>" and
    A4: "\<tau> \<longrightarrow> ch"   
   shows "\<tau> \<longrightarrow> \<theta>"
   using prems by auto

lemma MMI_syl5eq: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "C = A"   
   shows "\<phi> \<longrightarrow> C = B"
   using prems by auto

lemma MMI_eqcomi: assumes A1: "A = B"   
   shows "B = A"
   using prems by auto

lemma MMI_3eqtr: assumes A1: "A = B" and
    A2: "B = C" and
    A3: "C = D"   
   shows "A = D"
   using prems by auto

lemma MMI_mpbir: assumes Amin: "\<psi>" and
    Amaj: "\<phi> \<longleftrightarrow> \<psi>"   
   shows "\<phi>"
   using prems by auto

lemma MMI_syl3an3: assumes A1: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>" and
    A2: "\<tau> \<longrightarrow> ch"   
   shows "( \<phi> \<and> \<psi> \<and> \<tau> ) \<longrightarrow> \<theta>"
   using prems by auto

lemma MMI_3eqtrd: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> B = C" and
    A3: "\<phi> \<longrightarrow> C = D"   
   shows "\<phi> \<longrightarrow> A = D"
   using prems by auto

lemma MMI_syl5: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ch )" and
    A2: "\<theta> \<longrightarrow> \<psi>"   
   shows "\<phi> \<longrightarrow> ( \<theta> \<longrightarrow> ch )"
   using prems by auto

lemma MMI_exp3a: assumes A1: "\<phi> \<longrightarrow> ( ( \<psi> \<and> ch ) \<longrightarrow> \<theta> )"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ( ch \<longrightarrow> \<theta> ) )"
   using prems by auto

lemma MMI_com12: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ch )"   
   shows "\<psi> \<longrightarrow> ( \<phi> \<longrightarrow> ch )"
   using prems by auto

lemma MMI_3imp: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> ( ch \<longrightarrow> \<theta> ) )"   
   shows "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"
   using prems by auto;

(********* addsub12t-subidt *************)

lemma MMI_3eqtr3: assumes A1: "A = B" and
    A2: "A = C" and
    A3: "B = D"   
   shows "C = D"
   using prems by auto

lemma (in MMIsar0) MMI_opreq1i: assumes A1: "A = B"   
   shows 
  "( A \<ca> C ) = ( B \<ca> C )"
  "( A \<cs> C ) = ( B \<cs> C )"
  "( A \<cdot> C ) = ( B \<cdot> C )"
   using prems by auto;

lemma MMI_eqtr3: assumes A1: "A = B" and
    A2: "A = C"   
   shows "B = C"
   using prems by auto

lemma MMI_dedth: assumes A1: "A = if ( \<phi> , A , B ) \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "ch"   
   shows "\<phi> \<longrightarrow> \<psi>"
   using prems by auto

lemma MMI_id: 
   shows "\<phi> \<longrightarrow> \<phi>"
  by auto

lemma MMI_eqtr3d: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> A = C"   
   shows "\<phi> \<longrightarrow> B = C"
   using prems by auto

lemma MMI_sylan2: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch" and
    A2: "\<theta> \<longrightarrow> \<psi>"   
   shows "( \<phi> \<and> \<theta> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_adantl: assumes A1: "\<phi> \<longrightarrow> \<psi>"   
   shows "( ch \<and> \<phi> ) \<longrightarrow> \<psi>"
   using prems by auto

lemma (in MMIsar0) MMI_opreq12: 
   shows 
  "( A = B \<and> C = D ) \<longrightarrow> ( A \<ca> C ) = ( B \<ca> D )"
  "( A = B \<and> C = D ) \<longrightarrow> ( A \<cs> C ) = ( B \<cs> D )"
  by auto

lemma MMI_anidms: assumes A1: "( \<phi> \<and> \<phi> ) \<longrightarrow> \<psi>"   
   shows "\<phi> \<longrightarrow> \<psi>"
   using prems by auto;

(******** subid1t-neg11 *************************)

lemma MMI_anabsan2: assumes A1: "( \<phi> \<and> ( \<psi> \<and> \<psi> ) ) \<longrightarrow> ch"   
   shows "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_3simp2: 
   shows "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<psi>"
  by auto

lemma MMI_3simp3: 
   shows "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> ch"
  by auto

lemma MMI_sylbir: assumes A1: "\<psi> \<longleftrightarrow> \<phi>" and
    A2: "\<psi> \<longrightarrow> ch"   
   shows "\<phi> \<longrightarrow> ch"
   using prems by auto

lemma MMI_3eqtr3g: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "A = C" and
    A3: "B = D"   
   shows "\<phi> \<longrightarrow> C = D"
   using prems by auto

lemma MMI_3bitr: assumes A1: "\<phi> \<longleftrightarrow> \<psi>" and
    A2: "\<psi> \<longleftrightarrow> ch" and
    A3: "ch \<longleftrightarrow> \<theta>"   
   shows "\<phi> \<longleftrightarrow> \<theta>"
   using prems by auto;

(************ negcon1-subeq0t**************)

lemma MMI_3bitr3: assumes A1: "\<phi> \<longleftrightarrow> \<psi>" and
    A2: "\<phi> \<longleftrightarrow> ch" and
    A3: "\<psi> \<longleftrightarrow> \<theta>"   
   shows "ch \<longleftrightarrow> \<theta>"
   using prems by auto

lemma MMI_eqcom: 
   shows "A = B \<longleftrightarrow> B = A"
  by auto

lemma MMI_syl6bb: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "ch \<longleftrightarrow> \<theta>"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> \<theta> )"
   using prems by auto

lemma MMI_3bitr3d: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> \<theta> )" and
    A3: "\<phi> \<longrightarrow> ( ch \<longleftrightarrow> \<tau> )"   
   shows "\<phi> \<longrightarrow> ( \<theta> \<longleftrightarrow> \<tau> )"
   using prems by auto

lemma MMI_syl3an2: assumes A1: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>" and
    A2: "\<tau> \<longrightarrow> \<psi>"   
   shows "( \<phi> \<and> \<tau> \<and> ch ) \<longrightarrow> \<theta>"
   using prems by auto;

(********neg0-0re ********************)

lemma MMI_df_rex: 
   shows "( \<exists> x \<in> A . \<phi>(x) ) \<longleftrightarrow> ( \<exists> x . ( x \<in> A \<and> \<phi>(x) ) )"
  by auto;

lemma MMI_mpbi: assumes Amin: "\<phi>" and
    Amaj: "\<phi> \<longleftrightarrow> \<psi>"   
   shows "\<psi>"
   using prems by auto

lemma MMI_mp3an12: assumes A1: "\<phi>" and
    A2: "\<psi>" and
    A3: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "ch \<longrightarrow> \<theta>"
   using prems by auto

lemma MMI_syl5bb: assumes A1: "\<phi> \<longrightarrow> ( \<psi> \<longleftrightarrow> ch )" and
    A2: "\<theta> \<longleftrightarrow> \<psi>"   
   shows "\<phi> \<longrightarrow> ( \<theta> \<longleftrightarrow> ch )"
   using prems by auto

lemma MMI_eleq1a: 
   shows "A \<in> B \<longrightarrow> ( C = A \<longrightarrow> C \<in> B )"
  by auto

lemma MMI_sylbird: assumes A1: "\<phi> \<longrightarrow> ( ch \<longleftrightarrow> \<psi> )" and
    A2: "\<phi> \<longrightarrow> ( ch \<longrightarrow> \<theta> )"   
   shows "\<phi> \<longrightarrow> ( \<psi> \<longrightarrow> \<theta> )"
   using prems by auto

lemma MMI_19_23aiv: assumes A1: "\<forall>x. \<phi>(x) \<longrightarrow> \<psi>"   
   shows "( \<exists> x . \<phi>(x) ) \<longrightarrow> \<psi>"
   using prems by auto;

lemma MMI_eqeltrrd: assumes A1: "\<phi> \<longrightarrow> A = B" and
    A2: "\<phi> \<longrightarrow> A \<in> C"   
   shows "\<phi> \<longrightarrow> B \<in> C"
   using prems by auto

lemma MMI_syl2an: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch" and
    A2: "\<theta> \<longrightarrow> \<phi>" and
    A3: "\<tau> \<longrightarrow> \<psi>"   
   shows "( \<theta> \<and> \<tau> ) \<longrightarrow> ch"
   using prems by auto;

(*********** mulid2t-muladdt *********************)

lemma MMI_adantrl: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( \<phi> \<and> ( \<theta> \<and> \<psi> ) ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_ad2ant2r: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( ( \<phi> \<and> \<theta> ) \<and> ( \<psi> \<and> \<tau> ) ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_adantll: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( ( \<theta> \<and> \<phi> ) \<and> \<psi> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_anandirs: assumes A1: "( ( \<phi> \<and> ch ) \<and> ( \<psi> \<and> ch ) ) \<longrightarrow> \<tau>"   
   shows "( ( \<phi> \<and> \<psi> ) \<and> ch ) \<longrightarrow> \<tau>"
   using prems by auto

lemma MMI_adantlr: assumes A1: "( \<phi> \<and> \<psi> ) \<longrightarrow> ch"   
   shows "( ( \<phi> \<and> \<theta> ) \<and> \<psi> ) \<longrightarrow> ch"
   using prems by auto

lemma MMI_an42s: assumes A1: "( ( \<phi> \<and> \<psi> ) \<and> ( ch \<and> \<theta> ) ) \<longrightarrow> \<tau>"   
   shows "( ( \<phi> \<and> ch ) \<and> ( \<theta> \<and> \<psi> ) ) \<longrightarrow> \<tau>"
   using prems by auto;

(******* muladd11t-muladd*****************************)

lemma MMI_mp3an2: assumes A1: "\<psi>" and
    A2: "( \<phi> \<and> \<psi> \<and> ch ) \<longrightarrow> \<theta>"   
   shows "( \<phi> \<and> ch ) \<longrightarrow> \<theta>"
   using prems by auto;


end