(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{func1.thy}*}
theory func1 = func + Fol1:

text{*We define the notion of function that preserves a collection here.
Given two collection of sets a function preserves the collections if 
the inverse image of sets in one collection belongs to the second one.
This notion does not have a name in romantic math. It is used to define 
continuous functions in Topology\_ZF\_2 theory. 
We define it here so that we can 
use it for other purposes, like defining measurable functions.
Recall that f-``(A) means the inverse image of the set $A$.*}

constdefs
  PresColl :: "[i,i,i]\<Rightarrow>o"
  "PresColl(f,S,T) == \<forall> A\<in>T. f-``(A)\<in>S";

section{*Properties of functions, function spaces and (inverse) images.*}

text{*If a function maps $A$ into another set, then $A$ is the 
  domain of the function.*}

lemma func1_1_L1: assumes A:"f:A\<rightarrow>C" shows "domain(f) = A"
proof-;
  from A show ?thesis using domain_of_fun by simp
qed;

text{*A first-order version of Pi\_type. *}

lemma func1_1_L1A: assumes A1: "f:X\<rightarrow>Y" and A2: "\<forall>x\<in>X. f`(x) \<in> Z"
  shows "f:X\<rightarrow>Z"
proof -;
  from A2 have T1: "!!x. x\<in>X \<Longrightarrow> f`(x) \<in> Z" by simp;
  from A1 T1 show "f:X\<rightarrow>Z" by (rule Pi_type);
qed;

text{*There is a value for each argument.*}

lemma func1_1_L2: assumes A:"f:X\<rightarrow>Y" "x\<in>X" 
  shows "\<exists>y\<in>Y. <x,y> \<in> f";  
proof-;
    from A have "f`(x) \<in> Y" using apply_type by simp;
    moreover from A have "<x,f`(x)>\<in> f" using apply_Pair by simp;
    ultimately show ?thesis by auto;
qed; 

text{*Inverse image of any set is contained in the domain.*}

lemma func1_1_L3: assumes A:"f:X\<rightarrow>Y" shows "f-``(D) \<subseteq> X"
proof-;
   have "\<forall>x. x \<in>  f -`` D \<longrightarrow> x \<in> domain(f)"
      using  vimage_iff domain_iff by auto;
    with A have "\<forall>x. x \<in> f -`` D \<longrightarrow> x\<in> X" using func1_1_L1 by simp
    then show ?thesis by auto;
qed;

text{*The inverse image of the range is the domain.*}

lemma func1_1_L4: assumes A:"f:X\<rightarrow>Y" shows "f-``(Y) = X"
proof
  from A show "f -`` Y \<subseteq> X" using func1_1_L3 by simp;
  from A show "X \<subseteq> f -`` Y" using func1_1_L2 vimage_iff by auto;
qed;

text{*The arguments belongs to the domain and values to the range.*}

lemma func1_1_L5: 
  assumes A1: "<x,y> \<in> f" and A2: "f:X\<rightarrow>Y"  
  shows "x\<in>X \<and> y\<in>Y"
proof;
  from A1 A2 show "x\<in>X" using apply_iff by simp;
  with A2 have "f`(x)\<in> Y" using apply_type by simp;
  with A1 A2 show "y\<in>Y" using apply_iff by simp;
qed;

text{*The (argument, value) pair belongs to the graph of the function.*}

lemma func1_1_L5A: 
  assumes A1: "f:X\<rightarrow>Y" "x\<in>X" "y = f`(x)"
  shows "<x,y> \<in> f" "y \<in> range(f)" 
proof -;
  from A1 show "<x,y> \<in> f" using apply_Pair by simp;
  then show "y \<in> range(f)" using rangeI by simp;
qed;

text{*The range of function thet maps $X$ into $Y$ is contained in $Y$.*}

lemma func1_1_L5B: 
  assumes  A1:"f:X\<rightarrow>Y" shows "range(f) \<subseteq> Y"
proof;
  fix y assume "y \<in> range(f)"
  then obtain x where "<x,y> \<in> f"
    using range_def converse_def domain_def by auto;
  with A1 show "y\<in>Y" using func1_1_L5 by blast;
qed;

text{*The image of any set is contained in the range.*}

lemma func1_1_L6: assumes A:"f:X\<rightarrow>Y" shows "f``(B)\<subseteq>Y"
proof; 
  fix y
  assume A1:"y \<in> f `` B" show "y \<in> Y";
  proof-;
    from A1 obtain x where "<x,y>\<in> f" using image_iff by auto;
    with A show "y \<in> Y" using func1_1_L5 by fast;
  qed;
qed;

text{*The inverse image of any set is contained in the domain.*}

lemma func1_1_L6A: assumes A1:"f:X\<rightarrow>Y" shows "f-``(A)\<subseteq>X"
proof;
  fix x
  assume A2: "x\<in>f-``(A)" show "x\<in>X"
  proof -;
    from A2 obtain y where "<x,y>\<in> f" using vimage_iff by auto;
    with A1 show  "x\<in>X" using func1_1_L5 by fast;
  qed;
qed;

text{*Inverse image of a greater set is greater.*}

lemma func1_1_L7: assumes A1:"A\<subseteq>B" "function(f)" 
  shows "f-``(A)\<subseteq> f-``(B)"
proof -;
  from A1 show ?thesis using function_vimage_Diff by auto
qed;

text{*Image of a greater set is greater.*}

lemma func1_1_L8: assumes A1:"A\<subseteq>B"  shows "f``(A)\<subseteq> f``(B)"
proof -;
  from A1 show ?thesis using image_Un by auto;
qed;

text{* A set is contained in the the inverse image of its image.
There is similar theorem in equalities.thy (function\_image\_vimage)
which shows that the image of inverse image of a set 
is contained in the set.*}

lemma func1_1_L9: assumes A1:"f:X\<rightarrow>Y" and A2:"A\<subseteq>X"
  shows "A \<subseteq> f-``(f``(A))"
proof -
  from A2 have "\<forall>x\<in>A. x\<in>X" by auto;
  with A1 have "\<forall>x\<in>A. <x,f`(x)> \<in> f"  using apply_Pair by simp
  then have "\<forall>x\<in>A. x \<in> f-``(f``(A))" using image_iff by auto
  thus ?thesis by auto;
qed;

text{*A technical lemma needed to make the func1\_1\_L11 proof more clear.*}

lemma func1_1_L10: 
  assumes A1:"f \<subseteq> X\<times>Y" and A2:"x\<in>X" and A3:"\<exists>!y. (y\<in>Y & <x,y> \<in> f)"
  shows "\<exists>!y. <x,y> \<in> f"
proof;
  from A3 show "\<exists>y. \<langle>x, y\<rangle> \<in> f" by auto;
  fix y n assume A4:"<x,y> \<in> f" and A5: "<x,n> \<in> f" show "y=n"; 
  proof -;
    from A1 A4 A5 A3 show "y=n" by auto;
  qed;
qed;


text{*If $f\subseteq X\times Y$ and for every $x\in X$ there is exactly 
one $y\in Y$ such that $(x,y)\in f$ then $f$ maps $X$ to $Y$.*}

lemma func1_1_L11: 
  assumes A1:"f \<subseteq> X\<times>Y" and A2:"\<forall>x\<in>X. \<exists>!y. y\<in>Y & <x,y> \<in> f"
  shows "f: X\<rightarrow>Y"
proof -;
  from A1 A2 show ?thesis using func1_1_L10 Pi_iff_old by simp;
qed;

text{*A set defined by a lambda-type expression is a fuction. There is a 
  similar lemma in func.thy, but I had problems with lamda expressions syntax
  so I could not apply it. This lemma is a workaround this. Besides, lambda 
  expressions are not readable.
*}

lemma func1_1_L11A: assumes A1:"\<forall>x\<in>X. b(x)\<in>Y"
  shows "{<x,y> \<in> X\<times>Y. b(x) = y} \<in> X\<rightarrow>Y"
proof -;
  let ?f = "{<x,y> \<in> X\<times>Y. b(x) = y}"
  have "?f \<subseteq> X\<times>Y" by auto;
  moreover have "\<forall>x\<in>X. \<exists>!y. y\<in>Y & <x,y> \<in> ?f"
  proof;
    fix x assume A2:"x\<in>X" 
    show "\<exists>!y. y\<in>Y \<and> \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}";
    proof;
      fix y y1
      assume A1:"y\<in>Y \<and> \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
	and A2:"y1\<in>Y \<and> \<langle>x, y1\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
      from A1 have "y = b(x)" by simp;
      moreover from A2 have "y1 = b(x)" by simp;
      ultimately show "y = y1" by simp;
    next;
      def y == "b(x)";
      with A2 A1 show 
        "\<exists>y. y\<in>Y & \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
	by simp;
    qed;
  qed;
  ultimately show "{<x,y> \<in> X\<times>Y. b(x) = y} \<in> X\<rightarrow>Y" 
    using func1_1_L11 by simp;
qed;

text{*The value of a function defined by a meta-function is this 
  meta-function.*}

lemma func1_1_L11B: 
  assumes A1: "f \<in> X\<rightarrow>Y" "x\<in>X"
  and A2: "f = {<x,y> \<in> X\<times>Y. b(x) = y}"
  shows "f`(x) = b(x)"
proof -;
  from A1 have "<x,f`(x)> \<in> f"
    using apply_iff by simp;
  with A2 show ?thesis by simp;
qed;
  
text{*The inverse image of an intersection of a nonempty collection of sets 
  is the intersection of the 
  inverse images. This generalizes function\_vimage\_Int
  which is proven for the case of two sets.*}

lemma  func1_1_L12:
  assumes A1:"B\<subseteq>Pow(Y)" and A2:"B\<noteq>0" and A3:"f:X\<rightarrow>Y"
  shows "f-``(\<Inter>B) = (\<Inter>U\<in>B. f-``(U))"
proof;
  from A2 show  "f -`` (\<Inter>B) \<subseteq> (\<Inter>U\<in>B. f -`` U)" by blast;
  show "(\<Inter>U\<in>B. f -`` U) \<subseteq> f -`` (\<Inter>B)"
  proof;
    fix x assume A4: "x \<in> (\<Inter>U\<in>B. f -`` U)";
    from A3 have "\<forall>U\<in>B. f-``(U) \<subseteq> X" using func1_1_L6A by simp;
    with A4 have "\<forall>U\<in>B. x\<in>X" by auto;
    with A2 have "x\<in>X" by auto;
    with A3 have "\<exists>!y. <x,y> \<in> f" using Pi_iff_old by simp;
    with A2 A4 have "\<exists>y\<in>\<Inter>B. <x,y>\<in>f" by auto;
    then show "x \<in> f -`` (\<Inter>B)" using vimage_iff by simp;
  qed
qed;

text{*If the inverse image of a set is not empty, then the set is not empty.
Proof by contradiction.*}

lemma func1_1_L13: assumes A1:"f-``(A)\<noteq>0" shows "A\<noteq>0"
proof (rule ccontr);
  assume A2:"\<not> A \<noteq> 0" from A2 A1 show False by simp;
qed;

text{*If the image of a set is not empty, then the set is not empty.
Proof by contradiction.*}

lemma func1_1_L13A: assumes A1:"f``(A)\<noteq>0" shows "A\<noteq>0"
proof (rule ccontr);
  assume A2:"\<not> A \<noteq> 0" from A2 A1 show False by simp;
qed;

text{*What is the inverse image of a singleton?*}

lemma func1_1_L14: assumes A1: "f\<in>X\<rightarrow>Y" 
  shows "f-``({y}) = {x\<in>X. f`(x) = y}"
proof -
  from A1 show ?thesis
    using func1_1_L6A vimage_singleton_iff apply_iff
    by auto;
qed;

text{* A more familiar definition of inverse image.*}

lemma func1_1_L15: assumes A1: "f:X\<rightarrow>Y"
  shows "f-``(A) = {x\<in>X. f`(x) \<in> A}"
proof -;
  have "f-``(A) = (\<Union>y\<in>A . f-``{y})" 
    by (rule vimage_eq_UN);
  with A1 show ?thesis using func1_1_L14 by auto;
qed;

text{*A more familiar definition of image.*}

lemma func1_1_L16: assumes A1: "f:X\<rightarrow>Y" and A2:"A\<subseteq>X"
  shows "f``(A) = {f`(x). x \<in> A}"
proof;
 from A1 show "f``(A) \<subseteq> {f`(x). x \<in> A}"
   using image_iff apply_iff by auto;
 show "{f`(x). x \<in> A} \<subseteq> f``(A)"
 proof;
   fix y assume "y \<in> {f`(x). x \<in> A}"
   then obtain x where "x\<in>A \<and> y = f`(x)"
     by auto;
   with A1 A2 show "y \<in> f``(A)"
     using apply_iff image_iff by auto;
 qed;
qed;

text{*An image of an image is the image of a composition.*}

lemma func1_1_L16A: assumes  A1:"f:X\<rightarrow>Y" and A2:"g:Y\<rightarrow>Z"
  and A3:"A\<subseteq>X"
  shows "g``(f``(A)) =  {g`(f`(x)). x\<in>A} "
proof -
  from A1 A3 have "{f`(x). x\<in>A} \<subseteq> Y"
    using apply_funtype by auto;
  with A2 have "g``{f`(x). x\<in>A} = {g`(f`(x)). x\<in>A}"
    using func1_1_L16 by auto;
  moreover from A1 A3 have
    "g``{f`(x). x\<in>A}= g``(f``(A)) "
    using func1_1_L16 by simp;
  ultimately show ?thesis by simp;
qed;
  
text{*What is the image of a set defined by a meta-fuction?*}

lemma func1_1_L17: 
  assumes A1:"f \<in> X\<rightarrow>Y" and A2:"\<forall>x\<in>A. b(x) \<in> X"
  shows "f``({b(x). x\<in>A}) = {f`(b(x)). x\<in>A}"
proof -;
  from A2 have "{b(x). x\<in>A} \<subseteq> X" by auto;
   with A1 have 
    "f``({b(x). x\<in>A}) = {f`(y). y \<in> {b(x). x\<in>A}}"
    using func1_1_L16 by simp;
  then show ?thesis by auto;
qed;

section{*Functions restricted to a set*}
 
text{*What is the inverse image of a set under a restricted fuction?*}

lemma func1_2_L1: assumes A1: "f:X\<rightarrow>Y" and A2:"B\<subseteq>X"
  shows "restrict(f,B)-``(A) = f-``(A) \<inter> B"
proof -;
  let ?g = "restrict(f,B)";
  from A1 A2 have "?g\<in> B\<rightarrow>Y" 
    using restrict_type2 by simp;
  with A2 A1 show "?g-``(A) = f-``(A) \<inter> B"
    using func1_1_L15 restrict_if by auto;
qed;
   
text{*A criterion for when one function is a restriction of another.
  The lemma below provides a result useful in the actual proof of the 
  criterion and applications.*}

lemma func1_2_L2: 
  assumes A1:"f:X\<rightarrow>Y" and A2:"g \<in> A\<rightarrow>Z" 
  and A3:"A\<subseteq>X" and A4:"f \<inter> A\<times>Z = g"
  shows "\<forall>x\<in>A. g`(x) = f`(x)"
proof;
  fix x assume "x\<in>A"
  with A2 have "<x,g`(x)> \<in> g" using apply_Pair by simp;
  with A4 A1 show "g`(x) = f`(x)"  using apply_iff by auto; 
qed;

text{*Here is the actual criterion.*}

lemma func1_2_L3: 
  assumes A1:"f:X\<rightarrow>Y" and A2:"g:A\<rightarrow>Z" 
  and A3:"A\<subseteq>X" and A4:"f \<inter> A\<times>Z = g"
  shows "g = restrict(f,A)"
proof;
  from A4 show "g \<subseteq> restrict(f, A)" using restrict_iff by auto;
  show "restrict(f, A) \<subseteq> g"
  proof;
    fix z assume A5:"z \<in> restrict(f,A)"
    then obtain x y where D1:"z\<in>f & x\<in>A  & z = <x,y>"
      using restrict_iff by auto;
    with A1 have "y = f`(x)" using apply_iff by auto;
    with A1 A2 A3 A4 D1 have "y = g`(x)" using func1_2_L2 by simp;
    with A2 D1 show "z\<in>g" using apply_Pair by simp;
  qed;
qed;

text{*Which function space a restricted function belongs to?*}

lemma func1_2_L4: 
  assumes A1:"f:X\<rightarrow>Y" and A2:"A\<subseteq>X" and A3:"\<forall>x\<in>A. f`(x) \<in> Z"
  shows "restrict(f,A) \<in> A\<rightarrow>Z"
proof -;
  let ?g = "restrict(f,A)"
  from A1 A2 have "?g : A\<rightarrow>Y" 
    using restrict_type2 by simp;
  moreover from A1 A3 have "!!x. (x\<in>A \<Longrightarrow> ?g`(x) \<in> Z)"
    using restrict by simp;
  ultimately show ?thesis by (rule Pi_type); 
qed;

section{*Constant functions*}

text{*We define constant($=c$) functions on a set $X$  in a natural way as 
  ConstantFunction$(X,c)$. *}

constdefs
  ConstantFunction :: "[i,i]\<Rightarrow>i"
  "ConstantFunction(X,c) == X\<times>{c}"

lemma func1_3_L1: 
  assumes A1:"c\<in>Y" shows "ConstantFunction(X,c) : X\<rightarrow>Y"
proof -;
   from A1 have 
    "X\<times>{c} = {<x,y> \<in> X\<times>Y. c = y}" by auto;
   with A1 show ?thesis
     using func1_1_L11A ConstantFunction_def
     by simp;
qed;

text{*Constant function is equal to the constant on its domain.*}

lemma func1_3_L2: assumes A1:"x\<in>X"
  shows "ConstantFunction(X,c)`(x) = c"
proof -;
  have "ConstantFunction(X,c) \<in> X\<rightarrow>{c}"
    using func1_3_L1 by simp;
  moreover from A1 have "<x,c> \<in> ConstantFunction(X,c)"
    using ConstantFunction_def by simp;
  ultimately show ?thesis using apply_iff by simp;
qed;
  
text{*This concludes func1.thy.*}

end