(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{OrderedGroup\_ZF.thy}*}

theory OrderedGroup_ZF = Group_ZF + Order_ZF + Finite_ZF_1:

text{*This theory file defines and shows the basic properties of (partially
  or linearly) ordered groups. We define the set of nonnegative elemnts 
  and the absolute value function. 
  We show that in linearly ordered groups finite sets are bounded 
  and provide a sufficient condition for bounded sets to be finite. This 
  allows to show in Int\_ZF.thy that subsets of integers are bounded iff 
  they are finite. *}

section{*Ordered groups*}

text{*This section defines ordered group.*}

text{*An ordered group is a group equipped with a partial order that is
  "translation invariant", that is if $a\leq b$ then $a\cdot g \leq b\cdot g$
  and $g\cdot a \leq g\cdot b$. We define the set of nonnegative elements
  in the obvious way as $G^+ =\{x\in G: 1 \leq x\}$. We also define 
  the absolute value as a ZF-function that is the 
  identity on $G^+$ and the group inverse on the rest of the group.
 *}

constdefs 

  IsAnOrdGroup :: "[i,i,i]\<Rightarrow>o"
  "IsAnOrdGroup(G,P,r) == 
  (IsAgroup(G,P) \<and> r\<subseteq>G\<times>G \<and> IsPartOrder(G,r) \<and> (\<forall>g\<in>G. \<forall>a b. 
  <a,b> \<in> r \<longrightarrow> <P`<a,g>,P`<b,g> > \<in> r \<and> <P`<g,a>,P`<g,b> > \<in> r ) )"

  Nonnegative :: "[i,i,i]\<Rightarrow>i"
  "Nonnegative(G,P,r) == {x\<in>G. <TheNeutralElement(G,P),x> \<in> r}"

  AbsoluteValue :: "[i,i,i]\<Rightarrow>i"
  "AbsoluteValue(G,P,r) == IdentityFunction(Nonnegative(G,P,r)) \<union> 
  restrict(GroupInv(G,P),G - Nonnegative(G,P,r))";


text{*We will use a similar notation for ordered groups as for the generic 
  groups.*}

locale group3 =

  fixes G :: "i"
  fixes P :: "i"
  fixes r :: "i"
  assumes ordGroupAssum:"IsAnOrdGroup(G,P,r)"

  fixes unit :: "i" ("\<one>")
  defines unit_def [simp]:"\<one> == TheNeutralElement(G,P)"

  fixes groper  :: "[i,i]\<Rightarrow>i" (infixl "\<cdot>" 70)
  defines groper_def [simp]: "a \<cdot> b == P`<a,b>"

  fixes inv :: "i\<Rightarrow>i" ("_\<inverse> " [90] 91)
  defines inv_def [simp]: "x\<inverse> == GroupInv(G,P)`(x)"

  fixes lesseq :: "[i,i]\<Rightarrow>o" (infix "\<lsq>" 68)
  defines lesseq_def [simp]: "a \<lsq> b == <a,b> \<in> r"

  fixes nonnegative :: "i" ("G\<^sup>+")
  defines nonnegative_def [simp]: "G\<^sup>+ == Nonnegative(G,P,r)"

  fixes abs :: "i\<Rightarrow>i" ("| _ |")
  defines abs_def [simp]: "|a| == AbsoluteValue(G,P,r)`(a)";

text{*In group3 context we can use the theorems proven in group0 context.*}

lemma (in group3) OrderedGroup_ZF_1_L1: "group0(G,P)"
proof -;
  from ordGroupAssum show ?thesis using IsAnOrdGroup_def group0_def
    by simp;
qed;

text{*The next lemma is just to see the definition of the nonnegative set
  in our notation.*}

lemma (in group3) OrderedGroup_ZF_1_L2: "g\<in>G\<^sup>+ \<longleftrightarrow> \<one>\<lsq>g"
proof -;
  from ordGroupAssum show ?thesis 
    using IsAnOrdGroup_def Nonnegative_def by auto;
qed;

text{*For total order if $g$ is not nonnegative, then it has to be negative.*}

lemma (in group3) OrderedGroup_ZF_1_L2A: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G-G\<^sup>+"
  shows "a\<lsq>\<one>"
proof -;
  from A2 have "a\<in>G" "\<not>(\<one>\<lsq>a)" using OrderedGroup_ZF_1_L2 by auto;
  with A1 show ?thesis 
    using IsTotal_def OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
qed;

text{*The group order is reflexive.*}

lemma (in group3) OrderedGroup_ZF_1_L3: assumes A1:"g\<in>G"
  shows "g\<lsq>g"
proof -;
  from A1 ordGroupAssum show ?thesis using
    IsAnOrdGroup_def IsPartOrder_def refl_def by simp;
qed;

text{*$1$ is nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L3A: "\<one>\<in>G\<^sup>+"
proof -;
 show ?thesis using OrderedGroup_ZF_1_L2  OrderedGroup_ZF_1_L3
   OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
qed;
  
text{*In this context $a \leq b$ implies that both $a$ and $b$ belong 
  to $G$.*}

lemma (in group3) OrderedGroup_ZF_1_L4: 
  assumes A1: "a\<lsq>b" shows "a\<in>G" "b\<in>G"
proof -;
  from ordGroupAssum A1 show "a\<in>G" "b\<in>G" 
    using IsAnOrdGroup_def by auto;
qed;

text{*It is good to have transitivity handy.*}

lemma (in group3) OrderedGroup_ZF_1_L4A:
  assumes A1: "a\<lsq>b" "b\<lsq>c" shows "a\<lsq>c"
proof -
  from ordGroupAssum have "trans(r)"
    using IsAnOrdGroup_def IsPartOrder_def
    by simp
  moreover from A1 have "<a,b> \<in> r \<and> <b,c> \<in> r" by simp;
  ultimately have "<a,c> \<in> r" by (rule Fol1_L3);
  thus ?thesis by simp;
qed;

text{*For linearly ordered groups elements in the nonnegative set are
  greater than those in the complement.*}

lemma (in group3) OrderedGroup_ZF_1_L4B:
  assumes A1: "r {is total on} G" 
  and A2: "a\<in>G\<^sup>+" and A3: "b\<in>G-G\<^sup>+"
  shows "b\<lsq>a"
proof -;
  from A1 A2 A3 have "b\<lsq>\<one>" "\<one>\<lsq>a"
    using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L2A by auto;
  thus ?thesis by (rule OrderedGroup_ZF_1_L4A);
qed;

text{*The nonnegative set is contained in the group.*}

lemma (in group3) OrderedGroup_ZF_1_L2B: "G\<^sup>+ \<subseteq> G"
proof -;
  show ?thesis using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L4
    by auto;
qed;

text{*Taking the inverse on both sides inverses the inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L5:
  assumes A1: "a\<lsq>b" shows "b\<inverse>\<lsq>a\<inverse>"
proof -;
  from A1 have T1: "a\<in>G" "b\<in>G" "a\<inverse>\<in>G" "b\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.group0_2_L5 by auto;
  with A1 ordGroupAssum have "a\<cdot>a\<inverse>\<lsq>b\<cdot>a\<inverse>" using IsAnOrdGroup_def
    by simp;
  with T1 ordGroupAssum have "b\<inverse>\<cdot>\<one>\<lsq>b\<inverse>\<cdot>(b\<cdot>a\<inverse>)"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L6 IsAnOrdGroup_def
    by simp;
  with T1 show ?thesis using
    OrderedGroup_ZF_1_L1 group0.group0_2_L2 group0.group0_2_L3
    group0.group0_2_L6 by simp;
qed;

text{*If an element is smaller that the unit, then its inverse is greater.*}

lemma (in group3) OrderedGroup_ZF_1_L5A: 
  assumes A1: "a\<lsq>\<one>" shows "\<one>\<lsq>a\<inverse>"
proof -
  from A1 have "\<one>\<inverse>\<lsq>a\<inverse>" using OrderedGroup_ZF_1_L5
    by simp;
  then show ?thesis using OrderedGroup_ZF_1_L1 group0.group0_2_L7A 
    by simp;
qed;
    
text{*If the order is total, the elements that do not belong
  to the positive set are negative. We also show here that the group inverse
  of an element that does not belong to the nonnegative set does belong to the
  nonnegative set.*}

lemma (in group3) OrderedGroup_ZF_1_L6: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G-G\<^sup>+"
  shows "a\<lsq>\<one>" "a\<inverse> \<in> G\<^sup>+" "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+" 
proof -; 
  from A2 have T1: "a\<in>G" "a\<notin>G\<^sup>+" "\<one>\<in>G" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
  with A1 show "a\<lsq>\<one>" using OrderedGroup_ZF_1_L2 IsTotal_def
    by auto;
  then show "a\<inverse> \<in> G\<^sup>+" using OrderedGroup_ZF_1_L5A OrderedGroup_ZF_1_L2
    by simp;
  with A2 show "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+"
    using restrict by simp;
qed;

section{*Bounded sets*}

text{*A bounded set can be translated to put it in  $G^+$ and then it is 
 still bounded above. *}

lemma (in group3) OrderedGroup_ZF_2_L1: 
  assumes A1: "\<forall>g\<in>A. L\<lsq>g \<and> g\<lsq>M"
  and A2: "S = RightTranslation(G,P,L\<inverse>)" 
  and A3: "a \<in> S``(A)"
  shows "a\<lsq>M\<cdot>L\<inverse>" "\<one>\<lsq>a"
proof -
  from A3 have "A\<noteq>0" using func1_1_L13A by fast;
  then obtain g where "g\<in>A" by auto
  with A1 have T1: "L\<in>G" "M\<in>G" "L\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
    group0.group0_2_L5 by auto;
  with A2 have "S : G\<rightarrow>G" using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover from A1 have T2: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  ultimately have "S``(A) = {S`(b). b\<in>A}" using func1_1_L16
    by simp;
  with A3 obtain b where T3: "b\<in>A" "a = S`(b)" by auto;
  with A1 ordGroupAssum T1 have "b\<cdot>L\<inverse>\<lsq>M\<cdot>L\<inverse>" "L\<cdot>L\<inverse>\<lsq>b\<cdot>L\<inverse>"
    using IsAnOrdGroup_def by auto;
  with T3 A2 T1 T2 show "a\<lsq>M\<cdot>L\<inverse>" "\<one>\<lsq>a"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L2 group0.group0_2_L6 
    by auto;
qed;

text{*Every bounded set is an image of a subset of an interval that starts at 
  $1$.*}

lemma (in group3) OrderedGroup_ZF_2_L2:
  assumes A1: "IsBounded(A,r)" 
  shows "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)";
proof (cases "A=0");
  assume A2: "A=0" 
  let ?B = "0"
  let ?g = "\<one>"
  let ?T = "ConstantFunction(G,\<one>)"
  have "?g\<in>G\<^sup>+" using OrderedGroup_ZF_1_L3A by simp;
  moreover have "?T : G\<rightarrow>G"
    using func1_3_L1 OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
  moreover from A2 have "A = T``(?B)" by simp;
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)" by simp;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;
next assume A3: "A\<noteq>0"
  with A1 obtain L U where D1: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U "
    using IsBounded_def IsBoundedBelow_def IsBoundedAbove_def
    by auto;
  with A3 have T1: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  from A3 obtain a where "a\<in>A" by auto;
  with D1 have T2: "L\<lsq>a" "a\<lsq>U" by auto;
  then have T3: "L\<in>G" "L\<inverse>\<in> G" "U\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.group0_2_L5 by auto;
  let ?T = "RightTranslation(G,P,L)"
  let ?B = "RightTranslation(G,P,L\<inverse>)``(A)"
  let ?g = "U\<cdot>L\<inverse>"
  have "?g\<in>G\<^sup>+"
  proof -
    from T2 have "L\<lsq>U" using OrderedGroup_ZF_1_L4A by fast;
    with ordGroupAssum T3 have "L\<cdot>L\<inverse>\<lsq>?g" 
      using IsAnOrdGroup_def by simp;
    with T3 show ?thesis using OrderedGroup_ZF_1_L1 group0.group0_2_L6
      OrderedGroup_ZF_1_L2 by simp;
  qed;
  moreover from T3 have "?T : G\<rightarrow>G"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover have "A = ?T``(?B)"
  proof -;
    from T3 T1 have "?T``(?B) = {a\<cdot>L\<inverse>\<cdot>L. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_5_L6
      by simp;
    moreover from T3 T1 have "\<forall>a\<in>A. a\<cdot>L\<inverse>\<cdot>L = a\<cdot>(L\<inverse>\<cdot>L)"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L3 by auto;
    ultimately have "?T``(?B) = {a\<cdot>(L\<inverse>\<cdot>L). a\<in>A}" by simp;
    with T3 have "?T``(?B) = {a\<cdot>\<one>. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L6 by simp;
    moreover from T1 have "\<forall>a\<in>A. a\<cdot>\<one>=a"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
    ultimately show ?thesis by simp
  qed
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)"
  proof
    fix y assume A4: "y \<in> ?B"
    def D2: S == "RightTranslation(G,P,L\<inverse>)"
    from D1 have T4: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U" by simp;
    moreover from D2 have T5: "S = RightTranslation(G,P,L\<inverse>)" 
      by simp; 
    moreover from A4 D2 have T6: "y \<in> S``(A)" by simp;
    ultimately have "y\<lsq>U\<cdot>L\<inverse>" using OrderedGroup_ZF_2_L1
      by blast;
    moreover from T4 T5 T6 have "\<one>\<lsq>y" by (rule OrderedGroup_ZF_2_L1);
    ultimately show "y \<in> Interval(r,\<one>,?g)" using Interval_def by auto;
  qed;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;  
qed;
      
text{*If every interval starting at $1$ is finite, then every bounded set is 
  finite. I find it interesting that this does not require the group to be 
  linearly ordered (the order to be total).*}

theorem (in group3) OrderedGroup_ZF_2_T1:
  assumes A1: "\<forall>g\<in>G\<^sup>+. Interval(r,\<one>,g) \<in> Fin(G)"
  and A2: "IsBounded(A,r)" 
  shows "A \<in> Fin(G)"
proof -
  from A2 have
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    using OrderedGroup_ZF_2_L2 by simp;
  then obtain B g T where D1: "g\<in>G\<^sup>+" "B \<subseteq> Interval(r,\<one>,g)" 
    and D2: "T : G\<rightarrow>G" "A = T``(B)" by auto;
  from D1 A1 have "B\<in>Fin(G)" using Fin_subset_lemma by blast;
  with D2 show ?thesis using Finite1_L6A by simp;
qed;

text{*In linearly ordered groups finite sets are bounded.*}

theorem (in group3) OrderedGroup_ZF_1_T2:
  assumes A1: "r {is total on} G" and A2: "B\<in>Fin(G)" 
  shows "IsBounded(B,r)"
proof -
  from ordGroupAssum A1 A2 show ?thesis using
    IsAnOrdGroup_def IsPartOrder_def Finite_ZF_1_T1
    by simp;
qed;

section{*Absolute value and the triangle inequality*};

text{*The goal of this section is to prove the triangle inequality for 
  ordered groups.*};

text{*Absolute value maps $G$ into $G$.*}

lemma (in group3) OrderedGroup_ZF_3_L1: 
  "AbsoluteValue(G,P,r) : G\<rightarrow>G"
proof -;
  let ?f = "IdentityFunction(G\<^sup>+)"
  let ?g = "restrict(GroupInv(G,P),G-G\<^sup>+)"
  have "?f : G\<^sup>+\<rightarrow>G\<^sup>+" using func_ZF_6_L1 by simp;
  then have "?f : G\<^sup>+\<rightarrow>G" using OrderedGroup_ZF_1_L2B
    by (rule fun_weaken_type);
  moreover have "?g : G-G\<^sup>+\<rightarrow>G"
  proof -;
    from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G" 
      using IsAnOrdGroup_def group0_2_T2 by simp;
    moreover have "G-G\<^sup>+ \<subseteq> G" by auto;
    ultimately show ?thesis using restrict_type2 by simp;
  qed;
  moreover have "G\<^sup>+\<inter>(G-G\<^sup>+) = 0" by blast;
  ultimately have "?f \<union> ?g : G\<^sup>+\<union>(G-G\<^sup>+)\<rightarrow>G\<union>G" 
    by (rule fun_disjoint_Un);
  moreover have "G\<^sup>+\<union>(G-G\<^sup>+) = G" using OrderedGroup_ZF_1_L2B
    by auto;
  ultimately show "AbsoluteValue(G,P,r) : G\<rightarrow>G" 
    using AbsoluteValue_def by simp;
qed;

text{*If $a\in G^+$, then $|a| = a$.*}

lemma (in group3) OrderedGroup_ZF_3_L2:
  assumes A1: "a\<in>G\<^sup>+" shows "|a| = a"
proof -
  from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G"
    using IsAnOrdGroup_def group0_2_T2 by simp;
  with A1 show ?thesis using
    func1_1_L1 OrderedGroup_ZF_1_L2B fun_disjoint_apply1
    AbsoluteValue_def func_ZF_6_L2 by simp;
qed;

text{*If $a\in G\setminus G^+$, then $|a| = a^{-1}$.*}

lemma (in group3) OrderedGroup_ZF_3_L3:
   assumes A1: "a \<in> G-G\<^sup>+" shows "|a| = a\<inverse>"
proof -
  have "domain(IdentityFunction(G\<^sup>+)) = G\<^sup>+"
    using func_ZF_6_L1 func1_1_L1 by auto;
  with A1 show ?thesis using fun_disjoint_apply2 AbsoluteValue_def
    restrict by simp;
qed;

text{*In linearly ordered groups the absolute value of any element 
  is in $G^+$.*}

lemma (in group3) OrderedGroup_ZF_3_L3A: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G"
  shows "|a| \<in> G\<^sup>+"
proof (cases "a\<in>G\<^sup>+");
  assume "a \<in> G\<^sup>+" then show "|a| \<in> G\<^sup>+" 
    using OrderedGroup_ZF_3_L2 by simp;
next assume "a \<notin> G\<^sup>+" 
  with A1 A2 show "|a| \<in> G\<^sup>+" using OrderedGroup_ZF_3_L3
    OrderedGroup_ZF_1_L6 by simp;
qed;
  
text{*For linearly ordered groups (where the order is total), the absolute
  value maps the group into the positive set.*}

lemma (in group3) OrderedGroup_ZF_3_L3B:
  assumes A1: "r {is total on} G"
  shows "AbsoluteValue(G,P,r) : G\<rightarrow>G\<^sup>+"
proof-;
  have "AbsoluteValue(G,P,r) : G\<rightarrow>G" using OrderedGroup_ZF_3_L1
    by simp;
  moreover from A1 have T2: 
    "\<forall>g\<in>G. AbsoluteValue(G,P,r)`(g)  \<in> G\<^sup>+" 
    using OrderedGroup_ZF_3_L3A by simp;
  ultimately show ?thesis by (rule func1_1_L1A);
qed;

text{*If $b$ is greater than both $a$ and $a^{-1}$, then $b$ is greater than
  $|a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L4: 
  assumes A1: "a\<lsq>b" and A2: "a\<inverse>\<lsq> b" 
  shows "|a|\<lsq> b"
proof (cases "a\<in>G\<^sup>+");
  assume "a\<in>G\<^sup>+" 
  with A1 show "|a|\<lsq> b" using OrderedGroup_ZF_3_L2 by simp;
next assume "a\<notin>G\<^sup>+"
  with A1 A2 show "|a|\<lsq> b" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_3_L3 by simp;
qed;

text{*In linearly ordered groups $a\leq |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L5: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G"
  shows "a \<lsq> |a|"
proof (cases "a\<in>G\<^sup>+");
  assume "a \<in> G\<^sup>+"
  with A2 show "a \<lsq> |a|" 
    using OrderedGroup_ZF_3_L2 OrderedGroup_ZF_1_L3 by simp;
next assume "a \<notin> G\<^sup>+"
  with A1 A2 show "a \<lsq> |a|"
    using OrderedGroup_ZF_3_L3A OrderedGroup_ZF_1_L4B by simp;
qed;

text{*$-a\leq |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L6: 
  assumes A1: "a\<in>G"
  shows "a\<inverse> \<lsq> |a|"
proof (cases "a\<in>G\<^sup>+")
  assume "a \<in> G\<^sup>+"
  then have T1: "\<one>\<lsq>a" and T2: "|a| = a" using OrderedGroup_ZF_1_L2  
    OrderedGroup_ZF_3_L2 by auto;
  then have "a\<inverse>\<lsq>\<one>\<inverse>" using OrderedGroup_ZF_1_L5 by simp;
  then have T3: "a\<inverse>\<lsq>\<one>" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L7A by simp;
  from T3 T1 have "a\<inverse>\<lsq>a" by (rule OrderedGroup_ZF_1_L4A);
  with T2 show "a\<inverse> \<lsq> |a|" by simp;
next assume A2: "a \<notin> G\<^sup>+"
  from A1 have "|a| \<in> G" 
    using OrderedGroup_ZF_3_L1 apply_funtype by auto;
  with ordGroupAssum have "|a| \<lsq> |a|" 
    using IsAnOrdGroup_def IsPartOrder_def refl_def by simp;
  with A1 A2 show "a\<inverse> \<lsq> |a|" using OrderedGroup_ZF_3_L3 by simp;
qed;

text{*We can multiply the sides of two inequalities.*}

lemma (in group3) OrderedGroup_ZF_3_L7:
  assumes A1: "a\<lsq>b" and A2: "c\<lsq>d"
  shows "a\<cdot>c \<lsq> b\<cdot>d"
proof -;
  from A1 A2 have "c\<in>G" "b\<in>G" using OrderedGroup_ZF_1_L4 by auto;
  with A1 A2 ordGroupAssum have "a\<cdot>c\<lsq> b\<cdot>c" "b\<cdot>c\<lsq>b\<cdot>d"
    using IsAnOrdGroup_def by auto;
  then show "a\<cdot>c \<lsq> b\<cdot>d" by (rule OrderedGroup_ZF_1_L4A);
qed;

text{*Triangle inequality for linearly ordered abelian groups. 
  It would be nice to drop commutativity or give an example that shows we
  can't do that.*}

theorem (in group3) OrderedGroup_ZF_3_T1:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and A3: "a\<in>G" "b\<in>G" 
  shows "|a\<cdot>b| \<lsq> |a|\<cdot>|b|"
proof -;
  from A1 A2 A3 have 
    "a \<lsq> |a|" "b \<lsq> |b|" "a\<inverse> \<lsq> |a|" "b\<inverse> \<lsq> |b|"
    using OrderedGroup_ZF_3_L5 OrderedGroup_ZF_3_L6 by auto;
  then have "a\<cdot>b \<lsq> |a|\<cdot>|b|" "a\<inverse>\<cdot>b\<inverse> \<lsq> |a|\<cdot>|b|"
    using OrderedGroup_ZF_3_L7 by auto;
  with A1 A3 show "|a\<cdot>b| \<lsq> |a|\<cdot>|b|"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L10 IsCommutative_def 
    OrderedGroup_ZF_3_L4 by simp;
qed;

text{*This concludes OrderedGroups\_.thy.*};

end