(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Order\_ZF.thy}*}

theory Order_ZF = Fol1:

text{*This theory file considers various notion related to order. We redefine
  the notions of a total order, linear order and partial order to have the 
  same terminology as wikipedia (I found it very consistent across different 
  areas of math). We also define and study the notions of intervals and bounded
  sets. We show the inclusion relations between the intervals with endpoints
  being in certain order. We also show that union of bounded sets are bounded. 
  This allows to show that finite sets are bounded in Finite\_ZF.thy.*}

section{*Linear order*}

text{*In this section we formulate the definitions related to order 
  relations.*}

text{*We define a linear order as a binary relation that is antisymmetric, 
  transitive and total. Note that this terminology is different than the
  one used the standard Order.thy file. The sets that are bounded below and 
  above are also defined, as are bounded sets. Empty sets are defined as 
  bounded.
  The notation for the definition of an interval may be mysterious for some
  readers, see Order\_ZF\_2\_L1 for more intuitive notation. 
  We aslo define the maximum (the greater of) two elemnts and the minmum 
  (the smaller of) two elements.*}

constdefs 

  IsTotal :: "[i,i]\<Rightarrow>o" (infixl "{is total on}" 65)
  "r {is total on} X == (\<forall>a\<in>X.\<forall>b\<in>X. <a,b> \<in> r \<or> <b,a> \<in> r)"

  IsLinOrder :: "[i,i]\<Rightarrow>o"
  "IsLinOrder(X,r) == ( antisym(r) \<and> trans(r) \<and> (r {is total on} X))"

  IsPartOrder :: "[i,i]\<Rightarrow>o"
  "IsPartOrder(X,r) == (refl(X,r) \<and> antisym(r) \<and> trans(r))"

  IsBoundedAbove :: "[i,i]\<Rightarrow>o"
  "IsBoundedAbove(A,r) == ( A=0 \<or> (\<exists>u. \<forall>x\<in>A. <x,u> \<in> r))"

  IsBoundedBelow :: "[i,i]\<Rightarrow>o"
  "IsBoundedBelow(A,r) == (A=0 \<or> (\<exists>l. \<forall>x\<in>A. <l,x> \<in> r))"

  IsBounded :: "[i,i]\<Rightarrow>o"
  "IsBounded(A,r) == (IsBoundedAbove(A,r) \<and> IsBoundedBelow(A,r))"

  Interval :: "[i,i,i]\<Rightarrow>i"
  "Interval(r,a,b) == r``{a} \<inter> r-``{b}"

  GreaterOf :: "[i,i,i]\<Rightarrow>i"
  "GreaterOf(r,a,b) == (if <a,b> \<in> r then b else a)"

  SmallerOf :: "[i,i,i]\<Rightarrow>i"
  "SmallerOf(r,a,b) == (if <a,b> \<in> r then a else b)"


text{*The essential condition to show that a total relation is reflexive.*}

lemma Order_ZF_1_L1: assumes A1: "r {is total on} X" and A2: "a\<in>X"
  shows "<a,a> \<in> r"
proof -;
  from A1 A2 show ?thesis using IsTotal_def by auto;
qed;

text{*A total relation is reflexive.*}

lemma Order_ZF_1_L1A:
  assumes A1: "r {is total on} X"
  shows "refl(X,r)"
proof -;
  from A1 show ?thesis using Order_ZF_1_L1 refl_def by simp;
qed;
text{*A linear order is partial order.*}

lemma Order_ZF_1_L2: assumes A1: "IsLinOrder(X,r)" 
  shows "IsPartOrder(X,r)"
proof -;
  from A1 show ?thesis 
    using IsLinOrder_def IsPartOrder_def refl_def Order_ZF_1_L1
    by auto;
qed;

section{*Intervals*}

text{*In this section we discuss intervals.*}

text{*The next lemma explains the notation of the definition of an interval.*}

lemma Order_ZF_2_L1: "x \<in> Interval(r,a,b) \<longleftrightarrow> <a,x> \<in> r \<and> <x,b> \<in> r"
proof -;
  show ?thesis using Interval_def by auto;
qed;

text{*Since there are some problems with applying the above lemma 
  (seems that simp and auto don't handle equivalence very well), we
  split Order\_ZF\_2\_L1 into two lemmas.*}

lemma Order_ZF_2_L1A: assumes A1: "x \<in> Interval(r,a,b)"
  shows "<a,x> \<in> r" "<x,b> \<in> r"
proof -
  from A1 show "<a,x> \<in> r" "<x,b> \<in> r" using Order_ZF_2_L1
    by auto;
qed;

text{*Order\_ZF\_2\_L1, implication from right to left.*}

lemma Order_ZF_2_L1B: assumes A1: "<a,x> \<in> r" "<x,b> \<in> r"
  shows "x \<in> Interval(r,a,b)"
proof -;
  from A1 show ?thesis using Order_ZF_2_L1 by simp;
qed;

text{*If the relation is reflexive, the endpoints belong to the interval.*}

lemma Order_ZF_2_L2: assumes A1: "refl(X,r)" 
  and A2: "a\<in>X" "b\<in>X" and A3: "<a,b>\<in>r"
  shows "a\<in>Interval(r,a,b)" "b\<in>Interval(r,a,b)"
proof -
  from A1 A2 A3 show "a\<in>Interval(r,a,b)" "b\<in>Interval(r,a,b)"
    using refl_def Order_ZF_2_L1 by auto;
qed;

text{*If $a,b,c,d$ are in this order, then $[b,c]\subseteq [a,d]$. We
  only need trasitivity for this to be true.*}

lemma Order_ZF_2_L3: 
  assumes A1: "trans(r)" and A2:"<a,b>\<in>r" "<b,c>\<in>r" "<c,d>\<in>r"
shows "Interval(r,b,c) \<subseteq> Interval(r,a,d)"
proof;
  fix x assume A3: "x \<in> Interval(r, b, c)"
  from A1 have "trans(r)" .
  moreover from A2 A3 have "<a,b> \<in> r \<and> <b,x> \<in> r" using Order_ZF_2_L1A
    by simp;
  ultimately have T1: "<a,x> \<in> r" by (rule Fol1_L3);
  from A1 have "trans(r)" .
  moreover from A2 A3 have "<x,c> \<in> r \<and> <c,d> \<in> r" using Order_ZF_2_L1A
    by simp;
  ultimately have "<x,d> \<in> r" by (rule Fol1_L3);
  with T1 show "x \<in> Interval(r,a,d)" using Order_ZF_2_L1B
    by simp;
qed;

text{*For reflexive and antisymmetric relations the interval with equal 
  endpoints consists only of that endpoint.*}

lemma Order_ZF_2_L4: 
  assumes A1: "refl(X,r)" and A2: "antisym(r)" and A3: "a\<in>X"
  shows "Interval(r,a,a) = {a}"
proof;
  from A1 A3 have "<a,a> \<in> r" using refl_def by simp;
  with A1 A3 show "{a} \<subseteq> Interval(r,a,a)" using Order_ZF_2_L2 by simp;
  from A2 show "Interval(r,a,a) \<subseteq> {a}" using Order_ZF_2_L1A Fol1_L4
    by fast;
qed;

text{*For transitive relations the endpoints have to be in the relation for
  the interval to be nonempty.*}

lemma Order_ZF_2_L5: assumes A1: "trans(r)" and A2: "<a,b> \<notin> r"
  shows "Interval(r,a,b) = 0"
proof (rule ccontr);
  assume "Interval(r,a,b)\<noteq>0" then obtain x where "x \<in> Interval(r,a,b)"
    by auto;
  with A1 A2 show False using Order_ZF_2_L1A Fol1_L3 by fast;
qed;

section{*Bounded sets*}

text{*In this section we consider properties of bounded sets.*}

text{*For reflexive relations singletons are bounded.*}

lemma Order_ZF_3_L1: assumes A1: "refl(X,r)"
  and A2: "a\<in>X"
  shows "IsBounded({a},r)"
proof -
  from A1 A2 show ?thesis using 
    refl_def IsBoundedAbove_def IsBoundedBelow_def
    IsBounded_def by auto;
qed;

text{*For a total relation order, the greater of two elements, 
  as defined above is indeed greater of any of the two.*}

lemma Order_ZF_3_L2: assumes A1: "r {is total on} X"
  and A2: "x\<in>X" "y\<in>X"
  shows "<x,GreaterOf(r,x,y)> \<in> r" "<y,GreaterOf(r,x,y)> \<in> r"
proof -;
  from A1 A2 show 
    "<x,GreaterOf(r,x,y)> \<in> r" "<y,GreaterOf(r,x,y)> \<in> r"
    using IsTotal_def Order_ZF_1_L1 GreaterOf_def
    by auto;
qed;

text{*For total and transitive relation the union of two sets bounded 
  above is bounded above.*}

lemma Order_ZF_3_L3: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedAbove(A,r)" "IsBoundedAbove(B,r)"
  and A4: "r \<subseteq> X\<times>X"
  shows "IsBoundedAbove(A\<union>B,r)"
proof (cases "A=0 \<or> B=0");
  assume "A=0 \<or> B=0" 
  with A3 show ?thesis by auto;
next assume "\<not> (A = 0 \<or> B = 0)"
  then have T1: "A\<noteq>0" "B\<noteq>0" by auto;
  with A3 obtain u w where D1: "\<forall>x\<in>A. <x,u> \<in> r" "\<forall>x\<in>B. <x,w> \<in> r"
    using IsBoundedAbove_def by auto;
  let ?U = "GreaterOf(r,u,w)"
  from T1 A4 D1 have "u\<in>X" "w\<in>X" by auto;
  with A1 A2 D1 have "\<forall>x\<in>A\<union>B.<x,?U> \<in> r"
    using Order_ZF_3_L2 Fol1_L3 by blast;
  then show "IsBoundedAbove(A\<union>B,r)"
    using IsBoundedAbove_def by auto;
qed
  
text{*For total and transitive relations if a set $A$ is bounded above then 
  $A\cup \{a\}$ is bounded above.*}

lemma Order_ZF_3_L4: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedAbove(A,r)" and A4: "a\<in>X" and A5: "r \<subseteq> X\<times>X"
  shows "IsBoundedAbove(A\<union>{a},r)"
proof -
  from A1 have "refl(X,r)"
    using Order_ZF_1_L1A by simp;
  with A1 A2 A3 A4 A5 show ?thesis using
    Order_ZF_3_L1 IsBounded_def Order_ZF_3_L3 by simp;
qed;

text{*For a total relation order, the smaller of two elements, 
  as defined above is indeed smaller of any of the two*}

lemma Order_ZF_3_L5: assumes A1: "r {is total on} X"
  and A2: "x\<in>X" "y\<in>X"
  shows "<SmallerOf(r,x,y),x> \<in> r" "<SmallerOf(r,x,y),y> \<in> r"
proof -;
  from A1 A2 show 
    "<SmallerOf(r,x,y),x> \<in> r" "<SmallerOf(r,x,y),y> \<in> r"
    using IsTotal_def Order_ZF_1_L1 SmallerOf_def
    by auto;
qed

text{*For total and transitive relation the union of two sets bounded 
  below is bounded below.*}

lemma Order_ZF_3_L6: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedBelow(A,r)" "IsBoundedBelow(B,r)"
  and A4: "r \<subseteq> X\<times>X"
  shows "IsBoundedBelow(A\<union>B,r)"
proof (cases "A=0 \<or> B=0");
  assume "A=0 \<or> B=0" 
  with A3 show ?thesis by auto;
next assume "\<not> (A = 0 \<or> B = 0)"
  then have T1: "A\<noteq>0" "B\<noteq>0" by auto;
  with A3 obtain l m where D1: "\<forall>x\<in>A. <l,x> \<in> r" "\<forall>x\<in>B. <m,x> \<in> r"
    using IsBoundedBelow_def by auto;
  let ?L = "SmallerOf(r,l,m)"
  from T1 A4 D1 have T1: "l\<in>X" "m\<in>X" by auto;
  with A1 A2 D1 have "\<forall>x\<in>A\<union>B.<?L,x> \<in> r"
    using Order_ZF_3_L5 Fol1_L3 by blast;
  then show "IsBoundedBelow(A\<union>B,r)"
    using IsBoundedBelow_def by auto;
qed;

text{*For total and transitive relations if a set $A$ is bounded below then 
  $A\cup \{a\}$ is bounded below.*}

lemma Order_ZF_3_L7:
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedBelow(A,r)" and A4: "a\<in>X" and A5: "r \<subseteq> X\<times>X"
  shows "IsBoundedBelow(A\<union>{a},r)"
proof -;
  from A1 have "refl(X,r)"
    using Order_ZF_1_L1A by simp;
  with A1 A2 A3 A4 A5 show ?thesis using
    Order_ZF_3_L1 IsBounded_def Order_ZF_3_L6 by simp;
qed;

text{*For total and transitive relations unions of two bounded sets are 
  bounded.*}

theorem Order_ZF_3_T1: 
  assumes A1: "r {is total on} X" and A2: "trans(r)" 
  and A3: "IsBounded(A,r)" "IsBounded(B,r)"
  and A4: "r \<subseteq> X\<times>X"
  shows "IsBounded(A\<union>B,r)"
proof -;
  from A1 A2 A3 A4 show ?thesis 
    using Order_ZF_3_L3 Order_ZF_3_L6 Order_ZF_3_L7 IsBounded_def 
    by simp;
qed;

text{*For total and transitive relations if a set $A$ is bounded then 
  $A\cup \{a\}$ is bounded.*}

lemma Order_ZF_3_L8: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBounded(A,r)" and A4: "a\<in>X" and A5: "r \<subseteq> X\<times>X"
  shows "IsBounded(A\<union>{a},r)"
proof -;
  from A1 A2 A3 A4 A5 show ?thesis using
    Order_ZF_1_L1A Order_ZF_3_L1 Order_ZF_3_T1 by blast;
qed;

end
 