(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Int\_ZF.thy}*}

theory Int_ZF = OrderedGroup_ZF + Finite_ZF_1 + Int + Nat_ZF:;

text{*This theory file is an interface between the old-style Isabelle 
  (ZF logic) material on integers and the IsarMathLib project. Here we
  redefine the meta-level operations on integers 
  (addition and multiplication) to convert them to ZF-functions and show
  that integers form a commutative group with respect to addition and 
  commutative monid with respect to multiplication. Similarly, we redefine the
  order on integers as a relation, that is a subset of $Z\times Z$. 
  We show that a subset of intergers is bounded iff it is finite.*}

section{*Addition and multiplication as ZF-functions.*}

text{*In this section we provide definitions of addition and multiplication
  as subsets of $(Z\times Z)\times Z$.*}

constdefs

  IntegerAddition :: "i"
  "IntegerAddition == { <x,c> \<in> (int\<times>int)\<times>int. fst(x) $+ snd(x) = c}"

  IntegerMultiplication :: "i"
  "IntegerMultiplication == 
    { <x,c> \<in> (int\<times>int)\<times>int. fst(x) $\<times> snd(x) = c}";


text{*IntegerAddition and IntegerMultiplication are functions on 
  int$\times$int. *}

lemma Int_ZF_1_L1: 
  shows "IntegerAddition \<in> int\<times>int \<rightarrow> int"
  "IntegerMultiplication \<in> int\<times>int \<rightarrow> int"
proof -
  have
    "{<x,c> \<in> (int\<times>int)\<times>int. fst(x) $+ snd(x) = c} \<in> int\<times>int\<rightarrow>int" 
    "{<x,c> \<in> (int\<times>int)\<times>int. fst(x) $\<times> snd(x) = c} \<in> int\<times>int\<rightarrow>int"
    using func1_1_L11A by auto
  then show "IntegerAddition \<in> int\<times>int \<rightarrow> int" 
    "IntegerMultiplication \<in> int\<times>int \<rightarrow> int"
    using IntegerAddition_def IntegerMultiplication_def by auto
qed;

text{*IntegerAddition adds integers and IntegerMultiplication multiplies
  integers.*}

lemma Int_ZF_1_L2: assumes A1:"a \<in> int" "b \<in> int"
  shows "IntegerAddition`(<a,b>) = a $+ b"
  "IntegerMultiplication`(<a,b>) = a $\<times> b"
proof -;
  let ?x = "<a,b>"
  let ?c = "a $+ b"
  let ?d = "a $\<times> b"
  from A1 have "fst(?x) $+ snd(?x) = ?c" "fst(?x) $\<times> snd(?x) = ?d"
    by auto;
  with A1 have 
    "<?x,?c> \<in> {<x,c> \<in> (int\<times>int)\<times>int. fst(x) $+ snd(x) = c}"
    "<?x,?d> \<in> {<x,d> \<in> (int\<times>int)\<times>int. fst(x) $\<times> snd(x) = d}"
    by auto;
  then show "IntegerAddition`(?x) = ?c"
    "IntegerMultiplication`(?x) = ?d"
    using IntegerAddition_def IntegerMultiplication_def 
      Int_ZF_1_L1 apply_iff by auto
qed;
 
text{*Integer addition and multiplication are associative.*}

lemma Int_ZF_1_L3: 
  assumes A1:"x \<in> int" "y\<in>int" "z\<in>int"
  shows "IntegerAddition`(<IntegerAddition`(<x,y>),z>) = 
  IntegerAddition`( < x,IntegerAddition`(<y,z>)> )"
   "IntegerMultiplication`(<IntegerMultiplication`(<x,y>),z>) = 
  IntegerMultiplication`( < x,IntegerMultiplication`(<y,z>)> )"
proof -;
  from A1 show  "IntegerAddition`(<IntegerAddition`(<x,y>),z>) = 
    IntegerAddition`( < x,IntegerAddition`(<y,z>)> )"
    "IntegerMultiplication`(<IntegerMultiplication`(<x,y>),z>) = 
    IntegerMultiplication`( < x,IntegerMultiplication`(<y,z>)> )"
    using Int_ZF_1_L2 zadd_assoc zmult_assoc by auto;
qed;

text{*Integer addition and multiplication are commutative.*}

lemma Int_ZF_1_L4:
  assumes A1:"x \<in> int" "y\<in>int"
  shows "IntegerAddition` <x,y> = IntegerAddition` <y,x>"
  "IntegerMultiplication` <x,y> = IntegerMultiplication` <y,x>"
proof -;
  from A1 show "IntegerAddition` <x,y> = IntegerAddition` <y,x>"
    "IntegerMultiplication` <x,y> = IntegerMultiplication` <y,x>"
    using Int_ZF_1_L2 zadd_commute zmult_commute by auto;
qed;

text{*Zero is neutral for addition and one for multiplication.*}

lemma Int_ZF_1_L5: assumes A1:"x \<in> int"
  shows "IntegerAddition`< ($# 0), x > = x \<and>
   IntegerAddition`< x ,($# 0) > = x"
  "IntegerMultiplication` < ($# 1), x > = x \<and> 
  IntegerMultiplication`< x ,($# 1) > = x"
proof -;
  from A1 show "IntegerAddition`< ($# 0), x > = x \<and>
    IntegerAddition`< x ,($# 0) > = x"
    using Int_ZF_1_L2 zadd_int0 Int_ZF_1_L4 by simp;
  from A1 have T1: "IntegerMultiplication` < ($# 1), x > = x"
    using Int_ZF_1_L2 zmult_int1 by simp;
  moreover from A1 T1 have "IntegerMultiplication`< x ,($# 1) > = x"
    using Int_ZF_1_L4 by simp;
  ultimately show  "IntegerMultiplication` < ($# 1), x > = x \<and> 
    IntegerMultiplication`< x ,($# 1) > = x"
    by simp;
qed;
    

text{*Zero is neutral for addition and one for multiplication.*}

lemma Int_ZF_1_L6: "($# 0) \<in> int \<and> 
  (\<forall>x \<in> int. IntegerAddition`< ($# 0), x > = x \<and>
   IntegerAddition`< x ,($# 0) > = x)"
  "($# 1) \<in> int \<and> 
  (\<forall>x \<in> int. IntegerMultiplication` < ($# 1), x > = x \<and> 
  IntegerMultiplication`< x ,($# 1) > = x)"
proof -
  show "($# 0) \<in> int \<and> 
    (\<forall>x \<in> int. IntegerAddition`< ($# 0), x > = x \<and>
    IntegerAddition`< x ,($# 0) > = x)"
    "($# 1) \<in> int \<and> 
    (\<forall>x \<in> int. IntegerMultiplication` < ($# 1), x > = x \<and> 
    IntegerMultiplication`< x ,($# 1) > = x)"
    using Int_ZF_1_L5 by auto;
qed;

text{*Integers with addition and integers with multiplication
  form monoids.*}
 
theorem Int_ZF_1_T1: "IsAmonoid(int,IntegerAddition)"
  "IsAmonoid(int,IntegerMultiplication)"
proof -
   have  
    "\<exists>e \<in> int. \<forall>x \<in> int. IntegerAddition`<e, x > = x \<and> 
    IntegerAddition`<x,e> = x"
     "\<exists>e \<in> int. \<forall>x \<in> int. IntegerMultiplication`<e, x> = x \<and> 
    IntegerMultiplication`<x,e> = x"
     using Int_ZF_1_L6 by auto;
   then show "IsAmonoid(int,IntegerAddition)"
     "IsAmonoid(int,IntegerMultiplication)" using 
     IsAmonoid_def IsAssociative_def Int_ZF_1_L1 Int_ZF_1_L3 
     by auto;
qed;

(*text{*We can use theorems proven in the monoid0 context.*}

lemma Int_ZF_1_L7: shows "monoid0(int,IntegerAddition)"
proof -; 
  show ?thesis using Int_ZF_1_T1 monoid0_def by simp;
qed;*)

text{*Zero is the neutral element of the integers with addition
  and one is the neutral element of the integers with multiplication.*}

lemma Int_ZF_1_L8: 
  "($# 0) = TheNeutralElement(int,IntegerAddition)"
  "($# 1) = TheNeutralElement(int,IntegerMultiplication)"
proof -;
  have "monoid0(int,IntegerAddition)"
    using Int_ZF_1_T1 monoid0_def by simp;
  moreover have 
    "($# 0) \<in> int \<and>
    (\<forall>x \<in> int.(IntegerAddition` (< ($# 0), x >) = x) \<and> 
    (IntegerAddition` (< x ,($# 0) >) = x))"
    using Int_ZF_1_L6 by auto;
  ultimately show "($# 0) = TheNeutralElement(int,IntegerAddition)"
    by (rule monoid0.group0_1_L4);
  have "monoid0(int,IntegerMultiplication)"
    using Int_ZF_1_T1 monoid0_def by simp;
  moreover have "($# 1) \<in> int \<and> 
    (\<forall>x \<in> int. IntegerMultiplication` < ($# 1), x > = x \<and> 
    IntegerMultiplication`< x ,($# 1) > = x)"
    using Int_ZF_1_L6 by auto;
  ultimately show
    "($# 1) = TheNeutralElement(int,IntegerMultiplication)"
    by (rule monoid0.group0_1_L4);
qed;

text{*Each integer has an inverse (in the addition sense).*}

lemma Int_ZF_1_L9: assumes A1:"g \<in> int"
  shows 
  "\<exists> b \<in> int. IntegerAddition`<g,b> = 
  TheNeutralElement(int,IntegerAddition)"
proof -;
  from A1 have "IntegerAddition`<g,$-g> = 
    TheNeutralElement(int,IntegerAddition)"
    using Int_ZF_1_L2 Int_ZF_1_L8 by simp;
  thus ?thesis by auto;
qed;

text{*Integers with addition form an abelian group. This also shows
  that we can apply all theorems proven in the proof contexts (locales) 
  that require the assumpion that some pair of sets form a group like 
  locale group0.*}
 
theorem Int_ZF_1_T2: 
  "IsAgroup(int,IntegerAddition)"
  "IntegerAddition {is commutative on} int"
  "group0(int,IntegerAddition)"
proof -;
  show "IsAgroup(int,IntegerAddition)"
    using Int_ZF_1_T1 Int_ZF_1_L9 IsAgroup_def
    by simp;
  then show "group0(int,IntegerAddition)"
    using group0_def by simp;
  show "IntegerAddition {is commutative on} int"
    using Int_ZF_1_L4 IsCommutative_def by simp;
qed;

text{*What is the additive group inverse in the group of integers?*}

lemma Int_ZF_1_L9: assumes A1: "m\<in>int" 
  shows "$-m = GroupInv(int,IntegerAddition)`(m)"
proof -; 
  from A1 have "m\<in>int" "$-m \<in> int" "IntegerAddition`<m,$-m> = 
    TheNeutralElement(int,IntegerAddition)"
    using zminus_type Int_ZF_1_L2 Int_ZF_1_L8 by auto;
  then show ?thesis using Int_ZF_1_T2 group0.group0_2_L9 by fast;
qed;

section{*Integers as an ordered group*}

text{*In this section we define order on integers as a relation, that is a 
  subset of $Z\times Z$ and show that integers form an ordered group.*}

text{*We use the $\$\leq $ (higher order) relation defined in Int.thy to 
  define a subset of $Z\times Z$ that constitutes the ZF order relation 
  corresponding to it.*}

constdefs
  IntegerOrder :: "i" 
  "IntegerOrder == {p\<in>int\<times>int. fst(p) $\<le> snd(p)}"

text{*We will define a locale to introduce notation $x\leq y$ for integers.
  Also, we will write $m..n$ to denote the integer iterval with endpoints
  in $m$ and $n$.*}

locale int0 =

  fixes lesseq :: "[i,i]\<Rightarrow>o" (infix "\<lsq>" 60)
  defines lesseq_def [simp]: "m \<lsq> n == <m,n> \<in> IntegerOrder"

  fixes interval :: "[i,i]\<Rightarrow>i" (infix ".." 70)
  defines interval_def [simp]: "m..n == Interval(IntegerOrder,m,n)";

text{*The next lemma interprets the definition one way. *}

lemma (in int0) Int_ZF_2_L1: 
  assumes A1: "m\<in>int" "n\<in>int" and A2: "m $\<le> n"
  shows "m \<lsq> n"
proof -
  from A1 A2 have "<m,n> \<in> {x\<in>int\<times>int. fst(x) $\<le> snd(x)}" 
    by simp;
  then show ?thesis using IntegerOrder_def by simp;
qed;

text{*The next lemma interprets the definition the other way. *}

lemma (in int0) Int_ZF_2_L1A: assumes A1: "m \<lsq> n" 
  shows "m $\<le> n" "m\<in>int" "n\<in>int"
proof -;
  from A1 have  "<m,n> \<in> {p\<in>int\<times>int. fst(p) $\<le> snd(p)}"
    using IntegerOrder_def by simp;
  thus "m $\<le> n" "m\<in>int" "n\<in>int" by auto;
qed;

text{*Integer order is a relation on integers.*}

lemma Int_ZF_2_L1B: "IntegerOrder \<subseteq> int\<times>int"
proof;
  fix x assume "x\<in>IntegerOrder" 
  then have "x \<in> {p\<in>int\<times>int. fst(p) $\<le> snd(p)}"
    using IntegerOrder_def by simp;
  then show "x\<in>int\<times>int" by simp;
qed;

text{* The order on integers is reflexive.*}

lemma (in int0) Int_ZF_2_L2: "refl(int,IntegerOrder)"
proof -;
  show ?thesis using Int_ZF_2_L1 zle_refl refl_def
    by auto;
qed;

text{*The essential condition to show antisymmetry of the order on integers.*}

lemma (in int0) Int_ZF_2_L3: 
  assumes A1: "m \<lsq> n" "n \<lsq> m"  
  shows "m=n"
proof -;
  from A1 have "m $\<le> n" "n $\<le> m" "m\<in>int" "n\<in>int"
    using Int_ZF_2_L1A by auto;
  then show "m=n" using zle_anti_sym by auto;
qed;
  
text{*The order on integers is antisymmetric.*}

lemma (in int0) Int_ZF_2_L4: "antisym(IntegerOrder)"
proof -
  have "\<forall>m n. m \<lsq> n  \<and> n \<lsq> m \<longrightarrow> m=n"
    using Int_ZF_2_L3 by auto;
  then show ?thesis using imp_conj antisym_def by simp;
qed;

text{*The essential condition to show that the order on integers is 
  transitive.*}

lemma Int_ZF_2_L5: 
  assumes A1: "<m,n> \<in> IntegerOrder" "<n,k> \<in> IntegerOrder"
  shows "<m,k> \<in> IntegerOrder"
proof -;
  from A1 have T1: "m $\<le> n" "n $\<le> k" and T2: "m\<in>int" "k\<in>int"
    using int0.Int_ZF_2_L1A by auto;
  from T1 have "m $\<le> k" by (rule zle_trans);
  with T2 show ?thesis using int0.Int_ZF_2_L1 by simp;
qed;

text{*The order on integers is transitive.*}

lemma Int_ZF_2_L6: "trans(IntegerOrder)"
proof -;
  have "\<forall> m n k. 
    (\<langle>m, n\<rangle>\<in>IntegerOrder \<and> \<langle>n, k\<rangle>\<in>IntegerOrder \<longrightarrow> \<langle>m, k\<rangle>\<in> IntegerOrder)"
    using Int_ZF_2_L5 by blast;
  then show ?thesis by (rule Fol1_L2);
qed;

text{*The order on integers is a partial order.*}

lemma Int_ZF_2_L7: "IsPartOrder(int,IntegerOrder)"
proof -;
  show ?thesis using int0.Int_ZF_2_L2 int0.Int_ZF_2_L4 
    Int_ZF_2_L6 IsPartOrder_def by simp;
qed;

text{*The essential condition to show that the order on integers is 
  preserved by translations. *}

lemma (in int0) Int_ZF_2_L8: assumes A1: "k \<in> int" 
  and A2: "m \<lsq> n" 
  shows 
  "IntegerAddition`<m,k> \<lsq> IntegerAddition`<n,k> "
  "IntegerAddition`<k,m> \<lsq> IntegerAddition`<k,n> "
proof -;  
  from A2 have "m $\<le> n" and T1: "m\<in>int" "n\<in>int" 
    using Int_ZF_2_L1A by auto;
  then have "m $+ k $\<le> n $+ k" "k $+ m $\<le> k $+ n"
    using zadd_right_cancel_zle zadd_left_cancel_zle by auto;
  moreover from T1 A1 have 
    "m $+ k \<in> int" "n $+ k \<in> int"
    "k $+ m \<in> int" "k $+ n \<in> int"
    using Int_ZF_1_L2 Int_ZF_1_L1 apply_funtype by auto;
  ultimately have
    "(m $+ k) \<lsq> (n $+ k)"
    "(k $+ m) \<lsq> (k $+ n)"
    using Int_ZF_1_L2 Int_ZF_2_L1 by auto;
  moreover from A1 T1 have
    "m $+ k = IntegerAddition`<m,k>" "n $+ k = IntegerAddition`<n,k>"
    "k $+ m = IntegerAddition`<k,m>" "k $+ n = IntegerAddition`<k,n>"
    using Int_ZF_1_L2 by auto;
  ultimately show
    "IntegerAddition`<m,k> \<lsq> IntegerAddition`<n,k> "
    "IntegerAddition`<k,m> \<lsq> IntegerAddition`<k,n> "
    by auto;
qed;

text{*Integers form a linearly ordered group. We can apply all theorems
  proven in group3 context to integers. *}

theorem (in int0) Int_ZF_2_T1: 
  "IsAnOrdGroup(int,IntegerAddition,IntegerOrder)"
  "IntegerOrder {is total on} int"
  "group3(int,IntegerAddition,IntegerOrder)"
proof -;
  have "\<forall>k\<in>int. \<forall>m n. m \<lsq> n  \<longrightarrow> 
    IntegerAddition`<m,k> \<lsq> IntegerAddition`<n,k> \<and>
    IntegerAddition`<k,m> \<lsq> IntegerAddition`<k,n>"
    using Int_ZF_2_L8 by simp;
  then show "IsAnOrdGroup(int,IntegerAddition,IntegerOrder)" using
    Int_ZF_1_T2 Int_ZF_2_L1B Int_ZF_2_L7 IsAnOrdGroup_def
    by simp;
  then show "group3(int,IntegerAddition,IntegerOrder)"
    using group3_def by simp;
  show "IntegerOrder {is total on} int"
    using IsTotal_def zle_linear Int_ZF_2_L1 by auto;
qed;

text{*If a pair $(i,m)$ belongs to the order relation on integers and
  $i\neq m$, then $i<m$ in the sense of defined in the standard Isabelle's 
  Int.thy.*}

lemma (in int0) Int_ZF_2_L9: assumes A1: "i \<lsq> m" and A2: "i\<noteq>m"
  shows "i $< m"
proof -;
  from A1 have "i $\<le> m" "i\<in>int" "m\<in>int" 
    using Int_ZF_2_L1A by auto;
  with A2 show "i $< m" using zle_def by simp;
qed;

text{*Taking minus on both sides of inequality reverses it.*}

lemma (in int0) Int_ZF_2_L10: assumes A1: "k \<lsq> i"
  shows "$-i \<lsq> $-k"
proof -
  from A1 show ?thesis using Int_ZF_2_L1A
    Int_ZF_1_L9 Int_ZF_2_T1 group3.OrderedGroup_ZF_1_L5 by simp;
qed;


text{*Adding one to an integer corresponds to taking a successor for a natural
  number.*}

lemma (in int0) Int_ZF_2_L11: "i $+ $# n $+ ($# 1)  =  i $+ $# succ(n)"
proof -
  have "$# succ(n) = $#1 $+ $# n" using int_succ_int_1 by blast;
  then have "i $+ $# succ(n) = i $+ ($# n  $+ $#1)"
    using zadd_commute by simp;
  then show ?thesis using zadd_assoc by simp;
qed;

text{*Adding a natural number increases integers.*}

lemma (in int0) Int_ZF_2_L12: assumes A1: "i \<in> int" and A2: "n\<in>nat"
  shows "i \<lsq> i $+ $#n"
proof (cases "n = 0");
  assume "n = 0" 
  with A1 show "i \<lsq> i $+ $#n" using zadd_int0  Int_ZF_2_L2 refl_def
    by simp;
next;
  assume "n\<noteq>0" 
  with A2 obtain k where "k\<in>nat" "n = succ(k)" 
    using Nat_ZF_1_L3 by auto;
  with A1 show "i \<lsq> i $+ $#n"
    using zless_succ_zadd zless_imp_zle Int_ZF_2_L1 by simp;
qed;

text{*Adding one increases integers.*}

lemma (in int0) Int_ZF_2_L12A: assumes A1:"j\<lsq>k"
  shows "j\<lsq> k $+ $#1"
proof -;
  from A1 have T1:"j\<in>int" "k\<in>int" "j $\<le> k" 
    using Int_ZF_2_L1A by auto;  
  moreover from T1 have "k $\<le> k $+ $#1" using Int_ZF_2_L12 Int_ZF_2_L1A
    by simp;
  ultimately have "j $\<le> k $+ $#1" using zle_trans by fast;
  with T1 show ?thesis using Int_ZF_2_L1 by simp;
qed;

text{*If $k+1\leq m+n$, where $n$ is a non-zero natural number, then 
  $m\leq k$. *}

lemma (in int0) Int_ZF_2_L13: 
  assumes A1: "k\<in>int" "m\<in>int" and A2: "n\<in>nat" 
  and A3: "k $+ ($# 1) = m $+ $# succ(n)"
  shows "m \<lsq> k"
proof -;
  from A1 have "k\<in>int" "m $+ $# n \<in> int" by auto;
  moreover from A2 have "k $+ $# 1 = m $+ $# n $+ $#1"
    using Int_ZF_2_L11 by simp;
  ultimately have "k = m $+ $# n" using zadd_right_cancel by simp;
  with A1 A2 show ?thesis using Int_ZF_2_L12 by simp;
qed;

section{*Induction on integers.*}

text{*In this section we show some induction lemmas for integers. 
  The basic tools are the induction on natural numbers and the fact that
  integers can be written as a sum of a smaller integer and a natural number.*}

text{*An integer can be written a a sum of a smaller integer and a natural 
  number.*}

lemma (in int0) Int_ZF_3_L2: assumes A1: "i \<lsq> m"
  shows "\<exists>n\<in>nat. m = i $+ $# n"
proof (cases "i=m");
  let ?n = "0"
  assume A2: "i=m"  
  from A1 A2 have "?n \<in> nat" "m = i $+ $# ?n"
    using Int_ZF_2_L1A zadd_int0_right by auto;
  thus "\<exists>n\<in>nat. m = i $+ $# n" by blast;
next
  assume A3: "i\<noteq>m" 
  with A1 have "i $< m" "i\<in>int" "m\<in>int"   
    using Int_ZF_2_L9 Int_ZF_2_L1A by auto;
  then obtain k where D1: "k\<in>nat" "m = i $+ $# succ(k)"
    using zless_imp_succ_zadd_lemma by auto;
  let ?n = "succ(k)"
  from D1 have "?n\<in>nat" "m = i $+ $# ?n" by auto;
  thus "\<exists>n\<in>nat. m = i $+ $# n" by simp;
qed;

text{*Induction for integers, the induction step.*}

lemma (in int0) Int_ZF_3_L6: assumes A1: "i\<in>int" 
  and A2: "\<forall>m. i\<lsq>m \<and> Q(m) \<longrightarrow> Q(m $+ ($# 1))"
  shows "\<forall>k\<in>nat. Q(i $+ ($# k)) \<longrightarrow> Q(i $+ ($# succ(k)))"
proof;
  fix k assume A3: "k\<in>nat" show "Q(i $+ $# k) \<longrightarrow> Q(i $+ $# succ(k))"
  proof;
    assume A4: "Q(i $+ $# k)"
    from A1 A3 have "i\<lsq> i $+ ($# k)" using Int_ZF_2_L12
      by simp;
    with A4 A2 have "Q(i $+ ($# k) $+ ($# 1))" by simp;
    then show "Q(i $+ ($# succ(k)))" using Int_ZF_2_L11 by simp;
  qed
qed;

text{*Induction on integers.*}

theorem (in int0) Int_ZF_3_T1: 
  assumes A1: "i\<lsq>k" and A2: "Q(i)"
  and A3: "\<forall>m. i\<lsq>m \<and> Q(m) \<longrightarrow> Q(m $+ ($# 1))"
  shows "Q(k)"
proof -;
  from A1 obtain n where D1: "n\<in>nat" and D2: "k = i $+ $# n"
    using Int_ZF_3_L2 by auto;
  from A1 have T1: "i\<in>int" using Int_ZF_2_L1A by simp;
  from D1 have "n\<in>nat" .
  moreover from A1 have "Q(i $+ $#0)" 
    using Int_ZF_2_L1A zadd_int0 by simp;
  moreover from T1 A3 have 
    "\<forall>k\<in>nat. Q(i $+ ($# k)) \<longrightarrow> Q(i $+ ($# succ(k)))"
    by (rule Int_ZF_3_L6);
  ultimately have "Q(i $+ ($# n))" by (rule Nat_ZF_1_L2); 
  with D2 show "Q(k)" by simp;
qed;

text{*Another form of induction on integers. This rewrites the basic theorem 
   Int\_ZF\_3\_T1 substituting $P(-k)$ for $Q(k)$.*}

lemma (in int0) Int_ZF_3_L7: assumes A1: "i\<lsq>k" and A2: "P($-i)"
  and A3: "\<forall>m. i\<lsq>m \<and> P($-m) \<longrightarrow> P($-(m $+ ($# 1)))"
  shows "P($-k)"
proof -
  from A1 A2 A3 show "P($-k)" by (rule Int_ZF_3_T1);
qed;

text{*Another induction on integers. This rewrites Int\_ZF\_3\_L7
  substituting $-k$ for $k$ and $-i$ for $i$.*}

lemma (in int0) Int_ZF_3_L8: assumes A1: "k\<lsq>i" and A2: "P(i)" 
  and A3: "\<forall>m. $-i\<lsq>m \<and> P($-m) \<longrightarrow> P($-(m $+ ($# 1)))"
  shows "P(k)"
proof -
  from A1 have T1: "$-i\<lsq>$-k" using Int_ZF_2_L10 by simp;
  from A1 A2 have T2: "P($- $- i)" using Int_ZF_2_L1A zminus_zminus
    by simp;
  from T1 T2 A3 have "P($-($-k))" by (rule Int_ZF_3_L7);
  with A1 show "P(k)" using Int_ZF_2_L1A zminus_zminus by simp;
qed;

text{*An implication between two forms of induction steps.*}

lemma (in int0) Int_ZF_3_L9: assumes A1: "i\<in>int"
  and A2: "\<forall>n. n\<lsq>i \<and> P(n) \<longrightarrow> P(n $+ $-($#1))"    
  shows "\<forall>m. $-i\<lsq>m \<and> P($-m) \<longrightarrow> P($-(m $+ ($# 1)))"
proof;
  fix m show "$-i\<lsq>m \<and> P($-m) \<longrightarrow> P($-(m $+ ($# 1)))"
  proof;
    assume A3: "$- i \<lsq> m \<and> P($- m)"
    then have "$- i \<lsq> m" by simp;
    then have "$-m \<lsq> $- ($- i)" by (rule Int_ZF_2_L10);
    with A1 A2 A3 show "P($-(m $+ ($# 1)))"
      using zminus_zminus zminus_zadd_distrib by simp;
  qed;
qed;

text{*Backwards induction on integers.*}

lemma (in int0) Int_ZF_3_T2: assumes A1: "k\<lsq>i" and A2: "P(i)" 
  and A3: "\<forall>n. n\<lsq>i \<and> P(n) \<longrightarrow>P(n $+ $-($#1)) "
  shows "P(k)"
proof -
  from A1 have T1: "i\<in>int" using Int_ZF_2_L1A by simp;
  from T1 A3 have T2: "\<forall>m. $-i\<lsq>m \<and> P($-m) \<longrightarrow> P($-(m $+ ($# 1)))"
    by (rule Int_ZF_3_L9);
  from A1 A2 T2 show "P(k)" by (rule Int_ZF_3_L8)
qed;

section{*Bounded vs. finite subsets of integers*}

text{*The goal of this section is to establish that a subset of integers 
  is bounded is and only is it is finite. The fact that all finite sets 
  are bounded is already shown for all linearly ordered groups in 
  OrderedGroups\_ZF.thy. To show the other implication we
  show that all intervals starting at 0 are finite and then use a result from
  OrderedGroups\_ZF.thy.*}

text{*There are no integers between $k$ and $k+1$.*}

lemma (in int0) Int_ZF_4_L1: 
  assumes A1: "k\<in>int" "m\<in>int" "n\<in>nat" and A2: "k $+ $#1 = m $+ $#n"
  shows "m =  k $+ $#1 \<or> m \<lsq> k"
proof (cases "n=0");
  assume "n=0" 
  with A1 A2 show "m =  k $+ $#1 \<or> m \<lsq> k" 
    using zadd_int0 by simp;
next assume "n\<noteq>0" 
  with A1 obtain j where D1: "j\<in>nat" "n = succ(j)"
    using Nat_ZF_1_L3 by auto;
  with A1 A2 D1 show "m =  k $+ $#1 \<or> m \<lsq> k" 
    using Int_ZF_2_L13 by simp;
qed;

text{*If $j\in m..k+1$, then $j\in m..n$ or $j=k+1$.*}

lemma (in int0) Int_ZF_4_L2: assumes A1: "k\<in>int"
  and A2: "j \<in> m..(k $+ $#1)"
  shows "j \<in> m..k \<or> j \<in> {k $+ $#1}"
proof -;
  from A2 have T1: "m\<lsq>j" "j\<lsq>(k $+ $#1)" using Order_ZF_2_L1A 
    by auto;
  then have T2: "m\<in>int" "j\<in>int" using Int_ZF_2_L1A by auto; 
  from T1 obtain n where "n\<in>nat" "k $+ $#1 = j $+ $# n"
    using Int_ZF_3_L2 by auto;
  with A1 T1 T2 have "(m\<lsq>j \<and> j \<lsq> k) \<or> j \<in> {k $+ $#1}"
    using Int_ZF_4_L1 by auto;
  then show ?thesis using Order_ZF_2_L1B by auto;
qed;
  
text{*Extending an integer interval by one is the same as adding the new 
  endpoint.*}

lemma (in int0) Int_ZF_4_L3: assumes A1: "m\<lsq> k"
  shows "m..(k $+ $#1) = m..k \<union> {k $+ $#1}"
proof;
  from A1 have T1: "m\<in>int" "k\<in>int" using Int_ZF_2_L1A by auto;
  then show "m .. (k $+ $# 1) \<subseteq> m .. k \<union> {k $+ $# 1}"
    using Int_ZF_4_L2 by auto;
  from T1 have "m\<lsq> m" using Int_ZF_2_T1 group3.OrderedGroup_ZF_1_L3
    by simp;
  with T1 A1 have "m .. k \<subseteq> m .. (k $+ $# 1)"
    using Int_ZF_2_L12 Int_ZF_2_L6 Order_ZF_2_L3 by simp;
  with T1 A1 show "m..k \<union> {k $+ $#1} \<subseteq> m..(k $+ $#1)"
    using Int_ZF_2_L12A Int_ZF_2_L2 Order_ZF_2_L2 by auto;
qed;

text{*Integer intervals are finite - induction step.*}

lemma (in int0) Int_ZF_4_L4: 
  assumes A1: "i\<lsq>m" and A2: "i..m \<in> Fin(int)"
  shows "i..(m $+ $#1) \<in> Fin(int)"
proof -
  from A1 A2 show ?thesis using Int_ZF_4_L3 by simp;
qed;

text{*Integer intervals are finite.*}

lemma (in int0) Int_ZF_4_L5: assumes A1: "i\<in>int" "k\<in>int"
  shows "i..k \<in> Fin(int)"
proof (cases "i\<lsq> k");
  assume A2: "i\<lsq>k"
  moreover from A1 have "i..i \<in> Fin(int)"
    using Int_ZF_2_L2 Int_ZF_2_L4 Order_ZF_2_L4 by simp;
  moreover from A2 have 
    "\<forall>m. i\<lsq>m \<and> i..m \<in> Fin(int) \<longrightarrow> i..(m $+ $#1) \<in> Fin(int)"
    using Int_ZF_4_L4 by simp;
  ultimately show "i..k \<in> Fin(int)" by (rule Int_ZF_3_T1);
next assume "\<not> i \<lsq> k"
  then show "i..k \<in> Fin(int)" using Int_ZF_2_L6 Order_ZF_2_L5
    by simp;
qed;

text{*Bounded integer sets are finite.*}

lemma (in int0) Int_ZF_4_L6: assumes A1: "IsBounded(A,IntegerOrder)"
  shows "A \<in> Fin(int)"
proof -;
  have T1: "\<forall>m \<in> Nonnegative(int,IntegerAddition,IntegerOrder).
    $#0..m \<in> Fin(int)" 
  proof
    fix m assume "m \<in> Nonnegative(int,IntegerAddition,IntegerOrder)"
    then have "m\<in>int" using Int_ZF_2_T1 group3.OrderedGroup_ZF_1_L2B
      by auto;
    then show "$#0..m \<in> Fin(int)" using Int_ZF_4_L5 by simp
  qed;
  have "group3(int,IntegerAddition,IntegerOrder)"
    using Int_ZF_2_T1 by simp;
  moreover from T1 have "\<forall>m \<in> Nonnegative(int,IntegerAddition,IntegerOrder).
    Interval(IntegerOrder,TheNeutralElement(int,IntegerAddition),m) 
    \<in> Fin(int)" using Int_ZF_1_L8 by simp;
  moreover from A1 have "IsBounded(A,IntegerOrder)" .;
  ultimately show "A \<in> Fin(int)" by (rule group3.OrderedGroup_ZF_2_T1);
qed;

text{*A subset of integers is bounded iff it is finite..*}

theorem (in int0) Int_ZF_4_T1: "IsBounded(A,IntegerOrder)\<longleftrightarrow> A\<in>Fin(int)"
proof -
  show ?thesis using Int_ZF_4_L6 Int_ZF_2_T1 group3.OrderedGroup_ZF_1_T2
    by blast;
qed;

end






 
