(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Ring\_ZF.thy}*}

theory Ring_ZF = Group_ZF_2:

text{*This theory file covers basic facts about rings.*}

section{*Definition and basic properties*}

text{*In this section we define what is a ring and list the basic properties
  of rings. *}

text{*We say that three sets $(R,A,M)$ form a ring if $(R,A)$ is an abelian 
  group, $(R,M)$ is a monoid and $A$ is distributive with respect to $M$ on 
  $R$. $A$ represents the additive operation on $M$. 
  As such it is a subset of $(R\times R)\times R$ (recall that in ZF set theory
  functions are sets).
  Similarly $M$ is represents the multiplicative operation on $R$ and is also
  a subset of $(R\times R)\times R$.
  We  don't require the multiplicative operation to be commutative in the 
  definition of a ring.*}

constdefs 
  IsAring :: "[i,i,i]\<Rightarrow>o"
  "IsAring(R,A,M) == IsAgroup(R,A) \<and> (A {is commutative on} R) \<and> 
  IsAmonoid(R,M) \<and> IsDistributive(R,A,M)"


text{*Next we define a locale that will be used when considering rings.*}

locale ring0 =

  fixes R :: "i"
  fixes A :: "i"
  fixes M :: "i"
  assumes ringAssum: "IsAring(R,A,M)"

  fixes ringa :: "[i,i]\<Rightarrow>i" (infix "\<ra>" 69)
  defines ringa_def [simp]: "a\<ra> b == A`<a,b>"

  fixes ringminus :: "i\<Rightarrow>i" ("\<rm> _" 71)
  defines ringminus_def [simp]: "\<rm> a == GroupInv(R,A)`(a)"

  fixes ringsub :: "[i,i]\<Rightarrow>i" (infix "\<rs>" 69)
  defines ringsub_def [simp]: "a\<rs>b == a\<ra> (\<rm> b)"

  fixes ringm :: "[i,i]\<Rightarrow>i" (infix "\<cdot>" 70)
  defines ringm_def [simp]: "a\<cdot>b == M`<a,b>"

  fixes ringzero :: "i" ("\<zero>")
  defines ringzero_def [simp]: "\<zero> == TheNeutralElement(R,A)"

  fixes ringone :: "i" ("\<one>")
  defines ringone_def [simp]: "\<one> == TheNeutralElement(R,M)"

text{*In the ring0 context we can use theorems proven in some other contexts.*}

lemma (in ring0) Ring_ZF_1_L1: "group0(R,A)" "monoid0(R,M)"
proof -
  from ringAssum show "group0(R,A)"  "monoid0(R,M)"
    using IsAring_def group0_def monoid0_def by auto;
qed;

text{*Zero and one of the ring are element of the ring.*}

lemma (in ring0) Ring_ZF_1_L2: "\<zero> \<in> R" "\<one> \<in> R"
proof -
   show "\<zero> \<in> R" "\<one> \<in> R" using 
     Ring_ZF_1_L1 group0.group0_2_L2 monoid0.group0_1_L3 by auto;
qed;
  
text{*The next lemma lists the properties of a ring that require one element
  of a ring.*}

lemma (in ring0) Ring_ZF_1_L3: assumes A1: "a \<in> R"
  shows "\<rm>a \<in> R" "a\<ra>\<zero> = a" "\<zero>\<ra>a = a" "a\<cdot>\<one> = a" "\<one>\<cdot>a = a" 
proof -
  from A1 show "\<rm>a \<in> R" using Ring_ZF_1_L1 group0.group0_2_L5
    by simp;
  from A1 show 
    "a\<ra>\<zero> = a" "\<zero>\<ra>a = a" "a\<cdot>\<one> = a" "\<one>\<cdot>a = a"
    using Ring_ZF_1_L1 group0.group0_2_L2 monoid0.group0_1_L3
    by auto;
qed;

text{*Properties that require two elements of a ring.*}

lemma (in ring0) Ring_ZF_1_L4: assumes A1: "a\<in>R" "b\<in>R"
  shows "a\<ra>b \<in> R" "a\<rs>b \<in> R" "a\<cdot>b \<in> R" "a\<ra>b = b\<ra>a"
proof -
  from A1 show "a\<ra>b \<in> R" "a\<rs>b \<in> R" "a\<cdot>b \<in> R"
    using Ring_ZF_1_L1 Ring_ZF_1_L3
      group0.group0_2_L1 monoid0.group0_1_L1 by auto;
  from ringAssum A1 show "a\<ra>b = b\<ra>a" 
    using IsAring_def IsCommutative_def by simp;
qed;

text{*The additive operation in a ring is distributive with respect to the
  multiplicative operation.*}

lemma (in ring0) Ring_ZF_1_L5: assumes A1: "a\<in>R" "b\<in>R" "c\<in>R"
  shows "a\<cdot>(b\<ra>c) = a\<cdot>b \<ra> a\<cdot>c" "(b\<ra>c)\<cdot>a = b\<cdot>a \<ra> c\<cdot>a"
proof -; 
  from ringAssum A1 show 
    "a\<cdot>(b\<ra>c) = a\<cdot>b \<ra> a\<cdot>c" "(b\<ra>c)\<cdot>a = b\<cdot>a \<ra> c\<cdot>a"
    using IsAring_def IsDistributive_def by auto;
qed;

text{*Any element of a ring multiplied by zero is zero.*}

lemma (in ring0) Ring_ZF_1_L6: 
  assumes A1: "x\<in>R" shows "\<zero>\<cdot>x = \<zero>" "x\<cdot>\<zero> = \<zero>"
proof -
  def D1: a == "x\<cdot>\<one>";
  def D2: b == "x\<cdot>\<zero>"
  def D3: c == "\<one>\<cdot>x"
  def D4: d == "\<zero>\<cdot>x"
  from A1 D1 D2 D3 D4 have 
    "a \<ra> b = x\<cdot>(\<one> \<ra> \<zero>)" "c \<ra> d = (\<one> \<ra> \<zero>)\<cdot>x"
    using Ring_ZF_1_L2 Ring_ZF_1_L5 by auto;
  moreover from D1 D3 have "x\<cdot>(\<one> \<ra> \<zero>) = a" "(\<one> \<ra> \<zero>)\<cdot>x = c"
    using Ring_ZF_1_L2 Ring_ZF_1_L3 by auto;
  ultimately have "a \<ra> b = a" and T1: "c \<ra> d = c" by auto;
  moreover from A1 D1 D2 D3 D4 have 
    "a \<in> R" "b \<in> R" and T2: "c \<in> R" "d \<in> R"
    using Ring_ZF_1_L2 Ring_ZF_1_L4 by auto;
  ultimately have "b = \<zero>" using 
    Ring_ZF_1_L1 group0.group0_2_L7 by simp;
  moreover from T2 T1 have "d = \<zero>" using 
    Ring_ZF_1_L1 group0.group0_2_L7 by simp;
  moreover from D2 D4 have "b = x\<cdot>\<zero>" "d = \<zero>\<cdot>x" by auto; 
  ultimately show "x\<cdot>\<zero> = \<zero>" "\<zero>\<cdot>x = \<zero>" by auto;
qed;

text{*Negative can be pulled out of a product.*}

lemma (in ring0) Ring_ZF_1_L7: assumes A1: "a\<in>R" "b\<in>R"
  shows "(\<rm>a)\<cdot>b = \<rm>(a\<cdot>b)" "a\<cdot>(\<rm>b) = \<rm>(a\<cdot>b)" 
proof -
  from A1 have T1: 
    "a\<cdot>b \<in> R" "(\<rm>a) \<in> R" "((\<rm>a)\<cdot>b) \<in> R" 
    "(\<rm>b) \<in> R" "a\<cdot>(\<rm>b) \<in> R"
    using Ring_ZF_1_L3 Ring_ZF_1_L4 by auto;
  moreover have "(\<rm>a)\<cdot>b \<ra> a\<cdot>b = \<zero>" 
    and T2: "a\<cdot>(\<rm>b) \<ra> a\<cdot>b = \<zero>"
  proof -
    from A1 T1 have 
      "(\<rm>a)\<cdot>b \<ra> a\<cdot>b = ((\<rm>a)\<ra> a)\<cdot>b"
      "a\<cdot>(\<rm>b) \<ra> a\<cdot>b= a\<cdot>((\<rm>b)\<ra>b)"
      using Ring_ZF_1_L5 by auto;
    moreover from A1 have 
      "((\<rm>a)\<ra> a)\<cdot>b = \<zero>" "a\<cdot>((\<rm>b)\<ra>b) = \<zero>"
      using Ring_ZF_1_L1 group0.group0_2_L6 Ring_ZF_1_L6
      by auto;
    ultimately show 
      "(\<rm>a)\<cdot>b \<ra> a\<cdot>b = \<zero>" "a\<cdot>(\<rm>b) \<ra> a\<cdot>b = \<zero>" 
      by auto;
  qed;
  ultimately show "(\<rm>a)\<cdot>b = \<rm>(a\<cdot>b)"
    using Ring_ZF_1_L1 group0.group0_2_L9 by simp
  from T1 T2 show "a\<cdot>(\<rm>b) = \<rm>(a\<cdot>b)"
    using Ring_ZF_1_L1 group0.group0_2_L9 by simp;   
qed;

text{*Subtraction is distributive with respect to multiplication.*}

lemma (in ring0) Ring_ZF_1_L8: assumes A1: "a\<in>R" "b\<in>R" "c\<in>R"
  shows "a\<cdot>(b\<rs>c) = a\<cdot>b \<rs> a\<cdot>c" "(b\<rs>c)\<cdot>a = b\<cdot>a \<rs> c\<cdot>a"
proof -;
  from A1 show "a\<cdot>(b\<rs>c) = a\<cdot>b \<rs> a\<cdot>c" "(b\<rs>c)\<cdot>a = b\<cdot>a \<rs> c\<cdot>a"
    using Ring_ZF_1_L3 Ring_ZF_1_L5 Ring_ZF_1_L7 Ring_ZF_1_L4
    by auto;
qed;
end
