(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Group\_ZF.thy}*}

theory Group_ZF = func1:

text{*This theory will cover basics of group theory.*}

section{*Monoids.*}

text{*Monoid is a set with an associative operation and a neutral element.
  The operation is of course a function on $G\times G$ with values in $G$, 
  and therefore it is a subset of $(G\times G)\times G$. 
  Those who don't like that can go to HOL. 
  Monoid is like a group except that we don't require existence of 
  the inverse.*}

constdefs
  IsAmonoid :: "[i,i] \<Rightarrow> o"
  "IsAmonoid(G,f) ==
  f {is associative on} G \<and> 
  (\<exists>e\<in>G. (\<forall> g\<in>G. ( (f`(<e,g>) = g) \<and> (f`(<g,e>) = g))))";

text{*We use locales to define notation. This allows to separate notation 
and notion definitions. We would like to use additive notation for 
monoid, but unfortunately + is already taken.*}

locale monoid0 =
  fixes G :: "i"
  fixes f :: "i"
  assumes monoidAsssum:"IsAmonoid(G,f)"
  fixes monoper  :: "[i,i]\<Rightarrow>i" (infixl "\<oplus>" 70)
  defines monoper_def [simp]: "a \<oplus> b == f`<a,b>";

text{*The result of the monoid operation is in the monoid (carrier).*}

lemma (in monoid0) group0_1_L1: 
  assumes A1:"a\<in> G" "b\<in> G" shows "a\<oplus>b \<in> G"; 
proof -;
  from A1 monoidAsssum show ?thesis 
    using IsAmonoid_def IsAssociative_def apply_funtype 
    by auto
qed;

text{*There is only one neutral element in monoid.*}

lemma (in monoid0) group0_1_L2: 
  shows "\<exists>!e. e\<in> G \<and> (\<forall> g\<in>G. ( (e\<oplus>g = g) \<and> g\<oplus>e = g))"
proof;
  from monoidAsssum show 
    "\<exists>e. e\<in> G \<and> (\<forall> g\<in>G. e\<oplus>g = g \<and> g\<oplus>e = g)"
    using IsAmonoid_def by auto;
  fix e y
  assume A1:"e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)"
    and A2: "y \<in> G \<and> (\<forall>g\<in>G. y \<oplus> g = g \<and> g \<oplus> y = g)" show "e = y"; 
  proof -
    from A1 A2 have "y\<oplus>e = y" by simp;
    moreover from A1 A2 have "y\<oplus>e = e" by auto;
    ultimately show ?thesis by simp;
  qed;
qed;

text{*We could put the definition of neutral element anywhere, 
but it is only usable in conjuction 
with the above lemma.*}

constdefs 
 TheNeutralElement :: "[i,i]\<Rightarrow>i"
 "TheNeutralElement(G,f) == 
  ( THE e. (e\<in>G \<and> (\<forall> g\<in>G. ( (f`<e,g> = g) \<and> f`<g,e> = g))))";

text{*The neutral element is neutral.*}

lemma (in monoid0) group0_1_L3:
  assumes A1: "e = TheNeutralElement(G,f)"
  shows "e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)"
proof -;
  let ?n =  "THE b. b\<in> G \<and> (\<forall> g\<in>G. ( (b\<oplus>g = g) \<and> g\<oplus>b = g))";
  have "\<exists>!b. b\<in> G \<and> (\<forall> g\<in>G. ( (b\<oplus>g = g) \<and> g\<oplus>b = g))"
    using group0_1_L2 by simp;
  hence "?n\<in> G \<and> (\<forall> g\<in>G. ( (?n\<oplus>g = g) \<and> g\<oplus>?n = g))"
    by (rule theI);
  with A1 show ?thesis 
    using TheNeutralElement_def by simp;
qed;

text{*The monoid carrier is not empty.*}

lemma (in monoid0) group0_1_L3A: shows "G\<noteq>0"
proof -;
  have "TheNeutralElement(G,f) \<in> G" using group0_1_L3
    by simp;
  thus ?thesis by auto;
qed;

text{* The range of the monoid operation is the whole monoid carrier.*}

lemma (in monoid0) group0_1_L3B: shows "range(f) = G"
proof;
  from monoidAsssum have T1:"f : G\<times>G\<rightarrow>G"
     using IsAmonoid_def IsAssociative_def by simp;
  then show "range(f) \<subseteq> G" 
    using func1_1_L5B by simp;
  show "G \<subseteq> range(f)"
  proof;
    fix g assume A1:"g\<in>G"
    let ?e = "TheNeutralElement(G,f)"
    from A1 have "<?e,g> \<in> G\<times>G" "g = f`<?e,g>"
      using group0_1_L3 by auto;
    with T1 show "g \<in> range(f)"
      using func1_1_L5A by blast;
  qed;
qed;
      
text{*In a monoid a neutral element is the neutral element.*}

lemma (in monoid0) group0_1_L4: 
  assumes A1:"e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)"
  shows "e = TheNeutralElement(G,f)"
proof -
  let ?n =  "THE b. b\<in> G \<and> (\<forall> g\<in>G. ( (b\<oplus>g = g) \<and> g\<oplus>b = g))";
  have "\<exists>!b. b\<in> G \<and> (\<forall> g\<in>G. ( (b\<oplus>g = g) \<and> g\<oplus>b = g))"
    using group0_1_L2 by simp;
  moreover from A1 have 
    "e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)" .;
  ultimately have "(?n) = e" by (rule the_equality2);
  then show ?thesis using TheNeutralElement_def by simp;
qed;

text{*The next lemma shows that if the if we restrict the monoid operation to
  a subset of $G$ that contains the neutral element, then the 
  neutral element of the monoid operation is also neutral with the 
  restricted operation. This is used proven separately because it is
  used more than once.
*}

lemma (in monoid0) group0_1_L5:
   assumes A1:"\<forall>x\<in>H.\<forall>y\<in>H. x\<oplus>y \<in> H"
  and A2:"H\<subseteq>G"
  and A3:"e = TheNeutralElement(G,f)"
  and A4:"g = restrict(f,H\<times>H)"
  and A5:"e\<in>H"
  and A6:"h\<in>H"
  shows "(g`(<e,h>) = h) \<and> (g`(<h,e>) = h)";
proof -;
  from A4 A6 A5 have 
    "g`(<e,h>) = e\<oplus>h \<and> g`(<h,e>) = h\<oplus>e"
    using restrict_if by simp;
  with A3 A4 A6 A2 show 
    "g`(<e,h>) = h \<and> g`(<h,e>) = h"
    using  group0_1_L3 by auto;
qed;

text{*The next theorem shows that if the monoid operation is closed
on a subset of $G$ then this set is a (sub)monoid.
(although we do not define this notion). This will be 
useful when we study subgroups. *}

theorem (in monoid0) group0_1_T1: 
  assumes A1:"\<forall>x\<in>H.\<forall>y\<in>H. x\<oplus>y \<in> H"
  and A2:"H\<subseteq>G"
  and A3:"TheNeutralElement(G,f) \<in> H"
  shows "IsAmonoid(H,restrict(f,H\<times>H))"
proof -;
  let ?g = "restrict(f,H\<times>H)"
  let ?e = "TheNeutralElement(G,f)"
  from monoidAsssum have "f \<in> G\<times>G\<rightarrow>G" 
    using IsAmonoid_def IsAssociative_def by simp;
  moreover from A2 have "H\<times>H \<subseteq> G\<times>G" by auto;
  moreover from A1 have "\<forall>p \<in> H\<times>H. f`(p) \<in> H"
    by auto;
  ultimately have "?g \<in> H\<times>H\<rightarrow>H";
    using func1_2_L4 by simp;
  moreover have "\<forall> x \<in> H. \<forall> y \<in> H. \<forall> z \<in> H. 
    ( ?g`(<?g`(<x,y>),z>) = ?g`( < x,?g`(<y,z>)> ))"
  proof -
    from A1 have "\<forall> x \<in> H. \<forall> y \<in> H. \<forall> z \<in> H.
      ?g`(<?g`(<x,y>),z>) = x\<oplus>y\<oplus>z"
      using restrict_if by simp;
    moreover have "\<forall> x \<in> H. \<forall> y \<in> H. \<forall> z \<in> H.
      x\<oplus>y\<oplus>z = x\<oplus>(y\<oplus>z)"
    proof -;
      from monoidAsssum have 
	"\<forall> x \<in> G. \<forall> y \<in> G. \<forall> z \<in> G.
	x\<oplus>y\<oplus>z = x\<oplus>(y\<oplus>z)"
	using IsAmonoid_def IsAssociative_def 
	by simp;
      with A2 show ?thesis by auto;
    qed;
    moreover from A1 have "\<forall> x \<in> H. \<forall> y \<in> H. \<forall> z \<in> H.
      x\<oplus>(y\<oplus>z) = ?g`( < x,?g`(<y,z>)> )"
      using restrict_if by simp;
    ultimately show ?thesis by simp; 
  qed;
  moreover have 
    "\<exists>n\<in>H. (\<forall> h\<in>H. ( (?g`(<n,h>) = h) \<and> (?g`(<h,n>) = h)))"
  proof -;
    from A1 A2 A3 have 
      "\<forall> h\<in>H. ( (?g`(<?e,h>) = h) \<and> (?g`(<h,?e>) = h))"
      using group0_1_L5 by blast;
    with A3 show ?thesis by auto;
  qed;
  ultimately show ?thesis using IsAmonoid_def IsAssociative_def 
    by simp;
qed;
    
text{*Under the assumptions of group0\_1\_T1 the neutral element of a 
submonoid is the same as that of the monoid.*}

lemma group0_1_L6: 
  assumes A1:"IsAmonoid(G,f)"
  and A2:"\<forall>x\<in>H.\<forall>y\<in>H. f`<x,y> \<in> H"
  and A3:"H\<subseteq>G"
  and A4:"TheNeutralElement(G,f) \<in> H"
  shows "TheNeutralElement(H,restrict(f,H\<times>H)) = TheNeutralElement(G,f)"
proof -
  def D1: e == "TheNeutralElement(G,f)"
  def D2: g == "restrict(f,H\<times>H)"
  with A1 A2 A3 A4 have "monoid0(H,g)";
    using monoid0_def monoid0.group0_1_T1 
    by simp;
  moreover have 
    "e \<in> H \<and> (\<forall>h\<in>H. g`<e,h> = h \<and> g`<h,e> = h)"
  proof -;
    from A1 have "monoid0(G,f)" using monoid0_def 
      by simp;
    with A2 A3 D1 D2 A4 show ?thesis
      using monoid0.group0_1_L5 by blast;
  qed;
  ultimately have "e =  TheNeutralElement(H,g)"
    using monoid0.group0_1_L4 by auto;
  with D1 D2 show ?thesis by simp;
qed;
 
  
section{*Basic definitions and results for groups*}

text{*To define a group we take a monoid and add a requirement 
that the right inverse needs to exist for every element of the group.
We also define the group inverse as a relation on the group carrier.
Later we will show that this relation is a function. 
The GroupInv below is really the right inverse, understood as a function, 
that is a subset of $G\times G$.*}

constdefs

  IsAgroup :: "[i,i]\<Rightarrow>o"
  "IsAgroup(G,f) ==
  (IsAmonoid(G,f) \<and>
  (\<forall>g\<in>G. \<exists>b\<in>G. (f`<g,b> = TheNeutralElement(G,f))))"

  GroupInv :: "[i,i]\<Rightarrow>i"
  "GroupInv(G,f) == {<x,y> \<in> G\<times>G. f`<x,y> = TheNeutralElement(G,f)}";

text{*We will use the miltiplicative notation for groups.*}

locale group0 =
  fixes G :: "i"
  fixes f :: "i"
  assumes groupAsssum:"IsAgroup(G,f)"
  fixes e :: "i"
  defines e_def [simp]:"e == TheNeutralElement(G,f)"
  fixes groper  :: "[i,i]\<Rightarrow>i" (infixl "\<cdot>" 70)
  defines groper_def [simp]: "a \<cdot> b == f`<a,b>"
  fixes inv :: "i\<Rightarrow>i" ("_\<inverse> " [90] 91)
  defines inv_def [simp]: "x\<inverse> == GroupInv(G,f)`(x)";

text{*First we show a lemma that says that we can use theorems proven in
the monoid0 context (locale).*}

lemma (in group0) group0_2_L1: shows "monoid0(G,f)";
proof -;
  from groupAsssum show ?thesis using IsAgroup_def monoid0_def by simp;
qed;

text{*Technical lemma that allows to use $e$ as the neutral element of 
the group without referencing a list of lemmas and definitions.*}

lemma (in group0) group0_2_L2: 
  shows "e\<in>G \<and> (\<forall>g\<in>G.(e\<cdot>g = g \<and> g\<cdot>e = g))"
proof -;
  show ?thesis using group0_2_L1 monoid0.group0_1_L3 by simp;
qed;

text{*The group operation is associative. This is another technical lemma 
that allows to shorten the list of referenced lemmas in some proofs.*}

lemma (in group0) group0_2_L3: 
  assumes A1:"a\<in>G" "b\<in>G" "c\<in>G" shows "a\<cdot>(b\<cdot>c) = a\<cdot>b\<cdot>c"
proof -;
  from A1 groupAsssum show ?thesis 
    using IsAgroup_def IsAmonoid_def IsAssociative_def 
      group0_2_L1 monoid0.group0_1_L1 
    by simp;
qed;

text{*The group operation maps $G\times G$ into $G$. It is conveniet to have
  this fact easily accessible in the group0 context.*}

lemma (in group0) group0_2_L3A: shows "f : G\<times>G\<rightarrow>G"
proof -;
  from groupAsssum show ?thesis using
    IsAgroup_def IsAmonoid_def IsAssociative_def
    by simp;
qed;

  
text{*The definition of group requires the existence of the right inverse.
We show that this is also the left inverse.*}

theorem (in group0) group0_2_T1: 
  assumes A1: "g\<in>G" and A2:"b\<in>G" and A3: "g\<cdot>b = e"
  shows "b\<cdot>g = e";
proof -
  from A2 groupAsssum obtain c where D1:"c \<in> G \<and> b\<cdot>c = e" 
    using IsAgroup_def by auto;
  then have T1:"c\<in>G" by simp;
  have T2:"e\<in>G" using group0_2_L2 by simp;
  from A1 A2 have "b\<cdot>g = b\<cdot>g\<cdot>e" 
    using group0_2_L1 monoid0.group0_1_L1 group0_2_L2 by simp;
  also from A1 A2 T2 have "b\<cdot>g\<cdot>e = b\<cdot>(g\<cdot>e)"
    using group0_2_L3 by simp;
  also from D1 have " b\<cdot>(g\<cdot>e) = b\<cdot>(g\<cdot>(b\<cdot>c))" 
    by simp;
  also from  A1 A2 T1 have "b\<cdot>(g\<cdot>(b\<cdot>c)) = b\<cdot>(g\<cdot>b\<cdot>c)"
    using group0_2_L3 by simp;
  also from A3 A2 D1 have "b\<cdot>(g\<cdot>b\<cdot>c)= e" using group0_2_L2 by simp;
  finally show "b\<cdot>g = e" by simp;
qed;

text{*For every element of a group there is only one inverse.*}

lemma (in group0) group0_2_L4: 
  assumes A1:"x\<in>G" shows "\<exists>!y. y\<in>G \<and> x\<cdot>y = e"
proof;
  from A1 groupAsssum show "\<exists>y. y \<in> G \<and>  x\<cdot>y = e" 
    using IsAgroup_def by auto;
  fix y n
  assume A2:"y\<in>G \<and>  x\<cdot>y = e" and A3:"n\<in>G \<and> x\<cdot>n = e" show "y=n"
  proof -
    from A1 A2 have T1:"y\<cdot>x =e"
      using group0_2_T1 by simp;
    from A2 A3 have "y=y\<cdot>(x\<cdot>n)" 
      using group0_2_L2 by simp;
    also from A1 A2 A3 have "y\<cdot>(x\<cdot>n)= (y\<cdot>x)\<cdot>n" 
      using group0_2_L3 by blast;
    also from T1 A3 have "(y\<cdot>x)\<cdot>n = n" 
      using group0_2_L2 by simp;
    finally show "y=n" by simp;
  qed;
qed;

text{*The group inverse is a function that maps G into G.*}

theorem group0_2_T2: 
  assumes A1:"IsAgroup(G,f)" shows "GroupInv(G,f) \<in> G\<rightarrow>G"
proof -;
  have "GroupInv(G,f)\<subseteq>G*G" using GroupInv_def by auto;
  moreover from A1 have
    "\<forall>x\<in>G. \<exists>!y. y\<in> G \<and> <x,y> \<in> GroupInv(G,f)"
    using group0_def group0.group0_2_L4 GroupInv_def by simp;
  ultimately show ?thesis using func1_1_L11 by simp;
qed;

text{*We can think about the group inverse (the function) 
  as the inverse image of the neutral element.*}

theorem (in group0) group0_2_T3: shows "f-``{e} = GroupInv(G,f)"
proof -;
  from groupAsssum have "f \<in> G\<times>G \<rightarrow> G" 
    using IsAgroup_def IsAmonoid_def IsAssociative_def 
    by simp;
  then show "f-``{e} = GroupInv(G,f)"
    using func1_1_L14 GroupInv_def by auto;
qed;

text{*The inverse is in the group.*}

lemma (in group0) group0_2_L5: assumes A1:"x\<in>G" shows "x\<inverse>\<in>G"
proof -; 
  from groupAsssum have "GroupInv(G,f) \<in> G\<rightarrow>G" using group0_2_T2 by simp;
  with A1 show ?thesis using apply_type by simp;
qed;

text{*The notation for the inverse means what it is supposed to mean.*}

lemma (in group0) group0_2_L6: 
  assumes A1:"x\<in>G" shows "x\<cdot>x\<inverse> = e \<and> x\<inverse>\<cdot>x = e"
proof;
  from groupAsssum have "GroupInv(G,f) \<in> G\<rightarrow>G" 
    using group0_2_T2 by simp; 
  with A1 have "<x,x\<inverse>> \<in>  GroupInv(G,f)" 
    using apply_Pair by simp;
  then show "x\<cdot>x\<inverse> = e" using GroupInv_def by simp;
  with A1 show "x\<inverse>\<cdot>x = e" using group0_2_L5 group0_2_T1 by blast; 
qed;

text{*The next two lemmas state that unless we multiply by 
the neutral element, the result is always 
different than any of the operands.*}

lemma (in group0) group0_2_L7: 
  assumes A1:"a\<in>G" and A2:"b\<in>G" and A3:"a\<cdot>b = a"
  shows "b=e"
proof -;
  from A3 have "a\<inverse> \<cdot> (a\<cdot>b) = a\<inverse>\<cdot>a" by simp;
  with A1 A2 show ?thesis using
    group0_2_L5 group0_2_L3 group0_2_L6 group0_2_L2
    by simp;
qed;

lemma (in group0) group0_2_L8: 
  assumes A1:"a\<in>G" and A2:"b\<in>G" and A3:"a\<cdot>b = b"
  shows "a=e"
proof -;
  from A3 have "(a\<cdot>b)\<cdot>b\<inverse>  = b\<cdot>b\<inverse>" by simp;
  with A1 A2 have "a\<cdot>(b\<cdot>b\<inverse>)  = b\<cdot>b\<inverse>" using
    group0_2_L5 group0_2_L3 by simp;
  with A1 A2 show ?thesis 
    using group0_2_L6 group0_2_L2 by simp;
qed;

text{*The inverse of the neutral element is the neutral element.*}

lemma (in group0) group0_2_L7A: shows "e\<inverse>=e"
proof -
  show ?thesis using 
    group0_2_L2 group0_2_L5 group0_2_L6 group0_2_L7
    by blast;
qed;

text{*If a product of two elements of a group is equal to the neutral
element then they are inverses of each other. *}

lemma (in group0) group0_2_L9: 
  assumes A1:"a\<in>G" and A2:"b\<in>G" and A3:"a\<cdot>b = e" 
  shows "a = b\<inverse>" and "b = a\<inverse>"
proof -;
  from A3 have "a\<cdot>b\<cdot>b\<inverse> = e\<cdot>b\<inverse>" by simp; 
  with A1 A2 have "a\<cdot>(b\<cdot>b\<inverse>) = e\<cdot>b\<inverse>" using
    group0_2_L5 group0_2_L3 by simp;
  with A1 A2 show "a = b\<inverse>" using
    group0_2_L6 group0_2_L5 group0_2_L2 by simp;
  from A3 have "a\<inverse>\<cdot>(a\<cdot>b) = a\<inverse>\<cdot>e" by simp;
  with A1 A2 show "b = a\<inverse>" using 
    group0_2_L5 group0_2_L3 group0_2_L6 group0_2_L2
    by simp;
qed;

text{*What is the inverse of a product?*}

lemma (in group0) group0_2_L10:
  assumes A1:"a\<in>G" and A2:"b\<in>G" 
  shows " b\<inverse>\<cdot>a\<inverse> = (a\<cdot>b)\<inverse>"
proof -;
  from A1 A2 have 
    T1:"b\<inverse>\<in>G" and T2:"a\<inverse>\<in>G" and T3:"a\<cdot>b\<in>G" and T4:"b\<inverse>\<cdot>a\<inverse> \<in> G"
    using group0_2_L5 group0_2_L1 monoid0.group0_1_L1  
    by auto;
  from A1 A2 T4 have "a\<cdot>b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = a\<cdot>(b\<cdot>(b\<inverse>\<cdot>a\<inverse>))"
    using group0_2_L3 by simp;
  moreover from A2 T1 T2 have "b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = b\<cdot>b\<inverse>\<cdot>a\<inverse>"
    using group0_2_L3 by simp;
  moreover from A2 T2 have "b\<cdot>b\<inverse>\<cdot>a\<inverse> = a\<inverse>"
     using group0_2_L6 group0_2_L2 by simp;
  ultimately have "a\<cdot>b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = a\<cdot>a\<inverse>"
    by simp;
  with A1 have "a\<cdot>b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = e"
    using group0_2_L6 by simp;
  with T3 T4 show "b\<inverse>\<cdot>a\<inverse> = (a\<cdot>b)\<inverse>"
    using group0_2_L9 by simp;
qed;

text{*The inverse of the inverse is the element.*}

lemma (in group0) group0_2_L11:
  assumes  A1:"a\<in>G" shows "a = (a\<inverse>)\<inverse>"
proof -;
  from A1 show ?thesis
    using group0_2_L5 group0_2_L6 group0_2_L9
    by simp;
qed;

text{*What is the inverse of $ab^{-1}$ ?*}

lemma (in group0) group0_2_L12:
  assumes A1:"a\<in>G" and A2:"b\<in>G" 
  shows "(a\<cdot>b\<inverse>)\<inverse> = b\<cdot>a\<inverse>"
proof -;
  from A1 A2 have "(a\<cdot>b\<inverse>)\<inverse> = (b\<inverse>)\<inverse>\<cdot> a\<inverse>"
    using group0_2_L5 group0_2_L10 by simp;
  with A2 show ?thesis
    using group0_2_L11 by simp;
qed;

text{*The following lemma is used in real numbers construction 
in Real\_ZF.thy.*}

lemma (in group0) group0_2_L13:
  assumes A1:"s \<in> G\<rightarrow>G" "r \<in> G\<rightarrow>G"
  shows 
  "GroupInv(G,f)``{s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} = {r`(n)\<cdot>(s`(n))\<inverse>. n\<in>G}"
proof -;   
  from groupAsssum have "GroupInv(G,f) \<in> G\<rightarrow>G"
    using group0_2_T2 by simp;
  moreover from A1 have "\<forall>n\<in>G. s`(n)\<cdot>(r`(n))\<inverse> \<in> G"
    using apply_type group0_2_L5 group0_2_L1 monoid0.group0_1_L1
    by simp;
  ultimately have "GroupInv(G,f)``{s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} 
    = {(s`(n)\<cdot>(r`(n))\<inverse>)\<inverse>. n\<in>G}"
    using func1_1_L17 by simp;
  with A1 show ?thesis
    using apply_type group0_2_L12 by simp;
qed;

text{*We can insert a $b\cdot b^{-1}$ between two group elements. *}

lemma (in group0) group0_2_L14:
  assumes A1: "a\<in>G" and A2:"b\<in>G" and A3:"c\<in>G"
  shows "a\<cdot>c\<inverse>=(a\<cdot>b\<inverse>)\<cdot>(b\<cdot>c\<inverse>)"
proof -;
  from A1 A2 A3 have 
    T1:"c\<inverse>\<in>G" and T2:"b\<inverse>\<in>G" and T3:"a\<cdot>b\<inverse>\<in>G"
   using group0_2_L5 group0_2_L1 monoid0.group0_1_L1
   by auto;
 from A1 A2 have "a\<cdot>c\<inverse> =  a\<cdot>(b\<inverse>\<cdot>b)\<cdot>c\<inverse>"
   using group0_2_L2 group0_2_L6 by simp;
 with A1 T2 A2 T3 T1 show ?thesis
   using group0_2_L3 by simp;  
qed;

text{*Another lemma about rearranging a product.*}

lemma (in group0) group0_2_L15:
  assumes A1:"a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G"
  shows "(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse> = a\<cdot>(b\<cdot>d\<inverse>)\<cdot>a\<inverse>\<cdot>(a\<cdot>c\<inverse>)"
proof -
  from A1 have T1:
    "d\<inverse>\<in>G"  "c\<inverse>\<in>G" "a\<cdot>b\<in>G" "a\<cdot>(b\<cdot>d\<inverse>)\<in>G"
    using group0_2_L5 group0_2_L1 monoid0.group0_1_L1
    by auto;
  with A1 have "(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse> = (a\<cdot>b)\<cdot>(d\<inverse>\<cdot>c\<inverse>)"
    using group0_2_L10 by simp;
  also from A1 T1 have "(a\<cdot>b)\<cdot>(d\<inverse>\<cdot>c\<inverse>) = a\<cdot>(b\<cdot>d\<inverse>)\<cdot>c\<inverse>"
    using group0_2_L3 by simp;
  also from A1 T1 have "a\<cdot>(b\<cdot>d\<inverse>)\<cdot>c\<inverse> = a\<cdot>(b\<cdot>d\<inverse>)\<cdot>a\<inverse>\<cdot>(a\<cdot>c\<inverse>)"
    using group0_2_L14 by blast;
  finally show ?thesis by simp;
qed;
    
section{*Subgroups*}

text{*There are two common ways to define subgroups. One requires that the
group operations are closed in the subgroup. The second one defines subgroup 
as a subset of a group which is itself a group under the group operations.
We use the second approach because it results in shorter definition. We do not 
require $H$ to be a subset of $G$ as this can be inferred from our definition.
The rest of this section is devoted to proving the equivalence of these two
definitions of the notion of a subgroup. 
*}

constdefs 
  IsAsubgroup :: "[i,i]\<Rightarrow>o"
  "IsAsubgroup(H,f) == IsAgroup(H, restrict(f,H\<times>H))"

text{*Formally the group operation in a subgroup is different than in the
group as they have different domains. Of course we want to use the original 
operation with the asociated notation in the subgroup. The next couple of 
lemmas will allow for that.*}

text{*The neutral element of the subgroup is in the subgroup and it is 
both right and left neutral there. The notation is very ugly because
we don't want to introduce a separate notation for the subgroup operation.
*}

lemma group0_3_L1: 
  assumes A1:"IsAsubgroup(H,f)" 
  shows "TheNeutralElement(H,restrict(f,H\<times>H)) \<in> H"
  and 
  "\<forall>h\<in>H. restrict(f,H\<times>H)`<TheNeutralElement(H,restrict(f,H\<times>H)),h > = h"
  and 
  "\<forall>h\<in>H. restrict(f,H\<times>H)`<h,TheNeutralElement(H,restrict(f,H\<times>H)) > = h"
proof -;
  let ?b = "restrict(f,H\<times>H)"
  let ?n = "TheNeutralElement(H,restrict(f,H\<times>H))"
  from A1 have "group0(H,?b)"
    using IsAsubgroup_def group0_def by simp;
  then have T1:
    "?n \<in> H \<and> (\<forall>h\<in>H. (?b`<?n,h > = h \<and> ?b`<h,?n > = h))"
    by (rule group0.group0_2_L2);
  thus "?n \<in> H" by simp;
  from T1 show "\<forall>h\<in>H. ?b`<?n,h > = h" by auto;
  from T1 show "\<forall>h\<in>H. ?b`<h,?n> = h" by auto;
qed;

text{*Subgroup is contained in the group.*}

lemma (in group0) group0_3_L2: 
  assumes A1:"IsAsubgroup(H,f)"
  shows "H\<subseteq>G"
proof;
  fix h assume A2:"h\<in>H";
  let ?b = "restrict(f,H\<times>H)"
  let ?n = "TheNeutralElement(H,restrict(f,H\<times>H))"
   from A1 have "?b \<in> H\<times>H\<rightarrow>H" 
    using IsAsubgroup_def IsAgroup_def 
      IsAmonoid_def IsAssociative_def by simp;
  moreover from A2 A1 have "<?n,h> \<in> H\<times>H" 
    using group0_3_L1 by simp;
  moreover from A1 A2 have "h = ?b`<?n,h >" 
    using group0_3_L1 by simp;
  ultimately have "< <?n,h> , h > \<in> ?b" 
    using func1_1_L5A by blast;
  then have "< <?n,h> , h > \<in> f" using restrict_subset by auto;
  moreover from groupAsssum have "f \<in> G\<times>G \<rightarrow> G"
    using IsAgroup_def IsAmonoid_def IsAssociative_def 
    by simp
  ultimately have  "<?n,h> \<in> G\<times> G \<and> h\<in>G" using func1_1_L5 
    by blast;
  then show "h\<in>G" by simp;
qed;

text{*The group neutral element (denoted $e$ in the group0 context)
is a neutral element for the subgroup with respect to the froup action.*}

lemma (in group0) group0_3_L3:
  assumes A1:"IsAsubgroup(H,f)"
  shows "\<forall>h\<in>H. e\<cdot>h = h \<and> h\<cdot>e = h"
proof;
  fix h assume "h\<in>H" 
  with groupAsssum A1 show "e\<cdot>h = h \<and> h\<cdot>e = h"
    using group0_3_L2 group0_2_L2 by auto;
qed;

text{*The neutral element of a subgroup is the same as that of the group.*}

lemma (in group0) group0_3_L4: assumes A1:"IsAsubgroup(H,f)"
  shows "TheNeutralElement(H,restrict(f,H\<times>H)) = e"
proof -;
  let ?n =  "TheNeutralElement(H,restrict(f,H\<times>H))"
  from A1 have T1:"?n \<in> H" using group0_3_L1 by simp;
  with groupAsssum A1 have "?n\<in>G" using  group0_3_L2 by auto;
  with A1 T1 show ?thesis using 
     group0_3_L1 restrict_if group0_2_L7 by simp;
qed;

text{*The neutral element of the group (denoted $e$ in the group0 context)
belongs to every subgroup.*}

lemma (in group0) group0_3_L5: assumes A1:"IsAsubgroup(H,f)"
  shows "e\<in>H"
proof -;
  from A1 show "e\<in>H" using 
    group0_3_L1 group0_3_L4 by blast;
qed;

text{*Subgroups are closed with respect to the group operation.*}

lemma (in group0) group0_3_L6: assumes A1:"IsAsubgroup(H,f)"
  and A2:"a\<in>H" "b\<in>H"
  shows "a\<cdot>b \<in> H"
proof -; 
  let ?b = "restrict(f,H\<times>H)"
  from A1 have "monoid0(H,?b)" using
    IsAsubgroup_def IsAgroup_def monoid0_def by simp;
  with A2 have "?b` (<a,b>) \<in> H" using monoid0.group0_1_L1
    by blast;
 with A2 show "a\<cdot>b \<in> H" using restrict_if by simp;
qed;

text{*A preliminary lemma that we need to  show that taking the inverse 
in the subgroup is the same as taking the inverse
in the group. This theorem is formulated outside of the group0 locale
because I was unable to apply some lemmas proven in the group0 locale 
to the subgroup.*}

lemma group0_3_L7: 
  assumes A1:"IsAgroup(G,f)" 
  and A2:"IsAsubgroup(H,f)" and A3:"g=restrict(f,H\<times>H)"
  shows "GroupInv(G,f) \<inter> H\<times>H = GroupInv(H,g)"
proof -
  def D1: e == "TheNeutralElement(G,f)"
  def D2: e1 == "TheNeutralElement(H,g)"
  from A1 have T1:"group0(G,f)" using group0_def by simp;
  from A2 A3 have T2:"group0(H,g)" 
    using IsAsubgroup_def group0_def by simp 
  from T1 A2 A3 D1 D2 have "e1 = e" 
    using group0.group0_3_L4 by simp;
  with T1 D1 have "GroupInv(G,f) = f-``{e1}" 
    using group0.group0_2_T3 by simp;
  moreover have "g-``{e1} = f-``{e1} \<inter> H\<times>H"
  proof -;
    from A1 have "f \<in> G\<times>G\<rightarrow>G" 
      using IsAgroup_def IsAmonoid_def IsAssociative_def 
      by simp;
    moreover from T1 A2 have "H\<times>H \<subseteq> G\<times>G" 
      using group0.group0_3_L2 by auto;
    ultimately show "g-``{e1} = f-``{e1} \<inter> H\<times>H" 
      using A3 func1_2_L1 by simp;
  qed;
  moreover from T2 A3 D2 have "GroupInv(H,g) = g-``{e1}" 
    using group0.group0_2_T3 by simp;
  ultimately show ?thesis by simp;
qed;

text{*Using the lemma above we can show the actual statement.*}

theorem (in group0) group0_3_T1:
  assumes A1: "IsAsubgroup(H,f)" 
  and A2:"g=restrict(f,H\<times>H)"
  shows "GroupInv(H,g) = restrict(GroupInv(G,f),H)"
proof -;
  from groupAsssum have "GroupInv(G,f) \<in> G\<rightarrow>G" 
    using group0_2_T2 by simp;
  moreover from A1 A2 have "GroupInv(H,g) \<in> H\<rightarrow>H"
    using IsAsubgroup_def group0_2_T2 by simp;
  moreover from A1 have "H\<subseteq>G" 
    using group0_3_L2 by simp;
  moreover from groupAsssum A1 A2 have 
    "GroupInv(G,f) \<inter> H\<times>H = GroupInv(H,g)"
    using group0_3_L7 by simp;
  ultimately show ?thesis
    using func1_2_L3 by simp;
qed;

text{*A sligtly weaker, but more convenient in applications,
  reformulation of the above theorem.*}

theorem (in group0) group0_3_T2: 
  assumes A1: "IsAsubgroup(H,f)" 
  and A2:"g=restrict(f,H\<times>H)"
  shows "\<forall>h\<in>H. GroupInv(H,g)`(h) = h\<inverse>"
proof -
  from A1 A2 show ?thesis 
    using group0_3_T1 restrict_if by simp;
qed;

text{*Subgroups are closed with respect to taking the group inverse.
  Again, I was unable to apply group0\_2\_L5 directly to the group $H$. 
  This problem is worked around by repeating the (short) proof of 
  group0\_2\_L5 in the proof below.*}

theorem (in group0) group0_3_T3A: 
  assumes A1:"IsAsubgroup(H,f)" and A2:"h\<in>H"
  shows "h\<inverse>\<in> H"
proof -
  def D1: g == "restrict(f,H\<times>H)";
  from A1 D1 have  "GroupInv(H,g) \<in> H\<rightarrow>H"
    using IsAsubgroup_def group0_2_T2 by simp;
  with A2 have "GroupInv(H,g)`(h) \<in> H"
    using apply_type by simp;
  with A1 D1 A2 show "h\<inverse>\<in> H" using group0_3_T2 by simp;
qed;

text{*The next theorem states that a nonempty subset of 
of a group $G$ that is closed with respect to the group operation and 
taking the inverse is a subgroup of the group.*}

theorem (in group0) group0_3_T3:
  assumes A1: "H\<noteq>0"
  and A2: "H\<subseteq>G"
  and A3: "H {is closed with respect to} f"
  and A4: "\<forall>x\<in>H. x\<inverse> \<in> H"
  shows "IsAsubgroup(H,f)"
proof -;
  let ?g = "restrict(f,H\<times>H)"
  let ?n = "TheNeutralElement(H,?g)"
  from A3 have T0:"\<forall>x\<in>H.\<forall>y\<in>H. x\<cdot>y \<in> H"
    using IsOpClosed_def by simp;
  from A1 obtain x where "x\<in>H" by auto;
  with A4 T0 A2 have T1:"e\<in>H"
    using group0_2_L6 by blast;
  with T0 A2 have T2:"IsAmonoid(H,?g)"
    using group0_2_L1 monoid0.group0_1_T1
    by simp;
  moreover have "\<forall>h\<in>H.\<exists>b\<in>H. ?g`<h,b> = ?n"
  proof;
    fix h assume A5:"h\<in>H"
    with A4 A2 have "h\<cdot>h\<inverse> = e"
      using group0_2_L6 by auto;
    moreover from groupAsssum T0 A2 T1 have "e = ?n"
      using IsAgroup_def group0_1_L6 by auto;
    moreover from A5 A4 have "?g`<h,h\<inverse>> = h\<cdot>h\<inverse>"
      using restrict_if by simp;
    ultimately have "?g`<h,h\<inverse>> = ?n" by simp;
    with A5 A4 show "\<exists>b\<in>H. ?g`<h,b> = ?n" by auto;
  qed;
  ultimately show "IsAsubgroup(H,f)" using 
    IsAsubgroup_def IsAgroup_def by simp;
qed;

text{*Intersection of subgroups is a subgroup of each factor.*}

lemma group0_3_L7:
  assumes A1:"IsAgroup(G,f)"
  and A2:"IsAsubgroup(H1,f)"
  and A3:"IsAsubgroup(H2,f)"
  shows "IsAsubgroup(H1\<inter>H2,restrict(f,H1\<times>H1))"
proof -;
  let ?e = "TheNeutralElement(G,f)"
  let ?g = "restrict(f,H1\<times>H1)"
  from A1 have T1: "group0(G,f)"
    using group0_def by simp;
  from A2 have "group0(H1,?g)"
    using IsAsubgroup_def group0_def by simp;
  moreover have "H1\<inter>H2 \<noteq> 0"
  proof -
    from A1 A2 A3 have "?e \<in> H1\<inter>H2"
      using group0_def group0.group0_3_L5 by simp;
    thus ?thesis by auto;
  qed;
  moreover have T2:"H1\<inter>H2 \<subseteq> H1" by auto;
  moreover from T1 T2 A2 A3 have 
    "H1\<inter>H2 {is closed with respect to} ?g"
    using group0.group0_3_L6 IsOpClosed_def 
      func1_6_L7 func1_6_L5 by simp;
  moreover from T1 A2 A3 have 
    "\<forall>x \<in> H1\<inter>H2. GroupInv(H1,?g)`(x) \<in> H1\<inter>H2"
    using group0.group0_3_T2 group0.group0_3_T3A
    by simp;
  ultimately show ?thesis
    using group0.group0_3_T3 by simp;
qed;

text{*If one subgroup contains another then the latter is a subgroup of 
  the former.*}

section{*Abelian groups*}

text{*Here we will prove some facts specific to abelian groups. *}

text{*The following fact is useful for real numbers constuction 
  in Real\_ZF.thy.*}

lemma (in group0) group0_4_L1: 
  assumes A1:"f {is commutative on} G" 
  and A2:"a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G"
  shows "(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse> = (a\<cdot>c\<inverse>)\<cdot>(b\<cdot>d\<inverse>)"
proof -; 
  from A2 have T1:"c\<inverse>\<in>G" and T2:"d\<inverse>\<in>G" 
    and T3:"a\<cdot>b \<in> G" and T4:"a\<cdot>c\<inverse> \<in> G"
    using group0_2_L5  group0_2_L1 monoid0.group0_1_L1 by auto
  from A2 T2 T1 A1 have "(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse> = (a\<cdot>b)\<cdot>(c\<inverse>\<cdot>d\<inverse>)"
    using group0_2_L10 IsCommutative_def by simp;
  also from T3 T1 T2 A2 A1 have "(a\<cdot>b)\<cdot>(c\<inverse>\<cdot>d\<inverse>) = a\<cdot>(c\<inverse>\<cdot>b)\<cdot>d\<inverse>"
    using group0_2_L3 IsCommutative_def by simp;
  also from A2 T1 T4 T2 have "a\<cdot>(c\<inverse>\<cdot>b)\<cdot>d\<inverse> = (a\<cdot>c\<inverse>)\<cdot>(b\<cdot>d\<inverse>)"
    using group0_2_L3 by simp;
  finally show ?thesis by simp;
qed;

text{*In abelian groups, three elements can be rearranged by putting 
  the last one as first.*}

lemma (in group0) group0_4_L2A:
  assumes A1:"f {is commutative on} G"
  and A2:"a\<in>G" "b\<in>G" "c\<in>G"
  shows "a\<cdot>b\<cdot>c = c\<cdot>a\<cdot>b"
proof -;
  from A1 A2 have "a\<cdot>b\<cdot>c = c\<cdot>(a\<cdot>b)"
     using IsCommutative_def group0_2_L1 monoid0.group0_1_L1
    by simp;
  with A2 show ?thesis using
     group0_2_L1 monoid0.group0_1_L1 group0_2_L3
    by simp;
qed;

text{*Proving the facts about associative and commutative operations is 
  quite tedious in formalized mathematics. To a human the thing is simple:
  we can arrange the elements in any order and put parantheses wherever we 
  want, it is all the same. However, formalizing this statement would be 
  rather difficult (I think). The next lemma attempts a quasi-algorithmic
  approach to this type of problem. To prove that two expressions are equal, 
  we first strip one from parantheses, then rearrange the elements in proper 
  order, then put the parantheses where we want them to be. The algorithm for 
  rearrangement is easy to describe: we keep putting the first element 
  (from the right) that is in the wrong place at the left-most position
  until we get the proper arrangement. For the parantheses simp does it 
  very well.*}

lemma (in group0) group0_4_L2:
  assumes A1:"f {is commutative on} G"
  and A2:"a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G" "E\<in>G" "F\<in>G"
  shows "(a\<cdot>b)\<cdot>(c\<cdot>d)\<cdot>(E\<cdot>F) = (a\<cdot>(d\<cdot>F))\<cdot>(b\<cdot>(c\<cdot>E))"
proof -;
  from A2 have "(a\<cdot>b)\<cdot>(c\<cdot>d)\<cdot>(E\<cdot>F) = a\<cdot>b\<cdot>c\<cdot>d\<cdot>E\<cdot>F"
    using group0_2_L1 monoid0.group0_1_L1 group0_2_L3
    by simp;
  also have  "a\<cdot>b\<cdot>c\<cdot>d\<cdot>E\<cdot>F = a\<cdot>d\<cdot>F\<cdot>b\<cdot>c\<cdot>E"
  proof -
    from A1 A2 have "a\<cdot>b\<cdot>c\<cdot>d\<cdot>E\<cdot>F = F\<cdot>(a\<cdot>b\<cdot>c\<cdot>d\<cdot>E)"
      using IsCommutative_def group0_2_L1 monoid0.group0_1_L1 
      by simp;
    also from A2 have "F\<cdot>(a\<cdot>b\<cdot>c\<cdot>d\<cdot>E) = F\<cdot>a\<cdot>b\<cdot>c\<cdot>d\<cdot>E"
      using group0_2_L1 monoid0.group0_1_L1 group0_2_L3
      by simp;
    also from A1 A2 have "F\<cdot>a\<cdot>b\<cdot>c\<cdot>d\<cdot>E = d\<cdot>(F\<cdot>a\<cdot>b\<cdot>c)\<cdot>E"
      using IsCommutative_def group0_2_L1 monoid0.group0_1_L1
      by simp;
    also from A2 have "d\<cdot>(F\<cdot>a\<cdot>b\<cdot>c)\<cdot>E = d\<cdot>F\<cdot>a\<cdot>b\<cdot>c\<cdot>E"
      using group0_2_L1 monoid0.group0_1_L1 group0_2_L3
      by simp;
    also from A1 A2 have " d\<cdot>F\<cdot>a\<cdot>b\<cdot>c\<cdot>E = a\<cdot>(d\<cdot>F)\<cdot>b\<cdot>c\<cdot>E"
      using IsCommutative_def group0_2_L1 monoid0.group0_1_L1
      by simp;
    also from A2 have "a\<cdot>(d\<cdot>F)\<cdot>b\<cdot>c\<cdot>E = a\<cdot>d\<cdot>F\<cdot>b\<cdot>c\<cdot>E" 
      using group0_2_L1 monoid0.group0_1_L1 group0_2_L3
      by simp;
    finally show ?thesis by simp;
  qed;
  also from A2 have "a\<cdot>d\<cdot>F\<cdot>b\<cdot>c\<cdot>E = (a\<cdot>(d\<cdot>F))\<cdot>(b\<cdot>(c\<cdot>E))"
    using group0_2_L1 monoid0.group0_1_L1 group0_2_L3
    by simp;
  finally show ?thesis by simp;
qed;
  
text{*Another fact useful for real numbers construction 
  in Real\_ZF.thy.*}

lemma (in group0) group0_4_L3:
  assumes A1:"f {is commutative on} G" 
  and A2:"a\<in>G" "b\<in>G" and A3:"c\<in>G" "d\<in>G" "E\<in>G" "F\<in>G"
  shows "a\<cdot>b\<cdot>((c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse>) = (a\<cdot>(E\<cdot>c)\<inverse>)\<cdot>(b\<cdot>(F\<cdot>d)\<inverse>)"
proof -;
  from A3 have T1:
    "c\<inverse>\<in>G" "d\<inverse>\<in>G" "E\<inverse>\<in>G" "F\<inverse>\<in>G" "(c\<cdot>d)\<inverse>\<in>G" "(E\<cdot>F)\<inverse>\<in>G"
    using group0_2_L5 group0_2_L1 monoid0.group0_1_L1 
    by auto;
  from A2 T1 have 
    "a\<cdot>b\<cdot>((c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse>) = a\<cdot>b\<cdot>(c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse>"
    using group0_2_L1 monoid0.group0_1_L1 group0_2_L3
    by simp;
  also from A2 A3 have 
    "a\<cdot>b\<cdot>(c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse> = (a\<cdot>b)\<cdot>(d\<inverse>\<cdot>c\<inverse>)\<cdot>(F\<inverse>\<cdot>E\<inverse>)"
    using group0_2_L10 by simp;
   also from A1 A2 T1 have 
    "(a\<cdot>b)\<cdot>(d\<inverse>\<cdot>c\<inverse>)\<cdot>(F\<inverse>\<cdot>E\<inverse>) = (a\<cdot>(c\<inverse>\<cdot>E\<inverse>))\<cdot>(b\<cdot>(d\<inverse>\<cdot>F\<inverse>))"
    using group0_4_L2 by simp;
  also from A2 A3 have 
    "(a\<cdot>(c\<inverse>\<cdot>E\<inverse>))\<cdot>(b\<cdot>(d\<inverse>\<cdot>F\<inverse>)) = (a\<cdot>(E\<cdot>c)\<inverse>)\<cdot>(b\<cdot>(F\<cdot>d)\<inverse>)"
    using group0_2_L10 by simp;
  finally show ?thesis by simp;
qed;

text{*Another fact useful for real numbers construction 
  in Real\_ZF.thy.*}

lemma (in group0) group0_4_L4:
  assumes A1:"f {is commutative on} G" 
  and A2:"a\<in>G" "b\<in>G" "c\<in>G"
  shows "a\<inverse>\<cdot>(b\<inverse>\<cdot>c\<inverse>)\<inverse> = (a\<cdot>(b\<cdot>c)\<inverse>)\<inverse>"
proof -;
  from A2 have T1: 
    "b\<inverse>\<in>G" "c\<inverse>\<in>G" "b\<inverse>\<cdot>c\<inverse> \<in> G"
    using group0_2_L5 group0_2_L1 monoid0.group0_1_L1
    by auto;
  with A1 A2 show ?thesis
    using group0_2_L10 IsCommutative_def 
    by simp;
qed;

end