(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

HIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Real\_ZF\_1.thy}*}

theory Real_ZF_1 imports Real_ZF Int_ZF_2 OrderedField_ZF

begin

text{*In this theory file we continue the construction of real numbers started
  in @{text "Real_ZF.thy"}. We put here those parts of the construction that 
  can not be done in the general settings of abelian groups and require 
  integers, but do not require the axiom of choice.*}

section{*Definitions and notation*}

text{*In this section we define notions and notation needed for the rest of 
  the construction.*}


text{* The order on the set of real numbers is constructed by 
  specifying the set of 
  positive reals. This is defined as the projection of the set of positive
  slopes. A slope is positive if it takes an infinite number of 
  posititive values on the positive integers (see @{text "Int_ZF_2.thy"} 
  for properties of positive slopes). The order relation on real numbers
  is defined by prescribing the set of positive numbers (see section 
  @{text "Alternative definitions"} in @{text "OrderedGroup_ZF.thy."}). *}

constdefs

  "PositiveSlopes \<equiv> {s \<in> Slopes. 
  s``(PositiveIntegers) \<inter>  PositiveIntegers \<notin> Fin(int)}"

  "PositiveReals \<equiv> {SlopeEquivalenceRel``{s}. s \<in> PositiveSlopes}"

  "OrderOnReals \<equiv> OrderFromPosSet(RealNumbers,RealAddition,PositiveReals)"

text{*The next locale extends the locale @{text "real0"} to define notation 
  specific to the construction of real numbers. The notation follows the
  one defined in @{text "Int_ZF_2.thy"}. *}

locale real1 = real0 +

  fixes AlEq (infix "\<sim>" 68)
  defines AlEq_def [simp]: "s \<sim> r \<equiv> \<langle>s,r\<rangle> \<in> SlopeEquivalenceRel"

  fixes slope_add (infix "\<fp>" 70)
  defines slope_add_def [simp]: 
  "s \<fp> r \<equiv>  SlopeOp1`\<langle>s,r\<rangle>"

  fixes slope_comp (infix "\<circ>" 71)
  defines slope_comp_def [simp]: "s \<circ> r \<equiv>  SlopeOp2`\<langle>s,r\<rangle>"

  fixes slopes ("\<S>" )
  defines slopes_def [simp]: "\<S> \<equiv> AlmostHoms(int,IntegerAddition)"

  fixes posslopes ("\<S>\<^isub>+")
  defines posslopes_def [simp]: "\<S>\<^isub>+ \<equiv> PositiveSlopes"

  fixes slope_class ("[ _ ]")
  defines slope_class_def [simp]: "[f] \<equiv> SlopeEquivalenceRel``{f}"


  fixes slope_neg :: "i\<Rightarrow>i" ("\<fm>_" [90] 91)
  defines slope_neg_def [simp]: "\<fm>s \<equiv> GroupInv(int,IntegerAddition) O s"

section{*Multiplication of real numbers*}

text{*Multiplication of real numbers is defined as a projection of 
  composition of slopes onto the space of equivalence classes of slopes.
  Thus, the product of the real numbers given as classes of slopes $s$ and 
  $r$ is defined as the class of $s\circ r$. The goal of this section is to 
  show that multiplication defined this way is commutative.*}

text{*Let's recall a theorem from @{text "Int_ZF_2.thy"} that states that 
  if $f,g$ are slopes,
  then $f\circ g$ is equivalent to $g\circ f$. 
  Here we conclude from that that
  the classes of $f\circ g$ and $g\circ f$ are the same.*}

lemma (in real1) Real_ZF_1_1_L2: assumes A1: "f \<in> \<S>"  "g \<in> \<S>"
  shows "[f\<circ>g] = [g\<circ>f]"
proof -
  from A1 have "f\<circ>g \<sim> g\<circ>f" 
    using Slopes_def int1.Arthan_Th_9 SlopeOp1_def BoundedIntMaps_def
      SlopeEquivalenceRel_def SlopeOp2_def by simp;
  then show ?thesis using (*Real_ZF_1_1_L1*) Real_ZF_1_L11 equiv_class_eq
    by simp;
qed;

text{*Classes of slopes are real numbers.*}

lemma (in real1) Real_ZF_1_1_L3: assumes A1: "f \<in> \<S>"
  shows "[f] \<in> \<real>"
proof -
  from A1 have "[f] \<in> Slopes//SlopeEquivalenceRel"
    using Slopes_def quotientI by simp;
  then show "[f] \<in> \<real>" using RealNumbers_def by simp;  
qed;

text{*Each real number is a class of a slope.*}

lemma (in real1) Real_ZF_1_1_L3A: assumes A1: "a\<in>\<real>"
  shows "\<exists>f\<in>\<S> . a = [f]"
proof -
  from A1 have "a \<in> \<S>//SlopeEquivalenceRel" 
    using RealNumbers_def Slopes_def by simp
  then show ?thesis using quotient_def
    by simp;
qed;

text{*It is useful to have the definition of addition and multiplication 
  in the @{text "real1"} context notation.*}

lemma (in real1) Real_ZF_1_1_L4: 
  assumes A1: "f \<in> \<S>"  "g \<in> \<S>"
  shows
  "[f] \<ra> [g] = [f\<fp>g]"
  "[f] \<cdot> [g] = [f\<circ>g]"
proof -
  let ?r = "SlopeEquivalenceRel"
  have "[f]\<cdot>[g] = ProjFun2(\<S>,?r,SlopeOp2)`\<langle>[f],[g]\<rangle>"
    using RealMultiplication_def Slopes_def by simp;
  also from A1 have "\<dots> = [f\<circ>g]"
    using Real_ZF_1_L11 EquivClass_1_L10 Slopes_def
    by simp;
  finally show "[f] \<cdot> [g] = [f\<circ>g]" by simp;
  have "[f] \<ra> [g] = ProjFun2(\<S>,?r,SlopeOp1)`\<langle>[f],[g]\<rangle>"
    using RealAddition_def Slopes_def by simp;
  also from A1 have "\<dots> = [f\<fp>g]"
    using Real_ZF_1_L11 EquivClass_1_L10 Slopes_def
    by simp;
  finally show "[f] \<ra> [g] = [f\<fp>g]" by simp;
qed;
  
text{*Multiplication of real numbers is commutative.*}

theorem (in real1) real_mult_commute: assumes A1: "a\<in>\<real>"  "b\<in>\<real>"
  shows "a\<cdot>b = b\<cdot>a"
proof -
  from A1 have 
    "\<exists>f\<in>\<S> . a = [f]"
    "\<exists>g\<in>\<S> . b = [g]"
    using Real_ZF_1_1_L3A by auto;
  then obtain f g where 
    "f \<in> \<S>"  "g \<in> \<S>" and "a = [f]"  "b = [g]" 
    by auto;
  then show "a\<cdot>b = b\<cdot>a"
    using Real_ZF_1_1_L4 Real_ZF_1_1_L2 by simp;
qed;

text{*Multiplication is commutative on reals.*}

lemma real_mult_commutative: shows
  "RealMultiplication {is commutative on} RealNumbers"
  using real1.real_mult_commute IsCommutative_def
  by simp;

text{*$1$ is the class of identity function on integers. This is really shown
  in @{text "Real_ZF_1_L11"}, here we only rewrite it in the notation used
  in the @{text "real1"} context.*}

lemma (in real1) real_one_cl_identity: shows "[id(int)] = \<one>"
  using Real_ZF_1_L11 by simp;

text{*Two real numbers are equal iff the slopes that represent them are 
  almost equal. This is proven in @{text "Real_ZF_1_L13"}, here we just 
  rewrite it in the notation used in the @{text "real1"} context.*}

lemma (in real1) Real_ZF_1_1_L5: 
  assumes "f \<in> \<S>"  "g \<in> \<S>"
  shows "[f] = [g] \<longleftrightarrow> f \<sim> g"
  using prems Slopes_def Real_ZF_1_L13 by simp;

text{*If the pair of function belongs to the slope equivalence relation, then
  their classes are equal. This is convenient, because we don't need to assume
  that $f,g$ are slopes (follows from the fact that $f\sim g$).*}

lemma (in real1) Real_ZF_1_1_L5A: assumes "f \<sim> g"
  shows "[f] = [g]"
  using prems Real_ZF_1_L11 Slopes_def Real_ZF_1_1_L5
  by auto;

text{*Identity function on integers is a slope. 
  This is proven in @{text "Real_ZF_1_L13"}, here we just 
  rewrite it in the notation used in the @{text "real1"} context.*}

lemma (in real1) id_on_int_is_slope: shows "id(int) \<in> \<S>"
  using Real_ZF_1_L14 Slopes_def by simp;

text{*The next lemma is essentially the same as @{text " Real_ZF_1_L12"}, but
  written in the notation defined in the @{text "real1"} context. It states
  that if $f$ is a slope, then $-[f] = [-f]$.*}

lemma (in real1) Real_ZF_1_1_L6: assumes "f \<in> \<S>"
  shows "[\<fm>f] = \<rm>[f]"
  using prems Slopes_def SlopeEquivalenceRel_def Real_ZF_1_L12
  by simp;

text{*A result from @{text "Int_ZF_2.thy"}: the identity function on integers
  is not almost equal to any bounded function.*}

lemma (in real1) Real_ZF_1_1_L7:
  assumes A1: "f \<in> BoundedIntMaps"
  shows "\<not>(id(int) \<sim> f)"
  using prems Slopes_def SlopeOp1_def BoundedIntMaps_def 
    SlopeEquivalenceRel_def BoundedIntMaps_def int1.Int_ZF_2_3_L12 
  by simp;

text{*Zero is not one.*}

lemma (in real1) real_zero_not_one: shows "\<one>\<noteq>\<zero>"
proof -
  { assume A1: "\<one>=\<zero>"
    have "\<exists>f \<in> \<S>. \<zero> = [f]"
      using  Real_ZF_1_L4 Real_ZF_1_1_L3A by simp;
    with A1 have 
      "\<exists>f \<in> \<S>. [id(int)] = [f] \<and> [f] = \<zero>"
      using real_one_cl_identity by auto;
    then have False using Real_ZF_1_1_L5 Slopes_def 
      Real_ZF_1_L10 Real_ZF_1_1_L7 id_on_int_is_slope
      by auto;
  } then show "\<one>\<noteq>\<zero>" by auto;
qed;

section{*The order on reals*}

text{*In this section we show that the order relation defined by prescribing
  the set of positive reals as the projection of the set of positive
  slopes makes the ring of real numbers into an ordered ring.*}

text{*Positive slopes are slopes and positive reals are real.*}

lemma Real_ZF_1_2_L1: shows 
  "PositiveSlopes \<subseteq> Slopes"
  "PositiveReals \<subseteq> RealNumbers"
proof -
  have "PositiveSlopes = 
    {s \<in> Slopes. s``(PositiveIntegers) \<inter> PositiveIntegers \<notin> Fin(int)}"
    using PositiveSlopes_def by simp;
  then show "PositiveSlopes \<subseteq> Slopes" by (rule subset_with_property);
  then have 
    "{SlopeEquivalenceRel``{s}. s \<in> PositiveSlopes } \<subseteq> 
    Slopes//SlopeEquivalenceRel"
    using EquivClass_1_L1A by simp;
  then show "PositiveReals \<subseteq> RealNumbers"
    using PositiveReals_def RealNumbers_def by simp;
qed;
  
text{*Positive reals are the same as classes of a positive slopes.*}

lemma (in real1) Real_ZF_1_2_L2: 
  shows "a \<in> PositiveReals \<longleftrightarrow> (\<exists>f\<in>\<S>\<^isub>+. a = [f])"
proof
  assume "a \<in> PositiveReals"
  then have "a \<in> {([s]). s \<in> \<S>\<^isub>+}" using PositiveReals_def 
    by simp; (* it has to be ([a]), otherwise fails*)
  then show "\<exists>f\<in>\<S>\<^isub>+. a = [f]" by auto;
next assume "\<exists>f\<in>\<S>\<^isub>+. a = [f]"
  then have  "a \<in> {([s]). s \<in> \<S>\<^isub>+}" by auto;
  then show "a \<in> PositiveReals" using PositiveReals_def
    by simp;
qed;

text{*Let's recall from @{text "Int_ZF_2.thy"} that the sum and composition 
  of positive slopes is a positive slope.*}

lemma (in real1) Real_ZF_1_2_L3:
  assumes "f\<in>\<S>\<^isub>+"  "g\<in>\<S>\<^isub>+"
  shows 
  "f\<fp>g \<in> \<S>\<^isub>+"
  "f\<circ>g \<in> \<S>\<^isub>+"
  using prems Slopes_def PositiveSlopes_def PositiveIntegers_def
    SlopeOp1_def int1.sum_of_pos_sls_is_pos_sl
    SlopeOp2_def int1.comp_of_pos_sls_is_pos_sl
  by auto;


(*text{*A negative of a slope is a slope.*}

lemma (in real1) Real_ZF_1_2_L4: assumes "f \<in> \<S>"
  shows "(\<fm>f) \<in> \<S>"
  using prems Slopes_def BoundedIntMaps_def int1.Int_ZF_2_1_L12
  by simp;*)
  


text{*Bounded integer maps are not positive slopes.*}

lemma (in real1) Real_ZF_1_2_L5:
  assumes "f \<in> BoundedIntMaps"
  shows "f \<notin> \<S>\<^isub>+"
  using prems BoundedIntMaps_def Slopes_def PositiveSlopes_def
    PositiveIntegers_def int1.Int_ZF_2_3_L1A by simp;

 
text{*The set of positive reals is closed under addition and multiplication.
  Zero (the neutral element of addition) is not a positive number. *}

lemma (in real1) Real_ZF_1_2_L6: shows 
  "PositiveReals {is closed under} RealAddition"
  "PositiveReals {is closed under} RealMultiplication"
  "\<zero> \<notin> PositiveReals"
proof -
  { fix a fix b
    assume "a \<in> PositiveReals" and "b \<in> PositiveReals"
    then obtain f g where
      I: "f \<in> \<S>\<^isub>+"  "g \<in> \<S>\<^isub>+" and
      II: "a = [f]"  "b = [g]"
      using Real_ZF_1_2_L2 by auto;
    then have "f \<in> \<S>"  "g \<in> \<S>" using Real_ZF_1_2_L1 Slopes_def 
      by auto;
    with I II have 
      "a\<ra>b \<in> PositiveReals \<and> a\<cdot>b \<in> PositiveReals"
       using Real_ZF_1_1_L4 Real_ZF_1_2_L3 Real_ZF_1_2_L2
       by auto;
  } then show 
      "PositiveReals {is closed under} RealAddition"
      "PositiveReals {is closed under} RealMultiplication"
    using IsOpClosed_def
    by auto;
  { assume "\<zero> \<in> PositiveReals"
    then obtain f where "f \<in> \<S>\<^isub>+" and "\<zero> = [f]"
      using Real_ZF_1_2_L2 by auto;
    then have False
      using Real_ZF_1_2_L1 Slopes_def Real_ZF_1_L10 Real_ZF_1_2_L5
      by auto;
  } then show "\<zero> \<notin> PositiveReals" by auto;
qed;

text{*If a class of a slope $f$ is not zero, then either $f$ is 
  a positive slope or $-f$ is a positive slope. The real proof is in
  @{text "Int_ZF_2.thy."}*}

lemma (in real1) Real_ZF_1_2_L7: 
  assumes A1: "f \<in> \<S>" and A2: "[f] \<noteq> \<zero>"
  shows "(f \<in> \<S>\<^isub>+) Xor ((\<fm>f) \<in> \<S>\<^isub>+)"
  using prems Slopes_def SlopeEquivalenceRel_def BoundedIntMaps_def
    PositiveSlopes_def PositiveIntegers_def 
    Real_ZF_1_L10 int1.Int_ZF_2_3_L8 by simp;
  
text{*The next lemma rephrases @{text " Int_ZF_2_3_L10"} in the notation
  used in @{text "real1"} context.*}

lemma (in real1) Real_ZF_1_2_L8: 
  assumes A1: "f \<in> \<S>"  "g \<in> \<S>"
  and A2: "(f \<in> \<S>\<^isub>+) Xor (g \<in> \<S>\<^isub>+)"
  shows "([f] \<in> PositiveReals) Xor ([g] \<in> PositiveReals)"
  using prems PositiveReals_def SlopeEquivalenceRel_def Slopes_def 
    SlopeOp1_def BoundedIntMaps_def PositiveSlopes_def PositiveIntegers_def
    int1.Int_ZF_2_3_L10 by simp;

text{*The trichotomy law for the (potential) order on reals: if $a\neq 0$,
  then either $a$ is positive or $-a$ is potitive.*}

lemma (in real1) Real_ZF_1_2_L9: 
  assumes A1: "a\<in>\<real>" and A2: "a\<noteq>\<zero>"
  shows "(a \<in> PositiveReals) Xor ((\<rm>a) \<in> PositiveReals)"
proof -
  from A1 obtain f where I: "f \<in> \<S>"  "a = [f]"
    using Real_ZF_1_1_L3A by auto;
  with A2 have "([f] \<in> PositiveReals) Xor ([\<fm>f] \<in> PositiveReals)"
    using Slopes_def BoundedIntMaps_def int1.Int_ZF_2_1_L12
      Real_ZF_1_2_L7 Real_ZF_1_2_L8 by simp;
  with I show "(a \<in> PositiveReals) Xor ((\<rm>a) \<in> PositiveReals)"
    using Real_ZF_1_1_L6 by simp;
qed;

text{*Finally we are ready to prove that real numbers form an ordered ring.
  with no zero divisors.*}

theorem reals_are_ord_ring: shows
  "IsAnOrdRing(RealNumbers,RealAddition,RealMultiplication,OrderOnReals)"
  "OrderOnReals {is total on} RealNumbers"
  "PositiveSet(RealNumbers,RealAddition,OrderOnReals) = PositiveReals"
  "HasNoZeroDivs(RealNumbers,RealAddition,RealMultiplication)"
proof -
  let ?R = "RealNumbers"
  let ?A = "RealAddition"
  let ?M = "RealMultiplication"
  let ?P = "PositiveReals"
  let ?r = "OrderOnReals"
  let ?z = "TheNeutralElement(?R, ?A)"
  have I:
    "ring0(?R, ?A, ?M)" 
    "?M {is commutative on} ?R"
    "?P \<subseteq> ?R"
    "?P {is closed under} ?A"
    "TheNeutralElement(?R, ?A) \<notin> ?P"
    "\<forall>a\<in>?R. a \<noteq> ?z \<longrightarrow> (a \<in> ?P) Xor (GroupInv(?R, ?A)`(a) \<in> ?P)"
    "?P {is closed under} ?M" 
    "?r = OrderFromPosSet(?R, ?A, ?P)"
    using real0.Real_ZF_1_L3 real_mult_commutative Real_ZF_1_2_L1
      real1.Real_ZF_1_2_L6 real1.Real_ZF_1_2_L9 OrderOnReals_def
    by auto;
  then show "IsAnOrdRing(?R, ?A, ?M, ?r)" 
    by (rule ring0.ring_ord_by_positive_set)
  from I show "?r {is total on} ?R"
    by (rule ring0.ring_ord_by_positive_set)
  from I show "PositiveSet(?R,?A,?r) = ?P"
    by (rule ring0.ring_ord_by_positive_set)
  from I show "HasNoZeroDivs(?R,?A,?M)"
    by (rule ring0.ring_ord_by_positive_set);
qed;

section{*Inverting reals*}

text{*In this section we tackle the issue of existence of (multiplicative) 
  inverses of real numbers and and show that real numbers form an ordered field.
  *}

text{*We rewrite the theorem from @{text "Int_ZF_2.thy"} that shows
  that for every positive slope we can find one that is almost equal and
  has an inverse.*}

lemma (in real1) pos_slopes_have_inv: assumes "f \<in> \<S>\<^isub>+"
  shows "\<exists>g\<in>\<S>. f\<sim>g \<and> (\<exists>h\<in>\<S>. g\<circ>h \<sim> id(int))"
  using prems PositiveSlopes_def Slopes_def PositiveIntegers_def
    int1.pos_slope_has_inv SlopeOp1_def SlopeOp2_def 
    BoundedIntMaps_def SlopeEquivalenceRel_def
  by simp;

text{*The set of real numbers we are constructing is an ordered field.*}

theorem (in real1) reals_are_ord_field:
  shows "IsAnOrdField(RealNumbers,RealAddition,RealMultiplication,OrderOnReals)"
proof -
  let ?R = "RealNumbers"
  let ?A = "RealAddition"
  let ?M = "RealMultiplication"
  let ?r = "OrderOnReals"
  have "ring1(?R,?A,?M,?r)" and "\<zero> \<noteq> \<one>"
    using reals_are_ord_ring OrdRing_ZF_1_L2 real_zero_not_one
    by auto;
  moreover have
    "\<forall>a\<in>PositiveSet(?R,?A,?r). \<exists>b\<in>?R. a\<cdot>b = \<one>"
  proof
    fix a assume "a \<in> PositiveSet(?R,?A,?r)"
    then obtain f where I: "f\<in>\<S>\<^isub>+" and II: "a = [f]"
      using reals_are_ord_ring Real_ZF_1_2_L2 
      by auto;
    then have "\<exists>g\<in>\<S>. f\<sim>g \<and> (\<exists>h\<in>\<S>. g\<circ>h \<sim> id(int))"
      using pos_slopes_have_inv by simp;
    then obtain g where 
      III: "g\<in>\<S>" and IV: "f\<sim>g" and V: "\<exists>h\<in>\<S>. g\<circ>h \<sim> id(int)"
      by auto;
    from V obtain h where VII: "h\<in>\<S>" and VIII: "g\<circ>h \<sim> id(int)"
      by auto;
    from I III IV have "[f] = [g]"
      using Real_ZF_1_2_L1 Slopes_def Real_ZF_1_1_L5
      by auto;
    with II III VII VIII have "a\<cdot>[h] = \<one>"
      using Real_ZF_1_1_L4  Real_ZF_1_1_L5A real_one_cl_identity
      by simp;
    with VII show "\<exists>b\<in>?R. a\<cdot>b = \<one>" using Real_ZF_1_1_L3
      by auto;
  qed;
  ultimately show ?thesis using ring1.OrdField_ZF_1_L4
    by simp;
qed;

  
end