(*   This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{OrderedField\_ZF.thy}*}

theory OrderedField_ZF imports OrderedRing_ZF

begin

text{*This theory covers basic facts about ordered fiels.*}

section{*Definition and basic properties*}

text{*Ordered field is a notrivial ordered ring such that all 
  non-zero elements have an inverse. We define the notion of being a ordered 
  field as
  a statement about four sets. The first set, denoted @{text "K"} is the 
  carrier of the field. The second set, denoted @{text "A"} represents the 
  additive operation on @{text "K"} (recall that in ZF set theory functions 
  are sets. The third set @{text "M"} represents the multiplicative operation 
  on @{text "K"}. The fourth set @{text "r"} is the order 
  relation on @{text "K"}.*}

constdefs
  "IsAnOrdField(K,A,M,r) \<equiv> (IsAnOrdRing(K,A,M,r) \<and> 
  TheNeutralElement(K,A) \<noteq> TheNeutralElement(K,M) \<and>
  (\<forall>a\<in>K. a\<noteq>TheNeutralElement(K,A)\<longrightarrow>
  (\<exists>b\<in>K. M`\<langle>a,b\<rangle> = TheNeutralElement(K,M))))"

text{*The next context (locale) defines notation used for ordered fields. 
  We do that by extending the notation defined in the 
  @{text "ring1"} context that is used for oredered rings and 
  adding some assumptions to make sure we are 
  talking about ordered fields in this context. 
  We should rename the carrier from $R$ used in the @{text "ring1"} 
  context to $K$, more appriopriate for fields. Theoretically the Isar locale
  facility supports such renaming, but we experienced diffculties using
  some lemmas from @{text "ring1"} locale after renaming. 
  *}

locale field1 = ring1 +
  
  assumes not_triv: "\<zero> \<noteq> \<one>"

  assumes inv_exists: "\<forall>a\<in>R. a\<noteq>\<zero> \<longrightarrow> (\<exists>b\<in>R. a\<cdot>b = \<one>)"

  fixes non_zero ("R\<^isub>0")
  defines non_zero_def[simp]: "R\<^isub>0 \<equiv> R-{\<zero>}"

  fixes inv ("_\<inverse> " [90] 91)
  defines inv_def[simp]: "a\<inverse> \<equiv> GroupInv(R\<^isub>0,restrict(M,R\<^isub>0\<times>R\<^isub>0))`(a)"

text{*The next lemma assures us that we are talking fields 
  in the @{text "field1"} context.*}

lemma (in field1) OrdField_ZF_1_L1: shows "IsAnOrdField(R,A,M,r)"
  using OrdRing_ZF_1_L1 not_triv inv_exists IsAnOrdField_def
  by simp;

text{*We can use theorems proven in the @{text "field1"} context whenever we
  talk about an ordered field.*}

lemma OrdField_ZF_1_L2: assumes "IsAnOrdField(K,A,M,r)"
  shows "field1(K,A,M,r)"
  using prems IsAnOrdField_def OrdRing_ZF_1_L2 ring1_def
    IsAnOrdField_def field1_axioms_def field1_def
  by auto;

text{*In ordered rings the existence of a right inverse for all positive
  elements implies the existence of an inverse for all non zero elements.*}

lemma (in ring1) OrdField_ZF_1_L3: 
  assumes A1: "\<forall>a\<in>R\<^isub>+. \<exists>b\<in>R. a\<cdot>b = \<one>" and A2: "c\<in>R"  "c\<noteq>\<zero>"
  shows "\<exists>b\<in>R. c\<cdot>b = \<one>"
proof (cases "c\<in>R\<^isub>+");
  assume "c\<in>R\<^isub>+"
  with A1 show "\<exists>b\<in>R. c\<cdot>b = \<one>" by simp;
next assume "c\<notin>R\<^isub>+"
  with A2 have "(\<rm>c) \<in> R\<^isub>+"
    using OrdRing_ZF_3_L2A by simp;
  with A1 obtain b where "b\<in>R" and "(\<rm>c)\<cdot>b = \<one>"
    by auto;
  with A2 have "(\<rm>b) \<in> R"  "c\<cdot>(\<rm>b) = \<one>"
    using Ring_ZF_1_L3 Ring_ZF_1_L7 by auto;
  then show "\<exists>b\<in>R. c\<cdot>b = \<one>" by auto;
qed;
  
text{*Ordered fields are easier to deal with, because it is sufficient 
  to show the existence of an inverse for the set of positive elements.*}

lemma (in ring1) OrdField_ZF_1_L4: 
  assumes "\<zero> \<noteq> \<one>" and "\<forall>a\<in>R\<^isub>+. \<exists>b\<in>R. a\<cdot>b = \<one>"
  shows "IsAnOrdField(R,A,M,r)"
  using prems OrdRing_ZF_1_L1 OrdField_ZF_1_L3 IsAnOrdField_def
  by simp;

end