(*    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Order\_ZF.thy}*}

theory Order_ZF imports Fol1 

begin

text{*This theory file considers various notion related to order. We redefine
  the notions of a total order, linear order and partial order to have the 
  same terminology as wikipedia (I found it very consistent across different 
  areas of math). We also define and study the notions of intervals and bounded
  sets. We show the inclusion relations between the intervals with endpoints
  being in certain order. We also show that union of bounded sets are bounded. 
  This allows to show that finite sets are bounded in Finite\_ZF.thy.*}

section{*Definitions*}

text{*In this section we formulate the definitions related to order 
  relations.*}

text{*We define a linear order as a binary relation that is antisymmetric, 
  transitive and total. Note that this terminology is different than the
  one used the standard Order.thy file. The sets that are bounded below and 
  above are also defined, as are bounded sets. Empty sets are defined as 
  bounded.
  The notation for the definition of an interval may be mysterious for some
  readers, see @{text "Order_ZF_2_L1"} for more intuitive notation. 
  We aslo define the maximum (the greater of) two elemnts and the minmum 
  (the smaller of) two elements. 
  We say that a set has a maximum (minimum) if it has an element that is 
  not smaller (not greater, resp.) that any other one. We show that
  under some conditions this element of the set is unique (if exists).
  The element with this
  property is called the maximum (minimum) of the set.*}

constdefs 

  IsTotal (infixl "{is total on}" 65)
  "r {is total on} X \<equiv> (\<forall>a\<in>X.\<forall>b\<in>X. <a,b> \<in> r \<or> <b,a> \<in> r)"

  "IsLinOrder(X,r) \<equiv> ( antisym(r) \<and> trans(r) \<and> (r {is total on} X))"

  "IsPartOrder(X,r) \<equiv> (refl(X,r) \<and> antisym(r) \<and> trans(r))"

  "IsBoundedAbove(A,r) \<equiv> ( A=0 \<or> (\<exists>u. \<forall>x\<in>A. <x,u> \<in> r))"

  "IsBoundedBelow(A,r) \<equiv> (A=0 \<or> (\<exists>l. \<forall>x\<in>A. <l,x> \<in> r))"

  "IsBounded(A,r) \<equiv> (IsBoundedAbove(A,r) \<and> IsBoundedBelow(A,r))"

  "Interval(r,a,b) \<equiv> r``{a} \<inter> r-``{b}"

  "GreaterOf(r,a,b) \<equiv> (if <a,b> \<in> r then b else a)"

  "SmallerOf(r,a,b) \<equiv> (if <a,b> \<in> r then a else b)"

  "HasAmaximum(r,A) \<equiv> \<exists>M\<in>A.\<forall>x\<in>A. <x,M> \<in> r"

  "HasAminimum(r,A) \<equiv> \<exists>m\<in>A.\<forall>x\<in>A. <m,x> \<in> r"

  "Maximum(r,A) \<equiv> THE M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)"

  "Minimum(r,A) \<equiv> THE m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)";


text{*The essential condition to show that a total relation is reflexive.*}

lemma Order_ZF_1_L1: assumes "r {is total on} X" and "a\<in>X"
  shows "<a,a> \<in> r" using prems IsTotal_def by auto;

text{*A total relation is reflexive.*}

lemma Order_ZF_1_L1A:
  assumes "r {is total on} X"
  shows "refl(X,r)" using prems Order_ZF_1_L1 refl_def by simp;

text{*A linear order is partial order.*}

lemma Order_ZF_1_L2: assumes "IsLinOrder(X,r)" 
  shows "IsPartOrder(X,r)" 
  using prems IsLinOrder_def IsPartOrder_def refl_def Order_ZF_1_L1
  by auto;

text{*Partial order that is total is linear.*}

lemma Order_ZF_1_L3: 
  assumes "IsPartOrder(X,r)" and "r {is total on} X"
  shows "IsLinOrder(X,r)"
  using prems IsPartOrder_def IsLinOrder_def
  by simp;

text{*Relation that is total on a set is total on any subset.*}

lemma Order_ZF_1_L4: assumes "r {is total on} X" and "A\<subseteq>X"
  shows "r {is total on} A"
  using prems IsTotal_def by auto;

section{*Intervals*}

text{*In this section we discuss intervals.*}

text{*The next lemma explains the notation of the definition of an interval.*}

lemma Order_ZF_2_L1: 
  shows "x \<in> Interval(r,a,b) \<longleftrightarrow> <a,x> \<in> r \<and> <x,b> \<in> r"
  using Interval_def by auto;

text{*Since there are some problems with applying the above lemma 
  (seems that simp and auto don't handle equivalence very well), we
  split @{text "Order_ZF_2_L1"} into two lemmas.*}

lemma Order_ZF_2_L1A: assumes "x \<in> Interval(r,a,b)"
  shows "<a,x> \<in> r"  "<x,b> \<in> r"
  using prems  Order_ZF_2_L1 by auto;

text{*@{text "Order_ZF_2_L1"}, implication from right to left.*}

lemma Order_ZF_2_L1B: assumes "<a,x> \<in> r"  "<x,b> \<in> r"
  shows "x \<in> Interval(r,a,b)"
  using prems Order_ZF_2_L1 by simp;

text{*If the relation is reflexive, the endpoints belong to the interval.*}

lemma Order_ZF_2_L2: assumes "refl(X,r)" 
  and "a\<in>X"  "b\<in>X" and "<a,b> \<in> r"
  shows 
  "a \<in> Interval(r,a,b)"  
  "b \<in> Interval(r,a,b)"  
  using prems refl_def Order_ZF_2_L1 by auto;

text{*Under the assumptions of @{text "Order_ZF_2_L2"}, the interval is 
  nonempty. *}

lemma Order_ZF_2_L2A: assumes "refl(X,r)" 
  and "a\<in>X"  "b\<in>X" and "<a,b> \<in> r"
  shows "Interval(r,a,b) \<noteq> 0"
proof -
  from prems have "a \<in> Interval(r,a,b)"
    using Order_ZF_2_L2 by simp;
  then show "Interval(r,a,b) \<noteq> 0" by auto;
qed;

text{*If $a,b,c,d$ are in this order, then $[b,c]\subseteq [a,d]$. We
  only need trasitivity for this to be true.*}

lemma Order_ZF_2_L3: 
  assumes A1: "trans(r)" and A2:"<a,b>\<in>r"  "<b,c>\<in>r"  "<c,d>\<in>r"
shows "Interval(r,b,c) \<subseteq> Interval(r,a,d)"
proof;
  fix x assume A3: "x \<in> Interval(r, b, c)"
  from A1 have "trans(r)" .
  moreover from A2 A3 have "<a,b> \<in> r \<and> <b,x> \<in> r" using Order_ZF_2_L1A
    by simp;
  ultimately have T1: "<a,x> \<in> r" by (rule Fol1_L3);
  from A1 have "trans(r)" .
  moreover from A2 A3 have "<x,c> \<in> r \<and> <c,d> \<in> r" using Order_ZF_2_L1A
    by simp;
  ultimately have "<x,d> \<in> r" by (rule Fol1_L3);
  with T1 show "x \<in> Interval(r,a,d)" using Order_ZF_2_L1B
    by simp;
qed;

text{*For reflexive and antisymmetric relations the interval with equal 
  endpoints consists only of that endpoint.*}

lemma Order_ZF_2_L4: 
  assumes A1: "refl(X,r)" and A2: "antisym(r)" and A3: "a\<in>X"
  shows "Interval(r,a,a) = {a}"
proof;
  from A1 A3 have "<a,a> \<in> r" using refl_def by simp;
  with A1 A3 show "{a} \<subseteq> Interval(r,a,a)" using Order_ZF_2_L2 by simp;
  from A2 show "Interval(r,a,a) \<subseteq> {a}" using Order_ZF_2_L1A Fol1_L4
    by fast;
qed;

text{*For transitive relations the endpoints have to be in the relation for
  the interval to be nonempty.*}

lemma Order_ZF_2_L5: assumes A1: "trans(r)" and A2: "<a,b> \<notin> r"
  shows "Interval(r,a,b) = 0"
proof (rule ccontr);
  assume "Interval(r,a,b)\<noteq>0" then obtain x where "x \<in> Interval(r,a,b)"
    by auto;
  with A1 A2 show False using Order_ZF_2_L1A Fol1_L3 by fast;
qed;

text{*If a relation is defined on a set, then intervals are subsets of that
  set.*}

lemma Order_ZF_2_L6: assumes A1: "r \<subseteq> X\<times>X"
  shows "Interval(r,a,b) \<subseteq> X"
  using prems Interval_def by auto;

section{*Bounded sets*}

text{*In this section we consider properties of bounded sets.*}

text{*For reflexive relations singletons are bounded.*}

lemma Order_ZF_3_L1: assumes "refl(X,r)" and "a\<in>X"
  shows "IsBounded({a},r)"
  using prems refl_def IsBoundedAbove_def IsBoundedBelow_def
    IsBounded_def by auto;

text{*Bounded sets are contained in the domain of the relation.*}

lemma Order_ZF_3_L1A: assumes "r \<subseteq> X\<times>X" 
  and "IsBoundedAbove(A,r)"
  shows "A\<subseteq>X" using prems IsBoundedAbove_def by auto;

text{*For a total relation, the greater of two elements, 
  as defined above, is indeed greater of any of the two.*}

lemma Order_ZF_3_L2: assumes "r {is total on} X"
  and "x\<in>X" "y\<in>X"
  shows "<x,GreaterOf(r,x,y)> \<in> r" "<y,GreaterOf(r,x,y)> \<in> r"
  "<SmallerOf(r,x,y),x> \<in> r" "<SmallerOf(r,x,y),y> \<in> r"
  using prems IsTotal_def Order_ZF_1_L1 GreaterOf_def  SmallerOf_def
  by auto;

text{*If $A$ is bounded above by $u$, $B$ is bounded above by $w$,
  then $A\cup B$ is bounded above by the greater of $u,w$.*}

lemma Order_ZF_3_L2B:  
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "u\<in>X" "w\<in>X" 
  and A4: "\<forall>x\<in>A. <x,u> \<in> r" "\<forall>x\<in>B. <x,w> \<in> r"
  shows "\<forall>x\<in>A\<union>B. \<langle>x,GreaterOf(r,u,w)\<rangle> \<in> r"
proof;
  let ?v = "GreaterOf(r,u,w)"
  from A1 A3 have T1: "<u,?v> \<in> r" and T2: "<w,?v> \<in> r"
    using Order_ZF_3_L2 by auto;
  fix x assume A5: "x\<in>A\<union>B" show "\<langle>x,?v\<rangle> \<in> r"
  proof (cases "x\<in>A");
    assume "x\<in>A"
    with A4 T1 have "<x,u> \<in> r \<and> <u,?v> \<in> r" by simp
    with A2 show "\<langle>x,?v\<rangle> \<in> r" by (rule Fol1_L3);
  next assume "x\<notin>A" 
    with A5 A4 T2 have "<x,w> \<in> r \<and> <w,?v> \<in> r" by simp;
    with A2 show "\<langle>x,?v\<rangle> \<in> r" by (rule Fol1_L3);
  qed;
qed;

text{*For total and transitive relation the union of two sets bounded 
  above is bounded above.*}

lemma Order_ZF_3_L3: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedAbove(A,r)" "IsBoundedAbove(B,r)"
  and A4: "r \<subseteq> X\<times>X"
  shows "IsBoundedAbove(A\<union>B,r)"
proof (cases "A=0 \<or> B=0");
  assume "A=0 \<or> B=0" 
  with A3 show ?thesis by auto;
next assume "\<not> (A = 0 \<or> B = 0)"
  then have T1: "A\<noteq>0" "B\<noteq>0" by auto;
  with A3 obtain u w where D1: "\<forall>x\<in>A. <x,u> \<in> r" "\<forall>x\<in>B. <x,w> \<in> r"
    using IsBoundedAbove_def by auto;
  let ?U = "GreaterOf(r,u,w)"
  from T1 A4 D1 have "u\<in>X" "w\<in>X" by auto;
  with A1 A2 D1 have "\<forall>x\<in>A\<union>B.<x,?U> \<in> r"
    using Order_ZF_3_L2B by blast;
  then show "IsBoundedAbove(A\<union>B,r)"
    using IsBoundedAbove_def by auto;
qed;
  
text{*For total and transitive relations if a set $A$ is bounded above then 
  $A\cup \{a\}$ is bounded above.*}

lemma Order_ZF_3_L4: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedAbove(A,r)" and A4: "a\<in>X" and A5: "r \<subseteq> X\<times>X"
  shows "IsBoundedAbove(A\<union>{a},r)"
proof -
  from A1 have "refl(X,r)"
    using Order_ZF_1_L1A by simp;
  with prems show ?thesis using
    Order_ZF_3_L1 IsBounded_def Order_ZF_3_L3 by simp;
qed;

text{*If $A$ is bounded below by $l$, $B$ is bounded below by $m$,
  then $A\cup B$ is bounded below by the smaller of $u,w$.*}

lemma Order_ZF_3_L5B:  
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "l\<in>X" "m\<in>X" 
  and A4: "\<forall>x\<in>A. <l,x> \<in> r" "\<forall>x\<in>B. <m,x> \<in> r"
  shows "\<forall>x\<in>A\<union>B. \<langle>SmallerOf(r,l,m),x\<rangle> \<in> r"
proof;
  let ?k = "SmallerOf(r,l,m)"
  from A1 A3 have T1: "<?k,l> \<in> r" and T2: "<?k,m> \<in> r"
    using Order_ZF_3_L2 by auto;
  fix x assume A5: "x\<in>A\<union>B" show "\<langle>?k,x\<rangle> \<in> r"
  proof (cases "x\<in>A");
    assume "x\<in>A"
    with A4 T1 have "<?k,l> \<in> r \<and> <l,x> \<in> r" by simp
    with A2 show "\<langle>?k,x\<rangle> \<in> r" by (rule Fol1_L3);
  next assume "x\<notin>A" 
    with A5 A4 T2 have "<?k,m> \<in> r \<and> <m,x> \<in> r" by simp;
    with A2 show "\<langle>?k,x\<rangle> \<in> r" by (rule Fol1_L3);
  qed;
qed;

text{*For total and transitive relation the union of two sets bounded 
  below is bounded below.*}

lemma Order_ZF_3_L6: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedBelow(A,r)" "IsBoundedBelow(B,r)"
  and A4: "r \<subseteq> X\<times>X"
  shows "IsBoundedBelow(A\<union>B,r)"
proof (cases "A=0 \<or> B=0");
  assume "A=0 \<or> B=0" 
  with A3 show ?thesis by auto;
next assume "\<not> (A = 0 \<or> B = 0)"
  then have T1: "A\<noteq>0" "B\<noteq>0" by auto;
  with A3 obtain l m where D1: "\<forall>x\<in>A. <l,x> \<in> r" "\<forall>x\<in>B. <m,x> \<in> r"
    using IsBoundedBelow_def by auto;
  let ?L = "SmallerOf(r,l,m)"
  from T1 A4 D1 have T1: "l\<in>X" "m\<in>X" by auto;
  with A1 A2 D1 have "\<forall>x\<in>A\<union>B.<?L,x> \<in> r"
    using Order_ZF_3_L5B by blast;
  then show "IsBoundedBelow(A\<union>B,r)"
    using IsBoundedBelow_def by auto;
qed;

text{*For total and transitive relations if a set $A$ is bounded below then 
  $A\cup \{a\}$ is bounded below.*}

lemma Order_ZF_3_L7:
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "IsBoundedBelow(A,r)" and A4: "a\<in>X" and A5: "r \<subseteq> X\<times>X"
  shows "IsBoundedBelow(A\<union>{a},r)"
proof -
  from A1 have "refl(X,r)"
    using Order_ZF_1_L1A by simp;
  with prems show ?thesis using
    Order_ZF_3_L1 IsBounded_def Order_ZF_3_L6 by simp;
qed;

text{*For total and transitive relations unions of two bounded sets are 
  bounded.*}

theorem Order_ZF_3_T1: 
  assumes "r {is total on} X" and "trans(r)" 
  and "IsBounded(A,r)" "IsBounded(B,r)"
  and "r \<subseteq> X\<times>X"
  shows "IsBounded(A\<union>B,r)"
  using prems Order_ZF_3_L3 Order_ZF_3_L6 Order_ZF_3_L7 IsBounded_def
  by simp;

text{*For total and transitive relations if a set $A$ is bounded then 
  $A\cup \{a\}$ is bounded.*}

lemma Order_ZF_3_L8: 
  assumes "r {is total on} X" and "trans(r)"
  and "IsBounded(A,r)" and "a\<in>X" and "r \<subseteq> X\<times>X"
  shows "IsBounded(A\<union>{a},r)"
  using prems Order_ZF_1_L1A Order_ZF_3_L1 Order_ZF_3_T1 by blast;

text{*A sufficient condition for a set to be bounded below.*}

lemma Order_ZF_3_L9: assumes A1: "\<forall>a\<in>A. <l,a> \<in> r"
  shows "IsBoundedBelow(A,r)"
proof -
  from A1 have "\<exists>l. \<forall>x\<in>A. <l,x> \<in> r"
    by auto;
  then show "IsBoundedBelow(A,r)"
    using IsBoundedBelow_def by simp;
qed;

text{*A sufficient condition for a set to be bounded above.*}

lemma Order_ZF_3_L10: assumes A1: "\<forall>a\<in>A. <a,u> \<in> r"
  shows "IsBoundedAbove(A,r)"
proof -
  from A1 have "\<exists>u. \<forall>x\<in>A. <x,u> \<in> r"
    by auto;
  then show "IsBoundedAbove(A,r)"
    using IsBoundedAbove_def by simp;
qed;

text{*Intervals are bounded. *}
(*proof that uses Order_ZF_3_L9 and Order_ZF_3_L10 and is not shorter *)
lemma Order_ZF_3_L11: shows 
  "IsBoundedAbove(Interval(r,a,b),r)"
  "IsBoundedBelow(Interval(r,a,b),r)"
  "IsBounded(Interval(r,a,b),r)"
proof -
  { fix x assume "x \<in> Interval(r,a,b)" 
    then have "<x,b> \<in> r"  "<a,x> \<in> r" 
      using Order_ZF_2_L1A by auto
  } then have 
      "\<exists>u. \<forall>x\<in>Interval(r,a,b). <x,u> \<in> r" 
      "\<exists>l. \<forall>x\<in>Interval(r,a,b). <l,x> \<in> r"
    by auto;
  then show 
    "IsBoundedAbove(Interval(r,a,b),r)"
    "IsBoundedBelow(Interval(r,a,b),r)"
    "IsBounded(Interval(r,a,b),r)"
    using IsBoundedAbove_def IsBoundedBelow_def IsBounded_def
    by auto;
qed;

text{*A subset of a set that is bounded below is bounded below.*}

lemma Order_ZF_3_L12: assumes "IsBoundedBelow(A,r)" and "B\<subseteq>A"
  shows "IsBoundedBelow(B,r)"
  using prems IsBoundedBelow_def by auto;

text{*A subset of a set that is bounded above is bounded above.*}

lemma Order_ZF_3_L13: assumes "IsBoundedAbove(A,r)" and "B\<subseteq>A"
  shows "IsBoundedAbove(B,r)"
  using prems IsBoundedAbove_def by auto;

text{*If for every element of $X$ we can find one in $A$ 
  that is greater, then the $A$ can not be bounded above.
  Works for relations that are total, transitive and antisymmetric.*}

lemma Order_ZF_3_L14:  
  assumes A1: "r {is total on} X" 
  and A2: "trans(r)" and A3: "antisym(r)"
  and A4: "r \<subseteq> X\<times>X" and A5: "X\<noteq>0" 
  and A6: "\<forall>x\<in>X. \<exists>a\<in>A. x\<noteq>a \<and> \<langle>x,a\<rangle> \<in> r"
  shows "\<not>IsBoundedAbove(A,r)";
proof -
  { from A5 A6 have I: "A\<noteq>0" by auto
    moreover assume "IsBoundedAbove(A,r)"
    ultimately obtain u where II: "\<forall>x\<in>A. <x,u> \<in> r"
      using IsBounded_def IsBoundedAbove_def by auto;
    with A4 I have "u\<in>X" by auto;
    with A6 obtain b where "b\<in>A" and III: "u\<noteq>b" and "\<langle>u,b\<rangle> \<in> r"
      by auto;
    with II have "\<langle>b,u\<rangle> \<in> r"  "\<langle>u,b\<rangle> \<in> r" by auto;
    with A3 have "b=u" by (rule Fol1_L4);
    with III have False by simp;
  } thus "\<not>IsBoundedAbove(A,r)" by auto;
qed;
    
section{*Maximum and minimum of a set*}

text{*In this section we show that maximum and minimum are unique if they 
  exist. We also show that union of sets that have maxima (minima) has a 
  maximum (minimum). We also show that singletons have maximum and minimum.
  All this allows to show (in Finite\_ZF.thy) that every finite set has 
  well-defined maximum and minimum. *}

text{*For antisymmetric relations maximum of a set is unique if it exists.*}

lemma Order_ZF_4_L1: assumes A1: "antisym(r)" and A2: "HasAmaximum(r,A)"
  shows "\<exists>!M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)"
proof;
  from A2 show "\<exists>M. M \<in> A \<and> (\<forall>x\<in>A. \<langle>x, M\<rangle> \<in> r)"
    using HasAmaximum_def by auto;
  fix M1 M2 assume 
    A2: "M1 \<in> A \<and> (\<forall>x\<in>A. \<langle>x, M1\<rangle> \<in> r)" "M2 \<in> A \<and> (\<forall>x\<in>A. \<langle>x, M2\<rangle> \<in> r)"
    then have "\<langle>M1,M2\<rangle> \<in> r" "\<langle>M2,M1\<rangle> \<in> r" by auto;
    with A1 show "M1=M2" by (rule Fol1_L4);
qed;

text{*For antisymmetric relations minimum of a set is unique if it exists.*}

lemma Order_ZF_4_L2: assumes A1: "antisym(r)" and A2: "HasAminimum(r,A)"
  shows "\<exists>!m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)"
proof;
  from A2 show "\<exists>m. m \<in> A \<and> (\<forall>x\<in>A. \<langle>m, x\<rangle> \<in> r)"
    using HasAminimum_def by auto;
  fix m1 m2 assume 
    A2: "m1 \<in> A \<and> (\<forall>x\<in>A. \<langle>m1, x\<rangle> \<in> r)" "m2 \<in> A \<and> (\<forall>x\<in>A. \<langle>m2, x\<rangle> \<in> r)"
    then have "\<langle>m1,m2\<rangle> \<in> r" "\<langle>m2,m1\<rangle> \<in> r" by auto;
    with A1 show "m1=m2" by (rule Fol1_L4);
qed;

text{*Maximum of a set has desired properties. *}

lemma Order_ZF_4_L3: assumes A1: "antisym(r)" and A2: "HasAmaximum(r,A)"
  shows "Maximum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>x,Maximum(r,A)\<rangle> \<in> r"
proof -; 
  let ?Max = "THE M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)" 
  from A1 A2 have "\<exists>!M. M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)"
    by (rule Order_ZF_4_L1);
  then have "?Max \<in> A \<and> (\<forall>x\<in>A. <x,?Max> \<in> r)"
    by (rule theI);
  then show "Maximum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>x,Maximum(r,A)\<rangle> \<in> r"
    using Maximum_def by auto;
qed;
  
text{*Minimum of a set has desired properties.*}
    
lemma Order_ZF_4_L4: assumes A1: "antisym(r)" and A2: "HasAminimum(r,A)"
  shows "Minimum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>Minimum(r,A),x\<rangle> \<in> r"
proof -; 
  let ?Min = "THE m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)" 
  from A1 A2 have "\<exists>!m. m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)"
    by (rule Order_ZF_4_L2);
  then have "?Min \<in> A \<and> (\<forall>x\<in>A. <?Min,x> \<in> r)"
    by (rule theI);
  then show "Minimum(r,A) \<in> A" "\<forall>x\<in>A. \<langle>Minimum(r,A),x\<rangle> \<in> r"
    using Minimum_def by auto;
qed;

text{*For total and transitive relations a union a of two sets that have 
  maxima has a maximum.*}

lemma Order_ZF_4_L5: 
  assumes A1: "r {is total on} (A\<union>B)" and A2: "trans(r)"
  and A3: "HasAmaximum(r,A)" "HasAmaximum(r,B)"
  shows "HasAmaximum(r,A\<union>B)"
proof -;
  from A3 obtain M K where 
    D1: "M\<in>A \<and> (\<forall>x\<in>A. <x,M> \<in> r)" "K\<in>B \<and> (\<forall>x\<in>B. <x,K> \<in> r)" 
    using HasAmaximum_def by auto
  let ?L = "GreaterOf(r,M,K)"
  from D1 have T1: "M \<in> A\<union>B" "K \<in> A\<union>B" 
    "\<forall>x\<in>A. <x,M> \<in> r" "\<forall>x\<in>B. <x,K> \<in> r"
    by auto;
  with A1 A2 have "\<forall>x\<in>A\<union>B.<x,?L> \<in> r" by (rule Order_ZF_3_L2B);
  moreover from T1 have "?L \<in> A\<union>B" using GreaterOf_def IsTotal_def 
    by simp;
  ultimately show "HasAmaximum(r,A\<union>B)" using HasAmaximum_def by auto;
qed;

text{*For total and transitive relations A union a of two sets that have 
  minima has a minimum.*}

lemma Order_ZF_4_L6: 
  assumes A1: "r {is total on} (A\<union>B)" and A2: "trans(r)"
  and A3: "HasAminimum(r,A)" "HasAminimum(r,B)"
  shows "HasAminimum(r,A\<union>B)"
proof -;
  from A3 obtain m k where 
    D1: "m\<in>A \<and> (\<forall>x\<in>A. <m,x> \<in> r)" "k\<in>B \<and> (\<forall>x\<in>B. <k,x> \<in> r)" 
    using HasAminimum_def by auto
  let ?l = "SmallerOf(r,m,k)"
  from D1 have T1: "m \<in> A\<union>B" "k \<in> A\<union>B" 
    "\<forall>x\<in>A. <m,x> \<in> r" "\<forall>x\<in>B. <k,x> \<in> r"
    by auto;
  with A1 A2 have "\<forall>x\<in>A\<union>B.<?l,x> \<in> r" by (rule Order_ZF_3_L5B);
  moreover from T1 have "?l \<in> A\<union>B" using SmallerOf_def IsTotal_def 
    by simp;
  ultimately show "HasAminimum(r,A\<union>B)" using HasAminimum_def by auto;
qed;

text{*Set that has a maximum is bounded above.*}

lemma Order_ZF_4_L7:
  assumes "HasAmaximum(r,A)"
  shows "IsBoundedAbove(A,r)"
  using prems HasAmaximum_def IsBoundedAbove_def by auto;

text{*Set that has a minimum is bounded below.*}

lemma Order_ZF_4_L8A:
  assumes "HasAminimum(r,A)"
  shows "IsBoundedBelow(A,r)"
  using prems HasAminimum_def IsBoundedBelow_def by auto;


text{*For reflexive relations singletons have a minimum and maximum.*}

lemma Order_ZF_4_L8: assumes "refl(X,r)" and "a\<in>X"
  shows "HasAmaximum(r,{a})" "HasAminimum(r,{a})"
  using prems refl_def HasAmaximum_def HasAminimum_def by auto;

text{*For total and transitive relations if we add an element to a set 
  that has a maximum, the set still has a maximum.*}

lemma Order_ZF_4_L9: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "A\<subseteq>X" and A4: "a\<in>X" and A5: "HasAmaximum(r,A)"
  shows "HasAmaximum(r,A\<union>{a})"
proof -;
  from A3 A4 have "A\<union>{a} \<subseteq> X" by auto;
  with A1 have "r {is total on} (A\<union>{a})"
    using Order_ZF_1_L4 by blast;
  moreover from A1 A2 A4 A5 have
    "trans(r)" "HasAmaximum(r,A)" by auto;
  moreover from A1 A4 have "HasAmaximum(r,{a})"
    using Order_ZF_1_L1A Order_ZF_4_L8 by blast;
  ultimately show "HasAmaximum(r,A\<union>{a})" by (rule Order_ZF_4_L5);
qed;

text{*For total and transitive relations if we add an element to a set 
  that has a minimum, the set still has a minimum.*}

lemma Order_ZF_4_L10: 
  assumes A1: "r {is total on} X" and A2: "trans(r)"
  and A3: "A\<subseteq>X" and A4: "a\<in>X" and A5: "HasAminimum(r,A)"
  shows "HasAminimum(r,A\<union>{a})"
proof -;
  from A3 A4 have "A\<union>{a} \<subseteq> X" by auto;
  with A1 have "r {is total on} (A\<union>{a})"
    using Order_ZF_1_L4 by blast;
  moreover from A1 A2 A4 A5 have
    "trans(r)" "HasAminimum(r,A)" by auto;
  moreover from A1 A4 have "HasAminimum(r,{a})"
    using Order_ZF_1_L1A Order_ZF_4_L8 by blast;
  ultimately show "HasAminimum(r,A\<union>{a})" by (rule Order_ZF_4_L6);
qed;
 
end
 