(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Group\_ZF.thy}*}

theory Group_ZF imports func_ZF 

begin;

text{*This theory file will cover basics of group theory.*}

section{*Monoids.*}

text{*Monoid is a set with an associative operation and a neutral element.
  The operation is of course a function on $G\times G$ with values in $G$, 
  and therefore it is a subset of $(G\times G)\times G$. 
  Those who don't like that can go to HOL. 
  Monoid is like a group except that we don't require existence of 
  the inverse.*}

constdefs
  "IsAmonoid(G,f) \<equiv>
  f {is associative on} G \<and> 
  (\<exists>e\<in>G. (\<forall> g\<in>G. ( (f`(<e,g>) = g) \<and> (f`(<g,e>) = g))))";

text{*We use locales to define notation. This allows to separate notation 
and notion definitions. We would like to use additive notation for 
monoid, but unfortunately + is already taken.*}

locale monoid0 =
  fixes G and f
  assumes monoidAsssum:"IsAmonoid(G,f)"
  
  fixes monoper (infixl "\<oplus>" 70)
  defines monoper_def [simp]: "a \<oplus> b \<equiv> f`<a,b>";

text{*The result of the monoid operation is in the monoid (carrier).*}

lemma (in monoid0) group0_1_L1: 
  assumes "a\<in>G" "b\<in>G" shows "a\<oplus>b \<in> G"; 
  using prems monoidAsssum IsAmonoid_def IsAssociative_def apply_funtype
  by auto;

text{*There is only one neutral element in monoid.*}

lemma (in monoid0) group0_1_L2: 
  "\<exists>!e. e\<in>G \<and> (\<forall> g\<in>G. ( (e\<oplus>g = g) \<and> g\<oplus>e = g))"
proof
  fix e y
  assume "e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)"
    and "y \<in> G \<and> (\<forall>g\<in>G. y \<oplus> g = g \<and> g \<oplus> y = g)"
  then have "y\<oplus>e = y" "y\<oplus>e = e" by auto;
  thus "e = y" by simp;
next from monoidAsssum show 
    "\<exists>e. e\<in> G \<and> (\<forall> g\<in>G. e\<oplus>g = g \<and> g\<oplus>e = g)"
    using IsAmonoid_def by auto;
qed;

text{*We could put the definition of neutral element anywhere, 
  but it is only usable in conjuction 
  with the above lemma.*}

constdefs 
 "TheIdentityElement(G,f) \<equiv> 
  ( THE e. e\<in>G \<and> (\<forall> g\<in>G. f`<e,g> = g \<and> f`<g,e> = g))";

text{*The neutral element is neutral.*}

lemma (in monoid0) group0_1_L3:
  assumes A1: "e = TheIdentityElement(G,f)"
  shows "e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)"
proof -;
  let ?n =  "THE b. b\<in> G \<and> (\<forall> g\<in>G. b\<oplus>g = g \<and> g\<oplus>b = g)";
  have "\<exists>!b. b\<in> G \<and> (\<forall> g\<in>G. b\<oplus>g = g \<and> g\<oplus>b = g)"
    using group0_1_L2 by simp;
  hence "?n\<in> G \<and> (\<forall> g\<in>G. ?n\<oplus>g = g \<and> g\<oplus>?n = g)"
    by (rule theI);
  with A1 show ?thesis 
    using TheIdentityElement_def by simp;
qed;

text{*The monoid carrier is not empty.*}

lemma (in monoid0) group0_1_L3A: "G\<noteq>0"
proof -;
  have "TheIdentityElement(G,f) \<in> G" using group0_1_L3
    by simp;
  thus ?thesis by auto;
qed;

text{* The range of the monoid operation is the whole monoid carrier.*}

lemma (in monoid0) group0_1_L3B: "range(f) = G"
proof;
  from monoidAsssum have T1:"f : G\<times>G\<rightarrow>G"
     using IsAmonoid_def IsAssociative_def by simp;
  then show "range(f) \<subseteq> G" 
    using func1_1_L5B by simp;
  show "G \<subseteq> range(f)"
  proof;
    fix g assume A1:"g\<in>G"
    let ?e = "TheIdentityElement(G,f)"
    from A1 have "<?e,g> \<in> G\<times>G" "g = f`<?e,g>"
      using group0_1_L3 by auto;
    with T1 show "g \<in> range(f)"
      using func1_1_L5A by blast;
  qed;
qed;
      
text{*In a monoid a neutral element is the neutral element.*}

lemma (in monoid0) group0_1_L4: 
  assumes A1: "e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)"
  shows "e = TheIdentityElement(G,f)"
proof -
  let ?n =  "THE b. b\<in> G \<and> (\<forall> g\<in>G. b\<oplus>g = g \<and> g\<oplus>b = g)";
  have "\<exists>!b. b\<in> G \<and> (\<forall> g\<in>G. b\<oplus>g = g \<and> g\<oplus>b = g)"
    using group0_1_L2 by simp;
  moreover from A1 have 
    "e \<in> G \<and> (\<forall>g\<in>G. e \<oplus> g = g \<and> g \<oplus> e = g)" .;
  ultimately have "(?n) = e" by (rule the_equality2);
  then show ?thesis using TheIdentityElement_def by simp;
qed;

text{*The next lemma shows that if the if we restrict the monoid operation to
  a subset of $G$ that contains the neutral element, then the 
  neutral element of the monoid operation is also neutral with the 
  restricted operation. This is proven separately because it is
  used more than once.
*}

lemma (in monoid0) group0_1_L5:
  assumes A1: "\<forall>x\<in>H.\<forall>y\<in>H. x\<oplus>y \<in> H"
  and A2: "H\<subseteq>G"
  and A3: "e = TheIdentityElement(G,f)"
  and A4: "g = restrict(f,H\<times>H)"
  and A5: "e\<in>H"
  and A6: "h\<in>H"
  shows "g`<e,h> = h \<and> g`<h,e> = h";
proof -;
  from A4 A6 A5 have 
    "g`<e,h> = e\<oplus>h \<and> g`<h,e> = h\<oplus>e"
    using restrict_if by simp;
  with A3 A4 A6 A2 show 
    "g`<e,h> = h \<and> g`<h,e> = h"
    using  group0_1_L3 by auto;
qed;

text{*The next theorem shows that if the monoid operation is closed
  on a subset of $G$ then this set is a (sub)monoid.
  (although we do not define this notion). This will be 
  useful when we study subgroups. *}

theorem (in monoid0) group0_1_T1: 
  assumes A1: "H {is closed under} f"
  and A2: "H\<subseteq>G"
  and A3: "TheIdentityElement(G,f) \<in> H"
  shows  "IsAmonoid(H,restrict(f,H\<times>H))"
proof -;
  let ?g = "restrict(f,H\<times>H)"
  let ?e = "TheIdentityElement(G,f)"
  from monoidAsssum have "f \<in> G\<times>G\<rightarrow>G" 
    using IsAmonoid_def IsAssociative_def by simp;
  moreover from A2 have "H\<times>H \<subseteq> G\<times>G" by auto;
  moreover from A1 have "\<forall>p \<in> H\<times>H. f`(p) \<in> H"
    using IsOpClosed_def by auto;
  ultimately have "?g \<in> H\<times>H\<rightarrow>H";
    using func1_2_L4 by simp;
  moreover have "\<forall>x\<in>H.\<forall>y\<in>H.\<forall>z\<in>H. 
    ?g`\<langle>?g`<x,y>,z\<rangle> = ?g`\<langle>x,?g`<y,z>\<rangle>"
  proof -
    from A1 have "\<forall>x\<in>H.\<forall>y\<in>H.\<forall>z\<in>H.
      ?g`\<langle>?g`<x,y>,z\<rangle> = x\<oplus>y\<oplus>z"
      using IsOpClosed_def restrict_if by simp;
    moreover have "\<forall>x\<in>H.\<forall>y\<in>H.\<forall>z\<in>H. x\<oplus>y\<oplus>z = x\<oplus>(y\<oplus>z)"
    proof -;
      from monoidAsssum have 
	"\<forall>x\<in>G.\<forall>y\<in>G.\<forall>z\<in>G. x\<oplus>y\<oplus>z = x\<oplus>(y\<oplus>z)"
	using IsAmonoid_def IsAssociative_def 
	by simp;
      with A2 show ?thesis by auto;
    qed;
    moreover from A1 have 
      "\<forall>x\<in>H.\<forall>y\<in>H.\<forall>z\<in>H. x\<oplus>(y\<oplus>z) = ?g`\<langle> x,?g`<y,z>\<rangle>"
      using IsOpClosed_def restrict_if by simp;
    ultimately show ?thesis by simp; 
  qed;
  moreover have 
    "\<exists>n\<in>H. (\<forall>h\<in>H. ?g`<n,h> = h \<and> ?g`<h,n> = h)"
  proof -;
    from A1 have "\<forall>x\<in>H.\<forall>y\<in>H. x\<oplus>y \<in> H"
      using IsOpClosed_def by simp;
    with A2 A3 have 
      "\<forall> h\<in>H. ?g`<?e,h> = h \<and> ?g`<h,?e> = h"
      using group0_1_L5 by blast;
    with A3 show ?thesis by auto;
  qed;
  ultimately show ?thesis using IsAmonoid_def IsAssociative_def 
    by simp;
qed;
    
text{*Under the assumptions of @{text " group0_1_T1"} the neutral element of a 
  submonoid is the same as that of the monoid.*}

lemma group0_1_L6: 
  assumes A1: "IsAmonoid(G,f)"
  and A2: "H {is closed under} f"
  and A3: "H\<subseteq>G"
  and A4: "TheIdentityElement(G,f) \<in> H"
  shows "TheIdentityElement(H,restrict(f,H\<times>H)) = TheIdentityElement(G,f)"
proof -
  def D1: e \<equiv> "TheIdentityElement(G,f)"
  def D2: g \<equiv> "restrict(f,H\<times>H)"
  with A1 A2 A3 A4 have "monoid0(H,g)";
    using monoid0_def monoid0.group0_1_T1 
    by simp;
  moreover have 
    "e \<in> H \<and> (\<forall>h\<in>H. g`<e,h> = h \<and> g`<h,e> = h)"
  proof -;
    from A1 A2 have "monoid0(G,f)" "\<forall>x\<in>H.\<forall>y\<in>H. f`<x,y> \<in> H"
      using monoid0_def IsOpClosed_def by auto;
    with A3 D1 D2 A4 show ?thesis
      using monoid0.group0_1_L5 by blast;
  qed;
  ultimately have "e =  TheIdentityElement(H,g)"
    using monoid0.group0_1_L4 by auto;
  with D1 D2 show ?thesis by simp;
qed;
 
section{*Basic definitions and results for groups*}

text{*To define a group we take a monoid and add a requirement 
  that the right inverse needs to exist for every element of the group.
  We also define the group inverse as a relation on the group carrier.
  Later we will show that this relation is a function. 
  The @{text "GroupInv"} below is really the right inverse, understood as a function, 
  that is a subset of $G\times G$.*}

constdefs
  "IsAgroup(G,f) \<equiv> 
  (IsAmonoid(G,f) \<and> (\<forall>g\<in>G. \<exists>b\<in>G. f`<g,b> = TheIdentityElement(G,f)))"

  "GroupInv(G,f) \<equiv> {<x,y> \<in> G\<times>G. f`<x,y> = TheIdentityElement(G,f)}"

text{*We will use the miltiplicative notation for groups.*}

locale group0 =
  fixes G and f
  assumes groupAsssum: "IsAgroup(G,f)"

  fixes e ("\<one>")
  defines e_def[simp]: "\<one> \<equiv> TheIdentityElement(G,f)"
  
  fixes groper (infixl "\<cdot>" 70)
  defines groper_def [simp]: "a \<cdot> b \<equiv> f`<a,b>"
  
  fixes inv ("_\<inverse> " [90] 91)
  defines inv_def[simp]: "x\<inverse> \<equiv> GroupInv(G,f)`(x)";

text{*First we show a lemma that says that we can use theorems proven in
  the @{text "monoid0"} context (locale).*}

lemma (in group0) group0_2_L1: "monoid0(G,f)"
  using groupAsssum IsAgroup_def monoid0_def by simp;

text{*Technical lemma that allows to use $e$ as the neutral element of 
the group without referencing a list of lemmas and definitions.*}

lemma (in group0) group0_2_L2: 
  shows "\<one>\<in>G \<and> (\<forall>g\<in>G.(\<one>\<cdot>g = g \<and> g\<cdot>\<one> = g))"
  using group0_2_L1 monoid0.group0_1_L3 by simp;

text{*The group is closed under the group operation. Used all the time,
  useful to have handy.*}

lemma (in group0) group_op_closed: assumes "a\<in>G" "b\<in>G"
  shows "a\<cdot>b \<in> G" using prems group0_2_L1 monoid0.group0_1_L1 
  by simp;

text{*The group operation is associative. This is another technical lemma 
  that allows to shorten the list of referenced lemmas in some proofs.*}

lemma (in group0) group_oper_assoc: 
  assumes "a\<in>G" "b\<in>G" "c\<in>G" shows "a\<cdot>(b\<cdot>c) = a\<cdot>b\<cdot>c"
  using groupAsssum prems IsAgroup_def IsAmonoid_def 
    IsAssociative_def group_op_closed by simp;

text{*The group operation maps $G\times G$ into $G$. It is conveniet to have
  this fact easily accessible in the group0 context.*}

lemma (in group0) group_oper_assocA: shows "f : G\<times>G\<rightarrow>G"
  using groupAsssum IsAgroup_def IsAmonoid_def IsAssociative_def
  by simp;
  
text{*The definition of group requires the existence of the right inverse.
  We show that this is also the left inverse.*}

theorem (in group0) group0_2_T1: 
  assumes A1: "g\<in>G" and A2:"b\<in>G" and A3: "g\<cdot>b = \<one>"
  shows "b\<cdot>g = \<one>";
proof -
  from A2 groupAsssum obtain c where D1:"c \<in> G \<and> b\<cdot>c = \<one>" 
    using IsAgroup_def by auto;
  then have T1:"c\<in>G" by simp;
  have T2:"\<one>\<in>G" using group0_2_L2 by simp;
  from A1 A2 T2 D1 have "b\<cdot>g =  b\<cdot>(g\<cdot>(b\<cdot>c))"
    using group_op_closed group0_2_L2 group_oper_assoc 
    by simp;
  also from  A1 A2 T1 have "b\<cdot>(g\<cdot>(b\<cdot>c)) = b\<cdot>(g\<cdot>b\<cdot>c)"
    using group_oper_assoc by simp;
  also from A3 A2 D1 have "b\<cdot>(g\<cdot>b\<cdot>c)= \<one>" using group0_2_L2 by simp;
  finally show "b\<cdot>g = \<one>" by simp;
qed;

text{*For every element of a group there is only one inverse.*}

lemma (in group0) group0_2_L4: 
  assumes A1:"x\<in>G" shows "\<exists>!y. y\<in>G \<and> x\<cdot>y = \<one>"
proof;
  from A1 groupAsssum show "\<exists>y. y\<in>G \<and>  x\<cdot>y = \<one>" 
    using IsAgroup_def by auto;
  fix y n
  assume A2:"y\<in>G \<and>  x\<cdot>y = \<one>" and A3:"n\<in>G \<and> x\<cdot>n = \<one>" show "y=n"
  proof -
    from A1 A2 have T1:"y\<cdot>x = \<one>"
      using group0_2_T1 by simp;
    from A2 A3 have "y = y\<cdot>(x\<cdot>n)" 
      using group0_2_L2 by simp;
    also from A1 A2 A3 have "\<dots> = (y\<cdot>x)\<cdot>n" 
      using group_oper_assoc by blast;
    also from T1 A3 have "\<dots> = n" 
      using group0_2_L2 by simp;
    finally show "y=n" by simp;
  qed;
qed;

text{*The group inverse is a function that maps G into G.*}

theorem group0_2_T2: 
  assumes A1:"IsAgroup(G,f)" shows "GroupInv(G,f) : G\<rightarrow>G"
proof -;
  have "GroupInv(G,f) \<subseteq> G\<times>G" using GroupInv_def by auto;
  moreover from A1 have
    "\<forall>x\<in>G. \<exists>!y. y\<in> G \<and> <x,y> \<in> GroupInv(G,f)"
    using group0_def group0.group0_2_L4 GroupInv_def by simp;
  ultimately show ?thesis using func1_1_L11 by simp;
qed;

text{*We can think about the group inverse (the function) 
  as the inverse image of the neutral element.*}

theorem (in group0) group0_2_T3: shows "f-``{\<one>} = GroupInv(G,f)"
proof -;
  from groupAsssum have "f : G\<times>G \<rightarrow> G" 
    using IsAgroup_def IsAmonoid_def IsAssociative_def 
    by simp;
  then show "f-``{\<one>} = GroupInv(G,f)"
    using func1_1_L14 GroupInv_def by auto;
qed;

text{*The inverse is in the group.*}

lemma (in group0) inverse_in_group: assumes A1:"x\<in>G" shows "x\<inverse>\<in>G"
proof -
  from groupAsssum have "GroupInv(G,f) : G\<rightarrow>G" using group0_2_T2 by simp;
  with A1 show ?thesis using apply_type by simp;
qed;

text{*The notation for the inverse means what it is supposed to mean.*}

lemma (in group0) group0_2_L6: 
  assumes A1:"x\<in>G" shows "x\<cdot>x\<inverse> = \<one> \<and> x\<inverse>\<cdot>x = \<one>"
proof;
  from groupAsssum have "GroupInv(G,f) : G\<rightarrow>G" 
    using group0_2_T2 by simp; 
  with A1 have "<x,x\<inverse>> \<in>  GroupInv(G,f)" 
    using apply_Pair by simp;
  then show "x\<cdot>x\<inverse> = \<one>" using GroupInv_def by simp;
  with A1 show "x\<inverse>\<cdot>x = \<one>" using inverse_in_group group0_2_T1 by blast; 
qed;

text{*The next two lemmas state that unless we multiply by 
  the neutral element, the result is always 
  different than any of the operands.*}

lemma (in group0) group0_2_L7: 
  assumes A1: "a\<in>G" and A2: "b\<in>G" and A3: "a\<cdot>b = a"
  shows "b=\<one>"
proof -;
  from A3 have "a\<inverse> \<cdot> (a\<cdot>b) = a\<inverse>\<cdot>a" by simp;
  with A1 A2 show ?thesis using
    inverse_in_group group_oper_assoc group0_2_L6 group0_2_L2
    by simp;
qed;

lemma (in group0) group0_2_L8: 
  assumes A1: "a\<in>G" and A2: "b\<in>G" and A3:"a\<cdot>b = b"
  shows "a=\<one>"
proof -;
  from A3 have "(a\<cdot>b)\<cdot>b\<inverse>  = b\<cdot>b\<inverse>" by simp;
  with A1 A2 have "a\<cdot>(b\<cdot>b\<inverse>)  = b\<cdot>b\<inverse>" using
    inverse_in_group group_oper_assoc by simp;
  with A1 A2 show ?thesis 
    using group0_2_L6 group0_2_L2 by simp;
qed;

text{*The inverse of the neutral element is the neutral element.*}

lemma (in group0) group0_2_L7A: shows "\<one>\<inverse>=\<one>"
  using group0_2_L2 inverse_in_group group0_2_L6 group0_2_L7 by blast;

text{*if $a^{-1} = 1$, then $a=1$. *}

lemma (in group0) group0_2_L8A:  
  assumes A1: "a\<in>G" and A2: "a\<inverse> = \<one>"
  shows "a = \<one>"
proof -
  from A1 have "a\<cdot>a\<inverse> = \<one>" using group0_2_L6 by simp
  with A1 A2 show "a = \<one>" using group0_2_L2 by simp
qed;

text{*If $a$ is not a unit, then its inverse is not either.*}

lemma (in group0) group0_2_L8B:
  assumes "a\<in>G" and "a \<noteq> \<one>"
  shows "a\<inverse> \<noteq> \<one>" using prems group0_2_L8A by auto;

text{*If $a^{-1}$ is not a unit, then a is not either.*}

lemma (in group0) group0_2_L8C:
  assumes "a\<in>G" and "a\<inverse> \<noteq> \<one>"
  shows "a\<noteq>\<one>"
  using prems group0_2_L8A group0_2_L7A by auto;

text{*If a product of two elements of a group is equal to the neutral
element then they are inverses of each other. *}

lemma (in group0) group0_2_L9: 
  assumes A1: "a\<in>G" and A2: "b\<in>G" and A3: "a\<cdot>b = \<one>" 
  shows "a = b\<inverse>"  "b = a\<inverse>"
proof -;
  from A3 have "a\<cdot>b\<cdot>b\<inverse> = \<one>\<cdot>b\<inverse>" by simp; 
  with A1 A2 have "a\<cdot>(b\<cdot>b\<inverse>) = \<one>\<cdot>b\<inverse>" using
    inverse_in_group group_oper_assoc by simp;
  with A1 A2 show "a = b\<inverse>" using
    group0_2_L6 inverse_in_group group0_2_L2 by simp;
  from A3 have "a\<inverse>\<cdot>(a\<cdot>b) = a\<inverse>\<cdot>\<one>" by simp;
  with A1 A2 show "b = a\<inverse>" using 
    inverse_in_group group_oper_assoc group0_2_L6 group0_2_L2
    by simp;
qed;

text{*It happens quite often that we know what is (have a meta-function for) 
  the right inverse in a group. The next lemma shows that the value 
  of the group inverse (function) is equal to the right inverse 
  (meta-function).*}

lemma (in group0) group0_2_L9A: 
  assumes A1: "\<forall>g\<in>G. b(g) \<in> G \<and> g\<cdot>b(g) = \<one>"
  shows "\<forall>g\<in>G. b(g) = g\<inverse>"
proof;
  fix g assume A2:"g\<in>G"
  moreover from A2 A1 have "b(g) \<in> G" by simp;
  moreover from A1 A2 have "g\<cdot>b(g) = \<one>" by simp;
  ultimately show "b(g) = g\<inverse>" by (rule group0_2_L9);
qed;
 
text{*What is the inverse of a product?*}

lemma (in group0) group0_2_L10:
  assumes A1: "a\<in>G" and A2: "b\<in>G" 
  shows " b\<inverse>\<cdot>a\<inverse> = (a\<cdot>b)\<inverse>"
proof -;
  from A1 A2 have 
    T1:"b\<inverse>\<in>G" and T2:"a\<inverse>\<in>G" and T3:"a\<cdot>b\<in>G" and T4:"b\<inverse>\<cdot>a\<inverse> \<in> G"
    using inverse_in_group group_op_closed 
    by auto;
  from A1 A2 T4 have "a\<cdot>b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = a\<cdot>(b\<cdot>(b\<inverse>\<cdot>a\<inverse>))"
    using group_oper_assoc by simp;
  moreover from A2 T1 T2 have "b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = b\<cdot>b\<inverse>\<cdot>a\<inverse>"
    using group_oper_assoc by simp;
  moreover from A2 T2 have "b\<cdot>b\<inverse>\<cdot>a\<inverse> = a\<inverse>"
     using group0_2_L6 group0_2_L2 by simp;
  ultimately have "a\<cdot>b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = a\<cdot>a\<inverse>"
    by simp;
  with A1 have "a\<cdot>b\<cdot>(b\<inverse>\<cdot>a\<inverse>) = \<one>"
    using group0_2_L6 by simp;
  with T3 T4 show "b\<inverse>\<cdot>a\<inverse> = (a\<cdot>b)\<inverse>"
    using group0_2_L9 by simp;
qed;

text{*The inverse of the inverse is the element.*}

lemma (in group0) group0_2_L11:
  assumes "a\<in>G" shows "a = (a\<inverse>)\<inverse>"
  using prems inverse_in_group group0_2_L6 group0_2_L9 by simp;

text{*If $a^{-1}\cdot b=1$, then $a=b$.*}

lemma (in group0) group0_2_L11A:
  assumes A1: "a\<in>G" and A2: "b\<in>G" and A3: "a\<inverse>\<cdot>b = \<one>"
  shows "a=b"
proof -
  from A1 have "a\<inverse> \<in> G"  "b\<in>G" using inverse_in_group
    by auto;
  then have "b = (a\<inverse>)\<inverse>" by (rule group0_2_L9);
  with A1 show "a=b" using group0_2_L11 by simp;
qed;

text{*What is the inverse of $ab^{-1}$ ?*}

lemma (in group0) group0_2_L12:
  assumes A1:"a\<in>G" "b\<in>G" 
  shows 
  "(a\<cdot>b\<inverse>)\<inverse> = b\<cdot>a\<inverse>"
  "(a\<inverse>\<cdot>b)\<inverse> = b\<inverse>\<cdot>a"
proof -
  from A1 have 
    "(a\<cdot>b\<inverse>)\<inverse> = (b\<inverse>)\<inverse>\<cdot> a\<inverse>"  "(a\<inverse>\<cdot>b)\<inverse> = b\<inverse>\<cdot>(a\<inverse>)\<inverse>"
    using inverse_in_group group0_2_L10 by auto;
  with A1 show  "(a\<cdot>b\<inverse>)\<inverse> = b\<cdot>a\<inverse>"  "(a\<inverse>\<cdot>b)\<inverse> = b\<inverse>\<cdot>a"
    using group0_2_L11 by auto
qed;

(*text{*We can insert a $b\cdot b^{-1}$ between two group elements.*}

lemma (in group0) group0_2_L14B:
  assumes A1: "a\<in>G" "b\<in>G" "c\<in>G"
  shows "a\<inverse>\<cdot>c = (a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c)"
proof -
  from A1 have T1: "a\<inverse> \<in> G"  "b\<inverse> \<in> G"  "a\<inverse>\<cdot>b \<in> G" 
    using inverse_in_group group_op_closed 
    by auto;
  with A1 have "a\<inverse>\<cdot>c =  a\<inverse>\<cdot>(b\<cdot>b\<inverse>)\<cdot>c"
    using group0_2_L2 group0_2_L6 by simp;
  also from A1 T1 have "\<dots> = (a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c)"
    using group_oper_assoc by simp;
  finally show "a\<inverse>\<cdot>c = (a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c)" by simp;
qed;*)

text{*four useful rearrangements: we can insert a $b\cdot b^{-1}$ 
  between two group elements (another version) and one about a product of 
  an element and inverse of a product, and two others.*}

lemma (in group0) group0_2_L14A:
  assumes A1: "a\<in>G"  "b\<in>G"  "c\<in>G"
  shows 
  "a\<cdot>c\<inverse>= (a\<cdot>b\<inverse>)\<cdot>(b\<cdot>c\<inverse>)"
  "a\<inverse>\<cdot>c = (a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c)"
  "a\<cdot>(b\<cdot>c)\<inverse> = a\<cdot>c\<inverse>\<cdot>b\<inverse>"
  "a\<cdot>(b\<cdot>c\<inverse>) = a\<cdot>b\<cdot>c\<inverse>"
proof -
  from A1 have T1: 
    "a\<inverse> \<in> G"  "b\<inverse>\<in>G"  "c\<inverse>\<in>G"  "a\<inverse>\<cdot>b \<in> G"  "a\<cdot>b\<inverse>\<in>G"
    using inverse_in_group group_op_closed
    by auto;
  from A1 have "a\<cdot>c\<inverse> =  a\<cdot>(b\<inverse>\<cdot>b)\<cdot>c\<inverse>"
    using group0_2_L2 group0_2_L6 by simp;
  with A1 T1 show "a\<cdot>c\<inverse>=(a\<cdot>b\<inverse>)\<cdot>(b\<cdot>c\<inverse>)"
    using group_oper_assoc by simp;
  from A1 T1 have "a\<inverse>\<cdot>c =  a\<inverse>\<cdot>(b\<cdot>b\<inverse>)\<cdot>c"
    using group0_2_L2 group0_2_L6 by simp;
  also from A1 T1 have "\<dots> = (a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c)"
    using group_oper_assoc by simp;
  finally show "a\<inverse>\<cdot>c = (a\<inverse>\<cdot>b)\<cdot>(b\<inverse>\<cdot>c)" by simp;
  from A1 have "a\<cdot>(b\<cdot>c)\<inverse> =a\<cdot>(c\<inverse>\<cdot>b\<inverse>)"
    using group0_2_L10 by simp;
  with A1 T1 show "a\<cdot>(b\<cdot>c)\<inverse> =a\<cdot>c\<inverse>\<cdot>b\<inverse>" 
    using group_oper_assoc by simp;
  from A1 T1 show "a\<cdot>(b\<cdot>c\<inverse>) = a\<cdot>b\<cdot>c\<inverse>"
    using group_oper_assoc by simp
qed;
  
text{*Another lemma about rearranging a product.*}

lemma (in group0) group0_2_L15:
  assumes A1:"a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G"
  shows "(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse> = a\<cdot>(b\<cdot>d\<inverse>)\<cdot>a\<inverse>\<cdot>(a\<cdot>c\<inverse>)"
proof -
  from A1 have T1:
    "d\<inverse>\<in>G"  "c\<inverse>\<in>G" "a\<cdot>b\<in>G" "a\<cdot>(b\<cdot>d\<inverse>)\<in>G"
    using inverse_in_group group_op_closed
    by auto;
  with A1 have "(a\<cdot>b)\<cdot>(c\<cdot>d)\<inverse> = (a\<cdot>b)\<cdot>(d\<inverse>\<cdot>c\<inverse>)"
    using group0_2_L10 by simp;
  also from A1 T1 have "\<dots> = a\<cdot>(b\<cdot>d\<inverse>)\<cdot>c\<inverse>"
    using group_oper_assoc by simp;
  also from A1 T1 have "\<dots> = a\<cdot>(b\<cdot>d\<inverse>)\<cdot>a\<inverse>\<cdot>(a\<cdot>c\<inverse>)"
    using group0_2_L14A by blast;
  finally show ?thesis by simp;
qed;

text{*We can cancel an element with its inverse that is written next to it.*}

lemma (in group0) group0_2_L16:
  assumes  A1: "a\<in>G" "b\<in>G"
  shows 
  "a\<cdot>b\<inverse>\<cdot>b = a"  
  "a\<cdot>b\<cdot>b\<inverse> = a"
proof -
  from A1 have "a\<cdot>b\<inverse>\<cdot>b = a\<cdot>(b\<inverse>\<cdot>b)" "a\<cdot>b\<cdot>b\<inverse> = a\<cdot>(b\<cdot>b\<inverse>)" 
    using inverse_in_group group_oper_assoc by auto;
  with A1 show "a\<cdot>b\<inverse>\<cdot>b = a" "a\<cdot>b\<cdot>b\<inverse> = a"
    using group0_2_L6 group0_2_L2 by auto;  
qed;
  

text{*A hard to clasify fact: adding a neutral element to a set that is 
  closed under the group operation results in a set that is closed under the 
  group operation.*}

lemma (in group0) group0_2_L17: 
  assumes A1: "H\<subseteq>G"
  and A2: "H {is closed under} f"
  shows "(H \<union> {\<one>}) {is closed under} f"
proof -;
  { fix a b assume A3: "a \<in> H \<union> {\<one>}"  "b \<in> H \<union> {\<one>}"
    have "a\<cdot>b \<in> H \<union> {\<one>}"
    proof (cases "a\<in>H")
      assume A4: "a\<in>H" show "a\<cdot>b \<in> H \<union> {\<one>}"
      proof (cases "b\<in>H")
	assume "b\<in>H"
	with A2 A4 show "a\<cdot>b \<in> H \<union> {\<one>}" using IsOpClosed_def
	  by simp;
      next assume "b\<notin>H" 
	with A1 A3 A4 show "a\<cdot>b \<in> H \<union> {\<one>}"
	  using group0_2_L2 by auto;
      qed;
    next assume "a\<notin>H"
      with A1 A3 show "a\<cdot>b \<in> H \<union> {\<one>}"
	using group0_2_L2 by auto;
    qed
  } then show "(H \<union> {\<one>}) {is closed under} f"
    using IsOpClosed_def by auto;
qed;
    
section{*Subgroups*}

text{*There are two common ways to define subgroups. One requires that the
  group operations are closed in the subgroup. The second one defines subgroup 
  as a subset of a group which is itself a group under the group operations.
  We use the second approach because it results in shorter definition. 
  We do not 
  require $H$ to be a subset of $G$ as this can be inferred from our 
  definition.
  The rest of this section is devoted to proving the equivalence of these two
  definitions of the notion of a subgroup. 
  *}

constdefs 
  "IsAsubgroup(H,f) \<equiv> IsAgroup(H, restrict(f,H\<times>H))"

text{*Formally the group operation in a subgroup is different than in the
  group as they have different domains. Of course we want to use the original 
  operation with the associated notation in the subgroup. The next couple of 
  lemmas will allow for that.*}

text{*The neutral element of the subgroup is in the subgroup and it is 
both right and left neutral there. The notation is very ugly because
we don't want to introduce a separate notation for the subgroup operation.
*}

lemma group0_3_L1: 
  assumes A1:"IsAsubgroup(H,f)"
  and A2: "n = TheIdentityElement(H,restrict(f,H\<times>H))"
  shows "n \<in> H"
  "\<forall>h\<in>H. restrict(f,H\<times>H)`<n,h > = h"
  "\<forall>h\<in>H. restrict(f,H\<times>H)`<h,n > = h"
proof -;
  let ?b = "restrict(f,H\<times>H)"
  let ?e = "TheIdentityElement(H,restrict(f,H\<times>H))"
  from A1 have "group0(H,?b)"
    using IsAsubgroup_def group0_def by simp;
  then have T1:
    "?e \<in> H \<and> (\<forall>h\<in>H. (?b`<?e,h > = h \<and> ?b`<h,?e > = h))"
    by (rule group0.group0_2_L2);
  with A2 show "n \<in> H" by simp;
  from T1 A2 show "\<forall>h\<in>H. ?b`<n,h > = h" "\<forall>h\<in>H. ?b`<h,n> = h"
    by auto;
qed;

(*lemma group0_3_L1: 
  assumes A1:"IsAsubgroup(H,f)"
  and A2: "e = TheIdentityElement(H,restrict(f,H\<times>H))"
  shows "TheIdentityElement(H,restrict(f,H\<times>H)) \<in> H"
  and 
  "\<forall>h\<in>H. restrict(f,H\<times>H)`<TheIdentityElement(H,restrict(f,H\<times>H)),h > = h"
  and 
  "\<forall>h\<in>H. restrict(f,H\<times>H)`<h,TheIdentityElement(H,restrict(f,H\<times>H)) > = h"
proof -;
  let ?b = "restrict(f,H\<times>H)"
  let ?n = "TheIdentityElement(H,restrict(f,H\<times>H))"
  from A1 have "group0(H,?b)"
    using IsAsubgroup_def group0_def by simp;
  then have T1:
    "?n \<in> H \<and> (\<forall>h\<in>H. (?b`<?n,h > = h \<and> ?b`<h,?n > = h))"
    by (rule group0.group0_2_L2);
  thus "?n \<in> H" by simp;
  from T1 show "\<forall>h\<in>H. ?b`<?n,h > = h" by auto;
  from T1 show "\<forall>h\<in>H. ?b`<h,?n> = h" by auto;
qed;*)

text{*Subgroup is contained in the group.*}

lemma (in group0) group0_3_L2: 
  assumes A1:"IsAsubgroup(H,f)"
  shows "H\<subseteq>G"
proof;
  fix h assume A2:"h\<in>H";
  let ?b = "restrict(f,H\<times>H)"
  let ?n = "TheIdentityElement(H,restrict(f,H\<times>H))"
   from A1 have "?b \<in> H\<times>H\<rightarrow>H" 
    using IsAsubgroup_def IsAgroup_def 
      IsAmonoid_def IsAssociative_def by simp;
  moreover from A2 A1 have "<?n,h> \<in> H\<times>H" 
    using group0_3_L1 by simp;
  moreover from A1 A2 have "h = ?b`<?n,h >" 
    using group0_3_L1 by simp;
  ultimately have "\<langle><?n,h>,h\<rangle> \<in> ?b" 
    using func1_1_L5A by blast;
  then have "\<langle><?n,h>,h\<rangle> \<in> f" using restrict_subset by auto
  moreover from groupAsssum have "f:G\<times>G\<rightarrow>G"
    using IsAgroup_def IsAmonoid_def IsAssociative_def 
    by simp;
  ultimately show "h\<in>G" using func1_1_L5 
    by blast;
qed;

text{*The group neutral element (denoted $1$ in the group0 context)
is a neutral element for the subgroup with respect to the froup action.*}

lemma (in group0) group0_3_L3:
  assumes A1:"IsAsubgroup(H,f)"
  shows "\<forall>h\<in>H. \<one>\<cdot>h = h \<and> h\<cdot>\<one> = h"
proof;
  fix h assume "h\<in>H" 
  with groupAsssum A1 show "\<one>\<cdot>h = h \<and> h\<cdot>\<one> = h"
    using group0_3_L2 group0_2_L2 by auto;
qed;

text{*The neutral element of a subgroup is the same as that of the group.*}

lemma (in group0) group0_3_L4: assumes A1:"IsAsubgroup(H,f)"
  shows "TheIdentityElement(H,restrict(f,H\<times>H)) = \<one>"
proof -;
  let ?n =  "TheIdentityElement(H,restrict(f,H\<times>H))"
  from A1 have T1:"?n \<in> H" using group0_3_L1 by simp;
  with groupAsssum A1 have "?n\<in>G" using  group0_3_L2 by auto;
  with A1 T1 show ?thesis using 
     group0_3_L1 restrict_if group0_2_L7 by simp;
qed;

text{*The neutral element of the group (denoted $1$ in the group0 context)
belongs to every subgroup.*}

lemma (in group0) group0_3_L5: assumes A1: "IsAsubgroup(H,f)"
  shows "\<one>\<in>H"
proof -;
  from A1 show "\<one>\<in>H" using group0_3_L1 group0_3_L4 
    by fast;
qed;

text{*Subgroups are closed with respect to the group operation.*}

lemma (in group0) group0_3_L6: assumes A1:"IsAsubgroup(H,f)"
  and A2:"a\<in>H" "b\<in>H"
  shows "a\<cdot>b \<in> H"
proof -; 
  let ?b = "restrict(f,H\<times>H)"
  from A1 have "monoid0(H,?b)" using
    IsAsubgroup_def IsAgroup_def monoid0_def by simp;
  with A2 have "?b` (<a,b>) \<in> H" using monoid0.group0_1_L1
    by blast;
 with A2 show "a\<cdot>b \<in> H" using restrict_if by simp;
qed;

text{*A preliminary lemma that we need to  show that taking the inverse 
  in the subgroup is the same as taking the inverse
  in the group.*}

lemma group0_3_L7A: 
  assumes A1:"IsAgroup(G,f)" 
  and A2:"IsAsubgroup(H,f)" and A3:"g=restrict(f,H\<times>H)"
  shows "GroupInv(G,f) \<inter> H\<times>H = GroupInv(H,g)"
proof -
  def D1: e \<equiv> "TheIdentityElement(G,f)"
  def D2: e1 \<equiv> "TheIdentityElement(H,g)"
  from A1 have T1:"group0(G,f)" using group0_def by simp;
  from A2 A3 have T2:"group0(H,g)" 
    using IsAsubgroup_def group0_def by simp 
  from T1 A2 A3 D1 D2 have "e1 = e" 
    using group0.group0_3_L4 by simp;
  with T1 D1 have "GroupInv(G,f) = f-``{e1}" 
    using group0.group0_2_T3 by simp;
  moreover have "g-``{e1} = f-``{e1} \<inter> H\<times>H"
  proof -;
    from A1 have "f \<in> G\<times>G\<rightarrow>G" 
      using IsAgroup_def IsAmonoid_def IsAssociative_def 
      by simp;
    moreover from T1 A2 have "H\<times>H \<subseteq> G\<times>G" 
      using group0.group0_3_L2 by auto;
    ultimately show "g-``{e1} = f-``{e1} \<inter> H\<times>H" 
      using A3 func1_2_L1 by simp;
  qed;
  moreover from T2 A3 D2 have "GroupInv(H,g) = g-``{e1}" 
    using group0.group0_2_T3 by simp;
  ultimately show ?thesis by simp;
qed;

text{*Using the lemma above we can show the actual statement: 
  taking the inverse in the subgroup is the same as taking the inverse
  in the group.*}

theorem (in group0) group0_3_T1:
  assumes A1: "IsAsubgroup(H,f)" 
  and A2:"g=restrict(f,H\<times>H)"
  shows "GroupInv(H,g) = restrict(GroupInv(G,f),H)"
proof -;
  from groupAsssum have "GroupInv(G,f) : G\<rightarrow>G" 
    using group0_2_T2 by simp;
  moreover from A1 A2 have "GroupInv(H,g) : H\<rightarrow>H"
    using IsAsubgroup_def group0_2_T2 by simp;
  moreover from A1 have "H\<subseteq>G" 
    using group0_3_L2 by simp;
  moreover from groupAsssum A1 A2 have 
    "GroupInv(G,f) \<inter> H\<times>H = GroupInv(H,g)"
    using group0_3_L7A by simp;
  ultimately show ?thesis
    using func1_2_L3 by simp;
qed;

text{*A sligtly weaker, but more convenient in applications,
  reformulation of the above theorem.*}

theorem (in group0) group0_3_T2: 
  assumes "IsAsubgroup(H,f)" 
  and "g=restrict(f,H\<times>H)"
  shows "\<forall>h\<in>H. GroupInv(H,g)`(h) = h\<inverse>"
  using prems group0_3_T1 restrict_if by simp;

text{*Subgroups are closed with respect to taking the group inverse.
  Again, I was unable to apply @{text "inverse_in_group"} directly to the group 
  $H$. This problem is worked around by repeating the (short) proof of 
  @{text "inverse_in_group"} in the proof below.*}

theorem (in group0) group0_3_T3A: 
  assumes A1:"IsAsubgroup(H,f)" and A2:"h\<in>H"
  shows "h\<inverse>\<in> H"
proof -
  def D1: g \<equiv> "restrict(f,H\<times>H)";
  with A1 have  "GroupInv(H,g) \<in> H\<rightarrow>H"
    using IsAsubgroup_def group0_2_T2 by simp;
  with A2 have "GroupInv(H,g)`(h) \<in> H"
    using apply_type by simp;
  with A1 D1 A2 show "h\<inverse>\<in> H" using group0_3_T2 by simp;
qed;

text{*The next theorem states that a nonempty subset of 
of a group $G$ that is closed under the group operation and 
taking the inverse is a subgroup of the group.*}

theorem (in group0) group0_3_T3:
  assumes A1: "H\<noteq>0"
  and A2: "H\<subseteq>G"
  and A3: "H {is closed under} f"
  and A4: "\<forall>x\<in>H. x\<inverse> \<in> H"
  shows "IsAsubgroup(H,f)"
proof -;
  let ?g = "restrict(f,H\<times>H)"
  let ?n = "TheIdentityElement(H,?g)"
  from A3 have T0:"\<forall>x\<in>H.\<forall>y\<in>H. x\<cdot>y \<in> H"
    using IsOpClosed_def by simp;
  from A1 obtain x where "x\<in>H" by auto;
  with A4 T0 A2 have T1:"\<one>\<in>H"
    using group0_2_L6 by blast;
  with A3 A2 have T2:"IsAmonoid(H,?g)"
    using group0_2_L1 monoid0.group0_1_T1
    by simp;
  moreover have "\<forall>h\<in>H.\<exists>b\<in>H. ?g`<h,b> = ?n"
  proof;
    fix h assume A5:"h\<in>H"
    with A4 A2 have "h\<cdot>h\<inverse> = \<one>"
      using group0_2_L6 by auto;
    moreover from groupAsssum A3 A2 T1 have "\<one> = ?n"
      using IsAgroup_def group0_1_L6 by auto;
    moreover from A5 A4 have "?g`<h,h\<inverse>> = h\<cdot>h\<inverse>"
      using restrict_if by simp;
    ultimately have "?g`<h,h\<inverse>> = ?n" by simp;
    with A5 A4 show "\<exists>b\<in>H. ?g`<h,b> = ?n" by auto;
  qed;
  ultimately show "IsAsubgroup(H,f)" using 
    IsAsubgroup_def IsAgroup_def by simp;
qed;

text{*Intersection of subgroups is a subgroup of each factor.*}

lemma group0_3_L7:
  assumes A1:"IsAgroup(G,f)"
  and A2:"IsAsubgroup(H\<^isub>1,f)"
  and A3:"IsAsubgroup(H\<^isub>2,f)"
  shows "IsAsubgroup(H\<^isub>1\<inter>H\<^isub>2,restrict(f,H\<^isub>1\<times>H\<^isub>1))"
proof -;
  let ?e = "TheIdentityElement(G,f)"
  let ?g = "restrict(f,H\<^isub>1\<times>H\<^isub>1)"
  from A1 have T1: "group0(G,f)"
    using group0_def by simp;
  from A2 have "group0(H\<^isub>1,?g)"
    using IsAsubgroup_def group0_def by simp;
  moreover have "H\<^isub>1\<inter>H\<^isub>2 \<noteq> 0"
  proof -
    from A1 A2 A3 have "?e \<in> H\<^isub>1\<inter>H\<^isub>2"
      using group0_def group0.group0_3_L5 by simp;
    thus ?thesis by auto;
  qed;
  moreover have T2:"H\<^isub>1\<inter>H\<^isub>2 \<subseteq> H\<^isub>1" by auto;
  moreover from T1 T2 A2 A3 have 
    "H\<^isub>1\<inter>H\<^isub>2 {is closed under} ?g"
    using group0.group0_3_L6 IsOpClosed_def 
      func_ZF_4_L7 func_ZF_4_L5 by simp;
  moreover from T1 A2 A3 have 
    "\<forall>x \<in> H\<^isub>1\<inter>H\<^isub>2. GroupInv(H\<^isub>1,?g)`(x) \<in> H\<^isub>1\<inter>H\<^isub>2"
    using group0.group0_3_T2 group0.group0_3_T3A
    by simp;
  ultimately show ?thesis
    using group0.group0_3_T3 by simp;
qed;

section{*Abelian groups*}

text{*Here we will prove some facts specific to abelian groups. *}

text{*Proving the facts about associative and commutative operations is 
  quite tedious in formalized mathematics. To a human the thing is simple:
  we can arrange the elements in any order and put parantheses wherever we 
  want, it is all the same. However, formalizing this statement would be 
  rather difficult (I think). The next lemma attempts a quasi-algorithmic
  approach to this type of problem. To prove that two expressions are equal, 
  we first strip one from parantheses, then rearrange the elements in proper 
  order, then put the parantheses where we want them to be. The algorithm for 
  rearrangement is easy to describe: we keep putting the first element 
  (from the right) that is in the wrong place at the left-most position
  until we get the proper arrangement. For the parantheses simp does it 
  very well.*}

lemma (in group0) group0_4_L2:
  assumes A1:"f {is commutative on} G"
  and A2:"a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G" "E\<in>G" "F\<in>G"
  shows "(a\<cdot>b)\<cdot>(c\<cdot>d)\<cdot>(E\<cdot>F) = (a\<cdot>(d\<cdot>F))\<cdot>(b\<cdot>(c\<cdot>E))"
proof -;
  from A2 have "(a\<cdot>b)\<cdot>(c\<cdot>d)\<cdot>(E\<cdot>F) = a\<cdot>b\<cdot>c\<cdot>d\<cdot>E\<cdot>F"
    using group_op_closed group_oper_assoc
    by simp;
  also have  "a\<cdot>b\<cdot>c\<cdot>d\<cdot>E\<cdot>F = a\<cdot>d\<cdot>F\<cdot>b\<cdot>c\<cdot>E"
  proof -
    from A1 A2 have "a\<cdot>b\<cdot>c\<cdot>d\<cdot>E\<cdot>F = F\<cdot>(a\<cdot>b\<cdot>c\<cdot>d\<cdot>E)"
      using IsCommutative_def group_op_closed 
      by simp;
    also from A2 have "F\<cdot>(a\<cdot>b\<cdot>c\<cdot>d\<cdot>E) = F\<cdot>a\<cdot>b\<cdot>c\<cdot>d\<cdot>E"
      using group_op_closed group_oper_assoc
      by simp;
    also from A1 A2 have "F\<cdot>a\<cdot>b\<cdot>c\<cdot>d\<cdot>E = d\<cdot>(F\<cdot>a\<cdot>b\<cdot>c)\<cdot>E"
      using IsCommutative_def group_op_closed
      by simp;
    also from A2 have "d\<cdot>(F\<cdot>a\<cdot>b\<cdot>c)\<cdot>E = d\<cdot>F\<cdot>a\<cdot>b\<cdot>c\<cdot>E"
      using group_op_closed group_oper_assoc
      by simp;
    also from A1 A2 have " d\<cdot>F\<cdot>a\<cdot>b\<cdot>c\<cdot>E = a\<cdot>(d\<cdot>F)\<cdot>b\<cdot>c\<cdot>E"
      using IsCommutative_def group_op_closed
      by simp;
    also from A2 have "a\<cdot>(d\<cdot>F)\<cdot>b\<cdot>c\<cdot>E = a\<cdot>d\<cdot>F\<cdot>b\<cdot>c\<cdot>E" 
      using group_op_closed group_oper_assoc
      by simp;
    finally show ?thesis by simp;
  qed;
  also from A2 have "a\<cdot>d\<cdot>F\<cdot>b\<cdot>c\<cdot>E = (a\<cdot>(d\<cdot>F))\<cdot>(b\<cdot>(c\<cdot>E))"
    using group_op_closed group_oper_assoc
    by simp;
  finally show ?thesis by simp;
qed;
  
text{*Another useful rearrangement.*}

lemma (in group0) group0_4_L3:
  assumes A1:"f {is commutative on} G" 
  and A2:"a\<in>G" "b\<in>G" and A3:"c\<in>G" "d\<in>G" "E\<in>G" "F\<in>G"
  shows "a\<cdot>b\<cdot>((c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse>) = (a\<cdot>(E\<cdot>c)\<inverse>)\<cdot>(b\<cdot>(F\<cdot>d)\<inverse>)"
proof -;
  from A3 have T1:
    "c\<inverse>\<in>G" "d\<inverse>\<in>G" "E\<inverse>\<in>G" "F\<inverse>\<in>G" "(c\<cdot>d)\<inverse>\<in>G" "(E\<cdot>F)\<inverse>\<in>G"
    using inverse_in_group group_op_closed 
    by auto;
  from A2 T1 have 
    "a\<cdot>b\<cdot>((c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse>) = a\<cdot>b\<cdot>(c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse>"
    using group_op_closed group_oper_assoc
    by simp;
  also from A2 A3 have 
    "a\<cdot>b\<cdot>(c\<cdot>d)\<inverse>\<cdot>(E\<cdot>F)\<inverse> = (a\<cdot>b)\<cdot>(d\<inverse>\<cdot>c\<inverse>)\<cdot>(F\<inverse>\<cdot>E\<inverse>)"
    using group0_2_L10 by simp;
   also from A1 A2 T1 have 
    "(a\<cdot>b)\<cdot>(d\<inverse>\<cdot>c\<inverse>)\<cdot>(F\<inverse>\<cdot>E\<inverse>) = (a\<cdot>(c\<inverse>\<cdot>E\<inverse>))\<cdot>(b\<cdot>(d\<inverse>\<cdot>F\<inverse>))"
    using group0_4_L2 by simp;
  also from A2 A3 have 
    "(a\<cdot>(c\<inverse>\<cdot>E\<inverse>))\<cdot>(b\<cdot>(d\<inverse>\<cdot>F\<inverse>)) = (a\<cdot>(E\<cdot>c)\<inverse>)\<cdot>(b\<cdot>(F\<cdot>d)\<inverse>)"
    using group0_2_L10 by simp;
  finally show ?thesis by simp;
qed;

text{*Some useful rearrangements for two elements of a group.*}

lemma (in group0) group0_4_L4:
  assumes A1:"f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G"
  shows 
  "b\<inverse>\<cdot>a\<inverse> = a\<inverse>\<cdot>b\<inverse>" 
  "(a\<cdot>b)\<inverse> = a\<inverse>\<cdot>b\<inverse>" 
  "(a\<cdot>b\<inverse>)\<inverse> = a\<inverse>\<cdot>b"
proof -;
  from A2 have T1: "b\<inverse>\<in>G" "a\<inverse>\<in>G" using inverse_in_group by auto;
  with A1 show "b\<inverse>\<cdot>a\<inverse> = a\<inverse>\<cdot>b\<inverse>" using IsCommutative_def by simp;
  with A2 show "(a\<cdot>b)\<inverse> = a\<inverse>\<cdot>b\<inverse>" using group0_2_L10 by simp;
  from A2 T1 have "(a\<cdot>b\<inverse>)\<inverse> = (b\<inverse>)\<inverse>\<cdot>a\<inverse>" using group0_2_L10 by simp;
  with A1 A2 T1 show "(a\<cdot>b\<inverse>)\<inverse> = a\<inverse>\<cdot>b" 
    using group0_2_L11 IsCommutative_def by simp;
qed;
 
text{*Another bunch of useful rearrangements.*}

lemma (in group0) group0_4_L4A: 
  assumes A1:"f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G" "c\<in>G"
  shows 
  "a\<cdot>b\<cdot>c = c\<cdot>a\<cdot>b" 
  "a\<inverse>\<cdot>(b\<inverse>\<cdot>c\<inverse>)\<inverse> = (a\<cdot>(b\<cdot>c)\<inverse>)\<inverse>" 
  "a\<cdot>(b\<cdot>c)\<inverse> = a\<cdot>b\<inverse>\<cdot>c\<inverse>" 
  "a\<cdot>(b\<cdot>c\<inverse>)\<inverse> = a\<cdot>b\<inverse>\<cdot>c" 
  "a\<cdot>b\<inverse>\<cdot>c\<inverse> = a\<cdot>c\<inverse>\<cdot>b\<inverse>"
proof -
  from A1 A2 have "a\<cdot>b\<cdot>c = c\<cdot>(a\<cdot>b)"
    using IsCommutative_def group_op_closed
    by simp;
  with A2 show "a\<cdot>b\<cdot>c = c\<cdot>a\<cdot>b" using
     group_op_closed group_oper_assoc
    by simp
  from A2 have T1: "b\<inverse>\<in>G" "c\<inverse>\<in>G" "b\<inverse>\<cdot>c\<inverse> \<in> G" "a\<cdot>b \<in> G"
    using inverse_in_group group_op_closed
    by auto;
  with A1 A2 show "a\<inverse>\<cdot>(b\<inverse>\<cdot>c\<inverse>)\<inverse> = (a\<cdot>(b\<cdot>c)\<inverse>)\<inverse>"
    using group0_2_L10 IsCommutative_def 
    by simp;
  from A1 A2 T1 have "a\<cdot>(b\<cdot>c)\<inverse> = a\<cdot>(b\<inverse>\<cdot>c\<inverse>)"
    using group0_2_L10 IsCommutative_def by simp;
  with A2 T1 show "a\<cdot>(b\<cdot>c)\<inverse> = a\<cdot>b\<inverse>\<cdot>c\<inverse>"
    using group_oper_assoc by simp;
  from A1 A2 T1 have "a\<cdot>(b\<cdot>c\<inverse>)\<inverse> = a\<cdot>(b\<inverse>\<cdot>(c\<inverse>)\<inverse>)"
    using group0_2_L10 IsCommutative_def by simp;
  with A2 T1 show "a\<cdot>(b\<cdot>c\<inverse>)\<inverse> = a\<cdot>b\<inverse>\<cdot>c"
    using group_oper_assoc group0_2_L11 by simp;
  from A1 A2 T1 have "a\<cdot>b\<inverse>\<cdot>c\<inverse> = a\<cdot>(c\<inverse>\<cdot>b\<inverse>)"
    using group_oper_assoc IsCommutative_def by simp;
  with A2 T1 show "a\<cdot>b\<inverse>\<cdot>c\<inverse> = a\<cdot>c\<inverse>\<cdot>b\<inverse>"
    using group_oper_assoc by simp
qed;

text{*Another useful rearrangement.*}

lemma (in group0) group0_4_L4B: 
  assumes "f {is commutative on} G"
  and "a\<in>G" "b\<in>G" "c\<in>G"
  shows "a\<cdot>b\<inverse>\<cdot>(b\<cdot>c\<inverse>) = a\<cdot>c\<inverse>"  
  using prems inverse_in_group group_op_closed 
    group0_4_L4 group_oper_assoc group0_2_L16 by simp;

text{*Another useful rearrangement.*}

lemma (in group0) group0_4_L4C: 
  assumes A1: "f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G" "c\<in>G"
  shows "a\<cdot>b\<cdot>c = a\<cdot>(c\<cdot>b)"
proof -
  from A1 A2 have "a\<cdot>b\<cdot>c = c\<cdot>a\<cdot>b"
    using group0_4_L4A by simp
  also from A1 A2 have "c\<cdot>a\<cdot>b = a\<cdot>c\<cdot>b"
    using IsCommutative_def by simp;
  also from A2 have "a\<cdot>c\<cdot>b = a\<cdot>(c\<cdot>b)"
    using group_oper_assoc by simp;
  finally show "a\<cdot>b\<cdot>c = a\<cdot>(c\<cdot>b)" by simp;
qed;
  
text{*We can put an element on the other side of an equation.*}

lemma (in group0) group0_4_L5A:
  assumes A1:  "a\<in>G" "b\<in>G" "c\<in>G"
  and A2: "c = a\<cdot>b"
  shows "c\<cdot>b\<inverse> = a"
proof-;
  from A2 A1 have "c\<cdot>b\<inverse> =  a\<cdot>(b\<cdot>b\<inverse>)"
    using inverse_in_group group_oper_assoc by simp;
  also from A1 have "\<dots> = a" using group0_2_L6 group0_2_L2
    by simp;
  finally show ?thesis by simp;
qed;

text{*Another rearrangement lemma.*}

lemma (in group0) group0_4_L5: assumes A1:"f {is commutative on} G" 
  and A2: "a\<in>G" "b\<in>G" "c\<in>G"
  and A3: "c = a\<cdot>b\<inverse>"
  shows "a = b\<cdot>c"
proof -; 
  from A2 A3 have "c\<cdot>(b\<inverse>)\<inverse> = a"
    using inverse_in_group group0_4_L5A
    by simp;
  with A1 A2 show ?thesis using 
     group0_2_L11 IsCommutative_def by simp;
qed;

text{*In abelian groups we can cancel an element with its inverse
  even if separated by another element.*}

lemma (in group0) group0_4_L6A: assumes A1:"f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G"
  shows 
  "a\<cdot>b\<cdot>a\<inverse> = b"
  "a\<inverse>\<cdot>b\<cdot>a = b"
proof -;
  from A1 A2 have 
    "a\<cdot>b\<cdot>a\<inverse> = a\<inverse>\<cdot>a\<cdot>b"
    using inverse_in_group group0_4_L4A by blast
  also from A2 have "\<dots> = b"
    using group0_2_L6 group0_2_L2 by simp;
  finally show "a\<cdot>b\<cdot>a\<inverse> = b" by simp;
  from A1 A2 have 
    "a\<inverse>\<cdot>b\<cdot>a = a\<cdot>a\<inverse>\<cdot>b"
    using inverse_in_group group0_4_L4A by blast;
  also from A2 have "\<dots> = b"
    using group0_2_L6 group0_2_L2 by simp;
  finally show "a\<inverse>\<cdot>b\<cdot>a = b" by simp
qed;

text{*In abelian groups we can cancel an element with its inverse
  even if separated by two other elements.*}

lemma (in group0) group0_4_L6B: assumes A1:"f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G" "c\<in>G" 
  shows 
  "a\<cdot>b\<cdot>c\<cdot>a\<inverse> = b\<cdot>c"
  "a\<inverse>\<cdot>b\<cdot>c\<cdot>a = b\<cdot>c"
proof -
   from A2 have 
     "a\<cdot>b\<cdot>c\<cdot>a\<inverse> = a\<cdot>(b\<cdot>c)\<cdot>a\<inverse>"
     "a\<inverse>\<cdot>b\<cdot>c\<cdot>a = a\<inverse>\<cdot>(b\<cdot>c)\<cdot>a"
    using group_op_closed group_oper_assoc inverse_in_group
    by auto;
  with A1 A2 show
    "a\<cdot>b\<cdot>c\<cdot>a\<inverse> = b\<cdot>c"
    "a\<inverse>\<cdot>b\<cdot>c\<cdot>a = b\<cdot>c"
    using group_op_closed group0_4_L6A
    by auto;
qed;

text{*In abelian groups we can cancel an element with its inverse
  even if separated by three other elements.*}

lemma (in group0) group0_4_L6C: assumes A1:"f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G"
  shows "a\<cdot>b\<cdot>c\<cdot>d\<cdot>a\<inverse> = b\<cdot>c\<cdot>d" 
proof -
  from A2 have "a\<cdot>b\<cdot>c\<cdot>d\<cdot>a\<inverse> = a\<cdot>(b\<cdot>c\<cdot>d)\<cdot>a\<inverse>"
    using group_op_closed group_oper_assoc
    by simp;
  with A1 A2 show ?thesis 
    using group_op_closed group0_4_L6A 
    by simp;
qed;

text{*Another couple of useful rearrangements.*}

lemma (in group0) group0_4_L6D: 
  assumes A1: "f {is commutative on} G"
  and A2: "a\<in>G"  "b\<in>G"  "c\<in>G"
  shows 
  "a\<cdot>b\<inverse>\<cdot>(a\<cdot>c\<inverse>)\<inverse> = c\<cdot>b\<inverse>"
  "(a\<cdot>c)\<inverse>\<cdot>(b\<cdot>c) = a\<inverse>\<cdot>b"
  "a\<cdot>(b\<cdot>(c\<cdot>a\<inverse>\<cdot>b\<inverse>)) = c"
proof -
  from A2 have T1: 
    "a\<inverse> \<in> G"  "b\<inverse> \<in> G"  "a\<cdot>b\<inverse> \<in> G"  "c\<inverse>\<cdot>a\<inverse> \<in> G"
    "c\<cdot>a\<inverse> \<in> G"
    using inverse_in_group group_op_closed 
    by auto;
  with A1 A2 show "a\<cdot>b\<inverse>\<cdot>(a\<cdot>c\<inverse>)\<inverse> = c\<cdot>b\<inverse>"
    using group0_2_L12 group_oper_assoc group0_4_L6B
    IsCommutative_def by simp;
  from A2 T1 have "(a\<cdot>c)\<inverse>\<cdot>(b\<cdot>c) = c\<inverse>\<cdot>a\<inverse>\<cdot>b\<cdot>c"
    using group0_2_L10 group_oper_assoc by simp;
  also from A1 A2 T1 have "\<dots> = a\<inverse>\<cdot>b"
    using group0_4_L6B by simp;
  finally show "(a\<cdot>c)\<inverse>\<cdot>(b\<cdot>c) = a\<inverse>\<cdot>b"
    by simp;
  from A1 A2 T1 show "a\<cdot>(b\<cdot>(c\<cdot>a\<inverse>\<cdot>b\<inverse>)) = c"
    using group_oper_assoc group0_4_L6B group0_4_L6A
    by simp;  
qed

(*
text{*Another couple of useful rearrangements (to be deleted).*}

lemma (in group0) group0_4_L6DA: 
  assumes A1: "f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G" "c\<in>G"
  shows 
  "a\<cdot>b\<inverse>\<cdot>(a\<cdot>c\<inverse>)\<inverse> = c\<cdot>b\<inverse>"
  "(a\<cdot>c)\<inverse>\<cdot>(b\<cdot>c) = a\<inverse>\<cdot>b"
proof -
  from A2 have T1: 
    "a\<inverse> \<in> G"  "b\<inverse> \<in> G"  "a\<cdot>b\<inverse> \<in> G"  "c\<inverse>\<cdot>a\<inverse> \<in> G"
    using inverse_in_group group_op_closed 
    by auto;
  with A1 A2 show "a\<cdot>b\<inverse>\<cdot>(a\<cdot>c\<inverse>)\<inverse> = c\<cdot>b\<inverse>"
    using group0_2_L12 group_oper_assoc group0_4_L6B
    IsCommutative_def by simp;
  from A2 T1 have "(a\<cdot>c)\<inverse>\<cdot>(b\<cdot>c) = c\<inverse>\<cdot>a\<inverse>\<cdot>b\<cdot>c"
    using group0_2_L10 group_oper_assoc by simp;
  also from A1 A2 T1 have "\<dots> = a\<inverse>\<cdot>b"
    using group0_4_L6B by simp;
  finally show "(a\<cdot>c)\<inverse>\<cdot>(b\<cdot>c) = a\<inverse>\<cdot>b"
    by simp;
qed
*)
  
text{*Another rearrangement lemma.*}
 
lemma (in group0) group0_4_L7: assumes A1:"f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G"
  shows "a\<cdot>b\<cdot>c\<cdot>d\<inverse> = a\<cdot>d\<inverse>\<cdot> b\<cdot>c" 
proof -
  from A2 have T1:
    "b\<cdot>c \<in> G" "d\<inverse> \<in> G" "b\<inverse>\<in>G" "c\<inverse>\<in>G" 
    "d\<inverse>\<cdot>b \<in> G" "c\<inverse>\<cdot>d \<in> G" "(b\<cdot>c)\<inverse> \<in> G" 
    using group_op_closed inverse_in_group 
    by auto;
  with A1 A2 have "a\<cdot>b\<cdot>c\<cdot>d\<inverse> = a\<cdot>(d\<inverse>\<cdot>b\<cdot>c)"
    using group_oper_assoc group0_4_L4A by simp;
  also from A2 T1 have "a\<cdot>(d\<inverse>\<cdot>b\<cdot>c) = a\<cdot>d\<inverse>\<cdot>b\<cdot>c"
    using group_oper_assoc by simp; 
  finally show "a\<cdot>b\<cdot>c\<cdot>d\<inverse> = a\<cdot>d\<inverse>\<cdot> b\<cdot>c" by simp;
qed;

text{*Yet another rearrangement lemma.*}

lemma (in group0) group0_4_L8: 
  assumes A1: "f {is commutative on} G"
  and A2: "a\<in>G" "b\<in>G" "c\<in>G" "d\<in>G"
  shows 
  "a\<cdot>(b\<cdot>c)\<inverse> = (a\<cdot>d\<inverse>\<cdot>c\<inverse>)\<cdot>(d\<cdot>b\<inverse>)"
  "a\<cdot>b\<cdot>(c\<cdot>d) = c\<cdot>a\<cdot>(b\<cdot>d)"
  "a\<cdot>b\<cdot>(c\<cdot>d) = a\<cdot>c\<cdot>(b\<cdot>d)"
proof -
  from A2 have T1:
    "b\<cdot>c \<in> G" "a\<cdot>b \<in> G" "d\<inverse> \<in> G" "b\<inverse>\<in>G" "c\<inverse>\<in>G" 
    "d\<inverse>\<cdot>b \<in> G" "c\<inverse>\<cdot>d \<in> G" "(b\<cdot>c)\<inverse> \<in> G" 
    using group_op_closed inverse_in_group 
    by auto;
  from A2 have "a\<cdot>(b\<cdot>c)\<inverse> = a\<cdot>c\<inverse>\<cdot>b\<inverse>" using group0_2_L14A by blast;
  moreover from A2 have "a\<cdot>c\<inverse> = (a\<cdot>d\<inverse>)\<cdot>(d\<cdot>c\<inverse>)" using group0_2_L14A
    by blast;
  ultimately have "a\<cdot>(b\<cdot>c)\<inverse> = (a\<cdot>d\<inverse>)\<cdot>(d\<cdot>c\<inverse>)\<cdot>b\<inverse>" by simp;
  with A1 A2 T1 have "a\<cdot>(b\<cdot>c)\<inverse>= a\<cdot>d\<inverse>\<cdot>(c\<inverse>\<cdot>d)\<cdot>b\<inverse>"
    using IsCommutative_def by simp;
  with A2 T1 show "a\<cdot>(b\<cdot>c)\<inverse> = (a\<cdot>d\<inverse>\<cdot>c\<inverse>)\<cdot>(d\<cdot>b\<inverse>)"
    using group_op_closed group_oper_assoc by simp;
  from A2 T1 have "a\<cdot>b\<cdot>(c\<cdot>d) = a\<cdot>b\<cdot>c\<cdot>d"
    using group_oper_assoc by simp;
  also have "a\<cdot>b\<cdot>c\<cdot>d = c\<cdot>a\<cdot>b\<cdot>d"
  proof -;
    from A1 A2 have "a\<cdot>b\<cdot>c\<cdot>d = c\<cdot>(a\<cdot>b)\<cdot>d"
      using IsCommutative_def group_op_closed
      by simp;
    also from A2 have "\<dots> = c\<cdot>a\<cdot>b\<cdot>d"
      using group_op_closed group_oper_assoc
      by simp;
    finally show ?thesis by simp;
  qed;
  also from A2 have "c\<cdot>a\<cdot>b\<cdot>d =  c\<cdot>a\<cdot>(b\<cdot>d)"
    using group_op_closed group_oper_assoc
    by simp;
  finally show "a\<cdot>b\<cdot>(c\<cdot>d) = c\<cdot>a\<cdot>(b\<cdot>d)" by simp;
  with A1 A2 show "a\<cdot>b\<cdot>(c\<cdot>d) = a\<cdot>c\<cdot>(b\<cdot>d)"
    using IsCommutative_def by simp
qed;

section{*Translations*}

text{*In this section we consider translations. Translations are maps 
  $T: G\rightarrow G$ of the form $T_g (a) = g\cdot a$ or 
  $T_g (a) = a\cdot g$. We also consider two-dimensional translations
  $T_g : G\times G \rightarrow G\times G$, where 
  $T_g(a,b) = (a\cdot g, b\cdot g)$ or$T_g(a,b) = (g\cdot a, g\cdot b)$. 
  *}

constdefs
  "RightTranslation(G,P,g) \<equiv> {<a,b> \<in> G\<times>G. P`<a,g> = b}"

  "LeftTranslation(G,P,g) \<equiv> {<a,b> \<in> G\<times>G. P`<g,a> = b}"
 
  "RightTranslation2(G,P,g) \<equiv> 
  {<x,y> \<in> (G\<times>G)\<times>(G\<times>G). \<langle>P`<fst(x),g>, P`<snd(x),g>\<rangle> = y}"

  "LeftTranslation2(G,P,g) \<equiv> 
  {<x,y> \<in> (G\<times>G)\<times>(G\<times>G). \<langle>P`<g,fst(x)>, P`<g,snd(x)>\<rangle> = y}"

text{*Translations map $G$ into $G$. Two dimensional translations map
  $G\times G$ into itself.*}

lemma (in group0) group0_5_L1: assumes A1: "g\<in>G"
  shows "RightTranslation(G,f,g) : G\<rightarrow>G"
  "LeftTranslation(G,f,g) : G\<rightarrow>G"
  "RightTranslation2(G,f,g) : (G\<times>G)\<rightarrow>(G\<times>G)"
  "LeftTranslation2(G,f,g) : (G\<times>G)\<rightarrow>(G\<times>G)"
proof -
  from A1 have "\<forall>a\<in>G. a\<cdot>g \<in> G" "\<forall>a\<in>G. g\<cdot>a \<in> G"
    "\<forall>x \<in> G\<times>G.  <fst(x)\<cdot>g, snd(x)\<cdot>g> \<in> G\<times>G"
    "\<forall>x \<in> G\<times>G.  <g\<cdot>fst(x),g\<cdot>snd(x)> \<in> G\<times>G"
    using group_oper_assocA apply_funtype by auto;
  then show "RightTranslation(G,f,g) : G\<rightarrow>G" 
    "LeftTranslation(G,f,g) : G\<rightarrow>G"
    "RightTranslation2(G,f,g) : (G\<times>G)\<rightarrow>(G\<times>G)"
    "LeftTranslation2(G,f,g) : (G\<times>G)\<rightarrow>(G\<times>G)"
    using RightTranslation_def LeftTranslation_def
      RightTranslation2_def LeftTranslation2_def func1_1_L11A
    by auto;
qed;

text{*The values of the translations are what we expect.*}

lemma (in group0) group0_5_L2: assumes A1: "g\<in>G" "a\<in>G"
  shows 
  "RightTranslation(G,f,g)`(a) = a\<cdot>g"
  "LeftTranslation(G,f,g)`(a) = g\<cdot>a"
  using prems group0_5_L1 RightTranslation_def LeftTranslation_def
    func1_1_L11B by auto;

text{*The values of the two-dimensional translations are what we expect.*}

lemma (in group0) group0_5_L3: assumes A1: "g\<in>G" "a\<in>G" "b\<in>G"
  shows "RightTranslation2(G,f,g)`<a,b> = <a\<cdot>g,b\<cdot>g>"
  "LeftTranslation2(G,f,g)`<a,b> = <g\<cdot>a,g\<cdot>b>"
  using prems RightTranslation2_def LeftTranslation2_def
    group0_5_L1 func1_1_L11B by auto;


text{*Composition of left translations is a left translation by the product.*}

lemma (in group0) group0_5_L4: assumes A1:"g\<in>G" "h\<in>G" "a\<in>G"
  and A2: "T\<^isub>g = LeftTranslation(G,f,g)" "T\<^isub>h = LeftTranslation(G,f,h)"
  shows "T\<^isub>g`(T\<^isub>h`(a)) = g\<cdot>h\<cdot>a"
  "T\<^isub>g`(T\<^isub>h`(a)) = LeftTranslation(G,f,g\<cdot>h)`(a)"
proof -;
  from A1 have T1:"h\<cdot>a\<in>G" "g\<cdot>h\<in>G"
    using group_oper_assocA apply_funtype by auto;
  with A1 A2 show "T\<^isub>g`(T\<^isub>h`(a)) = g\<cdot>h\<cdot>a"
    using group0_5_L2 group_oper_assoc by simp;
  with A1 A2 T1 show 
    "T\<^isub>g`(T\<^isub>h`(a)) = LeftTranslation(G,f,g\<cdot>h)`(a)"
    using group0_5_L2 group_oper_assoc by simp;
qed;

text{*Composition of right translations is a right translation by 
  the product.*}

lemma (in group0) group0_5_L5: assumes A1:"g\<in>G" "h\<in>G" "a\<in>G"
  and A2: "T\<^isub>g = RightTranslation(G,f,g)" "T\<^isub>h = RightTranslation(G,f,h)"
  shows "T\<^isub>g`(T\<^isub>h`(a)) = a\<cdot>h\<cdot>g"
  "T\<^isub>g`(T\<^isub>h`(a)) = RightTranslation(G,f,h\<cdot>g)`(a)"
proof -
  from A1 have T1: "a\<cdot>h\<in>G" "h\<cdot>g \<in>G"
    using group_oper_assocA apply_funtype by auto;
  with A1 A2 show "T\<^isub>g`(T\<^isub>h`(a)) = a\<cdot>h\<cdot>g" 
    using group0_5_L2 group_oper_assoc by simp;
  with A1 A2 T1 show 
    "T\<^isub>g`(T\<^isub>h`(a)) = RightTranslation(G,f,h\<cdot>g)`(a)"
    using group0_5_L2 group_oper_assoc by simp;
qed;

text{*The image of a set under a composition of translations is the same as
  the image under translation by a product.*}

lemma (in group0) group0_5_L6: assumes A1: "g\<in>G" "h\<in>G" and A2: "A\<subseteq>G"
  and A3: "T\<^isub>g = RightTranslation(G,f,g)" "T\<^isub>h = RightTranslation(G,f,h)"
  shows "T\<^isub>g``(T\<^isub>h``(A)) = {a\<cdot>h\<cdot>g. a\<in>A}"
proof -;
  from A2 have T1:"\<forall>a\<in>A. a\<in>G" by auto;
  from A1 A3 have "T\<^isub>g : G\<rightarrow>G" "T\<^isub>h : G\<rightarrow>G"
    using group0_5_L1 by auto;
  with A1 A2 T1 A3 show "T\<^isub>g``(T\<^isub>h``(A)) = {a\<cdot>h\<cdot>g. a\<in>A}"
    using func1_1_L16A group0_5_L5 by simp;
qed;

end