(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Real\_ZF.thy}*}

theory Real_ZF = EquivClass1 + Fol1 + Finite1 + Group_ZF:

text{*The goal of this theory file is to provide a contruction of the set of 
  real numbers. There are several ways construct real numbers. 
  Most common start from the rational numbers and use Dedekind cuts 
  or Cauchy sequences. Following
  our desire for adventure we will use neither 
  of these and instead formalize a 
  construction devised by Norbert A'Campo in 2003 (look for 
  "A natural construction for the real numbers" on www.scroogle.org  or your 
  favourite search engine). This contruction magically makes the real 
  numbers directly from the group (!) of integers by defining reals as
  eqivalence classes of slopes. 
  Slopes are certain type of maps from the set of integers into itself.
*}

section{*Basic Definitions*}

text{*We start from a definition of the set of slopes and the slope 
equivalence, using a generic group instead of the group of integers.
See the  Real\_ZF\_1\_L1 for more readable notation.
*}

constdefs
  SetOfSlopes :: "[i,i]\<Rightarrow>i"
  "SetOfSlopes(G,f) == {s \<in> G\<rightarrow>G.
  {f` < s`(f`<m,n>) , (GroupInv(G,f)`(f`<s`(m),s`(n)>)) >.
  <m,n> \<in> G\<times>G } \<in> Fin(G\<times>G)}"
  
  SlopesEquivRel :: "[i,i]\<Rightarrow>i"
  "SlopesEquivRel(G,f) == 
  {<s,r> \<in> SetOfSlopes(G,f) \<times> SetOfSlopes(G,f). 
  {f`<s`(n), GroupInv(G,f)`(r`(n))> . n\<in>G} \<in> Fin(G)}"

text{*This lemma provides more readable notation for the SetOfSlopes 
  definition. Not really intended to be used in proofs, but just to see the
  definition in the notation defined in the group0 locale.*}

lemma (in group0) Real_ZF_1_L1:
  shows "SetOfSlopes(G,f) = 
  {s \<in> G\<rightarrow>G. { s`(m\<cdot>n)\<cdot>(s`(m)\<cdot>s`(n))\<inverse>. <m,n> \<in> G\<times>G} \<in> Fin(G\<times>G) }"
proof -;
  show ?thesis using SetOfSlopes_def by simp;
qed;

text{*This lemma provides more readable notation for the SlopesEquivRel
  definition.*}

lemma (in group0) Real_ZF_1_L2:
  shows "SlopesEquivRel(G,f) = 
  {<s,r> \<in> SetOfSlopes(G,f) \<times> SetOfSlopes(G,f).
  {s`(n)\<cdot>(r`(n))\<inverse> . n \<in> G} \<in> Fin(G)}"
proof -;
  show ?thesis using SlopesEquivRel_def by simp;
qed;

text{*SlopesEquivRel is reflexive.*}

lemma (in group0) Real_ZF_1_L3: assumes A1: "G\<noteq>0"
  shows "refl(SetOfSlopes(G,f),SlopesEquivRel(G,f))"
proof -;
  have 
    "\<forall>s \<in> SetOfSlopes(G,f). {s`(n)\<cdot>(s`(n))\<inverse> . n \<in> G} = {e}"
  proof;
    fix s assume "s \<in> SetOfSlopes(G,f)"
    with A1 show 
      "{s`(n)\<cdot>(s`(n))\<inverse> . n \<in> G} = {e}"
      using SetOfSlopes_def apply_type group0_2_L6 by auto;
  qed;
  thus "refl(SetOfSlopes(G,f),SlopesEquivRel(G,f))"
    using group0_2_L2 SlopesEquivRel_def refl_def by simp; 
qed;

text{* SlopesEquivRel is symmetric.*}

lemma (in group0) Real_ZF_1_L4: 
  shows "sym(SlopesEquivRel(G,f))"
proof -;
  have "\<forall>x y. <x,y> \<in> SlopesEquivRel(G,f) \<longrightarrow>  
    <y,x> \<in> SlopesEquivRel(G,f)"
  proof (rule double_forall_implies);
    fix s r assume "<s,r> \<in> SlopesEquivRel(G,f)"
     then have 
      T1:"s \<in> SetOfSlopes(G,f)" and T2:"r \<in> SetOfSlopes(G,f)"
      and T3:"{s`(n)\<cdot>(r`(n))\<inverse> . n \<in> G} \<in> Fin(G)"
      using SlopesEquivRel_def by auto;
    from T1 T2 have T4:"s \<in> G\<rightarrow>G" and T5:"r \<in> G\<rightarrow>G"
      using SetOfSlopes_def by auto;
    from groupAsssum T3 have
      "GroupInv(G,f)``{s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} \<in> Fin(G)"
      using group0_2_T2 Finite1_L6A by blast;
    with T4 T5 T1 T2 show "<r,s> \<in> SlopesEquivRel(G,f)"
      using group0_2_L13 SlopesEquivRel_def
      by simp;
  qed;
  then show ?thesis using sym_def by simp;
qed;

text{*Proving transitivity of SlopesEquivRel is a little more involved. 
  We want to show that if the sets $\{s(n)\cdot (r(n))^{-1}\}_{n\in G}$ and 
   $\{r(n)\cdot (q(n))^{-1}\}_{n\in G}$ are finite, then the set
   $\{s(n)\cdot (Q(n))^{-1}\}_{n\in G}$ is also finite.
  The lemma below shows that 
  $$\{s(n)\cdot (Q(n))^{-1}\}_{n\in G} \subseteq 
  f(\{s(n)\cdot (r(n))^{-1}\}_{n\in G}\times \{r(n)\cdot (q(n))^{-1}\}_{n\in G}.$$
  Recall that in the group0 context $f:G\times G\rightarrow G$ is the group
  operation: $f(x,y)=x\cdot y$ .*} 

lemma (in group0) Real_ZF_1_L5:
  assumes A1:"s \<in> G\<rightarrow>G" and A2:"r \<in> G\<rightarrow>G" and A3:"q \<in> G\<rightarrow>G"
  shows 
  "{s`(n)\<cdot>(q`(n))\<inverse>. n\<in>G} \<subseteq> 
  f``({s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} \<times> {r`(n)\<cdot>(q`(n))\<inverse>. n\<in>G})"
proof -;
  from A1 A2 A3 have 
    T0:"\<forall>n\<in>G. s`(n)\<in>G \<and> r`(n)\<in>G \<and> q`(n)\<in>G"
   using apply_type by simp;
  have "\<forall>n\<in>G. s`(n)\<cdot>(q`(n))\<inverse> = 
    (s`(n)\<cdot>(r`(n))\<inverse>) \<cdot> (r`(n)\<cdot>(q`(n))\<inverse>)"
  proof;
    fix n assume "n\<in>G"
    with T0 have 
      T1:"s`(n)\<in>G" and T2:"r`(n)\<in>G" and T3:"q`(n)\<in>G"
      by auto;
    from T1 T2 T3 show
      "s`(n)\<cdot>(q`(n))\<inverse> = 
      (s`(n)\<cdot>(r`(n))\<inverse>) \<cdot> (r`(n)\<cdot>(q`(n))\<inverse>)"
      by (rule group0_2_L14);
  qed
  hence
    "{s`(n)\<cdot>(q`(n))\<inverse>. n\<in>G} = 
    {(s`(n)\<cdot>(r`(n))\<inverse>) \<cdot> (r`(n)\<cdot>(q`(n))\<inverse>). n\<in>G}"
    by simp;
  moreover have
     "{(s`(n)\<cdot>(r`(n))\<inverse>) \<cdot> (r`(n)\<cdot>(q`(n))\<inverse>). n\<in>G} \<subseteq> 
    f``({s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} \<times> {r`(n)\<cdot>(q`(n))\<inverse>. n\<in>G})"
    proof -;
      from T0 have 
	"\<forall>n\<in>G. <(s`(n)\<cdot>(r`(n))\<inverse>), (r`(n)\<cdot>(q`(n))\<inverse>)> \<in> G\<times>G"
	using group0_2_L5 group0_2_L1 monoid0.group0_1_L1
	by simp;
      hence
	"{<(s`(n)\<cdot>(r`(n))\<inverse>), (r`(n)\<cdot>(q`(n))\<inverse>)>. n\<in>G} \<subseteq> G\<times>G"
	by auto;
      moreover from groupAsssum have "f \<in> G\<times>G\<rightarrow>G"
	using IsAgroup_def IsAmonoid_def by simp;
      ultimately have 
	"{(s`(n)\<cdot>(r`(n))\<inverse>) \<cdot> (r`(n)\<cdot>(q`(n))\<inverse>). n\<in>G} =
	f``{<(s`(n)\<cdot>(r`(n))\<inverse>), (r`(n)\<cdot>(q`(n))\<inverse>)>. n\<in>G}"
	using func1_1_L16 by auto;
      moreover have 
	"f``({<(s`(n)\<cdot>(r`(n))\<inverse>), (r`(n)\<cdot>(q`(n))\<inverse>)>. n\<in>G}) \<subseteq> 
	f``({s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} \<times> {r`(n)\<cdot>(q`(n))\<inverse>. n\<in>G})"
	(is "f``(?A) \<subseteq> f``(?B)")
	using func1_1_L8 by blast;
      ultimately show ?thesis by simp;    
    qed;
  ultimately show ?thesis by simp;
qed;


text{*SlopesEquivRel is transitive.*}

lemma (in group0) Real_ZF_1_L6: 
assumes A1: 
  "\<langle>s, r\<rangle> \<in> SlopesEquivRel(G,f)" 
  "\<langle>r, q\<rangle> \<in> SlopesEquivRel(G,f)"
  shows "\<langle>s, q\<rangle> \<in> SlopesEquivRel(G,f)" 
proof -;
  from A1 have
    T1:"s \<in> G\<rightarrow>G" "r \<in> G\<rightarrow>G" "q \<in> G\<rightarrow>G"
    and T2:
    "{s`(n)\<cdot>(r`(n))\<inverse> . n \<in> G} \<in> Fin(G)"
    "{r`(n)\<cdot>(q`(n))\<inverse> . n \<in> G} \<in> Fin(G)"
    and T3:
    "s \<in> SetOfSlopes(G,f)" "q \<in> SetOfSlopes(G,f)"
    using SlopesEquivRel_def SetOfSlopes_def by auto;
  from T2 groupAsssum have
    "f``({s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} \<times> {r`(n)\<cdot>(q`(n))\<inverse>. n\<in>G})
    \<in> Fin(G)"
    using Finite1_L14 IsAgroup_def IsAmonoid_def
    Finite1_L6A by auto;
     moreover from T1 have "{s`(n)\<cdot>(q`(n))\<inverse>. n\<in>G} \<subseteq> 
    f``({s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} \<times> {r`(n)\<cdot>(q`(n))\<inverse>. n\<in>G})"
    using Real_ZF_1_L5 by simp;
  ultimately have "{s`(n)\<cdot>(q`(n))\<inverse>. n\<in>G} \<in> Fin(G)"
    using Fin_subset by simp;
  with T3 show ?thesis using SlopesEquivRel_def
    by simp;
qed;

    
text{*SlopesEquivRel is an equivalence relation on the set of slopes.*}

theorem (in group0) Real_ZF_1_T1: assumes A1: "G\<noteq>0"
  shows "equiv(SetOfSlopes(G,f),SlopesEquivRel(G,f))"
proof -;
  let ?r = "SlopesEquivRel(G,f)"
  have 
    "\<forall>x y z. (\<langle>x, y\<rangle> \<in> ?r  \<and>  \<langle>y, z\<rangle> \<in> ?r \<longrightarrow> \<langle>x, z\<rangle> \<in> ?r)"
    using Real_ZF_1_L6 by blast;
  then have "trans(SlopesEquivRel(G,f))"
     using Fol1_L2 by blast;
   with A1 show ?thesis using
      Real_ZF_1_L3 Real_ZF_1_L4 SlopesEquivRel_def equiv_def
     by auto;
qed;

text{*Next we define the set of slope classes. This will become the set of real
   numbers when we replace the generic group with the addidtive group of 
  integers. We also define an operation on slopes (as a meta-function) in the 
  natural way as induced by the group operation. This is called SlopeMetaOp1.
  SlopeOp1 is the ZF-function corresponding to SlopeMetaOp1
   SlopeClassOp1 is the projection of SlopeOp1
  on the classes of slopes (the quotient space). 
  This operation will become the addition of real numbers one 
  day.
  The future SlopeMetaOp2 will be the composition of slopes and 
  will be turned into multiplication of real numbers.
 *}

constdefs

  SlopeClasses :: "[i,i]\<Rightarrow>i"
  "SlopeClasses(G,f) == (SetOfSlopes(G,f)//SlopesEquivRel(G,f))"

  SlopeMetaOp1 :: "[i,i,i,i]\<Rightarrow>i"
  "SlopeMetaOp1(G,f,s,r) == {<n,m> \<in> G\<times>G. m = f`<s`(n),r`(n)>}"

  SlopeOp1 :: "[i,i]\<Rightarrow>i"
  "SlopeOp1(G,f) ==  
  {<c,d> \<in> (SetOfSlopes(G,f)\<times> SetOfSlopes(G,f))\<times> SetOfSlopes(G,f).
  d = SlopeMetaOp1(G,f,fst(c),snd(d))}"

  SlopeClassOp1 :: "[i,i]\<Rightarrow>i"
  "SlopeClassOp1(G,f) == 
  ProjFun2(SetOfSlopes(G,f),SlopesEquivRel(G,f),SlopeOp1(G,f))"

text{*The first thing to prove is that SlopeOp1 can be projected to the 
  quotient space. In romantic math we need to show that the definition of 
  the projection does not depend on the choice of class representants. 
  Here we show that SlopeOp1 is Congruent2, as defined in EquivClass1.thy.*};



end







