(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Finite1.thy}*}
theory Finite1 = Finite + func1:

section{*Some lemmas about the finite powerset*}

text{*Intersection of a collection is contained in every element 
of the collection.*}
lemma ZF11: assumes A: "A \<in> M" shows "\<Inter>M \<subseteq> A";
  proof
    fix x;
    assume A1: "x \<in> \<Inter>M"
    from A1 A show "x \<in> A" ..
  qed;

text{*Intersection of a nonempty collection M of subsets of 
X is a subset of X *}
lemma ZF12: assumes A1: "\<forall>A\<in> M. A\<subseteq>X" and A2:"M\<noteq>0" 
              shows "(\<Inter> M) \<subseteq> X";
proof -;
 from A2 have "\<forall> A\<in> M. (\<Inter> M \<subseteq> A)" using ZF11 by simp;
 with A1 A2 show  "(\<Inter> M) \<subseteq> X" by fast;
qed;

text{*Here we define a restriction of a collection of sets to a given set. 
In romantic math this is typically denoted $X\cap M$ and means 
$\{X\cap A : A\in M \} $. Note there is also restrict$(f,A)$ 
defined for relations in ZF.thy.*}

constdefs
  RestrictedTo :: "[i,i]\<Rightarrow>i" (infixl "{restricted to}" 70)
  "M {restricted to} X == {X \<inter>  A . A \<in> M}";

text{*In Topology\_ZF theory we consider induced topology that is 
obtained by taking a subset of a topological space. To show that a topology 
restricted to a subset is also a topology 
on that subset we may need a fact that 
if $T$ is a collection of sets and $A$ is a set then every finite collection
$\{ V_i \}$ is of the form $V_i=U_i \cap A$, where $\{U_i\}$ is a finite 
subcollection of $T$. This is one of those trivial 
facts that require suprisingly 
long formal proof. 
Actually, the need for this fact is avoided by requiring intersection 
two open sets to be open (rather than intersection of 
a finite number of open sets). 
Still, the fact is left here as an example of a proof by induction.*} 

text{*We will use Fin\_induct lemma from Finite.thy. 
First we define a property of
finite sets that we want to show. *}

constdefs
Prfin :: "[i,i,i]\<Rightarrow>o"
"Prfin(T,A,M) == ( (M = 0) | (\<exists>N\<in> Fin(T). \<forall>V\<in> M. \<exists> U\<in> N. (V = U\<inter>A)))";

text{*Now we show the main induction step in a separate lemma. This will make 
the proof of the theorem FinRestr below look short and nice. 
The premises of the ind\_step lemma are those needed by 
the main induction step in 
lemma Fin\_induct (see Finite.thy).*}

lemma ind_step: assumes  A: "\<forall> V\<in> TA. \<exists> U\<in> T. V=U\<inter>A" 
  and A1: "W\<in>TA" and A2:"M\<in> Fin(TA)" 
  and A3:"W\<notin>M" and A4: "Prfin(T,A,M)" 
  shows "Prfin(T,A,cons(W,M))";
proof (cases "M=0");
  assume A7: "M=0" show "Prfin(T, A, cons(W, M))";
  proof-;
    from A1 A obtain U where A5: "U\<in>T" and A6:"W=U\<inter>A" by fast;
    let ?N = "{U}"
    from A5 have T1: "?N \<in> Fin(T)" by simp;
    from A7 A6 have T2:"\<forall>V\<in> cons(W,M). \<exists> U\<in>?N. V=U\<inter>A" by simp;
    from A7 T1 T2 show "Prfin(T, A, cons(W, M))" 
      using Prfin_def by auto;   
  qed;
next;
  assume A8:"M\<noteq>0"; show "Prfin(T, A, cons(W, M))"
  proof-;
    from A1 A obtain U where A5: "U\<in>T" and A6:"W=U\<inter>A" by fast;
    from A8 A4 obtain N0 
      where A9: "N0\<in> Fin(T)" 
      and A10: "\<forall>V\<in> M. \<exists> U0\<in> N0. (V = U0\<inter>A)" 
      using Prfin_def by auto;
    let ?N = "cons(U,N0)";
    from A5 A9 have "?N \<in> Fin(T)" by simp;
    moreover from A10 A6 have "\<forall>V\<in> cons(W,M). \<exists> U\<in>?N. V=U\<inter>A" by simp;
    ultimately have "\<exists> N\<in> Fin(T).\<forall>V\<in> cons(W,M). \<exists> U\<in>N. V=U\<inter>A" by auto;
    with A8 show "Prfin(T, A, cons(W, M))" 
      using Prfin_def by simp;
  qed;
qed;

text{*Now we are ready to prove the statement we need.*}

theorem FinRestr0: assumes A: "\<forall> V\<in> TA. \<exists> U\<in> T. V=U\<inter>A"
  shows "\<forall> M\<in> Fin(TA). Prfin(T,A,M)";
proof;
  fix M;
  assume A1: "M\<in> Fin(TA)"
  have "Prfin(T,A,0)" using Prfin_def by simp;
  with A1 show "Prfin(T,A,M)" using ind_step by (rule Fin_induct);
qed

text{*This is a different form of the above theorem:*}

theorem ZF1FinRestr: 
  assumes A1:"M\<in> Fin(TA)" and A2: "M\<noteq>0" 
  and A3: "\<forall> V\<in> TA. \<exists> U\<in> T. V=U\<inter>A"
  shows "\<exists>N\<in> Fin(T). (\<forall>V\<in> M. \<exists> U\<in> N. (V = U\<inter>A)) & N\<noteq>0"
proof -;
  from A3 A1 have "Prfin(T,A,M)" using FinRestr0 by blast;
  then have "\<exists>N\<in> Fin(T). \<forall>V\<in> M. \<exists> U\<in> N. (V = U\<inter>A)" 
    using A2 Prfin_def by simp;
  then obtain N where 
    D1:"N\<in> Fin(T) & (\<forall>V\<in> M. \<exists> U\<in> N. (V = U\<inter>A))" by auto;
  with A2 have "N\<noteq>0" by auto;
  with D1 show ?thesis by auto;
qed;

(*
text{*Points that belong to disjoint sets can not belong to the same set.
  Not used anywhere.*}

lemma Finite1_L1: assumes A:"x\<in>U & y\<in>V & U\<inter>V=0" shows "x\<notin>V" and "y\<notin>U" 
proof -;
  from A show "x\<notin>V" by auto;
  from A show "y\<notin>U" by auto;
qed;
*)

text{*Purely technical lemma used in Topology\_ZF\_1 to show 
  that if a topology is$T_2$, then it is $T_1$.*}

lemma Finite1_L2: assumes A:"\<exists>U V. (U\<in>T & V\<in>T & x\<in>U & y\<in>V & U\<inter>V=0)"
  shows "\<exists>U\<in>T. (x\<in>U & y\<notin>U)"
proof -
  from A obtain U V where D1:"U\<in>T & V\<in>T & x\<in>U & y\<in>V & U\<inter>V=0" by auto;
  with D1 show ?thesis by auto;
qed;

text{*A collection closed with respect to taking a union of two sets 
is closed with respect to taking finite unions. Proof by induction with 
the induction step formulated in a separate lemma.*}

text{*The induction step:*}

lemma Finite1_L3_IndStep: 
  assumes A1:"\<forall>A B. ((A\<in>C & B\<in>C) \<longrightarrow> A\<union>B\<in>C)" 
  and A2: "A\<in>C" and A3:"N\<in>Fin(C)" and A4:"A\<notin>N" and A5:"\<Union>N \<in> C" 
  shows "\<Union>cons(A,N) \<in> C"
proof -
  have "\<Union> cons(A,N) = A\<union> \<Union>N" by blast;
  with A1 A2 A5 show ?thesis by simp;
qed;

text{*The lemma:*}

lemma Finite1_L3:
  assumes A1:"0 \<in> C" and A2:"\<forall>A B. ((A\<in>C & B\<in>C) \<longrightarrow> A\<union>B\<in>C)" and 
  A3:"N\<in> Fin(C)"
  shows "\<Union>N\<in>C"
proof -;
  from A1 have "\<Union>0 \<in> C" by simp;
  with A3 show "\<Union>N\<in> C" using Finite1_L3_IndStep by (rule Fin_induct)
qed;

text{*A collection closed with respect to taking a intersection of two sets 
is closed with respect to taking finite intersections. Proof by induction with 
the induction step formulated in a separate lemma. This is sligltly more 
involved than the union case in Finite1\_L3, because the intersection
of empty collection is undefined (or should be treated as such). 
This is why we define the property to be proven for finite sets as a constdef.
*}

constdefs 
  IntPr :: "[i,i]\<Rightarrow>o"
  "IntPr(T,N) == (N = 0 | \<Inter>N \<in> T)"

text{*The induction step.*}

lemma Finite1_L4_IndStep:
  assumes A1:"\<forall>A B. ((A\<in>T & B\<in>T) \<longrightarrow> A\<inter>B\<in>T)"
  and A2: "A\<in>T" and A3:"N\<in>Fin(T)" and A4:"A\<notin>N" and A5:"IntPr(T,N)"
  shows "IntPr(T,cons(A,N))"
proof (cases "N=0")
  assume A6:"N=0" show "IntPr(T,cons(A,N))";
  proof-
    from A6 A2 show "IntPr(T, cons(A, N))" using IntPr_def by simp
  qed;
  next;
  assume A7:"N\<noteq>0" show "IntPr(T, cons(A, N))"
  proof -;
    from A7 A5 A2 A1 have "\<Inter>N \<inter> A \<in> T" using IntPr_def by simp;
    moreover from A7 have "\<Inter>cons(A, N) = \<Inter>N \<inter> A" by auto;
    ultimately show "IntPr(T, cons(A, N))" using IntPr_def by simp
  qed;
qed;

text{*The lemma.*}

lemma Finite1_L4:  assumes A1:"\<forall>A B. ((A\<in>T & B\<in>T) \<longrightarrow> A\<inter>B\<in>T)"
  and A2:"N\<in>Fin(T)"
  shows "IntPr(T,N)"
proof -;
  have "IntPr(T,0)" using IntPr_def by simp;
  with A2 show "IntPr(T,N)"  using Finite1_L4_IndStep by (rule Fin_induct);
qed;

text{*Next is a restatement of the above lemma that 
does not depend on the IntPr 
meta-function. This is the version that will be most likely used outside
of this theory file.*}

lemma Finite1_L5: 
  assumes A1: "\<forall>A B. ((A\<in>T & B\<in>T) \<longrightarrow> A\<inter>B\<in>T)" 
  and A2:"N\<noteq>0" and A3:"N\<in>Fin(T)"
  shows "\<Inter>N \<in> T"
proof -;
  from A1 A3 have "IntPr(T,N)" using Finite1_L4 by simp;
  with A2 show ?thesis using IntPr_def by simp;
qed;

text{*The images of finite subsets by a meta-function are finite. 
For example in topology if we have a finite collection of sets, then closing 
each of them results in a finite collection of closed sets. 
The proof is by induction.*}

text{*The induction step:*}

lemma Finite1_L6_IndStep: 
  assumes A1:"\<forall>V\<in>B. K(V)\<in>C"
  and A2:"U\<in>B" and A3:"N\<in>Fin(B)" and A4:"U\<notin>N" and A5:"{K(V). V\<in>N}\<in>Fin(C)"
  shows "{K(V). V\<in>cons(U,N)} \<in> Fin(C)"
proof -;
  from A2 A1 A5 show ?thesis by simp;
qed;

text{*The lemma:*}

lemma Finite1_L6: assumes A1:"\<forall>V\<in>B. K(V)\<in>C" and A2:"N\<in>Fin(B)"
  shows "{K(V). V\<in>N}\<in>Fin(C)"
proof -;
  have "{K(V). V\<in>0}\<in>Fin(C)" by simp;
  with A2 show ?thesis using Finite1_L6_IndStep by (rule Fin_induct);
qed;

text{*The image of a finite set is finite.*}

lemma Finite1_L6A: assumes A1:"f\<in>X\<rightarrow>Y" and A2:"N\<in>Fin(X)"
  shows "f``(N) \<in> Fin(Y)"
proof -;
  from A1 have "\<forall>x\<in>X. f`(x) \<in> Y" 
    using apply_type by simp;
  moreover from A2 have "N\<in>Fin(X)" .;
  ultimately have "{f`(x). x\<in>N} \<in> Fin(Y)"
    by (rule Finite1_L6);
  with A1 A2 show ?thesis
    using FinD func1_1_L16 by simp;
qed;

text{*Next is an interesting set identity that is I thought I needed to 
prove sufficient 
conditions for a collection of sets to be a base for some topology. 
Not used there, but perhaps finds its use somewhere else. 
This should be in ZF1.thy.*}

lemma Finite1_L7: assumes A1:"C\<in>B" 
  shows "\<Union>C = \<Union> \<Union> {A\<in>B. \<Union>C = \<Union>A}"
proof;
  from A1 show "\<Union>C \<subseteq> \<Union>\<Union>{A \<in> B . \<Union>C = \<Union>A}" by blast;
  show "\<Union>\<Union>{A \<in> B . \<Union>C = \<Union>A} \<subseteq> \<Union>C"
  proof;
    fix x assume A3:"x\<in> \<Union>\<Union>{A \<in> B . \<Union>C = \<Union>A}" show "x\<in>\<Union>C";
    proof -;
      from A3 have "\<exists>K. \<exists>A. (x\<in>K & K\<in>A & A\<in>B & \<Union> C = \<Union> A)" by auto;
      then have "\<exists>K. \<exists>A. (x\<in>\<Union>A & \<Union> C = \<Union> A)" by auto;
      then obtain K A where "x\<in>\<Union>A & \<Union> C = \<Union> A" by auto;
      thus "x\<in>\<Union>C" by simp;
    qed
  qed
qed;

text{*Next we show an identity that is used to prove sufficiency 
of some condition for 
a collection of sets to be a base for a topology. Should be in ZF1.thy.*}

lemma Finite1_L8: assumes A1:"\<forall>U\<in>C. \<exists>A\<in>B. U = \<Union>A" 
  shows "\<Union>\<Union> {\<Union>{A\<in>B. U = \<Union>A}. U\<in>C} = \<Union>C"
proof;
  show "\<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A}) \<subseteq> \<Union>C" by blast;
  show "\<Union>C \<subseteq> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})"
  proof;
    fix x assume A2:"x \<in> \<Union>C" 
    show "x\<in> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})"
    proof -;
      from A2 obtain U where D1:"U\<in>C & x\<in>U" by auto;
      with A1 obtain A where D2:"A\<in>B & U = \<Union>A" by auto;
      from D1 D2 show "x\<in> \<Union>(\<Union>U\<in>C. \<Union>{A \<in> B . U = \<Union>A})" by auto;
    qed;
  qed;
qed;
   
text{*If an intersection a a collection is not empty, then the collection is
not empty. We are (ab)using the fact the the intesection of empty collection 
is defined to be empty and prove by contradiction. Should be in ZF1.thy*}

lemma Finite1_L9: assumes A1:"\<Inter>A \<noteq> 0" shows "A\<noteq>0"
proof (rule ccontr);
  assume A2:"\<not> A \<noteq> 0" from A2 A1 show False by simp;
qed;

text{*A property of sets defined by separation(?). Should be in ZF1.thy.*}

lemma Finite1_L10: assumes A1:"B\<subseteq>A" 
  shows "{x \<in> A. p(x)} \<inter> B = {x\<in>B. p(x)}"
proof -;
  from A1 show ?thesis by blast;
qed;

text{*Another property of sets defined by separation(?). 
  Should be in ZF1.thy.*}

lemma Finite1_L11: shows "{<x,y,z> \<in> X\<times>Y\<times>Z. D(x,y,z)} \<subseteq> X\<times>Y\<times>Z"
proof -;
  show ?thesis by auto;
qed;
  
text{*Cartesian product of finite sets is finite.*}

lemma Finite1_L12: assumes A1:"A \<in> Fin(A)" and A2:"B \<in> Fin(B)"
  shows "A\<times>B \<in> Fin(A\<times>B)"
proof -;
   have T1:"\<forall>a\<in>A. \<forall>b\<in>B. {<a,b>} \<in> Fin(A\<times>B)" by simp;
   have "\<forall>a\<in>A. {{<a,b>}. b \<in> B} \<in> Fin(Fin(A\<times>B))"
   proof;
     fix a assume A3:"a \<in> A"
     with T1 have  "\<forall>b\<in>B. {<a,b>} \<in> Fin(A\<times>B)" 
       by simp;
     moreover from A2 have "B \<in> Fin(B)" .;
     ultimately show "{{<a,b>}. b \<in> B} \<in> Fin(Fin(A\<times>B))"
       by (rule Finite1_L6)
   qed;
   then have "\<forall>a\<in>A. \<Union> {{<a,b>}. b \<in> B} \<in> Fin(A\<times>B)"
     using Fin_UnionI by simp;
   moreover have 
     "\<forall>a\<in>A. \<Union> {{<a,b>}. b \<in> B} = {a}\<times> B" by blast;
   ultimately have "\<forall>a\<in>A. {a}\<times> B \<in> Fin(A\<times>B)" by simp;
   moreover from A1 have "A \<in> Fin(A)" . 
   ultimately have "{{a}\<times> B. a\<in>A} \<in> Fin(Fin(A\<times>B))"
     by (rule Finite1_L6);
   then have "\<Union>{{a}\<times> B. a\<in>A} \<in> Fin(A\<times>B)"
     using Fin_UnionI by simp;
   moreover have "\<Union>{{a}\<times> B. a\<in>A} = A\<times>B" by blast;
   ultimately show ?thesis by simp;
qed;

text{*A finite subset is a finite subset of itself.*}

text{*We define the characterisic meta-function that is an identity
on a set and assigns a default value everywhere else.*}

constdefs
  Characteristic :: "[i,i,i]\<Rightarrow>i"
  "Characteristic(A,default,x) == (if x\<in>A then x else default)";

lemma Finite1_L13: 
  assumes A1:"A \<in> Fin(X)" shows "A \<in> Fin(A)"
proof (cases "A=0");
  assume "A=0" then show "A \<in> Fin(A)" by simp;
  next;
  assume A2: "A\<noteq>0" then obtain c where D1:"c\<in>A" 
    by auto;
  then have "\<forall>x\<in>X. Characteristic(A,c,x) \<in> A"
    using Characteristic_def by simp;
  moreover from A1 have "A \<in> Fin(X)" .;
  ultimately have 
    "{Characteristic(A,c,x). x\<in>A} \<in> Fin(A)" 
    by (rule Finite1_L6);
  moreover from D1 have 
    "{Characteristic(A,c,x). x\<in>A} = A"
    using Characteristic_def by simp;
  ultimately show "A \<in> Fin(A)" by simp;
qed;

text{*Cartesian product of finite subsets is a finite subset of 
  cartesian product.*}

lemma Finite1_L14: assumes A1:"A \<in> Fin(X)" and A2:"B \<in> Fin(Y)"
  shows "A\<times>B \<in> Fin(X\<times>Y)"
proof -;
  from A1 A2 have "A\<times>B \<subseteq> X\<times>Y" using FinD by auto;
  then have "Fin(A\<times>B) \<subseteq> Fin(X\<times>Y)" using Fin_mono by simp;
  moreover from A1 A2 have "A\<times>B \<in> Fin(A\<times>B)"
    using Finite1_L13 Finite1_L12 by simp;
  ultimately show ?thesis by auto;
qed;
  
end