/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import java.io.InputStream;
import java.io.IOException;
import javax.mail.Part;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.internet.MimeMultipart;
import org.apache.log4j.Logger;

/**
 * Encapsulation of information about an attachment to a mail, 
 * to be used in the mailbox model
 */
public class MailPart
{
    private final Logger log = Logger.getLogger(MailPart.class);

    private String fileName;
    private String contentType;
    private String size;
    private boolean inline;
    private boolean text;
    private boolean anonymousText;
    private Object content;
    //private InputStream inputStream;
    private Part part;
    private boolean isMultipart;
    private MailPart[] subParts;

    public MailPart(Part part) 
	throws ModelException
    {
	try {
	    if (part == null)
		throw new IllegalArgumentException("parameter part is empty");
            this.part = part;
	    
	    log.debug("MailPart() reading part content");
	    fileName = part.getFileName();
	    contentType = part.getContentType();
	    int i_size = part.getSize();
	    if (i_size < 1024)
		size = Integer.toString(i_size) + " b";
	    else
		size = Integer.toString(i_size / 1024) + " Kb";

	    String disposition = part.getDisposition();
	    if (disposition == null ||
		disposition.equalsIgnoreCase(Part.ATTACHMENT))
		inline = false;
	    else
		inline = true;
	    text = part.isMimeType("text/*");

	    isMultipart = part.isMimeType("multipart/*");

	    if (log.isDebugEnabled())
		log.debug("MailPart() read headers: type " + contentType
			  + ", inline? " + inline
			  + ", text? " + text
			  + ", isMultipart?" + isMultipart);
	    
	    if (isText()) {
		log.debug("MailPart() is of type text, getting content");
		content = part.getContent();
		anonymousText = (fileName == null || fileName.equals(""));
	    } else {
		anonymousText = false;
	    }

//	    if (anonymousText)
//            	inputStream = null;
//	    else {
//		log.debug("MailPart() now getting input stream on content");
//		inputStream = part.getInputStream();
//	    }

	    if (isMultipart) {
		MimeMultipart mm;
		try {
		    mm = (MimeMultipart)part.getContent();
		}
		catch (IOException e) {
		    throw new ModelException("internal error reading part content", e);
		}
		catch (MessagingException e) {
		    throw new ModelException("internal error reading part content", e);
		}
		subParts = new MailPart[mm.getCount()];
		for (int i = 0; i < mm.getCount(); i++)
		    subParts[i] = new MailPart(mm.getBodyPart(i));
	    }
	    else
		subParts = new MailPart[0];

	}
	catch (Exception e) {
	    log.error("error initializing MailPart", e);
	    throw new ModelException("error initializing MailPart: " + e.toString(), e);
	}

	log.debug("MailPart() successful");
    }

    public String getFileName() {
	return fileName;
    }

    public String getContentType() {
	return contentType;
    }

    public String getSize() {
	return size;
    }

    public boolean isInline() {
	return inline;
    }

    public boolean isText() {
	return text;
    }

    public boolean isAnonymousText() {
	return anonymousText;
    }

    public boolean isMultipart() {
	return isMultipart;
    }

    public MailPart[] getSubParts() {
	return subParts;
    }

    /**
     * If the part is of type text, returns the text itself.
     * Otherwise, returns the empty string.
     */
    public String getText() {
	if (isText())
	    return (String)content;
	else
	    return "";
    }

	// implementation note: we don't want to open the input
	// stream in the constructor in all cases.
    public InputStream getInputStream() throws IOException, MessagingException
    {
	if (! anonymousText)
		return part.getInputStream();
	else
		return null;
    }
}




