/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import java.util.Comparator;
import java.util.Iterator;
import java.util.TreeSet;
import java.util.Vector;
import javax.mail.Address;
import javax.mail.internet.InternetAddress;

import org.apache.log4j.Logger;


/**
 * Part of the client model.
 * Contains information about contacts.
 * Implementation note: this must be a bean, as it will be serialized as XML
 */
public class ContactList
{
    public final static String KEY = "ContactList";

    private final Logger log = Logger.getLogger(ContactList.class);
    // user types used in this class (including superclasses of used types, and 
    // classes needed by custom classes that this class aggregates !
    public final static Class[] TYPES = 
    {ContactList.class,Contact.class,
     Vector.class,
     InternetAddress.class, Address.class};


    private long key;
    private Vector contacts;

    public ContactList() {
	key = System.currentTimeMillis();
	contacts = new Vector();
	if (log.isDebugEnabled())
	    log.debug("ContactList() created" + this.toString());
    }

    public Object clone() throws CloneNotSupportedException {
	if (log.isDebugEnabled())
	    log.debug("clone() called on " + this.toString());
	return super.clone();
    }

    // setter and getter according to beans API -
    // should actually not use these to access the contacts
    public void setContacts(Vector contacts) {
	if (log.isDebugEnabled())
	    log.debug("setContacts() called, parameter has " 
		      + (contacts != null ? contacts.size() : 0) + " elements.");
	this.contacts = contacts;
    }
    public Vector getContacts() {
	if (log.isDebugEnabled())
	    log.debug("getContacts() called on " + this.toString());
	return (Vector)contacts.clone();
    }
    public void setKey(long newKey) {
	key = newKey;
    }
    public long getKey() {
	return key;
    }
    
    public int getLength() {
	return contacts.size();
    }

    public Contact getContact(int index) {
	return (Contact)contacts.elementAt(index);
    }
    public Iterator getIterator() {
	if (log.isDebugEnabled())
	    log.debug("getIterator() called");

	return contacts.iterator();
    }

    public void deleteContact(int index)
    {
	if (log.isDebugEnabled())
	    log.debug("deleteContact() called for index " + index);
	
	if (index < 0 || index > (contacts.size() - 1))
	    throw new IllegalArgumentException("index " + index + " out of bounds");
	contacts.remove(index);
    }

    public void setContact(int contactNr, Contact contact)
    {
	if (contactNr < 0 || contactNr > getLength() - 1)
	    throw new IllegalArgumentException("contactNr " + contactNr + " is out of bounds");
	if (contact == null)
	    throw new IllegalArgumentException("empty contact");

	contacts.setElementAt(contact, contactNr);
    }

    public void addContact(Contact newContact) 
	throws ContactDuplicateException
    {
	if (newContact == null)
	    throw new IllegalArgumentException("parameter newContact may not be empty");
	if (log.isDebugEnabled())
	    log.debug("addContact() called");
	

	// check if contact is already there,
	// based on e-mail equivalence
	String key = newContact.getEmailAddress().getAddress();
	if (key != null) {	    
	    Iterator it = contacts.iterator();
	    int i = 0;
	    while (it.hasNext()) {
		i++;
		Contact currentContact = (Contact)it.next();
		if (key.equals(currentContact.getEmailAddress().getAddress()))
		    throw new ContactDuplicateException(newContact, currentContact, i);
	    }
	}
	contacts.add(newContact);
    }

    public boolean isEmpty() {
	if (log.isDebugEnabled())
	    log.debug("isEmpty() called, returning " + (contacts.size() == 0));
	return contacts.size() == 0;
    }


    public void sort(Comparator c) {
	if (! isEmpty()) {
	    TreeSet newbs = new TreeSet(c);
	    newbs.addAll(contacts);
	    contacts = new Vector(newbs);
	}
    }

    /**
     * utility method ;)
     */
    public String getContactEmails(int[] indices) 
    {
	StringBuffer ret = new StringBuffer();
	if (indices != null) {
	    boolean isFirst = true;
	    for (int i = 0; i < indices.length; i++) {
		Contact c = getContact(indices[i]);
		if (! isFirst)
		    ret.append(", ");
		if (c != null)
		    ret.append(c.getEmailAddress().toString());
		isFirst = false;
	    }
	}
	return ret.toString();
    }

    /** for debugging */
    public String toString() {
	StringBuffer ret = new StringBuffer();
	ret.append("ContactList ")
	    .append(key)
	    .append(" with " + getLength() + " elements");
	return ret.toString();
    }
}
