/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller.action;

import java.util.Collections;
import java.util.Map;

// servlet api
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpServletRequest;

// cocoon
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.cocoon.acting.AbstractAction;

// for logging use log4j
import org.apache.log4j.Logger;

import net.jatec.ironmailer.controller.AbstractWorkerBean;
import net.jatec.ironmailer.controller.ControllerException;
import net.jatec.ironmailer.controller.MailWorkerBean;
import net.jatec.ironmailer.framework.HttpSessionManager;
import net.jatec.ironmailer.framework.StringTools;

/**
 * Action to verify if request lives within a valid session. 
 * If not, the action fails, and the application can inform the user she needs to login.
 *
 * @author jwk
 */
public class SessionCheckAction extends AbstractAction
{
    private final Logger log = Logger.getLogger(SessionCheckAction.class);

    /**
     * Interface implementation
     * @throws Exception according to interface, but should never happen.
     * If there is some error, null is returned and the exception is logged.
     */
    public Map act(Redirector redirector, SourceResolver resolver, 
		   Map objectModel, String source,
		   Parameters par) 
	throws Exception 
    {
        HttpServletRequest req =(HttpServletRequest)objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);	

	try {
	    log.debug("act() called");

	    // Check that a session exists. If not, action fails
	    if (! HttpSessionManager.getInstance().isInValidSession(req)) {
		log.debug("act() : no session, failing");
		return null;
	    }

	    // Insure that some kind of worker bean exists in the session
	    // If not, there is something wrong: for example the servlet
	    // container threw up the worker bean but kept the session
	    HttpSession session = (HttpSession)req.getSession(false);
	    AbstractWorkerBean wb = (AbstractWorkerBean)session.getAttribute("workerBean");
	    if (wb == null) {
		log.warn("in a valid session but no worker bean found - probably unclean restart from servlet container");
		return null;
	    }

	    log.debug("act() : session OK, returning");
	    return Collections.EMPTY_MAP;
	}
	catch (Throwable t) {
	    // All errors result in the action failing, and in the setting
	    // of error fields in the request for later display by the GUI
	    log.error("act() got some unexpected error, probably due to a setup error", t);
	    log.debug("act() error fiels: " + t.toString() + ", message " + t.getMessage() + ", cause " + t.getCause());
	    req.setAttribute("error.content", t.toString());
	    req.setAttribute("error.stack", StringTools.getStackTraceAsString(t));		
	    return null;
	}
	
    }
}

