/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;

import java.util.Enumeration;
import java.util.Hashtable;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.FolderNotFoundException;
import javax.mail.MessagingException;
import javax.mail.Message;
import javax.mail.Store;
import org.apache.log4j.Logger;

import net.jatec.ironmailer.framework.MailTools;
import net.jatec.ironmailer.model.ApplicationConfiguration;
import net.jatec.ironmailer.model.MailboxOverview;
import net.jatec.ironmailer.model.MailFolder;
import net.jatec.ironmailer.model.MailFolderHeader;
import net.jatec.ironmailer.model.MailMessage;
import net.jatec.ironmailer.model.MessageHeader;
import net.jatec.ironmailer.model.ModelException;

/**
 * Good stuff to handle mailbox
 */
public class MailboxController
{
    private final static Logger log = Logger.getLogger(MailboxController.class);
    private MailboxOverview mailboxOverview;
    private Hashtable mailFolders;
    private Hashtable mailMessages;
    private ApplicationConfiguration applicationConfiguration;
    private Store store;

    public MailboxController(Store store, ApplicationConfiguration applicationConfiguration) 
	throws ControllerException
    {
	this.store = store;
	this.applicationConfiguration = applicationConfiguration;
	// initialize mailbox info
	setMailboxOverview(true);

    }

    public MailboxOverview getMailboxOverview() 
    {
	return mailboxOverview;
    }


    public void setMailboxOverview(boolean refresh)
	throws ControllerException
    {
	log.debug("setMailboxOverview() called, refresh=" + refresh);
	try {
	    if (mailboxOverview == null ||
		refresh) {
		// construct mailboxOverview
		log.debug("setMailboxOverview() creating overview");

		// if refresh mode, close any previous cache
		if (refresh)
		    closeCache(true);

		mailboxOverview = 
		    (new MailboxFetcher()).createMailboxOverview(store, applicationConfiguration);
		log.debug("setMailboxOverview() successfully created overview");
	    }
	} 
	catch (UserInstallationException e) {
	    throw new ControllerException("user installation is fishy", e);
	}
	catch (MessagingException e) {
	    throw messaging2controller("error in setMailboxOverview", e);
	}
    }


    public void closeCache(boolean expunge) {
	// close the mail folders
	if (mailFolders != null) {
	    for (Enumeration e = mailFolders.elements();
		 e.hasMoreElements();) {
		try {
		    ((MailFolder)e.nextElement()).close(true);
		}
		catch (FolderNotFoundException ignored) {
		    // handle special case where folder was deleted so
		    // it can't be closed. It's simpler to catch the exception
		    // here and ignore it, rather than complicate the
		    // cache handling.
		}
		catch (MessagingException ignored) {
		    log.warn("unexpected exception while closing cache (ignoring): " + e);
		}
	    }
	    mailFolders = null;
	}
	mailMessages = null;
	mailboxOverview = null;

    }


    /**
     * To be called by controller action, when details for a folder are 
     * requested
     */ 
    public void setMailFolder(int nr, boolean forceRefresh)
	throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("setMailFolder() called for nr " + nr + ", forceRefresh? " + forceRefresh);

	if (mailFolders == null)
	    mailFolders = new Hashtable();
	
	if (! forceRefresh &&
	    ((MailFolder)mailFolders.get(new Integer(nr)) != null)) {
	    log.debug("setMailFolder() not doing anything, folder is in cache");
	    // already in cache
	    return;
	}

	try {
	    if (log.isDebugEnabled())
		log.debug("setMailFolder() creating folder information");

	    MailFolderHeader mfh = getMailboxOverview().getMailFolderHeader(nr);
	    // sanity check
	    if (mfh == null) throw new ControllerException("something fishy: could not retrieve mail folder header for number " + nr);

	    if (log.isDebugEnabled())
		log.debug("setMailFolder() got mail folder header");

	    MailFolder mf = new MailFolder(mfh);
	    // sanity check
	    if (mf == null) throw new ControllerException("something fishy: could not instantiate MailFolder object for number " + nr);

	    if (log.isDebugEnabled())
		log.debug("setMailFolder() instantiated MailFolder object");

	    mailFolders.put(new Integer(nr), mf);

	    if (log.isDebugEnabled())
		log.debug("setMailFolder() successful for folder " 
			  + mf.getMailFolderHeader().getName() 
			  + ", isOpen? " + mf.getMailFolderHeader().getFolder().isOpen());
	} 
	catch (MessagingException e) {
	    throw messaging2controller("could not retrieve folder information for nr " + nr, e);
	}
	catch (ModelException e) {
	    throw new ControllerException("error initializing model for nr " + nr, e);
	}

    }


    /**
     * Accessor for GUI - will only work if controller previously called
     * setMailFolder(), otherwise returns null
     */
    public MailFolder getMailFolder(int nr) 
    {
	if (log.isDebugEnabled())
	    log.debug("getMailFolder() called for nr " + nr);
	MailFolder ret;

	if (mailFolders == null) {
	    log.warn("getMailFolder() returning null for nr " + nr + ", this is probably a setup error: check the GUI");
	    ret = null;
	}
	else
	    ret = (MailFolder)mailFolders.get(new Integer(nr));
	
	if (log.isDebugEnabled())
	    log.debug("getMailFolder() returning " + ret);

	return ret;
    }
 
    /** 
     * utility method, for whoever needs to retrieve a message header
     * @param folderNr - index of the folder containing the message
     * @param messageNr - index (starting at 0) of the desired message
     * @throws ControllerException when header does not exist
     */
    public MessageHeader getMessageHeader(int folderNr, int messageNr)
	throws ControllerException
    {
	MailFolder mf = getMailFolder(folderNr);
	if (mf == null)
	    throw new ControllerException("getMessageHeader() internal error: mail folder for nb " + folderNr + " is not there." + folderNr, null);
	MessageHeader header = (MessageHeader)mf.getMessageHeaders().get(messageNr);
	if (header == null)
	    throw new ControllerException("unexpected error: no header for message nr " + messageNr, null);

	if (log.isDebugEnabled())
	    log.debug("getMessageHeader() successful reading from folder " 
		      + mf.getMailFolderHeader().getName());

	return header;
    }


    /**
     * To be called by controller action, when content of a message is 
     * requested
     */ 
    public void setMessage(int folderNr, int p_messageNr)
	throws ControllerException
    {
	log.debug("setMessage() called with parameters folderNr=" + folderNr + " and p_messageNr=" + p_messageNr);
	
	// message are traditionally numbered starting at 1, but we need 0
	int messageNr = p_messageNr - 1;

	MailFolder mf = getMailFolder(folderNr);
	if (mf == null)
	    throw new ControllerException("setMessage() got illegal folder selected, number " + folderNr, null);
	if (mailMessages == null)
	    mailMessages = new Hashtable();

	// see if already in cache
	Hashtable messagesInFolder = (Hashtable)mailMessages.get(mf);
	if (messagesInFolder != null &&
	    (MailMessage)messagesInFolder.get(new Integer(messageNr)) != null)
	    return;

	// not yet in cache, fetch message and store
	log.debug("setMessage() fetching fresh message...");
	try {
	    MessageHeader header = getMessageHeader(folderNr, messageNr);
	    MailMessage mm = new MailMessage(header);
	    
	    // got the message, put in cache
	    if (messagesInFolder == null) {
		// first message retrieved for folder
		messagesInFolder = new Hashtable();
	    }
	    messagesInFolder.put(new Integer(messageNr), mm);
	    mailMessages.put(new Integer(folderNr), messagesInFolder);
	}
	catch (MessagingException e) {
	    throw messaging2controller("could not retrieve message for folderNr " + folderNr + ", messageNr=" + messageNr, e);
	}
	// when message if first fetched, need to update folder information
	// (flag "seen" may have changed)
	setMailFolder(folderNr, true);

	log.debug("setMessage() successful");
    }

    /**
     * Accessor for GUI - will only work if controller previously called
     * setMessage()
     */
    public MailMessage getMailMessage(int folderNr, int p_messageNr)
    {
	log.debug("getMailMessage() called with folderNr=" + folderNr
		  + " and messageNr=" + p_messageNr);
	// message are traditionally numbered starting at 1, but we need 0
	int messageNr = p_messageNr - 1;
	MailMessage ret = null;

	if (mailMessages != null) {
	    Hashtable messagesInFolder = 
		(Hashtable)mailMessages.get(new Integer(folderNr));
	    if (messagesInFolder != null)
		ret = (MailMessage)messagesInFolder.get(new Integer(messageNr));
	    else
		log.warn("getMailMessage() internal setup error, messagesInFolder is null");
	} else {
	    log.warn("getMailMessage() internal setup error, mailMessages is null");
	}
	log.debug("getMailMessage() returning, is null? " + (ret == null));
	return ret;
    }


    public void setFlag(int folderNr, int gui_messageNr, Flags.Flag flag, boolean value) throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("setFlag() called with parameters folderNr=" + folderNr + ", gui_messageNr=" + gui_messageNr + ", flag to set=" + flag + ", value=" + value);
	
	try {
	    MessageHeader mh = getMessageHeader(folderNr, gui_messageNr - 1);
	    Message m = mh.getMessage();
	    m.setFlag(flag, value);
	}
	catch (MessagingException e) {
	    throw messaging2controller("error setting flag " + flag, e);
	}
	
	// refresh folder information
	setMailFolder(folderNr, true);

	log.debug("setFlag() done");
    }


    /**
     * To be called by controller action, when flag of a message is 
     * to be set
     */ 
    public void setMessagesFlag(int folderNr, int[] msgIndices, String flag, boolean value)
	throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("setMessagesFlag() called with parameters folderNr=" + folderNr + ", flag to set: " + flag + "nb of messages handled " + msgIndices.length);
	
	if (msgIndices.length > 0) {
	    Flags.Flag mflag = MailTools.getStringAsFlag(flag);
	    if (mflag == null)
		throw new ControllerException("illegal flag setting: no support for flag called " + flag, null);

	    try {
		for (int i = 0; i < msgIndices.length; i++) {
		    int p_messageNr = msgIndices[i];
		    int messageNr = p_messageNr - 1;
		    // could use method setFlag() from above, for copy the code for efficiency
		    MessageHeader mh = getMessageHeader(folderNr, messageNr);
		    Message m = mh.getMessage();
		    m.setFlag(mflag, value);
		}
	    }
	    catch (MessagingException e) {
		throw messaging2controller("error setting flag " + flag, e);
	    }
	
	    // refresh folder information
	    setMailFolder(folderNr, true);
	}
	log.debug("setMessageFlag() done");
    }


    public void copyOrMoveMessages(int fromFolderNr, int toFolderNr,
				   int[] msgIndices, boolean isCopy)
	throws ControllerException
    {
	// Sanity check
	if (fromFolderNr < 0) throw new ControllerException("illegal usage: fromFolderNr may not be negative");
	if (toFolderNr < 0) throw new ControllerException("illegal usage: toFolderNr may not be negative");
	if (msgIndices == null) throw new ControllerException("illegal usage: no msgIndices provided");

	log.debug("copyOrMoveMessages() called from folder " + fromFolderNr + " to folder nr " + toFolderNr + " isCopy?" + isCopy + " for " + msgIndices.length + " messages");


	// Get messages to copy/move
	MailFolder thisMailFolder = getMailFolder(fromFolderNr);
	Message[] msgs = new Message[msgIndices.length];
	for (int i = 0; i < msgIndices.length; i++) {
	    // messages numbered starting at 1; array indexed at 0 so do -1
	    int index = msgIndices[i] - 1;
	    log.debug("copyOrMoveMessages() marking message nr " + index);
	    msgs[i] = thisMailFolder.getMessageHeader(index).getMessage();
	}

	Folder thisFolder = thisMailFolder.getMailFolderHeader().getFolder();
	Folder destinationFolder = getMailboxOverview().getMailFolderHeader(toFolderNr).getFolder();

	if (isCopy) {
	    log.debug("copyOrMoveMessages() performing copy of " + msgs.length + " messages");
	    try {
		thisFolder.copyMessages(msgs, destinationFolder);
	    }
	    catch (MessagingException e) {
		throw messaging2controller("error copying messages", e);
	    }
	} 
	else {
	    log.debug("copyOrMoveMessages() performing move of " + msgs.length + " messages");
	    FolderManager.moveMessages(thisFolder, destinationFolder, msgs);
	}

	// refreshing of information, in any case: 
	// overview info
	// source and destination folder
	setMailboxOverview(true);
	setMailFolder(fromFolderNr, true);
	setMailFolder(toFolderNr, true);
    }

    public static ControllerException messaging2controller(String info, MessagingException e) {
	if (log.isDebugEnabled())
	    log.debug("messaging2controller: got MessagingException " + e);
	
	// default transformed exception
	ControllerException ret = new ControllerException(info, e);

	// try to de-obfuscate JavaMail exception handling
	Exception child = e.getNextException();
	if (child != null) {
	    if (child instanceof java.io.IOException)
		ret = new BackendException(info, child);
	}

	return ret;
    }
}

