/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;

import java.net.URL;
import javax.mail.Flags;
import javax.mail.Store;
import javax.mail.MessagingException;
import javax.mail.search.SearchTerm;

import org.apache.log4j.Logger;
import org.apache.avalon.framework.parameters.Parameters;

import net.jatec.ironmailer.controller.action.ActionDispatcher;
import net.jatec.ironmailer.model.ApplicationConfiguration;
import net.jatec.ironmailer.model.ComposeInfo;
import net.jatec.ironmailer.model.ComposeReference;
import net.jatec.ironmailer.model.Contact;
import net.jatec.ironmailer.model.ContactDuplicateException;
import net.jatec.ironmailer.model.ContactList;
import net.jatec.ironmailer.model.MailboxOverview;
import net.jatec.ironmailer.model.MailPart;
import net.jatec.ironmailer.model.MailFolder;
import net.jatec.ironmailer.model.MailFolderHeader;
import net.jatec.ironmailer.model.MailMessage;
import net.jatec.ironmailer.model.MessageHeader;
import net.jatec.ironmailer.model.SearchMessagesResults;
import net.jatec.ironmailer.model.SendInfo;
import net.jatec.ironmailer.model.user.UserPreferences;

/**
 * Main class for controller.
 * Responsabilities: 
 * - coordinate the controller layer.
 * - provide setter/accessors to the GUI.
 * - handle errors.
 */
public class MailWorkerBean extends AbstractWorkerBean
{
    private final Logger log = Logger.getLogger(MailWorkerBean.class);
    private int testCtr;
    private String testString;
    private ApplicationConfiguration applicationConfiguration;
    private UserPreferences userPreferences;
    private boolean userPreferencesAvailable;
    private ComposeInfo lastComposeInfo;
    private MailSender mailSender;
    private SendInfo sendInfo;
    private GeneralStore userConfStore;
    private MailboxController mailboxController;
    private AttachmentController attachmentController;
    private FolderManager folderManager;
    private ActionDispatcher actionDispatcher;
    private Throwable lastException;
    private String firstPage;
    private ContactListController contactListController;
    private UserInput userInput;
    private SearchMessagesResults searchMessagesResults;

    public void initialize(ServerConnection serverConnection, URL contextURL, Parameters par)
	throws IllegalArgumentException, Exception
    {
	super.initialize(serverConnection, contextURL, par);

	if (log.isDebugEnabled())
	    log.debug("initialize() local initializations");

	// initialize applicationConfiguration
	applicationConfiguration = new ApplicationConfiguration(contextURL, par);

	// initialize mailSender
	mailSender = new MailSender(serverConnection, applicationConfiguration);

	// initialize folderManager
	Store store = ((MailServerConnection)serverConnection).getStore();
	folderManager = new FolderManager(store);

	// initialize mailboxController
	mailboxController = new MailboxController(store, applicationConfiguration);
	attachmentController = new AttachmentController(mailboxController);

	if (log.isDebugEnabled())
	    log.debug("initialize() setting up user preferences");
	// initialize user preferences
	// if it doesn't work, it's not fatal but make a note
	try {
	    initUserPreferences();
	    userPreferencesAvailable = true;
	} 
	catch (ControllerException e) {
	    log.warn("user preferences not available", e);
	    userPreferencesAvailable = false;
	}

	// initialize contacts manager
	contactListController = new ContactListController((MailServerConnection)getServerConnection(), applicationConfiguration);

	actionDispatcher = new ActionDispatcher();
    }

    private void closeCache(boolean expunge) 
	throws MessagingException
    {
	if (log.isDebugEnabled())
	    log.debug("closeCache() called with expunge " + expunge);

	if (mailboxController != null)
	    mailboxController.closeCache(expunge);

	if (lastComposeInfo != null)
	    lastComposeInfo.destroy();
    }

    public void setTestCtr(int i) {
	testCtr = i;
    }

    public int getTestCtr() {
	return testCtr;
    }

    public void setTestString(String s) {
	testString = s;
    }

    public String getTestString() {
	return testString;
    }

    public void close()
	throws Exception
    {
	if (log.isDebugEnabled())
	    log.debug("close() specific closings");

	if (userConfStore != null)
	    userConfStore.close();

	contactListController.close();

	closeCache(true);

	// after cleaning ourselves up, let super class do cleanups
	// (such as release connections)
	super.close();
    }

    public ActionDispatcher getActionDispatcher() {
	return actionDispatcher;
    }

    /**
     * Call this to force refresh of mailbox
     */
    public void setMailboxOverview(boolean refresh)
	throws ControllerException
    {
	try {
	    mailboxController.setMailboxOverview(refresh);
	}
	catch (ControllerException e) {
	    doError(e);
	}
    }

    /**
     * Accessor for GUI
     */
    public MailboxOverview getMailboxOverview() 
    {
	return mailboxController.getMailboxOverview();
    }

    /**
     * To be called by controller action, when details for a folder are 
     * requested
     */ 
    public void setMailFolder(int nr, boolean forceRefresh)
	throws ControllerException
    {
	mailboxController.setMailFolder(nr, forceRefresh);
    }

    /**
     * Accessor for GUI - will only work if controller previously called
     * setMailFolder(), otherwise returns null
     */
    public MailFolder getMailFolder(int nr) 
    {
	return mailboxController.getMailFolder(nr);
    }
 
    /** 
     * utility method, for whoever needs to retrieve a message header
     * @param folderNr - index of the folder containing the message
     * @param messageNr - index (starting at 0) of the desired message
     * @throws ControllerException when header does not exist
     */
    public MessageHeader getMessageHeader(int folderNr, int messageNr)
	throws ControllerException
    {
	MessageHeader ret = null;
	try {
	    ret = mailboxController.getMessageHeader(folderNr, messageNr);
	}
	catch (ControllerException e) {
	    doError(e);
	}
	return ret;
    }

    /**
     * To be called by controller action, when content of a message is 
     * requested
     */ 
    public void setMessage(int folderNr, int p_messageNr)
	throws ControllerException
    {
	try {
	    mailboxController.setMessage(folderNr, p_messageNr);
	}
	catch (ControllerException e) {
	    doError(e);
	}
    }

    /**
     * Accessor for GUI - will only work if controller previously called
     * setMessage()
     */
    public MailMessage getMailMessage(int folderNr, int p_messageNr)
    {
	return mailboxController.getMailMessage(folderNr, p_messageNr);
    }

    /**
     * To be called by controller action, when flag of a message is 
     * to be set
     */ 
    public void setMessagesFlag(int folderNr, int[] msgIndices, String flag, boolean value)
	throws ControllerException
    {
	try {
	    mailboxController.setMessagesFlag(folderNr, msgIndices, flag, value);
	}
	catch (ControllerException e) {
	    doError(e);
	}
    }

    /**
     * To be called by controller action, when content of an attachment is 
     * requested
     */ 
    public void setMailAttachment(int folderNr, int messageNr, 
				  int attachmentNr)
	throws ControllerException
    {
	try {
	    attachmentController.setMailAttachment(folderNr, messageNr, attachmentNr);
	}
	catch (ControllerException e) {
	    doError(e);
	}
    }

    /**
     * Accessor for GUI - will only work if controller previously called
     * setMessageAttachment() with the same parameters.
     * Otherwise returns null
     */
    public MailPart getMailAttachment(int folderNr, int messageNr, 
					    int attachmentNr)
    {
	return attachmentController.getMailAttachment(folderNr, messageNr, attachmentNr);
    }

    /**
     * To be called by controller action, when content of an attachment 
     * part is requested
     */ 
    public void setMailAttachmentPart(int folderNr, int messageNr, 
				      int attachmentNr, int partNr)
	throws ControllerException
    {
	try {
	    attachmentController.setMailAttachmentPart(folderNr, messageNr, attachmentNr, partNr);
	}
	catch (ControllerException e) {
	    doError(e);
	}
    }

    /**
     * Accessor for GUI - will only work if controller previously called
     * setMailAttachmentPart() with the same parameters.
     * Otherwise returns null
     */
    public MailPart getMailAttachmentPart(int folderNr, int messageNr, 
					  int attachmentNr, int partNr)
    {
	return attachmentController.getMailAttachmentPart(folderNr, messageNr, attachmentNr, partNr);
    }

    /**
     * To be called by action
     */
    public void copyOrMoveMessages(int fromFolderNr, int toFolderNr,
				   int[] msgIndices, boolean isCopy)
	throws ControllerException
    {
	mailboxController.copyOrMoveMessages(fromFolderNr, toFolderNr, msgIndices, isCopy);
    }

    /**
     * To be called by controller action, when new composition of message 
     * requested
     */ 
    public void setComposeInfo(ComposeInfo ci)
	throws ControllerException
    {
	log.debug("setComposeInfo() called");
	lastComposeInfo = ci;
    }

    /**
     * Accessor for GUI - will only work if controller previously called
     * setComposeInfo()
     * If no compose info is given, returns null - in this case empty
     * composition screen to be shown
     * Also accessor for any action which needs to add stuff to the compose info
     */
    public ComposeInfo getComposeInfo()
    {
	return lastComposeInfo;
    }

    /**
     * Action calls this, after setting the attributes of lastComposeInfo
     * GUI must call getSendInfo afterwards to check the status
     */
    public void send()
	throws ControllerException
    {
	// Store mail information in ComposeInfo (for later edit if needed)
	if (lastComposeInfo == null)
	    doError(new ControllerException("worker bean setup error: trying to send but compose info is not set", null));

	try {
	    sendInfo = mailSender.send(lastComposeInfo);
	    if (sendInfo.isOK()) {
		ComposeReference ref = lastComposeInfo.getReference();
		if (ref != null && ref.isReply())
		    mailboxController.setFlag(ref.getFolderNr(),ref.getMessageNr(), Flags.Flag.ANSWERED, true);
	    }
	}
	catch (MessagingException e) {
	    // unexpected exception: clean sendInfo and propagate
	    sendInfo = null;
	    doError(new ControllerException("error sending", e));
	}
    }

    /**
     * To be called by the GUI after a call to send()
     */
    public SendInfo getSendInfo()
    {
	return sendInfo;
    }

    public void addFolder(String folderName, boolean isMessageFolder)
	throws ControllerException, FolderSetupException
    {
	log.debug("addFolder() called for name " + folderName);

	folderManager.addFolder(folderName, isMessageFolder);

	// refresh data
	mailboxController.setMailboxOverview(true);

	log.debug("addFolder() successfully created folder " + folderName);
    }

    public void deleteFolder(int folderNr)
	throws ControllerException
    {
	log.debug("deleteFolder() called for number " + folderNr);
	MailFolderHeader mfh = getMailboxOverview().getMailFolderHeader(folderNr);
	if (! mfh.isDeleteable())
	    doError(new ControllerException("setup error: trying to delete undeleteable folder nr " + folderNr, null));

	folderManager.deleteFolder(mfh.getFolder());

	// refresh data
	mailboxController.setMailboxOverview(true);

	log.debug("deleteFolder() successfully deleted folder nr " + folderNr);
    }


    private GeneralStore getUserConfStore() 
	throws ControllerException
    {
	if (userConfStore == null)
	    userConfStore = new GeneralStore((MailServerConnection)getServerConnection(), applicationConfiguration.getUserConfigurationFolder());
	return userConfStore;
    }


    /**
     * This loads the currently stored user preferences.
     */
    public void initUserPreferences()
	throws ControllerException
    {
	log.debug("initUserPreferences() called");
	try {
	    userPreferences = (UserPreferences)getUserConfStore().getObject(UserPreferences.KEY, UserPreferences.class);
	}
	catch (ControllerException e) {
	    doError(new ControllerException("error setting user preferences", e));
	}
	log.debug("getUserPreferences() : read current preferences, is null? " + (userPreferences == null));
    }

    public boolean isUserPreferencesAvailable() {
	return userPreferencesAvailable;
    }

    /**
     * Accessor for user preferences. Note that an empty object can have 2 causes:
     * user preferences don't work at all, so you should check that first; or
     * user preferences have never been set, which is ok.
     * @see isUserPreferencesAvailable()
     */
    public UserPreferences getUserPreferences()
    {
	if (log.isDebugEnabled()) {
	    log.debug("getUserPreferences() called, null ? " + (userPreferences == null));
	    log.debug("getUserPreferences() returning this value: " + userPreferences);
	}
	return userPreferences;
    }

    public void setUserPreferences(UserPreferences up)
	throws NoConnectionException, ControllerException
    {
	try {
	    getUserConfStore().writeAsXml(UserPreferences.KEY, up, UserPreferences.TYPES);
	}
	catch (ControllerException e) {
	    doError(new ControllerException("error setting user preferences", e));
	}
	userPreferences = up;
    }

    /** initializer for contact list: must be called by GUI controller
     * before the GUI tries to render contact list
     */
    public void setContactList()
	throws ControllerException
    {
	try {
	    contactListController.initContactList();
	}
	catch (ControllerException e) {
	    doError(new ControllerException("error storing contact list", e));
	}
    }

    public ContactListController getContactListController() {
	return contactListController;
    }

    public void searchMessages(SearchTerm criteria) 
	throws ControllerException
    {
	try {
	    MailSearcher searcher = new MailSearcher(mailboxController);
	    searchMessagesResults = searcher.searchMessages(criteria);
	}
	catch (ControllerException e) {
	    doError(e);
	}
    }

    public void setSearchMessagesResults(SearchMessagesResults results) {
	searchMessagesResults = results;
    }

    /** accessor for GUI
     */
    public SearchMessagesResults getSearchMessagesResults() {
	return searchMessagesResults;
    }

    public ApplicationConfiguration getApplicationConfiguration()
    {
	return applicationConfiguration;
    }

    public MailSender getMailSender() {
	return mailSender;
    }

    public Throwable getLastException() {
	return lastException;
    }

    public void setLastException(Throwable t) {
	lastException = t;
    }

    public UserInput getUserInput() {
	return userInput;
    }

    public void setUserInput(UserInput userInput) {
	this.userInput = userInput;
    }

    /** GUI initialization helper */
    public void setFirstPage(String firstPage) {
	this.firstPage = firstPage;
    }

    public String getFirstPage() {
	return firstPage;
    }

    private void doError(ControllerException e) 
	throws ControllerException
    {
	// store for possible retrieval by xsp
	setLastException(e);
	throw e;
    }
}
