/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import java.text.DateFormat;
import java.util.Date;
import java.util.Hashtable;
import java.util.Locale;
import javax.mail.Address;
import javax.mail.Flags;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.apache.log4j.Logger;

import net.jatec.ironmailer.framework.MailTools;
import net.jatec.ironmailer.framework.StringTools;

/**
 * Encapsulation of header information about a message, 
 * to be used in the mailbox model
 */
public class MessageHeader
{
    public static final Logger log = Logger.getLogger(MessageHeader.class);

    private String subject;
    private Date date;
    private String dateString;
    private String size;
    private InternetAddress fromAddress;
    private MimeMessage message;
    private String[] flags;
    private boolean isLastInFolder;
    private boolean hasAttachment;
    private boolean isTextOnly;

    public MessageHeader(Message p_message)
	throws IllegalArgumentException, MessagingException
    {
	this(p_message, false);
    }

    /**
     * Initializes all required readable properties.
     * This is necessary so GUI can later read them without fearing an
     * exception (exceptions are handled during object construction)
     */
    public MessageHeader(Message p_message, boolean p_isLastInFolder) 
	throws IllegalArgumentException, MessagingException
    {
	if (p_message == null)
	    throw new IllegalArgumentException("parameter message may not be null");

	if (p_message instanceof MimeMessage)
	    message = (MimeMessage)p_message;
	else
	    throw new IllegalArgumentException("parameter message of unsupported type " + p_message.getClass().getName());

	subject = message.getSubject();
	date = message.getReceivedDate();
	dateString = StringTools.toStandardString(date);
	size = StringTools.sizeToString(message.getSize());
	
	// handle from address
	Address[] a = message.getFrom();
	if (a != null &&
	    a.length >= 1)
	    if (a[0] instanceof InternetAddress)
		fromAddress = (InternetAddress)a[0];
	    else
		throw new MessagingException("unsupported address type " + a[0].getClass().getName());

	// is it a message consisting of text only ? (can make processing easier)
	isTextOnly = message.isMimeType("text/plain");

	// custom flag: is it a message with attachments ?
	hasAttachment = message.isMimeType("multipart/*");

	// handle flags
	Flags allFlags = message.getFlags();
	if (log.isDebugEnabled())
	    log.debug("MessageHeader() got flags: " + MailTools.toString(allFlags));

	Flags.Flag[] ff = allFlags.getSystemFlags();
	String[] userFlags = allFlags.getUserFlags();

// 	int nbFlags = ff.length + userFlags.length + (hasAttachment ? 1 : 0);
	// do not handle user flags for now
	int nbFlags = ff.length + (hasAttachment ? 1 : 0);

	flags = new String[nbFlags];
	for (int i = 0; i < ff.length; i++) {
	    Flags.Flag f = ff[i];
	    String flagAsString = MailTools.getFlagAsString(f);
	    if (flagAsString != null)
		flags[i] = flagAsString;
	    else
		flags[i] = "unknown";
	}
	for (int i = 0; i < userFlags.length; i++) {
	    // Note: we currently have no useful mappings for user flags.
	    // set to unknown, otherwise a non-meaningful flag will be set
	    //flags[ff.length + i] = userFlags[i];

	    // disable user flags for now
// 	    flags[ff.length + i] = "unknown";
	    log.info("Note: message has user flag " + userFlags[i] + ", ignoring");
	}

	if (hasAttachment)
	    flags[nbFlags - 1] = "attach";

	isLastInFolder = p_isLastInFolder;
   }

    public String getSubject() { return subject;}
    
    public Date getDate() { return date;}

    public String getDateString() { return dateString;}

    public String getFromName() {
	return fromAddress.getPersonal();
    }

    public String getFromEmail() {
	return fromAddress.getAddress();
    }

    public InternetAddress getFromAddress() {
	return fromAddress;
    }

    public String getSize() {
	return size;
    }

    public String[] getFlags() {
	return flags;
    }

    public MimeMessage getMessage() {
	return message;
    }

    public boolean isLastInFolder() {
	return isLastInFolder;
    }

    public boolean isTextOnly() {
	return isTextOnly;
    }

    public boolean hasAttachment() {
	return hasAttachment();
    }
}
