/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import javax.mail.internet.InternetAddress;
import org.apache.log4j.Logger;
import net.jatec.ironmailer.framework.MailTools;

/**
 * Part of the client model, representing information about a referenced
 * message during composition (for instance, when replying to a message)
 */
public class ComposeReference
{
    private final Logger log = Logger.getLogger(ComposeReference.class);

    private boolean isReply;
    private boolean isForward;
    private MailMessage reference;
    private int folderNr;
    private int messageNr;
    private boolean isReplyToAll;
    private boolean isForwardAsAttachment;

    public ComposeReference(boolean isReply, boolean isForward, MailMessage reference, int folderNr, int messageNr, boolean isReplyToAll, boolean isForwardAsAttachment) 
	throws IllegalArgumentException
    {
	if (log.isDebugEnabled())
	    log.debug("ComposeReference() called with isReply " + isReply + ", isForward " + isForward + ", folderNr " + folderNr + ", messageNr " + messageNr + ", isReplyToAll=" + isReplyToAll + ", isForwardAsAttachment=" + isForwardAsAttachment);

	if (isReply && isForward)
	    throw new IllegalArgumentException("only one of isReply or isForward may be specified");
	if (isReply || isForward)
	    if (reference == null)
		throw new IllegalArgumentException("reference may not be null when isReply or isForward is set");
	
	this.isReply = isReply;
	this.isForward = isForward;
	this.reference = reference;
	this.folderNr = folderNr;
	this.messageNr = messageNr;
	this.isReplyToAll = isReplyToAll;
	this.isForwardAsAttachment = isForwardAsAttachment;
    }
    
    public boolean isReply() {
	return isReply;
    }

    public boolean isForward() {
	return isForward;
    }

    public MailMessage getReference() {
	return reference;
    }

    public int getFolderNr() { return folderNr;}

    public int getMessageNr() { return messageNr;}

    public boolean isReplyToAll() {
	return isReplyToAll;
    }

    public boolean isForwardAsAttachment() {
	return isForwardAsAttachment;
    }

    /** 
     * Helper method, constructs list of reply addresses 
     * if appropriate. Otherwise returns empty string.
     * (Implementation note: this is not a part of the model
     * strictly speaking, but let's not be pedantic).
     */
    public String getReplyToAddresses() {
	if (! isReply)
	    return "";
	else {
	    if (getReference().hasReplyToAddresses())
		return MailTools.toString(getReference().getReplyToAddresses(), null);
	    else
		return MailTools.toString(getReference().getFromAddresses(), null);
	}
    }

    public String getReplyCcAddresses(InternetAddress excludeAddress) {
	if (isReply && isReplyToAll)
	    return MailTools.toString(getReference().getToAddresses(), getReference().getCcAddresses(), excludeAddress);
	else 
	    return "";
    }
}




