/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.framework.beans;

import java.beans.BeanInfo;
import java.beans.DefaultPersistenceDelegate;
import java.beans.Encoder;
import java.beans.Expression;
import java.beans.EventSetDescriptor;
import java.beans.Introspector;
import java.beans.IntrospectionException;
import java.beans.PersistenceDelegate;
import java.beans.PropertyDescriptor;
import java.beans.Statement;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.EventListener;

import org.apache.log4j.Logger;
import net.jatec.ironmailer.framework.FrameworkException;

/**
 * An implementation of the java.beans.PersistenceDelegate which does
 * not use any awt code, as opposed to the java.beans.DefaultPersistenceDelegate
 * It is essential to not refer to any awt stuff, if the code is to run on a
 * server (where graphic libraries need not be installed)
 *
 * The implementation of this class was inspired by the study of 
 * java.beans.DefaultPersistenceDelegate
 *
 * @author jwk
 */

public class NonGraphicPersistenceDelegate extends PersistenceDelegate 
{
    private final static Logger log = Logger.getLogger(NonGraphicPersistenceDelegate.class);

    private String[] constructor;
    private Boolean definesEquals;

    // Write out the properties of this instance.
    private void initBean(Class type, Object oldInstance, Object newInstance, Encoder out) 
    {
	if (log.isDebugEnabled())
	    log.debug("initBean() called");

        BeanInfo info = null;
        try {
            info = Introspector.getBeanInfo(type);
        } catch (IntrospectionException e) {
            out.getExceptionListener().exceptionThrown(e);
        }
	if (log.isDebugEnabled())
	    log.debug("initBean() got bean info");

        // Properties
        PropertyDescriptor[] propertyDescriptors = info.getPropertyDescriptors();
	if (log.isDebugEnabled())
	    log.debug("initBean() got propertyDescriptors");

        for (int i = 0; i < propertyDescriptors.length; ++i ) {
            try {
		if (log.isDebugEnabled())
		    log.debug("initBean() calling doProperty for i=" + i + ", propertyName=" + propertyDescriptors[i].getName());
                doProperty(type, propertyDescriptors[i], oldInstance, newInstance, out);
            }
            catch (Exception e) {
                out.getExceptionListener().exceptionThrown(e);
            }
        }
	
    }
    


    // ---------------------------------------------------------------------------------
    // Code directly inspired from DefaultPersistenceDelegate, needed because of the scope
    // declarations :(
    // ---------------------------------------------------------------------------------

    public NonGraphicPersistenceDelegate() {
        this(new String[0]);
    }

    public NonGraphicPersistenceDelegate(String[] constructorPropertyNames) {
        this.constructor = constructorPropertyNames;
    }

    private static boolean definesEquals(Class type) {
        try {
            type.getDeclaredMethod("equals", new Class[]{Object.class});
            return true;
        }
        catch(NoSuchMethodException e) {
            return false;
        }
    }

    private boolean definesEquals(Object instance) {
        if (definesEquals != null) {
            return (definesEquals == Boolean.TRUE);
        }
        else {
            boolean result = definesEquals(instance.getClass());
            definesEquals = result ? Boolean.TRUE : Boolean.FALSE;
            return result;
        }
    }

    protected boolean mutatesTo(Object oldInstance, Object newInstance) {
        return (constructor.length == 0) || !definesEquals(oldInstance) ?
            super.mutatesTo(oldInstance, newInstance) :
            oldInstance.equals(newInstance);
    }

    private static String capitalize(String propertyName) {
        return propertyName.substring(0, 1).toUpperCase() + propertyName.substring(1);
    }

    protected Expression instantiate(Object oldInstance, Encoder out) {
        int nArgs = constructor.length;
        Class type = oldInstance.getClass();
        Object[] constructorArgs = new Object[nArgs];
        for(int i = 0; i < nArgs; i++) {
            String name = constructor[i];

            Field f = null;
            try {
                f = type.getDeclaredField(name);
                f.setAccessible(true);
            }
            catch (NoSuchFieldException e) {}
            try {
                constructorArgs[i] = (f != null && !Modifier.isStatic(f.getModifiers())) ?
                    f.get(oldInstance) :
                    type.getMethod("get"+capitalize(name), new Class[0]).invoke(oldInstance, new Object[0]);
            }
            catch (Exception e) {
                // handleError(e, "Warning: Failed to get " + name + " property for " + oldInstance.getClass() + " constructor");
                out.getExceptionListener().exceptionThrown(e);
            }
        }
        return new Expression(oldInstance, oldInstance.getClass(), "new", constructorArgs);
    }

    
    private boolean isTransient(Class type, PropertyDescriptor pd) 
    {
	if (log.isDebugEnabled())
	    log.debug("isTransient() called on type " + type.getName());

        if (type == null) {
            return false;
        }
        Method getter = pd.getReadMethod();
        Class declaringClass = getter.getDeclaringClass();
        if (declaringClass == type) {
            return Boolean.TRUE.equals(pd.getValue("transient"));
        }
        return isTransient(type.getSuperclass(), pd);
    }

    private static boolean equals(Object o1, Object o2) {
        return (o1 == null) ? (o2 == null) : o1.equals(o2);
    }

    private void doProperty(Class type, PropertyDescriptor pd, Object oldInstance, Object newInstance, Encoder out) throws Exception {
	if (log.isDebugEnabled())
	    log.debug("doProperty() called on type " + type.getName() + " with pd name = " + pd.getName());
	try {
	    Method getter = pd.getReadMethod();
	    Method setter = pd.getWriteMethod();

	    if (log.isDebugEnabled())
		log.debug("doProperty() getter is "
			  + (getter != null ? getter.getName() : null)
			  + ", setter is "
			  + (setter != null ? setter.getName() : null)
		    );
	    
	    if (getter != null && setter != null && !isTransient(type, pd)) {
		
		if (log.isDebugEnabled())
		    log.debug("doProperty() getting expressions");
		
		Expression oldGetExp = new Expression(oldInstance, getter.getName(), new Object[]{});
		Expression newGetExp = new Expression(newInstance, getter.getName(), new Object[]{});
		Object oldValue = oldGetExp.getValue();
		Object newValue = newGetExp.getValue();
		
		if (log.isDebugEnabled())
		    log.debug("doProperty() calling first writeExpression with " + oldGetExp);
		
		out.writeExpression(oldGetExp); 

		if (log.isDebugEnabled())
		    log.debug("doProperty() after first writeExpression with " + oldGetExp);

		if (!equals(newValue, out.get(oldValue))) { 
		    // Search for a static constant with this value; 
		    Object e = (Object[])pd.getValue("enumerationValues"); 
		    if (e instanceof Object[] && Array.getLength(e) % 3 == 0) { 
			Object[] a = (Object[])e; 
			for(int i = 0; i < a.length; i = i + 3) { 
			    try { 
				Field f = type.getField((String)a[i]); 
				if (f.get(null).equals(oldValue)) { 
				    out.remove(oldValue); 
				    out.writeExpression(new Expression(oldValue, f, "get", new Object[]{null}));
				}
			    }
			    catch (Exception ex) {}
			}
		    }
		    invokeStatement(oldInstance, setter.getName(), new Object[]{oldValue}, out);
		}
	    }
	}
	catch (UnsatisfiedLinkError e) {
	    log.error("doProperty() catching error on type " + type.getClass(),e);
	    throw new FrameworkException("doProperty() got error on type " + type.getClass() + ", error is " + e.toString(), e);
	}
	if (log.isDebugEnabled())
	    log.debug("doProperty() done");
    }


    static void invokeStatement(Object instance, String methodName, Object[] args, Encoder out) {
	if (log.isDebugEnabled())
	    log.debug("invokeStatement() called with methodName=" + methodName);

        out.writeStatement(new Statement(instance, methodName, args));
    }

    protected void initialize(Class type, Object oldInstance, Object newInstance, Encoder out) {
        super.initialize(type, oldInstance, newInstance, out);
        if (oldInstance.getClass() == type) { // !type.isInterface()) {
            initBean(type, oldInstance, newInstance, out);
        }
    }


    /*pp*/ static Class typeToClass(Class type) {
        return type.isPrimitive() ? typeNameToClass(type.getName()) : type;

    }

    /*pp*/ static Class typeNameToClass(String typeName) {
        typeName = typeName.intern();
        if (typeName == "boolean") return Boolean.class;
        if (typeName == "byte") return Byte.class;
        if (typeName == "char") return Character.class;
        if (typeName == "short") return Short.class;
        if (typeName == "int") return Integer.class;
        if (typeName == "long") return Long.class;
        if (typeName == "float") return Float.class;
        if (typeName == "double") return Double.class;
        if (typeName == "void") return Void.class;
        return null;
    }

    /*pp*/ static Field typeToField(Class type) {
        try {
            return typeToClass(type).getDeclaredField("TYPE");
        }
        catch (NoSuchFieldException e) {
            return null;
        }
    }

}
