/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller.action;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpServletRequest;

// cocoon
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.cocoon.acting.AbstractAction;

// for logging use log4j
import org.apache.log4j.Logger;

import net.jatec.ironmailer.controller.ControllerException;
import net.jatec.ironmailer.controller.MailWorkerBean;
import net.jatec.ironmailer.framework.HttpSessionManager;
import net.jatec.ironmailer.framework.StringTools;
import net.jatec.ironmailer.model.MailboxOverview;
import net.jatec.ironmailer.model.user.UserPreferences;

/**
 * Bootstrapping action: determines entry page for user, depending on preferences
 * @author wk
 */
public class FirstPageAction extends AbstractAction
{
    private final Logger log = Logger.getLogger(FirstPageAction.class);
    
    /**
     * Interface implementation
     * @throws Exception according to interface, but should never happen.
     * If there is some error, null is returned and the exception is logged.
     */
    public Map act(Redirector redirector, SourceResolver resolver, 
		   Map objectModel, String source,
		   Parameters par) 
	throws Exception 
    {
        HttpServletRequest req =(HttpServletRequest)objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);	

	try {
	    log.debug("act() called");

	    // 1. Check that a session exists. If not, action fails
	    if (! HttpSessionManager.getInstance().isInValidSession(req))
		return null;

	    // 2. Get worker bean (insure there is one in the session)
	    HttpSession session = (HttpSession)req.getSession(false);
	    MailWorkerBean wb = (MailWorkerBean)session.getAttribute("workerBean");
	    if (wb == null)
		throw new Exception("setup error: in a valid session but no worker bean found");
	    wb.setLastException(null);

	    String firstPageUri;

	    int initFolderNr = getInitFolder(wb);

	    if (initFolderNr >= 0) {
		log.debug("act() initFolderNr=" + initFolderNr);
		String folderPagePrefix;
		try { 
		    folderPagePrefix = par.getParameter("folderPagePrefix");
		}
		catch (ParameterException e) {
		    log.error("setup error: initial folder specified, but I do not know how to reference it", e);
		    throw e;
		}

 		firstPageUri = folderPagePrefix + initFolderNr;
	    }
	    else {
		log.debug("act() using parameter defaultfirstpage to bootstrap navigation, as there is no user preference");
		// if (no other desired page)
		String defaultUri;
		try { 
		    defaultUri = par.getParameter("defaultfirstpage");
		}
		catch (ParameterException e) {
		    log.error("FirstPageAction has a setup error: no user preference for first page, and no default parameter was set", e);
		    throw e;
		}
		firstPageUri = defaultUri;
	    }

	    log.debug("act() set firstPageUri to " + firstPageUri);

	    wb.setFirstPage(firstPageUri);
	    // don't use the nextpage variable stuff for now. The sitemap could use that, but it gets complicated
	    // with the frames and all, so write it instead in the worker bean
 	    HashMap actionMap = new HashMap();
 	    actionMap.put("nextpage", firstPageUri);

 	    return actionMap;
	}
	catch (Throwable t) {
	    // All other errors result in the action failing, and in the setting
	    // of error fields in the request for later display by the GUI
	    log.error("act() got some unexpected error, probably due to a setup error", t);
	    log.debug("act() error fiels: " + t.toString() + ", message " + t.getMessage() + ", cause " + t.getCause());
	    req.setAttribute("error.content", t.toString());
	    req.setAttribute("error.stack", StringTools.getStackTraceAsString(t));		
	    return null;
	}
	
    }

    /** Returns the user's initial folder number, or -1 if not selected 
     */
    private int getInitFolder(MailWorkerBean wb) throws ControllerException
    {
	int initFolder = -1;
	UserPreferences up = wb.getUserPreferences();
	String s_initFolder = (up != null ? up.getInitialFolder() : null);
	if (s_initFolder != null) {
	    // ensure mailbox is loaded
	    wb.setMailboxOverview(false);
	    MailboxOverview mo = wb.getMailboxOverview();
	    initFolder = mo.getFolderIndex(s_initFolder);
	}
	return initFolder;
    }

}

