/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller.action;

import java.io.File;
import java.util.Map;
import javax.servlet.ServletRequest;
import org.apache.log4j.Logger;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.servlet.multipart.PartOnDisk;

import net.jatec.ironmailer.controller.ControllerException;
import net.jatec.ironmailer.controller.MailWorkerBean;
import net.jatec.ironmailer.controller.NoSelectionException;
import net.jatec.ironmailer.framework.ServletTools;
import net.jatec.ironmailer.model.AttachmentHolder;
import net.jatec.ironmailer.model.ComposeInfo;
import net.jatec.ironmailer.model.ModelException;

/**
 * Attach an uploaded file to a mail composition.
 * 
 * Implementation notes:
 * - in Cocoon 2.1.3, uploaded Parts are deleted when the request
 * is finished. This is no good for us, since we need to keep the parts
 * as long as the mail composition is going on - we may only delete them
 * once a mail has been succesfully sent. Since there seems to be no way
 * to configure this Cocoon behaviour, this actions renames the file that
 * was uploaded (to prevent Cocoon from deleting it).
 *
 * Distribution notes:
 * - in Cocoon 2.1.3, uploading files only works when the corresponding
 * entry in web.xml has been set to true (default is false)
 *
 * @author JWK
 */
public class FileAttachAction implements ActionDispatcher.ActionStrategy
{
    private final Logger log = Logger.getLogger(AttachAction.class);

    public void process(ServletRequest req, MailWorkerBean wb, Redirector redirector, Map objectModel) 
	throws ControllerException
    {
	log.debug("process() called");

	// Determine if request is a multipart or not
	String contentType = req.getContentType();
	if (contentType != null) contentType = contentType.trim();
	boolean isMultipart = (contentType != null &&
			       contentType.startsWith("multipart/form-data"));
	log.debug("act() isMultipart? " + isMultipart + ", contentType=" + contentType);
	if (! isMultipart)
	    throw new ControllerException("setup error: file attach expects a multipart form", null);

	// Get Cocoon request
	Request cocoonReq = ObjectModelHelper.getRequest(objectModel);
	if (cocoonReq == null)
	    throw new ControllerException("cocoon is fucked up: no request");

	try {
	    log.debug("act() handling addition of file to attachment");
	    
	    // get file name selected by user
	    Object o_file = cocoonReq.get("file");
	    if (o_file == null)
		throw new NoSelectionException("add called but no file was given");
	    
	    if (!(o_file instanceof PartOnDisk))
		throw new ControllerException("setup error: add received file but it's of type " + o_file.getClass().getName() + ", whereas it should be of type PartOnDisk", null);
	    PartOnDisk cocoonFile = (PartOnDisk)o_file;
	    File file1 = cocoonFile.getFile();
	    
	    // TO DO: do we need to do something about the / in the path ?
	    File saveDir = new File(file1.getParent() + "/save");
	    if (! saveDir.exists())
		if (! saveDir.mkdir())
		    throw new ControllerException("oops: cannot create save directory named " + saveDir.getPath());
	    
	    String newName = saveDir.getPath() + "/" 
		+ file1.getName();
	    if (log.isDebugEnabled())
		log.debug("saving uploaded file as " + newName);
	    File file = new File(newName);
	    if (!file1.renameTo(file))
		throw new ControllerException("unable to rescue uploaded file into " + newName);
	    
	    if (! file.exists())
		throw new ControllerException("something fishy: file " + file.getName() + " does not actually exist");
	    
	    if (log.isDebugEnabled()) {
		log.debug("act() addition of file, now getting reference to file");
		log.debug("act() PartOnDisk has name " + file.getName()
			  + ", size " + file.length());
	    }

	    ComposeInfo ci = wb.getComposeInfo();
	    if (ci == null)
		throw new ControllerException("setup error: want to add attachment information to a message being composed, but no compose info found");
	    AttachmentHolder ah = ci.getAttachmentHolder();
	    ah.add(file);
	    
	    // for debugging only
	    String fileName = file.getName();
	    log.debug("act() done attaching file " + fileName);
	    
	    // end of handling "add"
	}
	catch (ModelException e) {
	    // model exceptions don't cause the action to fail;
	    // instead we make a note in the worker bean for user-friendly
	    // handling of exception
	    log.info("act failed with model exception: " + e.toString());
	    if (wb != null)
		wb.setLastException(e);
	}
	catch (NoSelectionException e) {
	    // make a note in worker bean and return
	    if (log.isDebugEnabled())
		log.debug("act got NoSelectionException: " + e.toString());
	    if (wb != null)
		wb.setLastException(e);
	}

	log.debug("process completed");
    }

}
