/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.InputStreamReader;
import java.io.InputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Hashtable;
import java.util.Properties;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.FolderClosedException;
import javax.mail.Message;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.internet.MimeMessage;
import org.apache.log4j.Logger;

import net.jatec.ironmailer.framework.FrameworkException;
import net.jatec.ironmailer.framework.xml.XMLReader;
import net.jatec.ironmailer.framework.xml.XMLWriter;

/**
 * Given a MailServerConnection, provide a generic storing mechanism
 * in this backend.
 */
public class GeneralStore
{
    private final Logger log = Logger.getLogger(GeneralStore.class);
    private final static String UTF = "UTF-8";

    private Folder myFolder;
    private Hashtable entries;

    /**
     * Initializes GeneralStore by creating table of entries.
     */
    public GeneralStore(MailServerConnection backend, String generalStoreName)
	throws ControllerException
    {
	log.debug("GeneralStore() called for name " + generalStoreName);
	try {
	    Store store = backend.getStore();
	    myFolder = store.getFolder(generalStoreName);
	    
	    if (! myFolder.exists()) {
		log.debug("GeneralStore() folder " + generalStoreName + " does not yet exist in this backend, creating");
		myFolder.create(Folder.HOLDS_MESSAGES);
		myFolder.setSubscribed(false);
	    }
	    else {
		log.debug("GeneralStore() existing folder was found");
	    }
	    myFolder.open(Folder.READ_WRITE);

	    // Entries are represented by list of MimeMessage, the key
	    // is the Subject field in the MimeMessage. If duplicate keys
	    // exist, only the last will survive ... there may be only one !
	    Message[] m_entries = myFolder.getMessages();
	    // length of message array a good guess for hashtable init size
	    entries = new Hashtable(m_entries.length);
	    for (int i = 0; i < m_entries.length; i++) {
		MimeMessage mm = (MimeMessage)m_entries[i];
		if (! mm.isSet(Flags.Flag.DELETED)) {
		    String entryId = mm.getSubject();
		    entries.put(entryId, mm);
		}
	    }
	}
	catch (Throwable t) {
	    throw new ControllerException("error initializing GeneralStore for name " + generalStoreName, t);
	}
	log.debug("GeneralStore() successful");
    }

    public void close() 
	throws ControllerException
    {
	log.debug("close() called");
	try {
	    if (myFolder.isOpen())
		myFolder.close(true);
	}
	catch (Throwable t) {
	    throw new ControllerException("error closing GeneralStore", t);
	}
    }

    public boolean hasEntry(String entryId)
    {
	return entries.containsKey(entryId);
    }

    /**
     * If entryId is not found, returns null
     * @throws ControllerException if input stream could not be provided
     */
    private InputStream read(String entryId) 
	throws ControllerException
    {
	log.debug("read() called for id " + entryId);
	InputStream ret;

	try {
	    MimeMessage mm = (MimeMessage)entries.get(entryId);
	    if (mm != null)
		ret = mm.getInputStream();
	    else
		ret = null;
	}
	catch (Throwable t) {
	    throw new ControllerException("error writing retrieving input stream", t);
	}
	return ret;
    }

    /**
     * Note: if there is no object in the store, returns null object
     */
    public Object getObject(String entryId, Class expectedType)
	throws ControllerException
    {
	log.debug("getObject() called for entryId " + entryId);
	Object ret = null;
	InputStream datastream = read(entryId);
	if (datastream != null) {
	    log.debug("getObject() got data stream");
	    try {
		ret = (new XMLReader()).getObject(datastream);
	    }
	    catch (FrameworkException e) {
		throw new ControllerException("error retrieving data",e);
	    }
	    if (ret == null)
		throw new ControllerException("internal error: there is a datastream but I cannot read it");
	    if (! expectedType.isInstance(ret))
		throw new ControllerException("bad type read from store: expected " + expectedType.getClass().getName() + " but got " + ret.getClass().getName());
	}
	else {
	    ret = null;
	    log.debug("getObject() found no data for entryId " + entryId);
	}
	return ret;
    }


    public void write(String entryId, String input, String encoding) 
	throws ControllerException
    {
	log.debug("write() called for id " + entryId);
	MimeMessage m;
	try {
	    m = new MimeMessage(Session.getDefaultInstance(new Properties()));
	    m.setSubject(entryId);
 	    m.setText(input.toString(), encoding);
	}
	catch (Throwable t) {
	    throw new ControllerException("error writing to GeneralStore", t);
	}
	log.debug("write() successfully created message, now calling store");
	storeMessage(m);
	log.debug("write() completed");
    }


    public void writeAsXml(String entryId, Object o, Class[] types)
	throws ControllerException
    {
	if (log.isDebugEnabled())
	    log.debug("writeAsXml() called for entryId " + entryId);

	ByteArrayOutputStream os = new ByteArrayOutputStream();
	try {
	    new XMLWriter(os, o, types);
	}
	catch (FrameworkException e) {
	    throw new ControllerException("error writing data",e);
	}
	log.debug("writeAsXml() got data, now calling store");
	try {
	    write(entryId, os.toString(UTF), UTF);
	}
	catch (UnsupportedEncodingException e) {
	    throw new ControllerException("error converting string", e);
	}
    }

    private void storeMessage(MimeMessage m) throws ControllerException
    {
	try {
	    String entryId = m.getSubject();
	    MimeMessage oldMessage = (MimeMessage)entries.get(entryId);
	    log.debug("storeMessage() check 1");

	    if (oldMessage != null) {
		log.debug("storeMessage() detected old message, removing");
		oldMessage.setFlag(Flags.Flag.DELETED, true);
		// remove from table, to be sure
		entries.remove(entryId);
		
		// purge this folder
		myFolder.expunge();
		log.debug("storeMessage() expunged old message");
	    }

	    // store message in folder
	    Message[] msgs = new Message[1];
	    msgs[0] = m;
	    log.debug("storeMessage() check 2");
	    myFolder.appendMessages(msgs);

	    log.debug("storeMessage() check 3");
	    // add message to table
	    entries.put(entryId, m);
	    log.debug("storeMessage() check 4");
	}
	catch (FolderClosedException e) {
	    throw new NoConnectionException("no connection", e);
	}
	catch (Exception e) {
	    throw new ControllerException("error writing to GeneralStore", e);
	}
	log.debug("storeMessage() completed");
    }

}


