/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import java.util.Enumeration;
import java.util.Vector;
import javax.mail.Address;
import javax.mail.Header;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import org.apache.log4j.Logger;

import net.jatec.ironmailer.framework.StringTools;

/**
 * Encapsulation of information about a mail message, to be used
 * in the mailbox model
 */
public class MailMessage
{
    private final Logger log = Logger.getLogger(MailMessage.class);
    private MessageHeader header;
    private InternetAddress[] fromAddresses;
    private InternetAddress[] toAddresses;
    private InternetAddress[] ccAddresses;
    private InternetAddress[] replyToAddresses;
    private String body;
    private MailPart[] attachments;
    private int attachmentOffset;
    private Multipart myMultipart;

    public MailMessage(MessageHeader header) 
	throws MessagingException
    {
	try {
	    if (header == null)
		throw new IllegalArgumentException("parameter header is empty");
	    MimeMessage message = header.getMessage();
	    if (message == null)
		throw new Exception("illegal header parameter: contains empty message");

	    this.header = header;
	   
	    log.debug("MailMessage() reading addresses from message ...");
	    Address[] tmp0 = message.getFrom();
	    if (tmp0 != null &&
		tmp0.length > 0) {
		fromAddresses = new InternetAddress[tmp0.length];
		for (int i = 0; i < tmp0.length; i++)
		    fromAddresses[i] = (InternetAddress)tmp0[i];
	    } else
		fromAddresses = new InternetAddress[0];
	    
	    Address[] tmp1 = message.getRecipients(Message.RecipientType.TO);
	    if (tmp1 != null &&
		tmp1.length > 0) {
		toAddresses = new InternetAddress[tmp1.length];
		for (int i = 0; i < tmp1.length; i++)
		    toAddresses[i] = (InternetAddress)tmp1[i];
	    } else
		toAddresses = new InternetAddress[0];
	    
	    Address[] tmp2 = header.getMessage().
		getRecipients(Message.RecipientType.CC);
	    if (tmp2 != null &&
		tmp2.length > 0) {
		ccAddresses = new InternetAddress[tmp2.length];
		for (int i = 0; i < tmp2.length; i++)
		    ccAddresses[i] = (InternetAddress)tmp2[i];
	    } else
		ccAddresses = new InternetAddress[0];

	    Address[] tmp3 = message.getReplyTo();
	    if (tmp3 != null &&
		tmp3.length > 0) {
		replyToAddresses = new InternetAddress[tmp3.length];
		for (int i = 0; i < tmp3.length; i++)
		    replyToAddresses[i] = (InternetAddress)tmp3[i];
	    } else
		replyToAddresses = new InternetAddress[0];

	    // prepare body
	    String contentType = message.getContentType();
	    log.debug("MailMessage() preparing to read body; content type of message is: " + contentType);
	    if (header.isTextOnly())
		log.debug("MailMessage() got a message containing text only");
	    
	    Object content = message.getContent();
	    if (content instanceof String) {
		log.debug("MailMessage() handling content of type String");
 		body = (String)content;
		// hack for stripping extra linefeeds, so GUI can display it easier
		//body = StringTools.toDisplayableString((String)content);
		//body = StringTools.toHtmlDisplayableString((String)content);
		attachments = new MailPart[0];
	    }
	    else if (content instanceof Multipart) {
		log.debug("MailMessage() handling content of type Multipart");
		// if first part is text-only, use that as body
		Multipart mp = (Multipart)content;
		int nbParts = mp.getCount();
		if (nbParts < 1) {
		    body = "";
		    attachments = new MailPart[0];
		}
		else {
		    log.debug("MailMessage() determining body parts");
		    MailPart firstPart = new MailPart(mp.getBodyPart(0));
		    attachmentOffset = 0;
		    if (firstPart.isAnonymousText()) {
			log.debug("MailMessage() handling case where first body part is plain text");
			body = firstPart.getText();
			attachments = new MailPart[nbParts - 1];
			attachmentOffset = 1;
		    } 
		    else {
			log.debug("MailMessage() handling case where first body part is NOT plain text, handling all parts as attachments");
			body = "";
			attachments = new MailPart[nbParts];
			attachments[0] = firstPart;
		    }
		    for (int i = 1; i < nbParts; i++) {
			log.debug("MailMessage() now handling attachment nb " + i);
			MailPart ma = new MailPart(mp.getBodyPart(i));
			attachments[i - attachmentOffset] = ma;
		    }
		}
		// save reference on multi-part contents: needed for multiple reading of attachments
		myMultipart = mp;
	    }
	    else {
		log.warn("MailMessage() got strange message, content type: " + message.getContentType() + ", attempting to transform to standard mail");
		// Situation: we have a MimeMessage, but we need a MimeBodyPart. I can't find a simple way to transform it, so ...
		MimeBodyPart mb = new MimeBodyPart();
		for (Enumeration headers = message.getAllHeaders(); headers.hasMoreElements(); ) {
			Header thisheader = (Header)headers.nextElement();
			mb.setHeader(thisheader.getName(), thisheader.getValue());
			if (log.isDebugEnabled())
				log.debug("MailMessage() workaround: setting header " + thisheader.getName() + " to value " + thisheader.getValue());
		}
		mb.setDisposition(Part.ATTACHMENT);
		log.debug("MailMessage() set all header fields of constructed body part");
		mb.setContent(content, contentType);
		log.debug("MailMessage() done setting content of constructed body part");
		// try to force content type : doesn't seem to work otherwise
		mb.setHeader("Content-Type", contentType);
		log.debug("MailMessage() constructed body part now claims it is of type " + mb.getContentType());
		myMultipart = new MimeMultipart();
		myMultipart.addBodyPart(mb);
		attachments = new MailPart[1];
		//attachments[0] = new MailPart(myMultipart.getBodyPart(0));
		attachments[0] = new MailPart(message);

		log.debug("MailMessage(), done constructing work-around multipart");
		//throw new Exception("unsupported mail content type: " + message.getContentType() + ", class " + content.getClass().getName());
	    }
	    // done reading message
	}
	catch (Exception e) {
	    log.error("error initializing MailMessage", e);
	    throw new MessagingException("error initializing MailMessage: " + e.toString(), e);
	}
	log.debug("MailMessage() successful");
    }

    public MessageHeader getMessageHeader() {
	return header;
    }

    public InternetAddress[] getFromAddresses() {
	return fromAddresses;
    }
	   
    public InternetAddress[] getToAddresses() {
	return toAddresses;
    }
	   
    public InternetAddress[] getCcAddresses() {
	return ccAddresses;
    }
	   
    public InternetAddress[] getReplyToAddresses() {
	return replyToAddresses;
    }

    public boolean hasReplyToAddresses() {
	return replyToAddresses != null 
	    && replyToAddresses.length > 0;
    }

    public boolean hasBody() {
	return body != null;
    }

    public String getBody() {
	return body;
    }

    /** 
     * returns a "fresh" handle every time.
     * i.e. caller may consume content of attachment.
     */
    public MailPart getMailAttachment(int index) 
	throws ModelException
    {
	MailPart ret = attachments[index];
	if (myMultipart != null) {
	    Part p;
	    try {
		p = myMultipart.getBodyPart(index + attachmentOffset);
	    }
	    catch (MessagingException e) {
		throw new ModelException("internal error, cannot access body part", e);
	    }
		       
	    MailPart fresh = new MailPart(p);
	    attachments[index] = fresh;
	}
	else
	    log.warn("getMailAttachment() returning, but cannot refresh body part due to empty multipart structure");
	return ret;
    }

    public MailPart[] getMailAttachments() {
	return attachments;
    }

}




