/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.model;

import java.util.Vector;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import org.apache.log4j.Logger;

/**
 * Encapsulation of information about a folder, to be used
 * in the mailbox model
 */
public class MailFolder
{
    private final Logger log = Logger.getLogger(MailFolder.class);
    private MailFolderHeader header;
    private Vector messageHeaders;

    public MailFolder(MailFolderHeader header) 
	throws MessagingException, ModelException
    {
	if (! header.isSelectable())
	    throw new ModelException("illegal usage of MailFolder model: you are attempting to open the folder " + header.getName() + ", but it is marked as not selectable");

	this.header = header;

	Folder folder = header.getFolder();

	// open folder, if necessary
	if (! folder.isOpen()) {
	    // important : we MUST open folder as read-write, otherwise
	    // new messages are not fetched into folder !
	    folder.open(Folder.READ_WRITE);
	} 

	Message[] messages = folder.getMessages();
	log.debug("MailFolder() retrieved " + messages.length + " messages for folder " + header.getName() + "; header indicates " + header.getNbMessages() + ", folder indicates " + folder.getMessageCount() + ", isOpen? " + folder.isOpen() + ", folder is " + folder.toString());
	messageHeaders = new Vector(messages.length);
	for (int i = 0; i < messages.length; i++) {
	    boolean isLast = (i == (messages.length - 1));
	    MessageHeader mh = new MessageHeader(messages[i], isLast);
	    messageHeaders.add(mh);
	}
    }

    public MailFolderHeader getMailFolderHeader() {
	return header;
    }

    public Vector getMessageHeaders() {
	return messageHeaders;
    }
	   
    public MessageHeader getMessageHeader(int nr) {
	return (MessageHeader)messageHeaders.get(nr);
    }

    public void close(boolean expunge) 
	throws MessagingException
    {
	Folder folder = header.getFolder();
	log.debug("close() called for folder " + folder.getName() + " with expunge flag " + expunge + " isOpen? " + folder.isOpen() + ", folder is " + folder.toString());

	// special case: folder should be open, but if it was unexpectedly
	// closed, and we want to expunge then reopen it
	if (expunge && ! folder.isOpen())
	    folder.open(Folder.READ_WRITE);

	if (folder.isOpen()) {
	    log.debug("close() closing folder " + folder.getName() + " with expunge flag " + expunge);
	    folder.close(expunge);
	}
    }

}




