/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.framework;

import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import javax.mail.Address;
import javax.mail.Flags;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.InternetHeaders;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.search.AndTerm;
import javax.mail.search.NotTerm;
import javax.mail.search.OrTerm;
import javax.mail.search.RecipientStringTerm;
import javax.mail.search.SearchTerm;
import javax.mail.search.SubjectTerm;
import javax.mail.search.StringTerm;
import org.apache.log4j.Logger;


public class MailTools
{
    public static final Logger log = Logger.getLogger(MailTools.class);

    public static SearchTerm[] getBaseTerms(SearchTerm term) {
	Vector results = new Vector();
	addBaseTerms(term, results);
	SearchTerm[] ret = new SearchTerm[results.size()];
	for (int i = 0; i < ret.length; i++) {
	    ret[i] = (SearchTerm)results.get(i);
	}
	return ret;
    }

    public static void addBaseTerms(SearchTerm term, Vector results) {
	if (term instanceof NotTerm)
	    addBaseTerms(((NotTerm)term).getTerm(),results);
	else
	    if (term instanceof AndTerm) {
		SearchTerm[] terms = ((AndTerm)term).getTerms();
		for (int i = 0; i < terms.length; i++)
		    addBaseTerms(terms[i],results);
	    }
	    else
		if (term instanceof OrTerm) {
		SearchTerm[] terms = ((OrTerm)term).getTerms();
		for (int i = 0; i < terms.length; i++)
		    addBaseTerms(terms[i],results);
		}
		else
		    if (term instanceof StringTerm) 
			results.add(term);
	// else ignore for now
    }

    public static MatchPart[] getMatchParts(SearchTerm[] criteria,
					    Class thisClass,
					    String content) {
	if (criteria == null) {
	    log.warn("getMatchParts() called on empty criteria, returning nothing");
	    return new MatchPart[0];
	}
	if (! StringTerm.class.isAssignableFrom(thisClass)) {
	    log.warn("getMatchParts() called on class " + thisClass.getName() + " which is not a StringTerm, returning nothing");
	    return new MatchPart[0];
	}

	MatchPart[] ret = null;
	for (int i = 0; i < criteria.length; i++) {
	    if (thisClass.isInstance(criteria[i])) {
		String pattern = ((StringTerm)(criteria[i])).getPattern();
		// TO DO: the following could be a problem depending on 
		// locale. Resulting string in lower case need not be same
		// length as original string. How to solve this ??
		int index = (content.toLowerCase()).indexOf(pattern.toLowerCase());
		if (index >= 0) {
		    MatchPart match1 = new MatchPart(content.substring(0,index),false);
		    MatchPart match2 = new MatchPart(content.substring(index,index+pattern.length()),true);
		    MatchPart match3 = new MatchPart(content.substring(index+pattern.length()),false);
		    ret = new MatchPart[]{match1,match2,match3};
		}
	    }
	}
	if (ret == null)
	    // no matches in this content, probably user searched somewhere else
	    ret = new MatchPart[]{new MatchPart(content,false)};

	return ret;
    }
					     

    public static SearchTerm getRecipientAddressSearchTerm(String criteria) {
	if (! StringTools.hasInfo(criteria))
	    return null;

	SearchTerm t1 = new RecipientStringTerm(Message.RecipientType.TO, criteria);
	SearchTerm t2 = new RecipientStringTerm(Message.RecipientType.CC, criteria);
	SearchTerm t3 = new RecipientStringTerm(Message.RecipientType.BCC, criteria);
	SearchTerm[] t = new SearchTerm[]{t1,t2,t3};
	return new OrTerm(t);
    }

    /** useful for debugging */
    public static String toString(Flags allFlags) {
	StringBuffer ret = new StringBuffer();
	ret.append("System flags={");
	Flags.Flag[] ff = allFlags.getSystemFlags();
	for (int i = 0; i < ff.length; i++) {
	    Flags.Flag f = ff[i];
	    String flagAsString = MailTools.getFlagAsString(f);
	    ret.append(flagAsString).append(" ");
	}
	ret.append("}");
	
	String[] userFlags = allFlags.getUserFlags();
	ret.append(", User flags={");
	for (int i = 0; i < userFlags.length; i++)
	    ret.append(userFlags[i]).append(" ");
	ret.append("}");
	return ret.toString();
    }

    public static String toString(Address[] as) {
	if (as == null)
	    return "";
	else {
	    StringBuffer ret = new StringBuffer();
	    for (int i = 0; i < as.length; i++) {
		if (i > 0)
		    ret.append(", ");
		ret.append(as[i].toString());
	    }
	    return ret.toString();
	}
    }

    public static String toString(InternetAddress[] ia1, InternetAddress[] ia2, InternetAddress excludeAddress) {
	StringBuffer ret = new StringBuffer();
	boolean ia1NotEmpty = (ia1 != null && ia1.length > 0);
	boolean ia2NotEmpty = (ia2 != null && ia2.length > 0);

	if (ia1NotEmpty)
	    ret.append(toString(ia1, excludeAddress));

	if (ia1NotEmpty && ia2NotEmpty)
	    ret.append(", ");

	if (ia2NotEmpty)
	    ret.append(toString(ia2, excludeAddress));

	return ret.toString();
    }

    public static String toString(InternetAddress[] ia, InternetAddress excludeAddress) {
	if (ia != null) {
	    log.debug("toString() called for " + ia.length + " addresses, excluding " + excludeAddress);
	    StringBuffer ret = new StringBuffer();
	    for (int i = 0; i < ia.length; i++) {
		// determine if this address is to be added
		if (excludeAddress == null ||
		    ! excludeAddress.getAddress().equals(ia[i].getAddress())) {
		    if (i > 0)
			ret.append(", ");
		    String name = ia[i].getPersonal();
		    if (name != null) {
			if (name.startsWith("\""))
			    ret.append(name);
			else
			    ret.append("\"")
				.append(name)
				.append("\" ");
		    }
		    ret.append("<")
			.append(ia[i].getAddress())
			.append(">");
		}
	    }
	    return ret.toString();
	}
	else
	    return "";
    }

    private static final Hashtable flags2String;
    static {
	flags2String = new Hashtable(6);
	flags2String.put(Flags.Flag.ANSWERED, "answered");
	flags2String.put(Flags.Flag.DELETED, "deleted");
	flags2String.put(Flags.Flag.DRAFT, "draft");
	flags2String.put(Flags.Flag.FLAGGED, "flagged");
	flags2String.put(Flags.Flag.RECENT, "recent");
	flags2String.put(Flags.Flag.SEEN, "seen");
    }

    private static final Hashtable strings2flag;
    static {
	strings2flag = new Hashtable(6);
	strings2flag.put("answered", Flags.Flag.ANSWERED);
	strings2flag.put("deleted", Flags.Flag.DELETED);
	strings2flag.put("draft", Flags.Flag.DRAFT);
	strings2flag.put("flagged", Flags.Flag.FLAGGED);
	strings2flag.put("recent", Flags.Flag.RECENT);
	strings2flag.put("seen", Flags.Flag.SEEN);
    }
    
    public static String getFlagAsString(Flags.Flag flag) {
	return (String)flags2String.get(flag);
    }

    public static Flags.Flag getStringAsFlag(String flag) {
	return (Flags.Flag)strings2flag.get(flag);
    }

    public static String toDisplayableString(MessagingException e) {
	StringBuffer ret = new StringBuffer();
	if (e != null) {
	    ret.append(e.getMessage().replaceAll("nested exception is:", "").replaceAll("\n",""));

	    // this code not needed right now: javamail currently puts the child messages also in
	    // the main message, so what's the point.
// 	    Exception next_e;
// 	    while ((next_e = e.getNextException()) instanceof MessagingException) {
// 		e = (MessagingException)e.getNextException();
// 		ret.append(" / ").append(e.getMessage());
// 	    }
	}
	return ret.toString();
    }

    /**
     * If string contains characters which can screw up an URL, such as a question mark,
     * they will be replaced.
     */
    public static String toURLString(String filename) {
	if (filename == null) return null;
	String ret = filename.replace('?', '_');

	return ret;
    }

    /**
     * When forwarding a message, I have a MimeMessage and I need to add it to another MimeMessage.
     * I have not found another way than transforming it into a MimeBodyPart, so that can be added.
     * If anyone has a better idea ... This sucks because I think information can get lost
     */
    public static MimeBodyPart[] toMimeBodyParts(MimeMessage msg, String textEncoding) throws Exception {
	
	boolean textOnly;
	MimeBodyPart[] ret;

	Object content = msg.getContent();
	if (content instanceof String) {
	    textOnly = true;
	}
	else 
	    if (content instanceof MimeMultipart) {
		textOnly = false;
	    }
	    else
		throw new Exception("cannot read MimeMessage, unknown class type " + content.getClass().getName());

	if (textOnly) {
	    ret = new MimeBodyPart[1];
	    MimeBodyPart part = new MimeBodyPart();
	    for (Enumeration headers = msg.getAllHeaderLines(); headers.hasMoreElements(); )
		part.addHeaderLine((String)headers.nextElement());
	    part.setFileName(toString(msg.getFrom()) + " - " + msg.getSubject() + " - " + msg.getSentDate());
	    part.setText((String)content, textEncoding);
	    // can't use the message's encoding, so I guess we need to provide it ourselves ??
	    //part.setText((String)content, msg.getEncoding());
	    ret[0] = part;
	}
	else {
	    MimeMultipart mm = (MimeMultipart)content;
	    ret = new MimeBodyPart[1];
	    MimeBodyPart part = new MimeBodyPart();
	    part.setFileName(toString(msg.getFrom()) + " - " + msg.getSubject() + " - " + msg.getSentDate());
	    part.setContent(mm);
	    ret[0] = part;
// 	    int nbParts = mm.getCount();
// 	    ret = new MimeBodyPart[nbParts];
// 	    InternetHeaders headerList = new InternetHeaders();
// 	    for (Enumeration headers = msg.getAllHeaderLines(); headers.hasMoreElements(); )
// 		headerList.addHeaderLine((String)headers.nextElement());

// 	    for (int i = 0; i < nbParts; i++) {
// 		// note : I'm not sure about this. We could just copy the body part into the new
// 		// array, but then what about the headers of the message ??
// 		MimeBodyPart part = new MimeBodyPart(headerList, mm.getBodyPart(i).getInputStream());
// 		ret[i] = part;
// 	    }

	}

	return ret;
    }
}
