/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller.action;

import java.util.Collections;
import java.util.Map;

import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpServletRequest;

// cocoon
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.cocoon.acting.AbstractAction;

// for logging use log4j
import org.apache.log4j.Logger;

import net.jatec.ironmailer.controller.MailWorkerBean;
import net.jatec.ironmailer.controller.NoSelectionException;
import net.jatec.ironmailer.framework.HttpSessionManager;
import net.jatec.ironmailer.framework.StringTools;


/**
 * @author wk
 */
public class ControllerAction extends AbstractAction
{
    private final Logger log = Logger.getLogger(ControllerAction.class);
    
    // this tag name must match the entry name in the dictionaries, for i18n of no-selection warning
    private final static String NO_SELECTION_USER_TAG = "general_noselection";

    /**
     * Interface implementation
     * @throws Exception according to interface, but should never happen.
     * If there is some error, null is returned and the exception is logged.
     */
    public Map act(Redirector redirector, SourceResolver resolver, 
		   Map objectModel, String source,
		   Parameters par) 
	throws Exception 
    {
        HttpServletRequest req 
	    = (HttpServletRequest)objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);	

	try {
	    log.debug("act() called");

	    // 1. Check that a session exists. If not, action fails
	    if (! HttpSessionManager.getInstance().isInValidSession(req))
		return null;

	    // 2. Get worker bean (insure there is one in the session)
	    HttpSession session = (HttpSession)req.getSession(false);
	    MailWorkerBean wb = (MailWorkerBean)session.getAttribute("workerBean");
	    if (wb == null)
		throw new Exception("setup error: in a valid session but no worker bean found");

	    // clean any cache from last request
	    wb.setLastException(null);
	    wb.setUserInput(null);

	    // 3. Check desired action, if specified
	    if (log.isDebugEnabled()) {
		log.debug("act() req.getContextPath()=" + req.getContextPath());
		log.debug("act() req.getPathInfo()=" + req.getPathInfo());
		log.debug("act() req.getPathTranslated()=" + req.getPathTranslated());
		log.debug("act() req.getQueryString()=" + req.getQueryString());
		log.debug("act() req.getRequestURI()=" + req.getRequestURI());
		log.debug("act() req.getServletPath()=" + req.getServletPath());
	    }
	    String action = req.getParameter("act");
	    log.debug("act() action=" + action);
	    if (action != null) {
		action = action.trim();
		ActionDispatcher ad = wb.getActionDispatcher();
		ad.processRequest(action, req, wb, redirector, objectModel);
	    }

	    log.debug("act() done handling action " + action);

	    // increment test counter.
	    // This is just for testing session handling (timeouts etc)
	    wb.setTestCtr(wb.getTestCtr() + 1);


	    return Collections.EMPTY_MAP;
	}
	catch (NoSelectionException e) {
	    if (log.isDebugEnabled())
		log.debug("act() handling NoSelectionAction");
	    req.setAttribute("warning.content", e.toString());
	    req.setAttribute("warning.usertag", NO_SELECTION_USER_TAG);
	    return Collections.EMPTY_MAP;
	}
	catch (Throwable t) {
	    // All other errors result in the action failing, and in the setting
	    // of error fields in the request for later display by the GUI
	    log.error("act() got some unexpected error, probably due to a setup error", t);
	    if (log.isDebugEnabled())
		log.debug("act() error fiels: " + t.toString() + ", message " + t.getMessage() + ", cause " + t.getCause());
	    req.setAttribute("error.content", t.toString());
	    req.setAttribute("error.stack", StringTools.getStackTraceAsString(t));		
	    return null;
	}
	
    }
}

