/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;

import java.util.Properties;
import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.SendFailedException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;

import org.apache.log4j.Logger;

import net.jatec.ironmailer.framework.MailTools;
import net.jatec.ironmailer.model.ApplicationConfiguration;
import net.jatec.ironmailer.model.AttachmentHolder;
import net.jatec.ironmailer.model.ComposeInfo;
import net.jatec.ironmailer.model.ComposeReference;
import net.jatec.ironmailer.model.SendInfo;


public class MailSender
{
    private final Logger log = Logger.getLogger(MailSender.class);
    private Session session;
    private Properties sessionProps;
    private boolean isDemoUser;
    private InternetAddress defaultFromUser;
    private ApplicationConfiguration applConf;

    public MailSender(ServerConnection sc, ApplicationConfiguration applConf) 
	throws MessagingException
    {
	sessionProps = new Properties();
	sessionProps.put("mail.smtp.host", sc.getHost());
	String currentUser = sc.getUser();
	String fromHost = applConf.getHost();
	String s_defaultFromUser = (new StringBuffer().append(currentUser).
	    append("@").
	    append(fromHost)).toString();
	try {
	    defaultFromUser = new InternetAddress(s_defaultFromUser);
	}
	catch (AddressException e) {
	    throw new MessagingException("invalid from address, could not initialize sender; from address is " + s_defaultFromUser + ", exception is " + e.toString());
	}
	this.applConf = applConf;
	isDemoUser = currentUser.equals(applConf.getDemoUser());
	log.debug("MailSender() successful initialization, using default from-address " + s_defaultFromUser);
    }

    private Session getSession() {
	if (session == null)
	    session = Session.getDefaultInstance(sessionProps);
	return session;
    }

    public InternetAddress getDefaultFromUser() {
	return defaultFromUser;
    }

    public SendInfo send(ComposeInfo ci)
	throws MessagingException, ControllerException
    {
	log.debug("send() called");
	if (ci == null) throw new ControllerException("internal error, ComposeInfo parameter is null");

	SendInfo ret = null;
	Session s = getSession();
	MimeMessage m = new MimeMessage(s);

	try {	    
	    InternetAddress from = ci.getFrom();
	    if (from == null) throw new ControllerException("internal error, from address is null");
	    m.setFrom(from);
	    
	    String s_replyTo = ci.getReplyTo();
	    if (s_replyTo != null && !(s_replyTo.equals(""))) {
		Address[] ra = new Address[1];
		ra[0] = new InternetAddress(s_replyTo);
		log.debug("send() set reply to " + ra[0].toString());
		m.setReplyTo(ra);
	    }
	    
	    m.setRecipients(Message.RecipientType.TO, ci.getTo());
	    if (ci.getCc() != null)
		m.setRecipients(Message.RecipientType.CC, ci.getCc());
	    if (ci.getBcc() != null)
		m.setRecipients(Message.RecipientType.BCC, ci.getBcc());

	    if (log.isDebugEnabled())
		log.debug("send() setting subject to " + ci.getSubject());

	    m.setSubject(ci.getSubject(), applConf.getEncoding());

	    if (log.isDebugEnabled())
		log.debug("send() subject in MimeMessage is now " + m.getSubject());

	    
	    AttachmentHolder ah = ci.getAttachmentHolder();
	    int nbAttachments = ah.getNbAttachments();
	    
	    ComposeReference cr= ci.getReference();
	    log.debug("send() is ComposeReference null ? " + (cr == null));
	    boolean isForwardAsAttachment = (cr != null && cr.isForwardAsAttachment());
	    
	    boolean needMultipart = 
		(nbAttachments > 0) || isForwardAsAttachment;
	    
	    
	    if (! needMultipart)
		m.setText(ci.getBody(), applConf.getEncoding());
	    else {
		log.debug("send() need to attach " + nbAttachments + " files");
		
		// message will be multi-part
		Multipart multipart = new MimeMultipart();
		
		// part 1 is body
		MimeBodyPart messageBodyPart = new MimeBodyPart();
		messageBodyPart.setText(ci.getBody(), applConf.getEncoding());
		multipart.addBodyPart(messageBodyPart);
		
		// other parts are the attachments
		// if forwarding something, go get it and attach it
		if (isForwardAsAttachment) {
		    MimeMessage reference = cr.getReference().getMessageHeader().getMessage();
		    log.debug("send() forwarding, got reference");
		    try {
			MimeBodyPart[] forwards = MailTools.toMimeBodyParts(reference, applConf.getEncoding());
			for (int i = 0; i < forwards.length; i++)
			    multipart.addBodyPart(forwards[i]);
		    }
		    catch (Exception e) {
			throw new ControllerException("could not read message to forward", e);
		    }
		    log.debug("send() forwarding, added referenced message as body part");
		}

		// any attachments that the user added
		for (int i = 0; i < nbAttachments; i++) {
// 		messageBodyPart = new MimeBodyPart();
// 		messageBodyPart.setFileName(ah.getAttachmentName(i));
// 		messageBodyPart.setDataHandler(new DataHandler(ah.getAttachmentDataSource(i)));
		    messageBodyPart = ah.getAttachment(i);
		    multipart.addBodyPart(messageBodyPart);
		}

		// put parts in message
		m.setContent(multipart);
	    }

	    log.debug("send() trying to send...");
	    checkAuthorized(m);
	    Transport.send(m);
	    ret = new SendInfo(m, null);
	}
	catch (SendFailedException e) {
	    if (log.isDebugEnabled())
		log.debug("send() handling SendFailedException");
	    ret = new SendInfo(m, e);
	}
	catch (AddressException e) {
	    if (log.isDebugEnabled())
		log.debug("send() handling AddressException");
	    ret = new SendInfo(m, e);
	}

	log.debug("send() done.");
	return ret;
    }

    private void checkAuthorized(MimeMessage m) 
	throws SendFailedException
    {
	if (isDemoUser) {
	    log.debug("checkAuthorized() for demo, checking addresses");
	    try {
		checkDemoAddresses(m.getRecipients(Message.RecipientType.TO));
		checkDemoAddresses(m.getRecipients(Message.RecipientType.CC));
		checkDemoAddresses(m.getRecipients(Message.RecipientType.BCC));
	    }
	    catch (SendFailedException e) {
		throw e;
	    }
	    catch (MessagingException e) {
		throw new SendFailedException("there was something fishy checking addresses for demo user, aborting", e);
	    }
	}
    }

    private void checkDemoAddresses(Address[] checks)
	throws SendFailedException
    {
	if (checks != null)
	    for (int i = 0; i < checks.length; i++) {
		InternetAddress ia = (InternetAddress)checks[i];
		if (! defaultFromUser.getAddress().equals(ia.getAddress()))
		    throw new SendFailedException("security check for demo user failed: this user may only send mails to himself, that is to " + defaultFromUser.getAddress() + " (you are trying to send to " + ia.getAddress() + ")");
	    }
    }
}
