/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;

import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Store;
import org.apache.log4j.Logger;

/**
 * Good stuff to handle folders
 */
public class FolderManager
{
    private final Logger log = Logger.getLogger(FolderManager.class);
    private Store store;

    public FolderManager(Store store) {
	this.store = store;
    }

    public Folder addFolder(String folderName, boolean isMessageFolder)
	throws IllegalArgumentException, FolderSetupException
    {
	if (folderName == null || folderName.equals(""))
	    throw new IllegalArgumentException("parameter folderName must contain a value");

	if (log.isDebugEnabled())
	    log.debug("addFolder() called for name " + folderName
		+ "isMessageFolder=" + isMessageFolder);

	int mode =
	    (isMessageFolder ? Folder.HOLDS_MESSAGES : Folder.HOLDS_FOLDERS);

	Folder f = null;
	try {
	    f = store.getFolder(folderName);
	    if (f.exists())
		throw new FolderSetupException(folderName, "folder already exists", null);
	    // TO DO: how to check for actual problem ??
	    if (! f.create(mode))
		throw new FolderSetupException(folderName, "create failed", null);
	    f.setSubscribed(true);
	}
	catch (MessagingException e) {
	    throw new FolderSetupException(folderName, "unexpected error", e);
	}
	return f;
    }

    public void deleteFolder(Folder folder)
	throws IllegalArgumentException, FolderSetupException
    {
	if (folder == null) 
	    throw new IllegalArgumentException("parameter folder may not be null");

	String folderName = folder.getName();
	log.debug("deleteFolder() called for folder named " + folderName);
	try {
	    folder.setSubscribed(false);
	    if (folder.isOpen())
		folder.close(false);
	    if (! folder.delete(true))
		throw new FolderSetupException(folderName, "delete folder failed for an unknown reason", null);
	}
	catch (MessagingException e) {
	    throw new FolderSetupException(folderName, "unexpected error in delete folder", e);
	}
    }

    /**
     * JavaMail API doesn't seem to have functionality for moving messages
     * from one folder to another, so put it here !
     */
    public static void moveMessages(Folder fromFolder, Folder toFolder,
				    Message[] msgs)
	throws IllegalArgumentException, ControllerException
    {
	if (fromFolder == null || toFolder == null)
	    throw new IllegalArgumentException("parameters fromFolder and toFolder may not be null");

	try {
	    // copy the messages to other folder
	    fromFolder.copyMessages(msgs, toFolder);

	    // delete them from this folder
	    // NOTE : I only know how to do this by setting their flags
	    // to deleted, and then expunging the folder.
	    // This has the side effect of also expunging other messages
	    // in the folder which have been marked as deleted !?
	    for (int i = 0; i < msgs.length; i++) {
		Message msg = msgs[i];
		msg.setFlag(Flags.Flag.DELETED, true);
	    }
	    if (! fromFolder.isOpen())
		fromFolder.open(Folder.READ_WRITE);
	    fromFolder.expunge();
	}
	catch (MessagingException e) {
	    throw new ControllerException("error in moveMessages", e);
	}
    }
}

