/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller.action;

import java.net.URL;
import java.util.Collections;
import java.util.Map;

import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpUtils;
import javax.security.auth.login.LoginException;
import javax.security.auth.login.FailedLoginException;

// cocoon
import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.Constants;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.cocoon.acting.AbstractAction;

// for logging use log4j
import org.apache.log4j.Logger;

import net.jatec.ironmailer.controller.AbstractWorkerBean;
import net.jatec.ironmailer.controller.ServerConnection;
import net.jatec.ironmailer.framework.CocoonTools;
import net.jatec.ironmailer.framework.HttpSessionManager;
import net.jatec.ironmailer.framework.ServletTools;
import net.jatec.ironmailer.framework.StringTools;


/**
 * @author jwk
 */
public class LoginAction extends AbstractAction
{
    private final Logger log = Logger.getLogger(LoginAction.class);
    
    /**
     * Interface implementation
     * @throws Exception according to interface, but should never happen.
     * If there is some error, null is returned and the exception is logged.
     */
    public Map act(Redirector redirector, SourceResolver resolver, 
		   Map objectModel, String source,
		   Parameters par) 
	throws Exception 
    {
        HttpServletRequest req =(HttpServletRequest)objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);

	String username = null;
	String password = null;
	String server = null;
	String serverConnectionImplementation = null;
	String workerBeanImplementation = null;

	try {
	    if (log.isDebugEnabled()) {
		log.debug("act() called");
		log.debug("act() objectModel " + StringTools.toString(objectModel));
		log.debug("act() source " + source);
		log.debug("act() par " + CocoonTools.toString(par));
	    }

	    // Read Form parameters
	    username = ServletTools.getString(req,"username",true);
	    password = ServletTools.getString(req,"password",true);

	    // Read Cocoon configuration parameters (from sitemap)
	    // Note: if a parameter is not found, Cocoon will throw a
	    // ParameterException
	    server = par.getParameter("server");
	    serverConnectionImplementation = par.getParameter("serverConnectionImplementation");
	    workerBeanImplementation = par.getParameter("workerBeanImplementation");


	    log.debug("act() called for user " + username + " on server " + server);
	    ServerConnection sc = (ServerConnection)Class.forName(serverConnectionImplementation).newInstance();
	    log.debug("act() created ServerConnection instance");
	    sc.connect(server, username, password.toCharArray());
	    log.debug("act() successfully connected through ServerConnection");

	    // Location info
	    URL url = new URL(HttpUtils.getRequestURL(req).toString());
	    log.debug("act() called via url " + url.toString());
	    
	    // create session and store worker bean
	    HttpSession session = req.getSession();
	    AbstractWorkerBean workerBean = (AbstractWorkerBean)Class.forName(workerBeanImplementation).newInstance();
	    workerBean.initialize(sc, url, par);

	    // store worker bean in session
	    session.setAttribute("workerBean", workerBean);

	    // put a marker in the session for successful login
	    HttpSessionManager.getInstance().setLoginDone(session, username);

	    return Collections.EMPTY_MAP;
	}
	catch (Throwable t) {
	    // All errors result in the action failing, and in the setting
	    // of error fields in the request for later display by the login
	    // page. 
	    // We distinguish between FailedLoginException, which is expected
	    // when the user gives bad information; and other exceptions
	    // which are unexpected, and are therefore logged on a different
	    // level.
	    if (t instanceof FailedLoginException) {
		log.info("act() " + "failed authentication for user " 
			 + username + " (exception " + t.toString() + ")");
		req.setAttribute("error.content", t.toString());
		req.setAttribute("error.stack", "(not shown)");
	    } else {
		log.error("act() got some unexpected error, probably due to a setup error", t);
		req.setAttribute("error.content", t.toString());
		req.setAttribute("error.stack", StringTools.getStackTraceAsString(t));		
	    }
	    return null;
	}
	
    }
}

