/* 
 *   Copyright (C) 2002, 2003, 2004 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;

import java.util.Collection;
import java.util.Vector;
import javax.mail.Folder;
import javax.mail.Store;
import javax.mail.FolderNotFoundException;
import javax.mail.URLName;
import javax.mail.MessagingException;
import org.apache.log4j.Logger;

import net.jatec.ironmailer.model.ApplicationConfiguration;
import net.jatec.ironmailer.model.MailboxOverview;
import net.jatec.ironmailer.model.MailFolderHeader;
import net.jatec.ironmailer.model.ModelException;

/**
 * Part of the controller responsible for getting a mailbox
 */
public class MailboxFetcher
{
    private final Logger log = Logger.getLogger(MailboxFetcher.class);

    public MailboxOverview createMailboxOverview(Store store, ApplicationConfiguration applConf) 
	throws IllegalArgumentException, MessagingException, UserInstallationException, ControllerException
    {
	log.debug("createMailboxOverview() called");
	if (store == null) throw new IllegalArgumentException("parameter store must be given");

	URLName url;
	String user = "unknown";
	Vector mailFolderHeaders;

	try {
	    Folder rootFolder = store.getDefaultFolder();
	    log.debug("createMailboxOverview() got default folder");
	    url = rootFolder.getURLName();
	    user = url.getUsername();
	    log.debug("createMailboxOverview() going to get mail folders");
	    if ((rootFolder.getType() & Folder.HOLDS_FOLDERS) == 0)
		throw new UserInstallationException("root folder has no subfolders", user, applConf.getAdmin());
	    
	    // get all subscribed folders - these should correspond 
	    // to the mail folders to be presented to the user.
	    // From the API:
	    // - list("*") on "Personal" will return the whole hierarchy.
	    // - list("%") on "Personal" will return "Finance" and "Jokes".
	    // Note on imap implementation : this will return all names
	    // in the user's .mailboxlist file in the user home
	    // We must use * to get the whole hierarchy: strangely enough,
	    // if we use % we also in fact get all folders (bug of javamail?),
	    // but we lose the namespace if we do.
//	    Folder[] folders = rootFolder.listSubscribed("*");
//	    Folder[] folders = rootFolder.listSubscribed("%");
	    Folder[] folders = rootFolder.list();

	    if (folders == null)
		throw new UserInstallationException("user has no folders", user, applConf.getAdmin());
	    
	    // the nb of folders is a good approximation for the final size
	    // of the vector, which will be <= in size
	    if (log.isDebugEnabled())
		log.debug("createMailboxOverview() got " + folders.length + " potential mail folders");
	    mailFolderHeaders = new Vector(folders.length);

	    // be sure to retrieve INBOX at first place.
	    // Depending on how backend is configured, INBOX might appear anywhere
	    // or not at all. So get it first !
	    if (log.isDebugEnabled())
		log.info("retrieve primary folder");
	    Folder primaryFolder = store.getFolder(MailFolderHeader.PRIMARY_FOLDER_NAME);
	    if (primaryFolder.exists()) {
		try {
		    MailFolderHeader mfh = getMailFolderHeader(primaryFolder,0);
		    mailFolderHeaders.insertElementAt(mfh, 0);
		}
		catch (ModelException e) {
		    log.warn("unexpected error on primary folder",e);
		}
	    }
	    else {
		log.info("primary folder not found for " + user);
	    }


	    for (int i = 0; i < folders.length; i++) {
		if (folders[i].exists() && 
		    folders[i].isSubscribed() &&
		    (! MailFolderHeader.PRIMARY_FOLDER_NAME.equals(folders[i].getName()))
		    ) {
		    if (log.isDebugEnabled())
			log.debug("createMailboxOverview() fetching header for folder nr " + i);
		    try {
			MailFolderHeader mfh = getMailFolderHeader(folders[i], 0);
			mailFolderHeaders.add(mfh);
			
			if (log.isDebugEnabled())
			    log.debug("createMailboxOverview() got header for folder nr " + i);
		    }
		    catch (MessagingException e) {
			log.warn("createMailboxOverview() got an exception with content: " + e.toString() + ", propagating");
			throw e;
		    }
		    catch (ModelException e) {
			log.warn("createMailboxOverview() got an exception trying to create the model: " + e.toString() + ", creating a controller exception");
			throw new ControllerException("createMailboxOverview() got an exception trying to create the model: " + e.toString(), e);
		    }
		}
	    }
	}
	catch (FolderNotFoundException e) {
	    throw new UserInstallationException("shouldn't happen here: " + e.toString(), user, applConf.getAdmin());
	}

	if (log.isDebugEnabled()) 
	    log.debug("createMailboxOverview() completed, now returning the object");

	return new MailboxOverview(url, mailFolderHeaders);
    }

    private MailFolderHeader getMailFolderHeader(Folder f, int level) 
	throws FolderNotFoundException, MessagingException, ModelException
    {
	if (log.isDebugEnabled())
	    log.debug("getMailFolderHeader() called on folder " + f.getName());

	MailFolderHeader mfh = new MailFolderHeader(f, level);
	if (mfh.holdsFolders()) {
	    Folder[] subfs = f.listSubscribed();
	    if (log.isDebugEnabled())
		log.debug("getMailFolderHeader() in case where subfolders exist");

	    MailFolderHeader[] mfh_subs = new MailFolderHeader[subfs.length];
	    for (int i = 0; i < subfs.length; i++) {
		mfh_subs[i] = getMailFolderHeader(subfs[i], level+1);
	    }
	    mfh.setSubfolders(mfh_subs);
	}
	return mfh;
    }


}
