/* 
 *   Copyright (C) 2002, 2003 Jatec AG, Switzerland
 *
 * This file is part of IronMailer.
 *
 * IronMailer is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

package net.jatec.ironmailer.controller;

import java.util.Properties;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.AuthenticationFailedException;
import javax.mail.MessagingException;
import javax.security.auth.login.LoginException;
import javax.security.auth.login.FailedLoginException;
import org.apache.log4j.Logger;

/**
 * Represents a simplified view of a mail server connection.
 * The server to send is the same as the server where to get mails,
 * and only IMAP protocol is supported.
 * An instance can only be instantiated with valid credentials to open
 * a mailbox, so it can also be used as a login verificator.
 */
public class MailServerConnection implements ServerConnection
{
    private final Logger log = Logger.getLogger(MailServerConnection.class);

    private Session session;
    private Store store;
    private String user;
    private String host;

    public void connect(String host, String user, char[] password)
	throws IllegalArgumentException, FailedLoginException, LoginException
    {
	log.info("connect() called for host " + host + " and user " + user);
	if (host == null || host.equals("")) 
	    throw new IllegalArgumentException("parameter host must contain a value");
	if (user == null || user.equals("")) 
	    throw new IllegalArgumentException("parameter user must contain a value");
	if (password == null || password.length == 0)
	    throw new IllegalArgumentException("parameter password must contain a value");

	try {
	    Properties props = new Properties();
	    props.put("mail.imap.host", host);
	    session = Session.getInstance(props);
	    store = session.getStore("imap");
	    store.connect(host, user, String.valueOf(password));
	} 
	catch (AuthenticationFailedException e) {
	    log.info("connect() failed: " + e.toString());
	    throw new FailedLoginException(e.toString()); 
	}
	catch (Throwable t) {
	    log.error("connect() failed with unexpected error " + t.toString(), t);
	    throw new LoginException("unexpected connection exception " + t.toString());
	}

	// remember user
	this.user = user;
	this.host = host;
    }

    /**
     * After calling this, don't think you can access the 
     * members again ;)
     */
    public void disconnect() {
	log.info("disconnect() called");
	try {
	    store.close();
	} 
	catch (MessagingException e) {
	    log.info("disconnect() ignoring exception: " + e.toString());
	}
	store = null;
	session = null;
    }

    public Session getSession() {
	return session;
    }

    public Store getStore() {
	return store;
    }

    public String getUser() {
	return user;
    }

    public String getHost() {
	return host;
    }
}
