--  arch-debug.ads: Architecture-specific debug channels.
--  Copyright (C) 2023 streaksu
--
--  This program is free software: you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation, either version 3 of the License, or
--  (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program.  If not, see <http://www.gnu.org/licenses/>.

with Devices;

package Arch.Debug with
   Abstract_State => Debug_State,
   Initializes    => Debug_State
is
   --  This package implements read / write access to a target-specific debug
   --  channel, which could take the shape of a serial port or debug monitor.

   --  Whether the abstracted channel supports write only or read / write.
   Supports_Read : constant Boolean;

   --  Read byte device arrays atomically.
   --  @param Message Array to print.
   procedure Read (Message : out Devices.Operation_Data)
      with Global => (In_Out => Debug_State);

   --  Print a character message atomically.
   --  @param Message Character to print.
   procedure Print (Message : Character)
      with Global => (In_Out => Debug_State);

   --  Print a string message atomically.
   --  @param Message String to print.
   procedure Print (Message : String)
      with Global => (In_Out => Debug_State);

   --  Print byte device arrays atomically.
   --  @param Message Array to print.
   procedure Print (Message : Devices.Operation_Data)
      with Global => (In_Out => Debug_State);

private

   #if ArchName = """aarch64-stivale2"""
      Supports_Read : constant Boolean := False;
   #elsif ArchName = """arm-raspi2b"""
      Supports_Read : constant Boolean := False;
   #elsif ArchName = """sparc-leon3"""
      Supports_Read : constant Boolean := False;
   #elsif ArchName = """x86_64-multiboot2"""
      Supports_Read : constant Boolean := True;
   #end if;
end Arch.Debug;
