/*
* Test ipv6
* written by ale in milano on 31 Oct 2023

Copyright (C) 2023 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#include <syslog.h>
#include <popt.h>

// format of data packet
#include "config_names.h"
#include "dbstruct.h"

#include <arpa/inet.h>
#include <unistd.h>
#include <errno.h>

#include "ip_util.h"
#include "initial_count.h"
#include "percent_prob.h"
#include "in_range_ip6.h"
#include "rehabilitated_prob.h"
#include "test/pic.h"

static app_private ap;
static char const err_prefix[] = "TESTipv6";

#include "setsig_func.h"
#include <assert.h>

static char *db_block_name = IPQBDB_DATABASE_NAME;
static int version_opt, help_opt, verbose = -1, yes_destroy_db;
static pic_params pp = { IPQBDB_INITIAL_DECAY, IPQBDB_INITIAL_COUNT,
	40, 1, 1, PIC_HEIGHT};

static int initial_probability;
static double index_factor = IPQBDB_INDEX_FACTOR;

static struct poptOption opttab[] =
{
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"yes-destroy-db", 'y', POPT_ARG_NONE|POPT_ARGFLAG_OPTIONAL, &yes_destroy_db, 0,
	"Skip destroy-db confirmation", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	{"attacks-per-minute", 'a', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &pp.attacks_per_minute, 0,
	"Number of attacks per minute", "integer"},
	{"db-block", 'b', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_block_name, 0,
	"The database of blocked addresses", "filename"},
	{"initial-count", 'c', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &pp.initial_count, 0,
	"Set initial probability count", "integer"},
	{"index-factor", 'f', POPT_ARG_DOUBLE|POPT_ARGFLAG_SHOW_DEFAULT, &index_factor, 0,
	"Set range index factor", "float"},
	{"minutes-per-pixel", 'm', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &pp.minutes_per_pixel, 0,
	"Minutes per vertical pixel", "integer"},
	{"initial-decay", 't', POPT_ARG_DOUBLE|POPT_ARGFLAG_SHOW_DEFAULT, &pp.initial_decay, 0,
	"The time taken for the block probability to halve (seconds)", "float"},
	{"seed", 's', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &pp.seed, 0,
	"Seed for the sequence of pseudo-random  integers", "integer"},
	{"pic-height", 'p', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &pp.pic_height, 0,
	"The height of the picture, in pixels", "integer"},
	POPT_TABLEEND
};

//	struct abuse_profile
//	/*
//	* Kind of abuse being committed by infected IPs.
//	*/
//	{
//		double decay;
//		int skill, initial_count;
//		// variable part
//		time_t last;
//	} profile[] =
//	//	{
//		//   DECAY, SKILL, COUNT,  LAST}, // DESCRIPTION
//		{  21600.0,   400,     5, -1200}, // ESMTP command error
//		{  21600.0,   150,     5, -1200}, // SMTP auth dictionary attack
//		{  21600.0,    80,     5, -1200}, // dictionary attack
//		{     60.0,    20,     4, -1200}, // SPF failure
//		{  21600.0,    20,    15, -1200}, // Domain does not exist
//		{2592000.0,    10,     0, -1200}, // IP without abuse team
//		{ 172800.0,     7,     0, -1200}, // spamtraps in rcptfilter.sh
//		{  21600.0,     4,    15, -1200}, // Relaying denied
//		{ 172800.0,     1,     4, -1200}, // X-Spam-Flag: YES
//		{  86400.0,     1,     0, -1200}  // PHP scan
//	};
//
//	#define PROFILE_MAX (sizeof profile/sizeof profile[0])
//
//	#define PRO_MAX 3

typedef struct ip_addr_pool
{
	int prob;   // probability to be infected, percentage.
	char infected;
	char nu[3];
} ip_addr_pool;

#define POOL_MAX (IP_PER_DOT*PIC_WIDTH)

static int simplified_judge(DB *db6, unsigned char ip_addr[16], time_t now)
{
	int verdict = 0;  // 0 = accept, 1 block

	DBT key, data;
	DBC *dbc6 = NULL;
	ip_data_t ip_data;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	memset(&ip_data, 0, sizeof ip_data);
	data.ulen = data.size = sizeof ip_data;
	data.data = &ip_data;
	key.flags = data.flags = DB_DBT_USERMEM;

	int found = 0;
	int need_write = 0;
	unsigned char buf[16];

	int rc = db6->cursor(db6, NULL, &dbc6, DB_WRITECURSOR);
	if (rc == 0)
	{
		memcpy(buf, ip_addr, 16);
		key.data = buf;
		key.ulen = key.size = 16;
		rc = dbc6->get(dbc6, &key, &data,  DB_SET_RANGE);
		if (rc == 0)
		{
			if (in_range_ip6(ip_addr, buf, ip_data.plen))
				found = 1;
			else
				rc = DB_NOTFOUND;
		}
	}

	if (found && ip_data.probability > 0)
	{
		if ((ip_data.last_block + IPQBDB_RECENT_DROP_TIME >= now &&
			ip_data.rep_ruling < IPQBDB_REPETITION_MAX) ||
			ip_data.decay >= IPQBDB_NEVER_DECAY) // static block
		{
			verdict = 1;
			ip_data.last_block = now;
			ip_data.block_cnt += 1;
			ip_data.rep_ruling += 1;
			need_write = 1;
		}
		else
		{
			// rehabilitate
			time_t delta = now - ip_data.last_update;
			if (IPQBDB_UPDATE_TICK_TIME < delta && ip_data.decay > 0.0)
			{
				unsigned int const prob =
					rehabilitated_prob(ip_data.probability,
						delta, ip_data.decay);
				if (prob < (unsigned) ip_data.probability)
				{
					ip_data.probability = prob;
					ip_data.last_update = now;
					/*
					* Was setting need_write = 1 in version 1.x,
					* but rehabilitation works the same if it is
					* upgraded in steps or all in one (up to
					* rounding.)  Omitting this write implies
					* write_block() must also rehabilitate.
					*/
				}
			}

			// rule
			int const toss = rand();
			if (toss < ip_data.probability)
			{
				verdict = 1;
				ip_data.last_block = now;
				ip_data.block_cnt += 1;
				ip_data.rep_ruling = 0;
				need_write = 1;
			}
		}

		/*
		* update the record (re-insert it if it was deleted meanwhile)
		*/
		if (need_write)
		{
			rc = dbc6->put(dbc6, &key, &data,  DB_CURRENT);
			if (rc)
			{
				report_error(&ap, LOG_CRIT,
					"cannot write record: %s\n", db_strerror(rc));
				return -1;
			}
		}
	}
	else if (rc != DB_NOTFOUND)
	{
		report_error(&ap, LOG_CRIT, "cannot read key: %s\n",
			rc? db_strerror(rc): "bad data record");
		return -1;
	}

	if (dbc6)
		dbc6->close(dbc6);

	return verdict;
}

static day_stats stats;
static long int ip_changed_count;

static int
write_block_preset(DB *db, pic_handle *pic, int ip_addr_ndx, time_t now)
/*
* Check if bad packet can go through.  In case block it.
*/
{
	assert(db);

	ip_u ip_addr;
	memset(&ip_addr, 0, sizeof ip_addr);
	ip_addr.ip = 6;
	ip_addr.u.ipv6[0] = 0x20;
	ip_addr.u.ipv6[1] = 1;
	ip_addr.u.ipv6[2] = 0xd;
	ip_addr.u.ipv6[3] = 0xb8;
	ip_addr.u.ipv6[7] = ip_addr_ndx / 256;
	ip_addr.u.ipv6[8] = ip_addr_ndx % 256;

	unsigned char ip[16];
	memcpy(ip, ip_addr.u.ipv6, sizeof ip);

	int rtc = simplified_judge(db, ip_addr.u.ipv6, now);
	if (rtc < 0)
		return 1; // error
	else if (rtc)
	{
		stats.blocked += 1;
		return 0; // packet blocked, cannot be caught
	}

	// once every 8 block a whole range
	if (rand() % 8 == 0)
	{
		int cplen = rand() % 56;  // complement
		if (cplen)
			ip_addr.plen = 128 - cplen;
	}

	int probability = initial_probability;
	if (write_block(db, &ip_addr, pp.initial_decay, 1, &probability, 0,
		NULL, NULL))
			return 1;

	pic_set_caught(pic, ip);

	if (memcmp(ip, ip_addr.u.ipv6, sizeof ip))
		++ip_changed_count;
	stats.caught += 1;
	return 0;
}

static int cannot_delete(switchable_fname *block_fname)
{
	FILE *out = stdout;
	if (!isatty(fileno(out)))
		out = stderr;
	if (!isatty(fileno(out)))
		return 1;
	make_six(block_fname);

	int yn;
	if (yes_destroy_db)
		yn = 'y';
	else
	{
		fprintf(out, "CAUTION: This destroys %s.  Continue? (y/n) ",
			block_fname->fname);

		yn = getchar();
	}

	int rtc = yn != 'y' && yn != 'Y';
	if (rtc == 0)
	{
		if (unlink(block_fname->fname) && errno != ENOENT)
		{
			fprintf(out, "Cannot delete %s: %s\n",
				block_fname->fname, strerror(errno));
			rtc = 1;
		}
	}

	return rtc;
}

static void set_pool_probability(ip_addr_pool *pool, int reversed)
/*
* Set the probability that each IP will be infected/ fixed.
* Also reset infected status.
*/
{
	int mod_left, mod_right;
	if (reversed)
	{
		mod_left = 3;
		mod_right = 1;
	}
	else
	{
		mod_left = 1;
		mod_right = 3;
	}
	int i = 0;
	while (i < POOL_MAX/2)
	{
		pool[i].infected = 0;
		pool[i++].prob = rand() & mod_left;
	}
	i += 48; // 3 dots
	while (i < POOL_MAX)
	{
		pool[i].infected = 0;
		pool[i++].prob = rand() & mod_right;
	}
}

int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s at %s\n",
			err_prefix, poptStrerror(rtc), poptBadOption(opt, 0));
		errs = 1;
	}
	else
	{
		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n"
			DB_VERSION_STRING "\n", err_prefix);
			errs = 2;
		}

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			fputs("Make sure to not mess up your working database.\n\n"
			"The test simulates attacks from random 2001:db8::/56 IPs\n"
			"and draws a histogram of the resulting records, where each\n"
			"dot stands for 16 IPs, painted in blue for ranges, red for\n"
			"singletons, green for both.  Attacking IPs can be blocked\n"
			"or caught, marking the latter with a black dot.  IPs on the\n"
			"right have a higher probability to attack than those on the\n"
			"left.  A narrow interval in between never attacks.  At the\n"
			"halfway all IPs are fixed and sides reversed.  Time grows\n"
			"downward.\n", stdout);
			errs = 2;
		}

		// popt sets verbose to 0 if no arg is given
		verbose += 1;

		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;

		if (poptPeekArg(opt) != NULL)
		{
			report_error(&ap, LOG_ERR, "unexpected argument: %s\n",
				poptGetArg(opt));
			errs = 1;
		}
	}

	if (errs == 1)
		poptPrintUsage(opt, stdout, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (pp.initial_count == 999)
		initial_probability = 0;
	else
	{
		if ((initial_probability = count2prob(pp.initial_count)) <= 0)
		{
			report_error(&ap, LOG_NOTICE,
				"count %d is too high: initial probability set to 1\n",
				pp.initial_count);
		}
		initial_probability += 1; // rounded
	}

	if (index_factor != IPQBDB_INDEX_FACTOR)
		set_ipqbdb_index_factor(index_factor);

	switchable_fname *block_fname = database_fname(db_block_name, &ap);
	if (errs == 0 && cannot_delete(block_fname))
		errs = 1;

	ip_addr_pool *pool = NULL;
	if (errs == 0)
	{
		pool = calloc(POOL_MAX, sizeof(ip_addr_pool));
		if (pool == NULL)
			errs = 1;
		else
			set_pool_probability(pool, 0);
	}

	pic_handle *pic = NULL;
	if (errs == 0)
	{
		stats.widest = 128;
		pic = pic_open(&pp, &stats);
		if (pic == NULL)
			errs = 1;
	}

	if (errs)
		rtc = 1;
	else
	{
		DB_ENV *db_env = NULL;
		DB *db6_block = NULL;
		if (block_fname)
			rtc = open_database(block_fname, &ap, &db_env, NULL, &db6_block);
		else
			rtc = 1;

		setsigs();
		srand(pp.seed);
		int const max_attacks_per_loop =
			pp.attacks_per_minute * pp.minutes_per_pixel;
		int attacks_per_loop = 1;
		time_t clock = 0, halfway =
			(60 * pp.minutes_per_pixel * pp.pic_height)/2;
		int day = 0;
		while (rtc == 0 && caught_signal == 0)
		{
			time_t eod = clock + 86400;
			int rows = 0;
			while (rtc == 0 && caught_signal == 0 && clock < eod)
			{
				set_simulation_clock(clock);
				for (int attacks = 0; attacks < attacks_per_loop;)
				{
					unsigned i = rand() % POOL_MAX;

					// catch virus
					if (pool[i].infected == 0 &&
						rand() % 100 < pool[i].prob)
							pool[i].infected = 1;

					// viral action
					if (pool[i].infected)
					{
						++attacks;
						rtc = write_block_preset(db6_block, pic, i, clock);
						if (rtc)
							goto out_of_loop;

						// eliminate virus
						if (rand() % 200 < pool[i].prob)
						{
							pool[i].infected = 0;
							stats.fixed += 1;
						}
					}

				}
				int is_halfway = 0;
				if (clock == halfway)
				{
					is_halfway = 1;
					set_pool_probability(pool, 1);
					clock += 84600;
					puts("One day break");
				}
				rtc = pic_draw_line(pic, clock, db6_block, is_halfway);
				clock += 60 * pp.minutes_per_pixel;
				++rows;
				if (attacks_per_loop < max_attacks_per_loop)
					++attacks_per_loop;
			}
			for (unsigned i = 0; i < POOL_MAX; ++i)
				if (pool[i].infected)
					stats.infected += 1;
			printf("End of day %d; infected %d, fixed %d, blocked %d, caught %d, ranges %d (max /%d), singletons %d\n",
				++day, stats.infected, stats.fixed, stats.blocked, stats.caught, stats.ranges/rows, stats.widest, stats.singletons/rows);
			memset(&stats, 0, sizeof stats);
			stats.widest = 128;
		}

	out_of_loop:

		set_simulation_clock(0);

		close_db(db6_block);
		close_dbenv(db_env, 1);
	}

	if (pic)
		pic_done(pic);

	if (ip_changed_count)
		printf("IPs changed after write_block() : %ld\n", ip_changed_count);
	free(block_fname);
	free(pool);
	return rtc;
}
