/*
* IPv6 comparison function written by ale in milano on 20 oct 2023

Copyright (C) 2023 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#include "in_range_ip6.h"
#include <string.h>
#include <assert.h>

int in_range_ip6(unsigned char const ip1[16],
	unsigned char const ip2[16], unsigned plen)
/*
* Check whether ip1 and ip2 are equal for the first plen bits.
* If ip1 and ip2 are the start and end points of a range, that is if
* they have the lowes 128-plen bits all set to 0 and 1 respectively,
* then an exact match is detected.
*
* Return 1 if in_range, 2 if exact, 0 if not.
*/
{
	assert(ip1);
	assert(ip2);

	int ndx = 0;
	if (plen > 128)
		return 0;

	if (plen == 0)
		return memcmp(ip1, ip2, 16) == 0? 2: 0;

	while (plen >= 8)
	{
		if (ip1[ndx] != ip2[ndx])
			return 0;
		ndx += 1;
		plen -= 8;
	}

	if (plen)
	{
		unsigned char mask = (1 << (8 - plen)) - 1;
		unsigned char nmask = ~mask;
		unsigned char i1 = ip1[ndx];
		unsigned char i2 = ip2[ndx];
		if ((i1 & nmask) != (i2 & nmask))
			return 0;

		if ((i1 & mask) != 0 || (i2 & mask) != mask)
			return 1;
	}

	while (ndx < 16)
	{
		if (ip1[ndx] != 0 || ip2[ndx] != 255)
			return 1;
		ndx += 1;
	}

	return 2;
}

#if defined TEST_MAIN
#include <stdio.h>
#include <stdlib.h>
#include <arpa/inet.h>

int main(int argc, char *argv[])
{
	while (argc > 3)
	{
		char *ip1_s = argv[1];
		char *ip2_s = argv[2];
		unsigned plen = atoi(argv[3]);
		char ip1[16];
		char ip2[16];

		if (plen > 128 || plen <= 0 ||
			inet_pton(AF_INET6, ip1_s, ip1) != 1 ||
			inet_pton(AF_INET6, ip2_s, ip2) != 1)
		{
			printf("arguments must be triple of ip1, ip2, prefix len\n");
			return 1;
		}

		int rtc = in_range_ip6(ip1, ip2, plen);
		printf("%s %c= %s mod /%d\n", ip1_s,
			rtc == 2? '=': rtc == 1? '~': '!', ip2_s, plen);
		argv += 3;
		argc -= 3;
	}

	return 0;
}
#endif // TEST_MAIN
