/*
* TESTrehabilitated_prob.c

Copyright (C) 2019-2021 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

#include "config_names.h"
#include "dbstruct.h"

#include "percent_prob.h"
#include "rehabilitated_prob.h"
#include "initial_count.h"

#include <string.h>



/*

 count = 4, decay = 60

 91.18.75.169 Mar 15 06:19:03 0 new decay: 60, prob: 6.25%, SPF failure
 91.18.75.169 Mar 15 06:19:28 0 old decay: 60, prob: 12.50%, SPF failure
 91.18.75.169 Mar 15 06:20:57 0 old decay: 60, prob: 9.81%, SPF failure
 91.18.75.169 Mar 15 12:24:17 0 old decay: 60, prob: 6.25%, SPF failure
 91.18.75.169 Mar 15 12:24:41 0 old decay: 60, prob: 9.47%, SPF failure
 91.18.75.169 Mar 15 12:25:30 0 old decay: 60, prob: 11.80%, SPF failure


 Get intervals:
 ps=$(date --date 06:19:03 +%s); for t in 06:19:28  06:20:57 12:24:17 12:24:41 12:25:30; do s=$(date --date "$t" "+%s"); diff=$((s - ps)); printf "%6d %s\n" "$diff" "$t"; ps=$s; done

 ./TESTrehabilitated_prob  4 60 delta 25 89 21800 24 49

 In theory, this should give the same numbers as logged.
 
 Unless the violation occurs in the same TCP connection and ibd-judge is
 only called on new connectons, a logged violation implies a packet was
 accepted, and possibly rehabilitated.  Getting 12.50% after 25 seconds
 implies there was no new connection.
 
 However, it doesn't.  It works with:
 
 ./TESTrehabilitated_prob  4 60 delta ~25 81 21800 24 41

 Looking at the mail log, it is possible to set intervals more precisely,
 but still don't get the exact numbers:

Mar 15 06:19:03 error,relay=91.18.75.169,port=21744: 517 SPF fail 44d37d4d.3080106@tana.it: Address does not pass the Sender Policy Framework

Mar 15 06:19:20 started,ip=[91.18.75.169],port=[21907]
Mar 15 06:19:28 error,relay=91.18.75.169,port=21907: 517 SPF fail 44d37d4d.3080106@tana.it: Address does not pass the Sender Policy Framework

Mar 15 06:20:08 started,ip=[91.18.75.169],port=[22016]
Mar 15 06:20:48 started,ip=[91.18.75.169],port=[22038]
Mar 15 06:20:57 error,relay=91.18.75.169,port=22038: 517 SPF fail 44d37d4d.3080106@tana.it: Address does not pass the Sender Policy Framework

Mar 15 12:24:09 started,ip=[91.18.75.169],port=[33278]
Mar 15 12:24:17 error,relay=91.18.75.169,port=33278: 517 SPF fail adriana@cavaliere.it: Address does not pass the Sender Policy Framework

Mar 15 12:24:33 started,ip=[91.18.75.169],port=[33474]
Mar 15 12:24:41 error,relay=91.18.75.169,port=33474: 517 SPF fail adriana@cavaliere.it: Address does not pass the Sender Policy Framework

Mar 15 12:25:22 started,ip=[91.18.75.169],port=[33659]
Mar 15 12:25:30 error,relay=91.18.75.169,port=33659: 517 SPF fail adriana@cavaliere.it: Address does not pass the Sender Policy Framework

 ./TESTrehabilitated_prob 4 60 delta :17 ~8 :40 :40 ~9 :21792 ~8 :16 ~8 :41 ~8

 works with (24 = :16 + ~8):

 ./TESTrehabilitated_prob 4 60 delta :17 ~8 :40 :41 ~9 :21792 ~8 24 :41 ~8
*/
int main(int argc, char *argv[])
{
	int steps = 10;
	int count = IPQBDB_INITIAL_COUNT;
	unsigned int probability  = count2prob(IPQBDB_INITIAL_COUNT);
	double decay = IPQBDB_INITIAL_DECAY;
	time_t delta = IPQBDB_UPDATE_TICK_TIME + 1;
	char *in_mode = "dec";
	enum {mode_dec, mode_hit, mode_delta} mode = mode_dec;

	probability += 1; // rounded

	// override
	if (argc > 1)
	{
		if (strcmp(argv[1], "--help") == 0 || strcmp(argv[1], "-h") == 0)
		{
			printf(
				"%s: show probability change for given initial-count and decay.\n"
				"\targuments (use dash (\"-\") to use default):\n"
				"\tinitial-count, default %d.\n"
				"\tdecay, default %g.\n"
				"\tmode (\"dec\", \"hit\", \"delta\"), default %s.\n"
				"\t\tdec: probability decay at regular intervals,\n"
				"\t\thit: getting caught at regular intervals,\n"
				"\t\tdelta: getting caught at given intervals.\n"
				"\tdelta (seconds between two steps), default %ld.\n"
				"\t\tif mode == delta, delta must be given for each interval,\n"
				"\t\tif argument starts with column (\":\") no hit,\n"
				"\t\tif starts with tilde (\"~\") no rehabilitation.\n"
				"\tsteps (times to repeat if mode != delta), default %d.\n\n",
				argv[0], count, decay, in_mode, delta, steps);
			return 0;
		}
		if (strcmp(argv[1], "-"))
		{
			count = atoi(argv[1]);
			probability  = count2prob(count);
		}
		if (argc > 2)
		{
			if (strcmp(argv[2], "-"))
				decay = atof(argv[2]);
			if (argc > 3)
			{
				if (strcmp(argv[3], "-"))
				{
					if (strcmp(argv[3], "dec") != 0)
					{
						if (strcmp(argv[3], "hit") == 0)
						{
							mode = mode_hit;
							in_mode = "hit";
						}
						else if (strcmp(argv[3], "delta") == 0)
						{
							mode = mode_delta;
							in_mode = "delta";
							steps = argc - 4;
						}
					}
				}
				if (mode != mode_delta && argc > 4)
				{
					if (strcmp(argv[4], "-"))
						delta = atol(argv[4]);
					if (argc > 5 && strcmp(argv[5], "-"))
							steps = atoi(argv[5]);
				}
			}
		}
	}

	printf("arguments:\ncount = %d, decay = %g, mode = %s\n\n",
		count, decay, in_mode);

	unsigned int hit_probability = probability;
	for (int i = 0; i < steps; ++i)
	{
		int rehab = 1, hit = 1, lead = ' ';
		if (mode == mode_delta)
		{
			char *arg = argv[i+4];
			switch (lead = *arg)
			{
				case '~': // no rehabilitation, just hit
				{
					rehab = 0;
					++arg;
					break;
				}
				case ':': // no hit, just rehabilitation
				{
					hit = 0;
					++arg;
					break;
				}
			}
			delta = atol(arg);
		}

		/*
		* For short intervals (less than IPQBDB_RECENT_DROP_TIME
		* since previous block) ibd-judge skips rehabilitation.
		* Also, for intervals less than IPQBDB_UPDATE_TICK_TIME,
		* ibd-judge does not update the record.
		*/
		unsigned int next_probability =
			rehab && delta > IPQBDB_UPDATE_TICK_TIME?
			rehabilitated_prob(probability, delta, decay): probability;

		if (mode == mode_hit || (mode == mode_delta && hit))
		/*
		* double probability, applying worsening logic like in
		* write_block() function in opendb2.c
		*/
		{
			if (RAND_MAX > next_probability &&
				RAND_MAX - next_probability >= next_probability)
					next_probability *= 2;
			else
				next_probability = RAND_MAX;

			if (next_probability < hit_probability)
				next_probability = hit_probability;

			if (next_probability > IPQBDB_PROBABILITY_THRESHOLD &&
				probability <= IPQBDB_PROBABILITY_THRESHOLD)
			{
				double new_decay = 2.0*decay;
				if (new_decay > IPQBDB_NEVER_DECAY)
					new_decay = IPQBDB_NEVER_DECAY;
				printf("\n    decay period doubled here %g -> %g\n\n",
					decay, new_decay);
				decay = new_decay;
			}
		}

		printf("%3d %c%6ld   %10d=%6.2f%% -> %18d=%6.2f%%\n",
			i, lead, delta,
			probability, percent_prob(probability),
			next_probability, percent_prob(next_probability));
		probability = next_probability;
	}

	return 0;
}
