/*
* opendb2.c

Copyright (C) 2008-2021 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <sys/types.h>
#include <db.h>

#include <syslog.h>
#include <math.h>

#include "config_names.h"
#include "dbstruct.h"

#include "utf8_util.h"
#include "in_range_ip6.h"
#include "rehabilitated_prob.h"

#include <assert.h>

/*
* retrieve the last used index for description table.
* db is the handle for the primary (bynumber) table.
* return -1 on error.
*/
int last_descr_ndx(DB *db)
{
	DBC *curs;
	int rtc = db->cursor(db, NULL, &curs, 0);
	if (rtc)
	{
		db->err(db, rtc, "cannot create cursor");
		return -1;
	}

	DBT key, data;
	char buf[1];
	int ndx = 0;
	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.data = &ndx;
	key.ulen = key.size = sizeof ndx;
	key.flags = DB_DBT_USERMEM;
	data.data = buf;
	data.ulen = sizeof buf;
	data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;

	rtc = curs->get(curs, &key, &data, DB_LAST);
	if (rtc == DB_NOTFOUND)
		ndx = 0; // descr table is empty, start from 1
	else if (rtc)
	{
		ndx = -1;
		db->err(db, rtc, "cannot read cursor");
	}

	curs->close(curs);
	return ndx;
}

/*
* get description number by descr, possibly inserting it
*/
int get_reason_id(int *reason_id, char *reason_string,
	app_private *ap, DB *db1, DB *db2, int verbose)
{
	DBT key1, key2, data;
	DBC *curs;
	union
	{
		bynumber_descr_t drec;
		char fill[sizeof(bynumber_descr_t) + IPQBDB_DESCR_MAX_LENGTH + 1];
	} u;
	int id = 0, rtc = 0;

	memset(&key1, 0, sizeof key1);
	memset(&key2, 0, sizeof key2);
	memset(&data, 0, sizeof data);
	memset(&u, 0, sizeof u);

	// write cursor: prevent writing the description table until close below
	rtc = db2->cursor(db2, NULL, &curs, DB_WRITECURSOR);
	if (rtc)
	{
		db2->err(db2, rtc, "cannot get write cursor");
		return rtc;
	}

	if ((key2.size = strlen(reason_string)) >= IPQBDB_DESCR_MAX_LENGTH)
	{
		char *end =
			begin_utf8(reason_string,
				reason_string + IPQBDB_DESCR_MAX_LENGTH - 1);
		*end = 0;
		key2.size = end - reason_string;
		report_error(ap, LOG_WARNING,
			"reason too long, max=%d: truncated to \"%s\"\n",
			IPQBDB_DESCR_MAX_LENGTH - 1, reason_string);
	}

	key2.data = reason_string;
	key2.size += 1; // always include terminating 0
	key1.data = &id;
	key1.size = key1.ulen = sizeof id;
	key1.flags = key2.flags = DB_DBT_USERMEM;
	
	strcpy(u.drec.descr, reason_string);
	size_t wsize = key2.size + offsetof(bynumber_descr_t, descr);

	data.data = &u.drec.chk;
	data.size = data.ulen = data.dlen = sizeof u.drec.chk;
	data.doff = offsetof(bynumber_descr_t, chk);
	data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;
	rtc = curs->pget(curs, &key2, &key1, &data, DB_SET);
	if (rtc == 0 && u.drec.chk == IPQBDB_CHK_SIGNATURE)
	{
		if (verbose)
			printf("found reason id %d for \"%s\"\n", id, reason_string);
		if (reason_id)
			*reason_id = id;
	}

	else if (rtc != DB_NOTFOUND)
	{
		db2->err(db2, rtc, "lookup %s", reason_string);
	}

	else if (db1 && (id = last_descr_ndx(db1)) >= 0)
	{
		++id;
		data.data = &u.drec;
		data.size = data.ulen = wsize;
		data.doff = data.dlen = 0;
		data.flags = DB_DBT_USERMEM;

		u.drec.chk = IPQBDB_CHK_SIGNATURE;
		u.drec.created = time(0);
		rtc = db1->put(db1, NULL, &key1, &data, DB_NOOVERWRITE);
		if (rtc == 0)
		{
			if (verbose)
				printf("inserted reason id %d for \"%s\"\n", id, reason_string);
			if (reason_id)
				*reason_id = id;
		}
	}
	
	curs->close(curs);
	
	if (rtc && (db1 || rtc != DB_NOTFOUND))
		db1->err(db1, rtc,
			"unable to insert id %d reason \"%s\"",
			id, reason_string);

	return rtc;
}

/*
* update the relevant *_cnt fields in the description.
* ip_data pointer indicates delete operation.
* if tloc is not NULL, it is used, otherwise current time is queried.
* return non-zero on error (logged)
*/
int set_descr_cnt(DB *db1, int descr_id, int is_new,
	ip_data_t const *dead, time_t const *tloc)
{
	assert(is_new == 0 || dead == NULL);

	bynumber_descr_t drec;
	DBT key, data;
	DBC *curs;
	char const *what = "dunno";
	int rtc;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.data = &descr_id;
	key.size = sizeof descr_id;
	key.flags = DB_DBT_USERMEM;
	data.data = &drec;
	data.size = data.ulen = data.dlen = sizeof drec;
	data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;

	// write cursor: prevent writing the description table until close below
	rtc = db1->cursor(db1, NULL, &curs, DB_WRITECURSOR);
	if (rtc)
	{
		db1->err(db1, rtc, "no write cursor");
		return rtc;
	}

	rtc = curs->get(curs, &key, &data, DB_SET);
	if (rtc == 0)
	{
		if (data.size == sizeof drec && drec.chk == IPQBDB_CHK_SIGNATURE)
		{
			time_t *tp = NULL;
			if (dead == NULL)
			{
				tp = &drec.last_added;
				drec.caught_cnt += 1;
				if (is_new)
					drec.rec_cnt += 1;
			}
			else
			{
				tp = &drec.last_removed;
				drec.block_cnt += dead->block_cnt;
				drec.decay_add_cnt += dead->decay_add_cnt;
			}

			if (tloc)
				*tp = *tloc;
			else
				time(tp);
			
			if ((rtc = db1->put(db1, NULL, &key, &data, 0)) != 0)
				what = "write";
		}
		else
		{
			rtc = 1;
			what = "record size/format";
		}
	}
	else what = "get";

	curs->close(curs);

	if (rtc)
		db1->err(db1, rtc, "set_descr_cnt on %s key=%d", what, descr_id);

	return rtc;
}

/*
* get description by number, with formatting options
*/
int get_descr(DB *db, int id, int opt, char *dest, size_t size)
{
	union
	{
		bynumber_descr_t drec;
		char fill[sizeof(bynumber_descr_t) + IPQBDB_DESCR_MAX_LENGTH + 1];
	} u;
	char *const descr = &u.drec.descr[0];
	char idbuf[32];
	char catbuf[32];
	size_t dlen = 0, ilen = 0, clen = 0, alloc = 0;
	int rtc;

	if (db)
	{
		DBT key, data;

		memset(&key, 0, sizeof key);
		memset(&data, 0, sizeof data);
		key.data = &id;
		key.size = sizeof id;
		key.flags = DB_DBT_USERMEM;
		data.data = &u;
		data.size = data.ulen = data.dlen = sizeof u;
		data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;

		rtc = db->get(db, NULL, &key, &data, 0);
		if (rtc == 0)
		{
			if (u.drec.chk == IPQBDB_CHK_SIGNATURE)
			{
				// data.size should include trailing 0
				dlen = data.size - 1 - offsetof(bynumber_descr_t, descr);
				if (dlen >= IPQBDB_DESCR_MAX_LENGTH)
					dlen = IPQBDB_DESCR_MAX_LENGTH - 1;
#if 0
				else if (dlen < 0)
					dlen = 0;
#endif
				if (descr[dlen] != 0)
				{
					char *t = begin_utf8(&descr[0], &descr[dlen]);
					dlen = t - &descr[0];
					descr[dlen] = 0;
				}
			}
			else
			{
				rtc = -1;
				db->errx(db, "description %d record corrupted or old format", id);
			}
		}
		else if (rtc != DB_NOTFOUND)
			db->err(db, rtc, "get descr %d", id);
	}
	else
		rtc = -1;

	if (rtc || (opt & get_descr_add_id))
	{
		ilen = sprintf(idbuf, "%d", id);
		alloc = ilen + 1; // separating space
		opt |= get_descr_add_id;
	}

	if (rtc == 0 && (opt & get_descr_add_category))
	{
		clen = sprintf(catbuf, "%d", u.drec.report_category);
		alloc += clen + 2; // comma and separating space
	}

	if (alloc >= size)
		goto error_exit;

	if (rtc || dlen == 0)
	/*
	* Record not found or data length == 0:
	* meta-description, parethesized if descr not quoted
	*/
	{
		char *p = descr;
		if ((opt & get_descr_quoted) == 0)
			*p++ = '(';

		if (rtc == 0)
		{
			assert(dlen == 0);
			strcpy(p, "void");
		}
		else if (rtc == DB_NOTFOUND)
			strcpy(p, "undef");
		else // error or db == NULL
			strcpy(p, "unavail");
		if ((opt & get_descr_quoted) == 0)
			strcat(descr, ")");

		dlen = strlen(descr);
		opt &= ~get_descr_quoted;
	}
	else if (opt & get_descr_quoted)
	{
		alloc += 2; // quotes only around original
	}

	/*
	* truncate description if too long.
	* shrink up to 5 chars of description as in "Hi..."
	* (size includes the trailing 0)
	*/
	if (alloc + dlen + 1 >= size)
	{
		if (dlen > 5 && alloc + 5 < size)
		{
			char *trunc = begin_utf8(descr, &descr[size - alloc - 4]);
			dlen = trunc - &descr[0];
			if (dlen < 2)
				goto error_exit;
			strcpy(trunc, "...");
			dlen += 3;
		}
		else
			goto error_exit;
	}

	alloc += dlen;
	alloc += 1; // terminating 0

	assert(alloc <= size);

	if (opt & get_descr_quoted)
		*dest++ = '"';
	if (dlen)
	{
		memcpy(dest, descr, dlen);
		dest += dlen;
	}
	if (opt & get_descr_quoted)
		*dest++ = '"';
	if (opt & get_descr_add_id)
	{
		*dest++ = ' ';
		memcpy(dest, idbuf, ilen);
		dest += ilen;
	}
	if (opt & get_descr_add_category)
	{
		*dest++ = ',';
		*dest++ = ' ';
		memcpy(dest, catbuf, clen);
		dest += clen;
	}
	*dest = 0;
	return 0;

	error_exit:
	{
		if (size && dest)
			*dest = 0;
		return -1;
	}
}


/*
* callback for describing the secondary key
*/
static int bydescr(DB *db2, DBT const* key1, DBT const* data1, DBT *key2)
{
	(void)db2; (void)key1;
	memset(key2, 0, sizeof *key2);
	key2->data = (char*)data1->data + offsetof(bynumber_descr_t, descr);
	if ((key2->size = data1->size - offsetof(bynumber_descr_t, descr)) >
		 sizeof(bynumber_descr_t)) key2->size = 0;
	return 0;
}


/*
* open the descr table, with a secondary index based on descrs
* along with a primary based on int. The env is already opened.
*/
int open_descrdb(char *fname, DB_ENV *db_env, DB** db1, DB** db2)
{
	int rtc;
	char const *what = "dunno";

	*db1 = *db2 = NULL;

	rtc = db_create(db1, db_env, 0);
	if (rtc)
	{
		what = "db1_create";
		goto error_exit;
	}

	rtc = (*db1)->open(*db1, NULL, fname, "bynumber", DB_BTREE, DB_CREATE, 0);
	if (rtc)
	{
		what = "DB->open(primary)";
		goto error_exit;
	}

	rtc = db_create(db2, db_env, 0);
	if (rtc)
	{
		what = "db2_create";
		goto error_exit;
	}


	rtc = (*db2)->open(*db2, NULL, fname, "bydescr", DB_HASH, DB_CREATE, 0);
	if (rtc)
	{
		what = "DB->open(secondary)";
		goto error_exit;
	}

	rtc = (*db1)->associate(*db1, NULL, *db2, bydescr, 0);
	if (rtc == 0)
		return 0;

	what = "DB->associate";

	error_exit:
	{
		db_env->err(db_env, rtc, "cannot %s", what);
		if (*db2)
		{
			(*db2)->close(*db2, 0);
			*db2 = NULL;
		}
		if (*db1)
		{
			(*db1)->close(*db1, 0);
			*db1 = NULL;
		}
	}
	return 1;
}

int check_whitelist_e(void *db_void, ip_u *ip_addr, double *decay,
	unsigned char end_range[16])
/*
* search the whitelist db for ip_addr and set decay if found.
* return values:
*  0: not found, the decay must be set by the caller to some initial default,
*  1: found, if decay is 0.0 the ip should not be blocked at all,
* -1: db error (logged).
*
* For IPv6, return the end range.
*/
{
	assert(db_void);
	assert(ip_addr);
	assert(ip_addr->ip == 4 || ip_addr->ip == 6);
	assert(decay);

	DBT key, data;
	ip_white_t whi;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);

	int const ip_is_4 = ip_addr->ip == 4;
	key.data = end_range;
	key.size = key.ulen = ip_is_4?
		sizeof ip_addr->u.ipv4: sizeof ip_addr->u.ipv6;
	memcpy(end_range, ip_addr->u.ip_data, key.size);

	key.flags = data.flags = DB_DBT_USERMEM;
	data.data = &whi;
	data.ulen = sizeof whi;

	int rtc;
	int get_rc;
	if (ip_is_4)
	{
		DB *db = db_void;
		get_rc = db->get(db, NULL, &key, &data, 0);
	}
	else
	{
		DBC *curs = db_void;
		if ((get_rc = curs->get(curs, &key, &data, DB_SET_RANGE)) == 0 &&
			!in_range_ip6(ip_addr->u.ip_data, key.data, whi.plen))
				get_rc = DB_NOTFOUND;
	}
	if (get_rc == 0)
	{
		if (data.size == sizeof whi && whi.chk == IPQBDB_CHK_SIGNATURE)
		{
			time_t now;
			if (whi.expiration > 0)
				time(&now);
			if (whi.expiration > 0 && now > whi.expiration)
				rtc = 0;
			else if (ip_is_4 ||
				in_range_ip6(end_range, ip_addr->u.ipv6, whi.plen))
			{
				int type = fpclassify(whi.decay);
				if ((type != FP_NORMAL && type != FP_ZERO) || whi.decay < 0.0)
					whi.decay = 0.0;
				*decay = whi.decay;
				rtc = 1;
			}
			else
				rtc = 0;
		}
		else
		{
			// db->errx(db, "bad whitelist record");
			rtc = -1;
		}
	}
	else if (get_rc == DB_NOTFOUND)
	{
		rtc = 0;
	}
	else
	{
		// db->err(db, get_rc, "get whitelist");
		rtc = -1;
	}

	return rtc;
}

int
check_whitelist(void *db, ip_u *ip_addr, double *decay)
{
	unsigned char e[16];
	return check_whitelist_e(db, ip_addr, decay, e);
}


static const unsigned int ipv6_ranges[] = {IPQBDB_IPV6_RANGES_COMMA};
#define IPV6_RANGES_MAX (int)(sizeof ipv6_ranges / sizeof ipv6_ranges[0])

static int ipqbdb_index_factor_100 = (int)(IPQBDB_INDEX_FACTOR * 100.0);

void set_ipqbdb_index_factor(double f)
{
	ipqbdb_index_factor_100 = (int)(f * 100.0);
}

double get_ipqbdb_index_factor(void)
{
	return (double)ipqbdb_index_factor_100/100.0;
}

int get_ipqbdb_index_factor_precision(void)
{
	return ipqbdb_index_factor_100 % 10 == 0? 1: 2;
}

static inline int nlz(uint32_t x)
/*
* Number of leading zeroes.  floor(log2(x)) == 31 - nlz(x),
* ceil(log2(x)) == 32 - nlz(x - 1).
*/
{
	if (x == 0) return 32;

	int n = 1;

	if ((x >> 16) == 0) {n = n + 16; x <<= 16;}
	if ((x >> 24) == 0) {n = n +  8; x <<=  8;}
	if ((x >> 28) == 0) {n = n +  4; x <<=  4;}
	if ((x >> 30) == 0) {n = n +  2; x <<=  2;}

	return n - (x >> 31);
}

static inline int factor_index(int factor, int probability)
/*
* log2(RAND_MAX) ~= 30.999..., assuming RAND_MAX = 2147483647.
* So log2(p/RAND_MAX) ~= log2(p) - 31 ~= 31 - nlz(p) - 31 = -nlz(p).
* To compute log2(factor * p/RAND_MAX), we add log2(factor), which
* is approx 31 - nlz(factor), so 31 - nlz(factor) - nlz(p);
*
* This is not so much better than using float numbers.  In a test,
* the above function run 100 million times on random numbers did so:
*
*    real	0m2.839s
*    user	0m2.833s
*    sys	0m0.004s
*
* While (int)floor(log2((double)y*(double)x/(double)RAND_MAX)) did so:
*
*    real	0m3.254s
*    user	0m3.248s
*    sys	0m0.005s
*/
{
#if RAND_MAX == 2147483647
	return 31 - nlz(factor) - nlz(probability);
#else
	return (int)floor(log2((double)factor*(double)probability/(double)RAND_MAX));
#endif
}

#if defined SIMULATION_TEST
static time_t simulation_clock;

void set_simulation_clock(time_t clock)
{
	simulation_clock = clock;
}
#endif

static int min_plen_limit(int prob)
{
	int index = factor_index(5 * ipqbdb_index_factor_100/100, prob) + 1;
	if (index > 0)
	{
		if (index >= IPV6_RANGES_MAX)
			index = IPV6_RANGES_MAX - 1;
		return ipv6_ranges[index];
	}

	return 96; // allow 32 bit bare minimum
}

static inline int range_allowed(ip_data_t *ip_data, int probability)
/*
* Return the minimum prefix width allowed with given count and
* probability.  Check against official_plen.
*
* Return 128 if no range is allowed.
*
*/
{
	unsigned int plen = 128;
	if (probability > 0)
	{
		int count = ip_data->caught_cnt < 5? ip_data->caught_cnt: 5;
		int index = factor_index(count * ipqbdb_index_factor_100/100, probability);
#if 0
		double const p = prob;
		double const c = ip_data->caught_cnt < 5? ip_data->caught_cnt: 5;
		int index = (int)floor(0.5 + log(c * IPQBDB_INDEX_FACTOR * p/RAND_MAX));
#endif
		if (index > 0)
		{
			if (index >= IPV6_RANGES_MAX)
				index = IPV6_RANGES_MAX - 1;
			if (ip_data->official_plen > 0 &&
				ip_data->official_plen > ipv6_ranges[index])
					plen = ip_data->official_plen;
			else
				plen = ipv6_ranges[index];
		}
	}
	return plen;
}

static inline int invalid_data(size_t data_size, ip_data_t *ip_data)
// return 0 if the data is valid
{
	if (data_size != sizeof *ip_data || ip_data->chk != IPQBDB_CHK_SIGNATURE)
		return 1;

	return 0;
}

#if !defined NDEBUG
static void assert_congruous(void const *key_data, void const *data_data)
{
	unsigned char const *ip = (unsigned char const *)key_data;
	ip_data_t const *ip_data = (ip_data_t const *)data_data;
	int plen = ip_data->plen;
	if (plen > 0)
	{
		assert(plen < 128);

		int slen = 128 - plen;
		int ndx = 15;
		for (; slen > 8; slen -= 8)
		{
			assert(ip[ndx] == 255);
			--ndx;
		}

		if (slen)
		{
			assert(ndx >= 0);
			int mask = (1 << slen) - 1;
			assert((ip[ndx] & mask) == mask);
		}
	}
}
#else // !NDEBUG
#define assert_congruous(x, y)  (void)0
#endif

static inline int
worsen_logic(ip_data_t *ip_data, ip_data_t *old_data, double decay,
	int reason_id, int initial_probability, int force, time_t now)
/*
* Double probability, apply worsening logic, set the reason to the
* "most severe" one.  Save the old data before doing so.
*
* Rehabilitate before worsening, since in 2.0 ibd-judge only does
* it in case it blocks (in which case it's not caught).
*
* Return rehabilitated probability.  This is either the record found
* or a candidate merger, so set the data as if it's going to be the
* culprit.  However, return the lowest probability found for range
* extension consideration.
*/
{
	memcpy(old_data, ip_data, sizeof *old_data);

	time_t delta = now - ip_data->last_update;
	if (IPQBDB_UPDATE_TICK_TIME < delta && ip_data->decay > 0.0)
	{
		unsigned int const prob =
			rehabilitated_prob(ip_data->probability,
				delta, ip_data->decay);
		if (prob < (unsigned) ip_data->probability)
			ip_data->probability = prob;
	}
	unsigned int return_prob = ip_data->probability;

	if (ip_data->decay < decay || (force & write_block_force_decay) != 0)
	{
		ip_data->decay = decay;
		ip_data->reason_id = reason_id;
	}
	ip_data->last_update = now;
	if (RAND_MAX > ip_data->probability &&
		RAND_MAX - ip_data->probability >= ip_data->probability)
			ip_data->probability *= 2;
	else
		ip_data->probability = RAND_MAX;

	if (ip_data->probability < initial_probability ||
		(force & write_block_force_probability) != 0)
	{
		ip_data->probability = initial_probability;
		ip_data->reason_id = reason_id;
	}

	if ((force & write_block_force_reason) != 0)
		ip_data->reason_id = reason_id;

	if (ip_data->probability > IPQBDB_PROBABILITY_THRESHOLD &&
		old_data->probability <= IPQBDB_PROBABILITY_THRESHOLD)
	{
		ip_data->decay *= 2.0;
		if (ip_data->decay > IPQBDB_NEVER_DECAY)
			ip_data->decay = IPQBDB_NEVER_DECAY;
		ip_data->decay_add_cnt += 1;
	}

	ip_data->caught_cnt += 1;

	return return_prob;
}

static int del_all_range(DBC *curs, ip_data_t *new_ip,
	unsigned char const *key_data, int plen, uint32_t dir)
/*
* Sweep all records belonging in the range identified by key_data
* and plen, proceding in the indicated dir from the currenrt curs.
* Return 0 on success.
*/
{
	DBT key, data;
	unsigned char ipv6[16];
	ip_data_t ip_data;
	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.size = key.ulen = 16;
	key.data = ipv6;
	data.size = data.ulen = sizeof ip_data;
	data.data = &ip_data;
	key.flags = data.flags = DB_DBT_USERMEM;
	int rtc = 0;
	while (rtc == 0)
	{
		rtc = curs->get(curs, &key, &data, dir);
		if (rtc == 0 && in_range_ip6(ipv6, key_data, plen))
		{
			if (ip_data.probability > new_ip->probability)
			{
				new_ip->probability = ip_data.probability;
				new_ip->reason_id = ip_data.reason_id;
			}
			if (ip_data.decay > new_ip->decay)
			{
				new_ip->decay = ip_data.decay;
				new_ip->reason_id = ip_data.reason_id;
				new_ip->decay_add_cnt = ip_data.decay_add_cnt;
			}
			new_ip->caught_cnt += ip_data.caught_cnt;
			rtc = curs->del(curs, 0);
		}
		else
			break;
	}

	if (rtc == DB_NOTFOUND)
		rtc = 0;

	return rtc;
}

static inline int del_range_both_sides(DBC *curs, ip_data_t *ip,
	unsigned char const *key_data, int plen)
/*
* Sweep all records belonging in the range identified by key_data
* and plen.  Return 0 on success.
*/
{

	int rtc = del_all_range(curs, ip, key_data, plen, DB_NEXT);
	if (rtc == 0)
		rtc = del_all_range(curs, ip, key_data, plen, DB_PREV);
	return rtc;
}

static void set_trailing_1s(int plen, unsigned char ip[16])
{
	assert(plen <= 128);

	int slen = 128 -plen;
	int ndx = 15;
	for (; slen > 8; slen -= 8)
		ip[ndx--] = 255;

	if (slen)
	{
		assert(ndx >= 0);
		ip[ndx] |= (1 << slen) - 1;
	}
}

#if defined SIMULATION_TEST
#define start_count_loops int count_loops = 0
#define do_count_loops ++count_loops
#define done_count_loops(x) store_loops(x, count_loops)
static FILE *stats_fp;
static void close_stats_fp(void)
{
	fclose(stats_fp);
	stats_fp = NULL;
}
void store_loops(char *what, int count_loops)
{
	if (stats_fp == NULL)
	{
		stats_fp = fopen("loops.txt", "w");
		if (stats_fp)
			fputs("what loops\n", stats_fp);
		assert(stats_fp);
		atexit(&close_stats_fp);
	}

	if (stats_fp)
		fprintf(stats_fp, "%s, %d\n", what, count_loops);
}
#else
#define start_count_loops (void)0
#define do_count_loops (void)0
#define done_count_loops(x) (void)0
#endif

int distance(unsigned char const ip1[16], unsigned char const ip2[16])
{
	unsigned char var[16];
	memcpy(var, ip2, 16);
	return 128 - range_ip(ip1, var, 16);
}

int write_block(DB *db, ip_u *ip_addr, double decay, int reason_id,
	int *probability, int force, write_block_cb verbose_fn, DB *db_d)
/*
* ip_addr can be changed on setting a range.
*/
{
	assert(db);
	assert(ip_addr);
	assert(ip_addr->ip == 4 || ip_addr->ip == 6);

	ip_data_t ip_data, old_data;
	DBT key, data;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);

	key.data = ip_addr->u.ip_data;
	key.size = key.ulen = ip_addr->ip == 4?
		sizeof ip_addr->u.ipv4: sizeof ip_addr->u.ipv6;
	key.flags = data.flags = DB_DBT_USERMEM;
	data.data = &ip_data;
	data.ulen = sizeof ip_data;
	memset(&ip_data, 0, sizeof ip_data);
	memset(&old_data, 0, sizeof old_data);

#if defined SIMULATION_TEST
	time_t now = simulation_clock;
#else
	time_t now = time(NULL);
#endif

	// a write cursor locks the database
	DBC *curs;
	int rtc = db->cursor(db, NULL, &curs, DB_WRITECURSOR);
	if (rtc)
	{
		db->err(db, rtc, "cannot create write cursor");
		return -1;
	}

	int rtc2 = 0, old = 1;  // old is to be reset if not found
	unsigned char ipv6[16]; // alt. key used for IPv6

	if (ip_addr->ip == 6)
	/*
	* IPv6 treatment provides for deleting stale records and check
	* extensibility.  Plain insertions are left at the bottom.
	*
	* Skip checking for extensions if ip_addr is found, or if ip_addr
	* is a range.  If ip_addr is found inside a range, check for
	* stale ranges.  In that case, delete and insert anew.
	*
	* For ranges, ip_addr has the start IP, which is converted
	* to the end of range for the key.  The range actually inserted
	* or extended and its plen are returned in ip_addr.
	*/
	{
		int first_loop = 1;
		memcpy(ipv6, key.data, sizeof ipv6);
		key.data = ipv6;

		int min_plen = min_plen_limit(*probability);
		int found = 0;
		int next_plen_allowed = 200, next_plen = -1, prev_plen = -1;

		DBC *prev_curs = NULL;

		rtc = curs->get(curs, &key, &data, DB_SET_RANGE);
		start_count_loops;
		while (rtc == 0)
		{
			do_count_loops;
			if (invalid_data(data.size, &ip_data))
			{
				rtc = 1;
				break;
			}

			assert_congruous(key.data, data.data);

			if (first_loop) // found?
			{
				if (memcmp(ip_addr->u.ipv6, ipv6, 16) == 0)
				{
					/*
					* ip_addr is a start of range, ipv6 an end,
					* so they can only be equal if plen == 0.
					*/
					if (ip_addr->plen == 0 && ip_data.plen == 0)
					{
						found = 3;
					}
					else
					{
						assert(ip_addr->plen != ip_data.plen);
						assert(ip_data.plen == 0);
						found = 2;
					}
				}
				else
				{
					found = in_range_ip6(ip_addr->u.ipv6, ipv6, ip_data.plen);
					if (found == 2 && ip_addr->plen == ip_data.plen)
						found = 3;
				}
			}

			int prob = worsen_logic(&ip_data, &old_data,
				decay, reason_id, *probability, force, now);

			next_plen_allowed = range_allowed(&ip_data, prob);
			if (first_loop)
			{
				if (found)
				/*
				* If the rehabilitated range doesn't include the 
				* caught IP any more, then it's stale. Delete it and 
				* insert the caught IP anew.  Except for exact match
				* of historic record (found == 3).
				* 
				* Otherwise save the worsened data without trying to 
				* extend the range, since we are about the limit.
				*/
				{
					if (found == 3)
					{
						assert_congruous(ipv6, data.data);
						rtc2 = curs->put(curs, NULL, &data, DB_CURRENT);
					}
					else
					{
						if (next_plen_allowed > ip_data.plen &&
							!in_range_ip6(ip_addr->u.ipv6, ipv6, next_plen_allowed))
						{
							rtc = curs->del(curs, 0);
							if (rtc == 0)
								rtc = DB_NOTFOUND;
						}
						else
						{
							if (ip_addr->plen >= ip_data.plen)
							{
								ip_addr->plen = ip_data.plen;
								assert_congruous(ipv6, data.data);
								rtc2 = curs->put(curs, NULL, &data, DB_CURRENT);
							}
							else
							{
								if (ip_addr->plen > 0)
								{
									set_trailing_1s(ip_addr->plen, ipv6);
									ip_data.plen = ip_addr->plen;
									rtc = del_range_both_sides(curs,
										&ip_data, ipv6, ip_addr->plen);
								}
								if (rtc == 0)
								{
									assert_congruous(key.data, data.data);
									rtc2 = curs->put(curs, &key, &data, DB_KEYFIRST);
								}
							}
						}
					}
					break;
				}

				if (ip_addr->plen > 0)
				{
					rtc = DB_NOTFOUND;
					break;
				}

				rtc = curs->dup(curs, &prev_curs,  DB_POSITION);
				if (rtc)
					break;
				first_loop = 0;
			}

			next_plen = range_ip(ip_addr->u.ipv6, ipv6, 16);
			if (next_plen >= next_plen_allowed || // possible, or
				next_plen < min_plen)             // off limit
					break;

			rtc = curs->get(curs, &key, &data, DB_NEXT);
		}
		done_count_loops(">");

		if (prev_curs && (rtc == 0 || rtc == DB_NOTFOUND))
		/*
		* Range extension logic.  Consider the previous record or,
		* if not found, the next; whichever would result in the
		* smaller range.
		*/
		{
			ip_data_t prev_ip_data;
			unsigned char prev_ipv6[16];
			data.data = &prev_ip_data;
			int prev_plen_allowed = 200;
			key.data = prev_ipv6;

			if (next_plen > min_plen)
				min_plen = next_plen; // useless to go further

			start_count_loops;
			for (;;)
			{
				do_count_loops;
				rtc = prev_curs->get(prev_curs, &key, &data, DB_PREV);
				if (rtc)
					break;

				int prob = worsen_logic(&prev_ip_data, &old_data,
						decay, reason_id, *probability, force, now);

				prev_plen_allowed = range_allowed(&prev_ip_data, prob);
				prev_plen = range_ip(ip_addr->u.ipv6, prev_ipv6, 16);
				if (prev_plen >= prev_plen_allowed || // possible, or
					prev_plen < min_plen)             // off limit
						break;
			}
			done_count_loops("<");

			if (rtc == 0 || rtc == DB_NOTFOUND)
			/*
			* If allowed to merge with the previous or next record,
			* then delete any intervening range and store the new
			* range as an update of the current recor.
			*/
			{
				int plen = 0;
				if (next_plen >= next_plen_allowed)
					plen = next_plen;

				if (prev_plen >= prev_plen_allowed &&
					prev_plen >= plen)
				{
					memcpy(ipv6, prev_ipv6, sizeof ipv6);
					memcpy(&ip_data, &prev_ip_data, sizeof ip_data);
					plen = prev_plen;
				}

				key.data = ipv6;
				data.data = &ip_data;
				if (plen > 0) // allowed
				{
					ip_addr->plen = plen;
					ip_data.plen = plen;
					first_in_range(ip_addr->u.ipv6, plen);
					set_trailing_1s(plen, ipv6);
					rtc = del_range_both_sides(curs,
						&ip_data, key.data, plen);
					if (rtc == 0)
					{
						assert_congruous(key.data, data.data);
						rtc2 = curs->put(curs, &key, &data, DB_KEYFIRST);
					}
				}
				else
					rtc = DB_NOTFOUND;
			}
		}
		if (prev_curs)
			prev_curs->close(prev_curs);
	}
	else // IPv4
	{
		rtc = curs->get(curs, &key, &data, DB_SET);
		if (rtc == 0 && invalid_data(data.size, &ip_data))
			rtc = 1;
		if (rtc == 0)
		{
			worsen_logic(&ip_data, &old_data,
				decay, reason_id, *probability, force, now);
			rtc2 = curs->put(curs, &key, &data, DB_CURRENT);
		}
	}

	if (rtc == DB_NOTFOUND)
	{
		rtc = old = 0;
		memset(&ip_data, 0, sizeof ip_data);
		ip_data.decay = decay;
		ip_data.last_update = ip_data.created = now;
		ip_data.probability = *probability;
		ip_data.reason_id = reason_id;
		ip_data.caught_cnt = 1;
		ip_data.chk = IPQBDB_CHK_SIGNATURE;
		data.size = data.ulen;

		if (ip_addr->ip == 6)
		{
			assert(key.data == ipv6);
			memcpy(ipv6, ip_addr->u.ipv6, sizeof ipv6);
			if (ip_addr->plen > 0) // insert range
			{
				int const plen = ip_addr->plen;
				ip_data.plen = plen;
				set_trailing_1s(plen, ipv6);
				rtc = del_range_both_sides(curs, &ip_data, key.data, plen);
			}
		}

		assert(key.data == ip_addr->u.ip_data || key.data == ipv6);
		assert(data.data == &ip_data);
		if (rtc == 0)
		{
			assert_congruous(key.data, data.data);
			rtc2 = curs->put(curs, &key, &data, DB_KEYFIRST);
		}
	}

	curs->close(curs);

	if (rtc == 1)
		db->errx(db, "bad block record");
	else if (rtc)
		db->err(db, rtc, "operating curs block");
	else if (rtc2)
		db->err(db, rtc2, "writing curs block");
	else
		*probability = ip_data.probability;

#if defined SIMULATION_TEST
	(void)old;
	(void)verbose_fn;
	(void)db_d;
#else
	if (rtc == 0 && rtc2 == 0)
	{
		if (db_d)
			set_descr_cnt(db_d, reason_id, !old, NULL, &now);
		if (verbose_fn)
			verbose_fn(ip_addr, key.data, reason_id, force,
				&ip_data, old? &old_data: NULL, db_d);
	}
#endif

	return rtc || rtc2;
}

void fputs_initial_count_help()
{
	fputs("\nThe initial-count determines the probability for newly inserted "
		"records,\nsuch that doubling the initial probability that many times "
		"results\nin a probability of 100%. Its max value is the number of "
		"bits in RAND_MAX.\n", stdout);
}
