// abuserdap.c written by vesely in milan on 22 Jul 2019
// query rdap at given source, then search for abuse addresses

// gcc -I source/jsmn -W -Wall -g -O0 -o abutest abuserdap.c report_error.c -lcurl (alfa)
// gcc -I . -W -Wall -g -O0 -o abuserdap abuserdap.c report_error.c -lcurl (beta)

/*
Copyright (C) 2019-2021 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

static const char usage[] = "[opt] rdap-url-or-file-or-IP\n"
	"where\n"
	"	opt is one or more of:\n"
	"	   -h      display help and exit\n"
	"	   -l      use syslog (LOG_MAIL), not stderr\n"
	"	   -m int  for increased max redirections (from 3)\n"
	"	   -x file exclude addresses in file\n"
	"	   -@      display collected exclusion tokens\n"
	"	   -s      display the data source\n"
	"	   -X      display the count of exclusions\n"
	"	   -v      for increased verbosity\n"
	"	  or\n"
	"	    --help (same as -h) or --version\n"
	"	rdap-url-or-file-or-IP is either the URL as per rfc7482\n"
	"	or a file containing json data or an IP address.\n\n"
	"An rdap-uri can be based on:";

static const char bootstrap_uri[] = "http://rdap.arin.net/bootstrap/ip/";

static const char *rdap_uri[] = {
	bootstrap_uri,
	"http://rdap.arin.net/registry/ip/",
	"https://rdap.db.ripe.net/ip/",
	"https://rdap.apnic.net/ip/",
	"https://rdap.afrinic.net/rdap/ip/",
	"https://rdap.lacnic.net/rdap/ip/"};

//#if defined HAVE_CONFIG
#include <config.h>
//#endif

#define ABUSERDAP_VERSION  PACKAGE_VERSION
#if !defined(ABUSERDAP_CACHE)
#define ABUSERDAP_CACHE "/var/lib/abuserdap"
#endif

#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <curl/curl.h>
#include <errno.h>
#include <ctype.h>
#include "report_error.h"

#define JSMN_STATIC // don't export jsmn functions
#define JSMN_PARENT_LINKS // needed to navigate
#define JSMN_STRICT // real JSON
#include "jsmn.h"

#include <assert.h>

typedef struct global
{
	char *arg;
	char *xclude;
	char *location;      // is this ever used?
	FILE *xfp;           // exclusion file (see grep_v())
	size_t datalen;
	app_private ap;
	int max_redir;       // default 3
	int excluded;        // number of excluded addresses
	int searched;        // calls of grep_v
	int rtc;
	int token_print: 1;  // opt -@
	int source_print: 1; // opt -s
	int ex_print: 1;     // opt -X
	int nu: 3;
	int ip_version: 2;   // 0=no set, 1=IPv4, 2=IPv6
	int verbose: 8;
} global;

static char *read_file(global *glo, size_t st_size)
{
	char *fname = glo->arg;
	char *data = malloc(st_size + 1);
	if (data)
	{
		FILE *fp = fopen(fname, "r");
		int good = 1;
		if (fp)
		{
			if (fread(data, st_size, 1, fp) == 1)
			{
				data[st_size] = 0;
				if (glo->verbose >= 1)
					report_error(&glo->ap, LOG_INFO,
						"Read %s\n", fname);
			}
			else
			{
				report_error(&glo->ap, LOG_ERR,
					"cannot read %s: %s\n", fname, strerror(errno));
				good = 0;
			}
			fclose(fp);
		}
		else
		{
			report_error(&glo->ap, LOG_ERR,
				"cannot open %s: %s\n", fname, strerror(errno));
			good = 0;
		}

		if (good)
			glo->datalen = st_size;
		else
		{
			free(data);
			data = NULL;
			glo->rtc = 2;
		}

		return data;
	}
	report_error(&glo->ap, LOG_CRIT, "Memory fault!!\n");
	abort();
}

typedef struct query_data
{
	global *glo;
	char *response, *location, *retry_after;
	char *buffer;
	// alloc always has 1 byte more, for trailing 0
	size_t b_alloc, b_used;
} query_data;

static inline int do_dup(char *p, char *off, char **dest)
/*
* Duplicate a non 0-terminated string p.  off points to the first
* byte off the string; that is, where a 0 would have been is the string
* were 0-terminated.  The result is 0-terminated, placed in dest.
* If dest is non NULL, it is freed before overwriting.
*/
{
	char *dup = NULL;

	if (p < off)
	{
		size_t len = off - p;
		while (len > 0 && isspace(((unsigned char*)p)[len - 1]))
			--len;
		if ((dup = malloc(len + 1)) == NULL)
			return -1;
		memcpy(dup, p, len);
		dup[len] = 0;
	}

	free(*dest);
	*dest = dup;

	return 1;
}

static inline int
headerdup(char *h_name, size_t h_len, char *ptr, char *off, char **dest)
/* duplicate header fields of interest for rdap_header */
{
	if (strncasecmp( h_name, ptr, --h_len) == 0)
	{
		char *p = ptr + h_len;
		while (p < off && *p != ':')
			++p;
		++p; // ':' or off
		while (p < off && isspace(*(unsigned char*)p))
			++p;
		return do_dup(p, off, dest);
	}

	return 0;
}

size_t rdap_header(char *ptr, size_t size, size_t nitems, void *userdata)
/* CURLOPT_HEADERFUNCTION callback */
{
	static char location[] = "Location";
	static char retry_after[] = "Retry-After";

	query_data *q = (query_data*) userdata;
	size_t len = size * nitems;
	char *off = ptr + len;

	if (q->glo->verbose >= 4)
	{
		int disp = len <= INT_MAX? len: INT_MAX;
		report_error(&q->glo->ap, LOG_DEBUG,
			"header: %.*s\n", disp, ptr);
	}

	int rtc = 0;
	if (q->response == NULL) // first line
	{
		rtc = do_dup(ptr, off, &q->response);
	}
	else
	{
		if (len > sizeof location &&
			(rtc = headerdup(location, sizeof location, ptr, off, &q->location)) > 0)
				return len;

		if (len > sizeof retry_after)
			rtc |= headerdup(retry_after, sizeof retry_after, ptr, off, &q->retry_after);
	}
	if (rtc < 0)
	{
		int disp = len > 30? 30: (int)len;
		report_error(&q->glo->ap, LOG_CRIT,
			"Memory fault at header %.*s (%zd)\n", disp, ptr, len);
		abort();
	}

	return len;
}

static size_t
rdap_receive(char *ptr, size_t size, size_t nmemb, void *userdata)
/* CURLOPT_WRITEFUNCTION callback */
{
	query_data *q = (query_data*) userdata;
	size_t avail = q->b_alloc - q->b_used;
	size_t const chunk = size * nmemb;
	if (avail <= chunk)
	{
		size_t new_alloc = 2*q->b_alloc + chunk;
		char *new_buffer = realloc(q->buffer, new_alloc);
		if (new_buffer == NULL)
		{
			free(q->buffer);
			report_error(&q->glo->ap, LOG_CRIT,
				"Memory fault at buffer %zd->%zd\n", q->b_alloc, new_alloc);
			abort();
		}
		q->buffer = new_buffer;
		q->b_alloc = new_alloc;
	}

	char *next = q->buffer + q->b_used;
	memcpy(next, ptr, chunk);
	next[chunk] = 0;
	q->b_used += chunk;

	return chunk;
}

static char *rdap_query(global *glo)
{
	char *url = glo->arg;

	// if multithreading, must call this before
	// if (curl_global_init(CURL_GLOBAL_ALL)) return NULL;

	CURL *curl = curl_easy_init();
	if (curl == NULL)
	{
		report_error(&glo->ap, LOG_ERR, "Cannot init curl\n");
		return NULL;
	}

	query_data q;
	memset(&q, 0, sizeof q);
	q.glo = glo;
	q.buffer = malloc(q.b_alloc = CURL_MAX_WRITE_SIZE);
	if (q.buffer == NULL)
	{
		report_error(&glo->ap, LOG_CRIT, "Memory fault!\n");
		abort();
	}

	char errorbuffer[CURL_ERROR_SIZE];
	errorbuffer[0]=0;
	CURLcode res = curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, errorbuffer);

	struct curl_slist *header = curl_slist_append(NULL, "Accept-Language: en");
	if (header) header = curl_slist_append(header, "Accept-Encoding: UTF-8");
	if (header) header = curl_slist_append(header,
		"Accept: application/json, application/rdap+json");
	if (header) header = curl_slist_append(header,
		"User-Agent: ipqbdb-reporter/" ABUSERDAP_VERSION);
	res |= curl_easy_setopt(curl, CURLOPT_HTTPHEADER, header);
	res |= curl_easy_setopt(curl, CURLOPT_TIMEOUT, 15L); // Thu 28 Apr 2022, was 600L
	res |= curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1L);
	res |= curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1L);
	res |= curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, rdap_header);
	res |= curl_easy_setopt(curl, CURLOPT_HEADERDATA, &q);
	res |= curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, rdap_receive);
	res |= curl_easy_setopt(curl, CURLOPT_WRITEDATA, &q);
	if (glo->verbose >= 3)
	{
		if (glo->ap.mode != error_report_stderr)
		{
			// TODO: open a pipe and fork a child that turns it to syslog
			// use CURLOPT_STDERR to tell curl to write to pipefd[1]
			report_error(&glo->ap, LOG_WARNING,
				"Curl verbosity not (yet) available with syslog\n");
		}
		else
			res |= curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
	}

	if (glo->ip_version)
	{
		long ip_resolve;
		switch (glo->ip_version)
		{
			case 1: ip_resolve = CURL_IPRESOLVE_V4; break;
			case 2: ip_resolve = CURL_IPRESOLVE_V6; break;
			default: ip_resolve = CURL_IPRESOLVE_WHATEVER; break;
		}
		res |= curl_easy_setopt(curl, CURLOPT_IPRESOLVE, ip_resolve);
	}

	if (res != CURLE_OK)
	{
		report_error(&glo->ap, LOG_CRIT,
			"Unsupported curl options? %s?\n", errorbuffer);
		glo->rtc = 2;
		free(q.buffer);
		q.buffer = NULL;
		goto cleanup;
	}

	for (int redirect = 0; ; ++redirect)
	{
		res = curl_easy_setopt(curl, CURLOPT_URL, url);
		if (res == CURLE_OK)
			res = curl_easy_perform(curl);

		long http_code;
		if (res == CURLE_OK)
			res = curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &http_code);
		if (res == CURLE_OK)
		{
			if (glo->verbose >= 3)
				report_error(&glo->ap, LOG_INFO,
					"URL %s: %ld%s%s\n", url, http_code,
					errorbuffer[0]? " ": "", errorbuffer);

			if (http_code >=400)
			{
				if (glo->verbose)
					report_error(&glo->ap, LOG_ERR,
						"%s returned %s\n", url, q.response);
				if (q.retry_after)
				// TODO: persistent across calls
				{
					report_error(&glo->ap, LOG_ERR,
						"Must not retry before %s\n", q.retry_after);
				}
				
				free(q.buffer);
				q.buffer = NULL;
				break;
			}

			if (http_code >= 300)
			{
				if (glo->verbose >= 1)
					report_error(&glo->ap, q.location? LOG_INFO: LOG_ERR,
						"Redirect %ld %s -> %s [step %d]\n", http_code, url,
						q.location? q.location: "NO LOCATION", redirect + 1);

				q.b_used = 0;
				if (redirect > glo->max_redir || !q.location)
					break;

				if (redirect > 0)
					free(url);
				url = strdup(q.location);
				continue;
			}

			if (http_code >= 200)
			{
				if (glo->verbose == 1)
					report_error(&glo->ap, LOG_INFO,
						"Success %ld for GET %s\n", http_code, url);
			}
			else
				report_error(&glo->ap, LOG_ERR,
					"Bad http_code: %ld for GET %s\n", http_code, url);
		}

		break;
	}

	if (res != CURLE_OK)
	{
		report_error(&glo->ap, LOG_ERR,
			"URL %s: %s\n", url, errorbuffer);
		glo->rtc = 2;
		free(q.buffer);
		q.buffer = NULL;
	}

	if (url != glo->arg)
	{
		free(url);
		url = glo->arg;
	}

cleanup:

	curl_slist_free_all(header);
	curl_easy_cleanup(curl);
	free(q.response);
	free(q.location);
	free(q.retry_after);

	if (q.buffer)
	{
		if (q.b_used)
		{
			q. buffer = realloc(q.buffer, q.b_used + 1);
			q.glo->datalen = q.b_used;
		}
		else
		{
			free(q.buffer);
			q.buffer = NULL;
		}
	}

	return q.buffer;
}

static char *file_or_IP(global *glo)
{
	char *forip = glo->arg;
	struct stat st;
	int rc;
	if ((rc = stat(forip, &st)) == 0 && (st.st_mode & S_IFMT) == S_IFREG)
	{
		return read_file(glo, st.st_size);
	}

	size_t foriplen = strlen(forip);
	if (strspn(forip, "0123456789.:abcdefABCDEF") == foriplen)
	{
		char arg[foriplen + sizeof bootstrap_uri];
		strcpy(&arg[0], bootstrap_uri);
		strcat(&arg[sizeof bootstrap_uri - 1], forip);
		glo->arg = &arg[0];
		return rdap_query(glo);
	}

	report_error(&glo->ap, LOG_ERR, "%s not an IP nor %s file: %s\n",
		forip, rc? "existing": "a regular", strerror(errno));
	glo->rtc = 2;
	return NULL;
}


typedef struct addrlst
{
	struct addrlst *next;
	char addr[];
} addrlst;

typedef struct json_data
{
	global *glo;
	char *data;
	jsmntok_t *tok;
	addrlst *albase, *tokbase;
	unsigned int addrcount, tokcount;
} json_data;

static void insert_addrlist(app_private *ap,
	addrlst **old, char *addr, size_t addrlen)
/*
* insert a new element at old, abort if out of memory
*/
{
	assert(old);

	addrlst *new_addr = malloc(sizeof(addrlst) + addrlen + 1);
	if (new_addr == NULL)
	{
		free(*old);
		report_error(ap, LOG_CRIT, "Memory fault\n");
		abort();
	}
	new_addr->next = *old;
	memcpy(&new_addr->addr[0], addr, addrlen);
	new_addr->addr[addrlen] = 0;
	*old = new_addr;
}

static int search_addrlist(addrlst **old, char *addr, size_t addrlen)
{
	while (*old != NULL)
	{
		if (addrlen == strlen((*old)->addr) &&
			strncasecmp(addr, (*old)->addr, addrlen) == 0)
				return 1; // found
		old = &(*old)->next;
	}

	return 0;
}

static int grep_v(json_data *jd, char *addr, size_t addrlen)
/*
* Check exclusion file, return 0 if not excluded.
*/
{
	global *glo = jd->glo;
	FILE *fp = glo->xfp;
	int rtc = 0;

	if (glo->searched > 0 && fseek(fp, 0L, SEEK_SET))
	{
		report_error(&glo->ap, LOG_CRIT,
			"cannot rewind %s: %s\n",
			glo->xclude, strerror(errno));
		return 0;
	}

	++glo->searched;
	char buf[4096], *addrz = strndup(addr, addrlen), *s;
	if (addrz == NULL)
		abort();

	while ((s = fgets(buf, sizeof buf, fp)) != NULL)
	{
		int ch;
		while ((ch = *(unsigned char*)s) != 0 && isspace(ch))
			++s;
		if (ch == 0 || ch == '#') // comment
			continue;

		char *tok = NULL;
		size_t toklen = 0;
		if (ch == '@') // token
		{
			tok = ++s;
			while ((ch = *(unsigned char*)s) != 0 && !isspace(ch))
				++s;
			if (ch == 0)
				continue;

			toklen = s - tok;
			*s++ = 0;
			while ((ch = *(unsigned char*)s) != 0 && isspace(ch))
				++s;
		}

		if (strstr(s, addrz))
		{
			glo->excluded += 1;
			if (glo->verbose >= 2)
				report_error(&glo->ap, LOG_INFO,
					"email \"%s\" excluded by option, token=%s\n",
					addrz, tok? tok: "NULL");

			if (tok && glo->token_print &&
				search_addrlist(&jd->tokbase, tok, toklen) == 0)
					insert_addrlist(&jd->glo->ap,
						&jd->tokbase, tok, toklen);

			rtc = 1;
			break;
		}
	}

	free(addrz);
	return rtc;
}

static void add_addrlst(json_data *jd, jsmntok_t *tok)
{
	if (tok == NULL)
		return;

	char *addr = jd->data + tok->start;
	size_t addrlen = tok->end - tok->start;
	if (addrlen >= INT_MAX)
		abort();

	/*
	* discard duplicate address (maybe one has {"pref": 1})
	*/
	if (search_addrlist(&jd->albase, addr, addrlen))
	{
		if (jd->glo->verbose >= 2)
			report_error(&jd->glo->ap, LOG_INFO,
				"email \"%.*s\" found again\n", (int)addrlen, addr);
		return;
	}

	/*
	* Check exclusion list
	* TODO: move this check at the end.
	*/
	if (jd->glo->xfp && grep_v(jd, addr, addrlen))
		return;

	/*
	* insert new address
	*/
	insert_addrlist(&jd->glo->ap, &jd->albase, addr, addrlen);
	jd->addrcount += 1;
	if (jd->glo->verbose >= 2)
		report_error(&jd->glo->ap, LOG_INFO, "address %u: %s\n",
			jd->addrcount, jd->albase->addr);
};

static int print_addrlst(addrlst **base, char const *sep)
/*
* Print one line with all entries (possibly 0) and free the list
*/
{
	assert(base);

	int rtc = 0;
	addrlst *prt = *base;
	while (prt)
	{
		addrlst *next = prt->next;
		printf("%s%s", prt->addr, next? sep: "\n");
		free(prt);
		prt = next;
		++rtc;
	}
	*base = NULL;

	if (rtc == 0)
		putchar('\n');

	return rtc;
}

// static string (ss) comparisons
static const char ss_email[] = "email";
static const char ss_vcard[] = "vcard";
static const char ss_vcardArray[] = "vcardArray";
static const char ss_objectClassName[] = "objectClassName";
static const char ss_entity[] = "entity";
static const char ss_abuse[] = "abuse";
static const char ss_type[] = "type";
static const char ss_text[] = "text";
static const char ss_roles[] = "roles";

static inline int
stringeq(json_data *jd, jsmntok_t *tok, const char *s, size_t slen)
{
	return tok && tok->type == JSMN_STRING &&
		slen == (size_t)(tok->end - tok->start) &&
		strncmp(jd->data + tok->start, s, slen) == 0;
}

static inline jsmntok_t *parent(json_data *jd, jsmntok_t *tok)
// for named token, "name": "value", name is the parent of value
{
	return tok && tok->parent >= 0? jd->tok + tok->parent: NULL;
}

static inline int is_array(jsmntok_t *tok)
{
	return tok && tok->type == JSMN_ARRAY;
}

static inline int is_object(jsmntok_t *tok)
{
	return tok && tok->type == JSMN_OBJECT;
}

static jsmntok_t *
find_member(json_data *jd, jsmntok_t *tok, const char *s, size_t slen)
// tok should be an object
{
	if (tok == NULL)
		return NULL;

	// children of this object have this index
	int parent = tok - jd->tok;
	int size = tok->size;
	while (size > 0)
	{
		++tok;
		if (tok->parent != parent)
			continue;

		--size;
		if (tok->is_key && stringeq(jd, tok, s, slen))
			return ++tok;
	}

	return NULL;
}

static jsmntok_t *find_next(jsmntok_t *tok)
// tok should be an array element
{
	if (tok == NULL)
		return NULL;

	int parent = tok->parent;
	while ((++tok)->parent >= parent)
	{
		if (tok->parent == parent)
			return tok;
	}

	return NULL;
}

static int toklen(jsmntok_t *tok)
{
	return tok? tok->end - tok->start: 0;
}

static char const *tokstring(char *data, jsmntok_t *tok)
{
	return tok && data? data + tok->start: "";
}

static void print_abuse_email(global *glo, char *data)
/*
* Parse data and print 
*/
{
	jsmn_parser parser;
	jsmn_init(&parser);

	int ntok = jsmn_parse(&parser, data, glo->datalen, NULL, 0);
	if (ntok > 0)
	{
		json_data jd;
		memset(&jd, 0, sizeof jd);
		jd.glo = glo;
		jd.data = data;
		if ((jd.tok = calloc(ntok, sizeof(jsmntok_t))) == NULL)
		{
			free(data);
			report_error(&glo->ap, LOG_CRIT, "Memory fault!!!\n");
			abort();
		}

		jsmn_init(&parser);
		int rc = jsmn_parse(&parser, data, glo->datalen, jd.tok, ntok);
		if (rc == ntok)
		{
			for (int i = 0; i < ntok; ++i)
			{
				jsmntok_t *tok = &jd.tok[i];
				if (stringeq(&jd, tok, ss_email, sizeof ss_email - 1))
				{
					/*
					* Are we in a jCard?
					* p should be [ "email", {}, "text" email-address ]
					* pp should be an array of properties
					* ppp an array of 2 elements: [ "vcard", pp ]
					* entity should be an entity specifying roles
					*/
					jsmntok_t *p, *pp, *ppp, *ppp_name, *entity;
					if (!is_array(p = parent(&jd, tok)) ||
						!is_array(pp = parent(&jd, p)) || 
						!is_array(ppp = parent(&jd, pp)) ||
						ppp->size != 2 ||
						!stringeq(&jd, ppp + 1, ss_vcard, sizeof ss_vcard - 1) ||
						!stringeq(&jd, ppp_name = parent(&jd, ppp),
							ss_vcardArray, sizeof ss_vcardArray - 1) ||
						!ppp_name->is_key ||
						!is_object(entity = parent(&jd, ppp_name)))
					{
						if (glo->verbose >= 3)
							report_error(&glo->ap, LOG_DEBUG,
								"skip email[%d] %.20s\n", i, tokstring(data, tok));
						continue;
					}

					jsmntok_t *class = find_member(&jd, entity,
						ss_objectClassName, sizeof ss_objectClassName - 1);
					if (!stringeq(&jd, class, ss_entity, sizeof ss_entity - 1))
					{
						if (glo->verbose >= 3)
							report_error(&glo->ap, LOG_DEBUG,
								"bad class tok[%d] \"%.*s\""
								", expected \"entity\"\n",
								i, toklen(tok), tokstring(data, tok));
						continue;
					}

					/*
					*   o  roles -- an array of strings, each signifying
					*      the relationship an object would have with
					*      its closest containing object
					*   https://tools.ietf.org/html/rfc7483#section-5.1
					*/
					jsmntok_t *roles = find_member(&jd, entity,
						ss_roles, sizeof ss_roles - 1);
					if (roles == NULL || !is_array(roles))
					{
						if (glo->verbose)
							report_error(&glo->ap, LOG_ERR,
								"Entity with jCard but %s roles in %s\n",
								roles == NULL? "no": "strange", glo->arg);
						continue;
					}

					int size = roles->size;
					while (size-- >= 0)
					{
						++roles;
						if (roles->type != JSMN_STRING)
						{
							if (glo->verbose)
								report_error(&glo->ap, LOG_ERR,
									"Invalid role for email[%d]\n", i);
							continue;
						}

						if (glo->verbose >= 3)
							report_error(&glo->ap, LOG_DEBUG,
								"role %.*s for email[%d]\n",
								toklen(roles), tokstring(data, roles), i);

						if (stringeq(&jd, roles, ss_abuse, sizeof ss_abuse - 1))
							break;
					}

					if (size < 0) // "abuse" role not found
					{
						if (glo->verbose >= 3)
							report_error(&glo->ap, LOG_DEBUG,
								"no abuse role for email[%d] %.20s\n",
								i, tokstring(data, tok));
						continue;
					}

					if (!is_object(++tok))
					{
						if (glo->verbose)
							report_error(&glo->ap, LOG_ERR,
								"Invalid email property in %s\n", glo->arg);
						continue;
					}

					/*
					*   2.  An object containing the parameters as described
					*       in Section 3.4.  If the property has no parameters,
					*       an empty object is used to represent that.
					*   https://tools.ietf.org/html/rfc7095#section-3.3
					*
					*   We look for "type" property, if it exists and is not
					*   "abuse", we consider that's not an abuse email address.
					*   (rfc6350 only specifies "home" and "work")
					*/

					jsmntok_t *type = find_member(&jd, tok,
						ss_type, sizeof ss_type - 1);

					if (glo->verbose >= 3)
						report_error(&glo->ap, LOG_DEBUG,
							"%stype %.*s for email[%d]\n",
							type? "": "no ",
							toklen(type), tokstring(data, type), i);
					if (type && !stringeq(&jd, type, ss_abuse, sizeof ss_abuse - 1))
						continue;

					tok = find_next(tok);
					if (!stringeq(&jd, tok, ss_text, sizeof ss_text - 1))
					{
						if (glo->verbose)
							report_error(&glo->ap, LOG_ERR,
								"Invalid email property type in %s\n", glo->arg);
						continue;
					}

					// the actual address should be the last element in the array
					add_addrlst(&jd, find_next(tok));
				}
			}

			int rtc = print_addrlst(&jd.albase, ", ");
			if (glo->verbose)
				report_error(&glo->ap, LOG_INFO,
					"%d abuse email address(es) found\n", rtc);
			if (glo->token_print)
			{
				rtc = print_addrlst(&jd.tokbase, " ");
				if (glo->verbose)
					report_error(&glo->ap, LOG_INFO,
						"%d exclude tokens collected\n", rtc);
			}
		}
		else
			report_error(&glo->ap, LOG_CRIT,
				"jsmn_parse returns %d and then %d in %s\n",
				ntok, rc, glo->arg);
		free(jd.tok);
	}
	else
		report_error(&glo->ap, LOG_ERR,
			"jsmn_parse returns %d (%s) in %s\n", ntok,
			ntok == JSMN_ERROR_NOMEM? "NOMEM":
			ntok == JSMN_ERROR_INVAL? "INVALID":
			ntok == JSMN_ERROR_PART? "PART": "?", glo->arg);
}

static int check_arg(char **dest, char *src, int opt)
{
	if (src != NULL && *dest == NULL) // Assume argv[argc] = NULL
	{
		*dest = src;
		return 0;
	}

	fprintf(stderr, "Argument mismatch for option -%c\n", opt);
	return 1;
}



int main(int argc, char *argv[])
{
	global glo;

	memset(&glo, 0, sizeof glo);
	if ((glo.ap.err_prefix = strrchr(argv[0], '/')) != NULL)
		glo.ap.err_prefix += 1;
	else
		glo.ap.err_prefix = argv[0];

	// bootstrap may lead to RIR, then RIR may delegate to national
	glo.max_redir = 3;

	for (int i = 1; i < argc;)
	{
		char *a = argv[i];
		if (a[0] == '-' && a[1] != 0)
		{
			if (strchr(a, 'h') != NULL || strcmp(a, "--help") == 0)
			{
				fprintf(stderr, "usage: %s %s\n", argv[0], usage);
				for (size_t i = 0; i < sizeof rdap_uri/sizeof rdap_uri[0]; ++i)
					fprintf(stderr, "	%s\n", rdap_uri[i]);
				return 0;
			}
			else if (strcmp(a, "--version") == 0)
			{
				curl_version_info_data *v = curl_version_info(CURLVERSION_NOW);
				fprintf(stderr,
					"%s " ABUSERDAP_VERSION " for ipqbdb, "
					"linked with libcurl %s (compiled " LIBCURL_VERSION ")\n",
					glo.ap.err_prefix,
					v && v->age>= 0? v->version: "???");
				return 0;
			}
			else
			{
				char ch;
				int next_arg = i;
				for (unsigned j = 1; (ch = a[j]) != 0; ++j)
					switch (ch)
					{
						case '4':
							glo.ip_version |= 1;
							break;
						case '6':
							glo.ip_version |= 2;
							break;
						case '@':
							glo.token_print = 1;
							break;
						case 'X':
							glo.ex_print = 1;
							break;
						case 'l':
						{
							openlog(glo.ap.err_prefix, LOG_PID,
								glo.ap.mode = LOG_DAEMON);
							break;
						}
						case 'm':
							++glo.max_redir;
							break;
						case 's':
							glo.source_print = 1;
							break;
						case 'v':
							++glo.verbose;
							break;
						case 'x':
							if (check_arg(&glo.xclude, argv[++next_arg], ch))
								return -1;
							break;
						default:
						{
							fprintf(stderr, "Invalid option %c in %s\n", ch, a);
							return 1;
						}
					}
				i = next_arg + 1;
			}
		}
		else if (glo.arg == NULL)
		{
			glo.arg = a;
			++i;
		}			
		else
		{
			fprintf(stderr, "Invalid argument %s at position %d\n", a, i);
			return 1;
		}
	}

	if (glo.arg)
	{
		if (glo.xclude)
		{
			glo.xfp = fopen(glo.xclude, "r");
			if (glo.xfp == NULL)
			{
				report_error(&glo.ap, LOG_ERR,
					"cannot open %s: %s\n", glo.xclude, strerror(errno));
				return 2;
			}
		}

		char *data, *scheme;
		if (strlen(glo.arg) > 8 &&
			strncasecmp("http", glo.arg, 4) == 0 &&
			(scheme = strstr(glo.arg, "://")) != NULL &&
			scheme <= glo.arg + 5)
		{
			data = rdap_query(&glo);
		}
		else
		{
			data = file_or_IP(&glo);
		}
			
		if (data)
		{
			print_abuse_email(&glo, data);
			if (glo.source_print)
				puts(glo.arg);
			if (glo.ex_print)
				printf("%d\n", glo.excluded);
			free(data);
			free(glo.location);
		}

		if (glo.xfp)
			fclose(glo.xfp);
	}

	return glo.rtc;
}
