/*
* ipq berkeley db daemon - ipqbdbd
* written by ale in milano on 10 sep 2008

Copyright (C) 2008 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#define _GNU_SOURCE 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <math.h>
#include <errno.h>

#define __USE_BSD 1
// for u_int in db.h
#include <sys/types.h>

// misc. ip stuff
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

// netfilter for iptables v1.3.6
#include <linux/netfilter.h>
#include <netinet/ip.h>

// this works with ip_queue.ko (CONFIG_IP_NF_QUEUE)
//#include <libipq/libipq.h> // from package iptables-dev

/*
how it works.... (attempt)
in net/netfilter/nf_queue.c, fun nf_queue()
  the packet is sent to queue_handler[pf]->outfn
  (todo: make sure this is what gets called for the NFQUEUE target)

  the [nf_]queue_handler (2nd parm) is set with fun nf_register_queue_handler
  it returns EEXIST, EBUSY, 0 resp. if the same, another, NULL handler is
  registered at that number (pf=1st parm). unregister sets to NULL.
  When NULL, packets are dropped. Pf is the protocol family (eg pf-inet?).

in net/netfilter/nfnetlink_queue.c the queue_handler is the same for all.
  it is a static struct, whose outfn is nfqnl_enqueue_packet. Since the
  queuenum comes from the originator, the nfqnl_instance is looked up here.

  The comment in unregister says:
  "This is a bug and a feature.  We can unregister other handlers(!)"

Using the test procedure in nfqnl_test.c (libnetfilter_queue) the
xt_NFQUEUE module gets automatically loaded on adding an iptables
rule, e.g., iptables -I interna_in 2 -p tcp --destination-port 13 -j NFQUEUE
on deleting the rule, the module remains loaded but its usage count
drops to 0 so that it can be removed with modprobe. The userspace program
does not receive a stop of some sort.
*/

// this works with .ko
// from package libnetfilter-queue-dev
#include <libnetfilter_queue/libnetfilter_queue.h>

// Berkeley DB v4.4
#include <db.h>

#include <popt.h>

// format of data packet
#include "config_names.h"
#include "dbstruct.h"

static app_private ap;
static char const err_prefix[] = "ipqbdbd";
//static int caught_signal;

static int verbose = -1;

// #if !defined NDEBUG
#define TRACE(v, x...) \
	while (ap.mode == 0 && verbose >= (v)) \
		{ fprintf(stderr, ## x); break; }
// #else
// #define TRACE(x...)
// #endif

static void sig_catcher(int sig)
{
#if !defined(NDEBUG)
	if (ap.mode == 0)
	{
		char buf[80];
		unsigned s = snprintf(buf, sizeof buf,
			"%s [%d]: received signal %s\n",
			err_prefix, (int)getpid(), strsignal(sig));
		if (s >= sizeof buf)
		{
			s = sizeof buf;
			buf[s - 1] = '\n';
		}
		write(2, buf, s);
	}
#endif
	if (sig == SIGABRT)
		syslog(LOG_CRIT, "aborting %s\n", err_prefix);
	caught_signal = sig;
}

typedef struct pkt_data
{
	time_t now;
	uint32_t ipaddr;
	uint32_t id;
} pkt_data;

// seems a callback may be called multiple times for a given packet...(?)
typedef struct ipaddr_cnt
{
	uint32_t cnt;
	uint32_t deadbeef;
	pkt_data pkt[4];
} ipaddr_cnt;

static int
packet_cb(struct nfq_q_handle *qh, struct nfgenmsg *nfmsg,
	      struct nfq_data *nfa, void *data)
{
	(void)qh; (void)nfmsg;
	ipaddr_cnt *cnt = (ipaddr_cnt*)data;
	if (cnt == NULL || cnt->deadbeef != 0xdeadbeef ||
		cnt->cnt >= sizeof cnt->pkt / sizeof cnt->pkt[0])
	{
		report_error(&ap, LOG_CRIT, "cb fucked up (%p->%x/%u)\n",
			cnt, cnt? cnt->deadbeef: 0, cnt? cnt->cnt: 0);
		return -1;
	}

	char *payload = NULL;
	int plen = nfq_get_payload(nfa, &payload);
	struct nfqnl_msg_packet_hdr *ph = nfq_get_msg_packet_hdr(nfa);
	TRACE(3, "in packet callback, len %d, hdr %s, cnt %u\n",
		plen, ph? "ok": "NO", cnt->cnt);

	if (ph && plen >= 16)
	{
		pkt_data *const pkt = &cnt->pkt[cnt->cnt++];

		// time if given, or 0
		struct timeval tv;
		if (nfq_get_timestamp(nfa, &tv))
			pkt->now = 0;
		else
			pkt->now = tv.tv_sec + (tv.tv_usec > 500000);

		// src address, in network order
		memcpy(&pkt->ipaddr, payload + 12, sizeof pkt->ipaddr);

		// id for verdict
		pkt->id = ntohl(ph->packet_id);
		TRACE(3, "pkt set: id=%d, src=%s, time=%ld\n",
			pkt->id, inet_ntoa(*(struct in_addr*)&pkt->ipaddr), pkt->now);
	}

	return 0;
}

static double percent_prob(int p)
{
	return 100.0 * (double)p / ((double)RAND_MAX);
}

static int
daemon_loop(struct nfq_handle *h, DB *db, int queue_num)
{
	DBT key, data;
	ip_data_t ip_data;
	uint32_t ip_addr;

	memset(&ip_data, 0, sizeof ip_data);
	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.flags = data.flags = DB_DBT_USERMEM;
	key.ulen = key.size = sizeof ip_addr;
	key.data = &ip_addr;
	data.ulen = data.size = sizeof ip_data;
	data.data = &ip_data;

	ipaddr_cnt cb_cnt;
	memset(&cb_cnt, 0, sizeof cb_cnt);
	cb_cnt.deadbeef = 0xdeadbeef;

	struct nfq_q_handle *qh =
		nfq_create_queue(h, queue_num, &packet_cb, &cb_cnt);
	if (qh == NULL)
	{
		report_error(&ap, LOG_CRIT, "cannot nfq_create_queue #%d\n", queue_num);
		return 1;
	}

	char buf[4096];
	if (nfq_set_mode(qh, NFQNL_COPY_PACKET, sizeof buf) < 0)
	{
		report_error(&ap, LOG_CRIT, "can't set packet_copy mode\n");
		nfq_destroy_queue(qh);
		return 1;
	}

	int fd = nfq_fd(h);
	int rv;

	while ((rv = recv(fd, buf, sizeof(buf), 0)) >= 0)
	{
		TRACE(2, "PACKET RECV size=%d (q#%d)\n", rv, queue_num);
		nfq_handle_packet(h, buf, rv); // it may go to a different queue_no

		if (cb_cnt.deadbeef != 0xdeadbeef ||
			cb_cnt.cnt > sizeof cb_cnt.pkt / sizeof cb_cnt.pkt[0])
		{
			report_error(&ap, LOG_CRIT, "cb fucked up (%x/%u)\n",
				cb_cnt.deadbeef, cb_cnt.cnt);
			break;
		}
		TRACE(3, "%d slot(s) filled...\n", cb_cnt.cnt);
		while (cb_cnt.cnt > 0)
		{
			pkt_data *const pkt = &cb_cnt.pkt[--cb_cnt.cnt];
			unsigned int verdict = NF_ACCEPT;
			time_t now = pkt->now? pkt->now: time(NULL);
			ip_addr = pkt->ipaddr;

			int rc = db->get(db, NULL, &key, &data, 0);
			if (rc == 0)
			{
				int need_write = 0;
				if (ip_data.probability <= 0)
				{
					TRACE(1, "ACCEPT %s: probability=%d\n",
						inet_ntoa(*(struct in_addr*)&ip_addr),
						ip_data.probability);
				}
				else if (ip_data.last_block + RECENT_DROP_TIME >= now ||
					ip_data.decay >= IPQBDB_NEVER_DECAY) // static block
				{
					verdict = NF_DROP;
					ip_data.last_block = now;
					ip_data.block_cnt += 1;
					need_write = 1;
					TRACE(1, "DROP %s %s\n",
						inet_ntoa(*(struct in_addr*)&ip_addr),
						ip_data.last_block + RECENT_DROP_TIME >= now?
							"again": "forever");
				}
				else // rehabilitate function
				{
					time_t delta = now - ip_data.last_update;
					if (UPDATE_TICK_TIME < delta && ip_data.decay > 0.0)
					{
						double d = (double)ip_data.probability *
								exp2(-(double)delta / ip_data.decay);
						unsigned int prob = isnormal(d)? (int)floor(0.5 + d): 0;
						if (prob < (unsigned) ip_data.probability)
						{
							ip_data.probability = prob;
							ip_data.last_update = now;
							need_write = 1;
						}
					}

					if (rand() < ip_data.probability)
					{
						verdict = NF_DROP;
						ip_data.last_block = now;
						ip_data.block_cnt += 1;
						need_write = 1;
					}

					TRACE(1, "%s %s: probability=%.2f%%, delta=%ld, reason=%d\n",
						verdict == NF_DROP? "DROP": "ACCEPT",
						inet_ntoa(*(struct in_addr*)&ip_addr),
						percent_prob(ip_data.probability),
						(long)delta, ip_data.reason_id);
				}

				/*
				* re-insert the record
				* in case it has been deleted since we read it
				*/
				if (need_write &&
					(rc = db->put(db, NULL, &key, &data, 0)) != 0)
				{
					syslog(LOG_CRIT,
						"cannot write record for %#x: %s\n",
							ip_addr, db_strerror(rc));
					TRACE(1, "DB ERROR writing: %s\n", db_strerror(rc));
				}
			}
			else if (rc != DB_NOTFOUND)
			{
				syslog(LOG_CRIT, "cannot read key %#x: %s\n",
					ip_addr, db_strerror(rc));
				TRACE(1, "DB ERROR reading: %s\n", db_strerror(rc));
			}
			else
			{
				TRACE(1, "ACCEPT %s: not in db\n",
					inet_ntoa(*(struct in_addr*)&ip_addr));
			}

			rc = nfq_set_verdict(qh, pkt->id, verdict, 0, NULL);
			if (rc < 0)
			{
				time_t later = time(NULL);
				syslog(LOG_CRIT,
					"cannot set verdict %s for %#x: nfq_err=%d (time diff=%ld)\n",
					verdict == NF_DROP? "DROP": "ACCEPT",
					ip_addr, nfq_errno, (long)(later - now));
			}
		}
	}

	// printf("unbinding from queue 0\n");
	nfq_destroy_queue(qh);

	// rv < 0 for EINTR
	rv = 0;

	if (caught_signal)
	{
		if (caught_signal == SIGHUP)
		{
			rv = -1;
			caught_signal = 0;
			syslog(LOG_INFO, "reopening on SIGHUP\n");
		}
		else
			syslog(LOG_INFO, "exiting on signal %s\n",
				strsignal(caught_signal));
	}

	return rv;
}

static int setsigs(void)
{
	int rtc;
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = sig_catcher;

	rtc =
		sigaction(SIGPIPE, &act, NULL) ||
		sigaction(SIGILL, &act, NULL)  ||
		sigaction(SIGHUP, &act, NULL)  ||
		sigaction(SIGINT, &act, NULL)  ||
		sigaction(SIGQUIT, &act, NULL) ||
		sigaction(SIGPWR,  &act, NULL) ||
		sigaction(SIGABRT, &act, NULL) ||
		sigaction(SIGTERM, &act, NULL);

	act.sa_handler = SIG_IGN;
	rtc |=
		sigaction(SIGALRM, &act, NULL) ||
		sigaction(SIGIO,   &act, NULL) ||
		sigaction(SIGUSR1, &act, NULL) ||
		sigaction(SIGUSR2, &act, NULL);
//		sigaction(SIGEMT,  &act, NULL);
	return rtc;
}

static char *db_name = IPQBDB_DATABASE_NAME;
static int version_opt, help_opt, no_daemon_opt, no_cleanup_opt;
static int queue_num;
static struct poptOption opttab[] =
{
	{"db-block", 'b', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_name, 0,
	"The database where IPv4 addresses are looked up", "filename"},
	{"queue-num", 'Q', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &queue_num, 0,
	"The queue number for the NFQUEUE target.", "queue"},
	{"no-daemon", 'D', POPT_ARG_NONE, &no_daemon_opt, 0,
	"Stay foreground and use stderr", NULL},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"no-db-cleanup", '\0', POPT_ARG_NONE, &no_cleanup_opt, 0,
	"On exit don't cleanup environment (__db.00? files) if not still busy", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	POPT_TABLEEND
};

int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s\n",
			err_prefix, poptStrerror(rtc));
		errs = 1;
	}
	else
	{
		if (poptPeekArg(opt) != NULL)
		{
			fprintf(stderr, "%s: unexpected argument: %s\n",
				err_prefix, poptGetArg(opt));
			errs = 1;
		}

		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n", err_prefix);
			errs = 2;
		}

		// popt sets verbose to 0 if no arg is given
		// otherwise it stays -1
		verbose += 1;

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			errs = 2;
		}
	}

	if (errs == 1)
		poptPrintUsage(opt, stdout, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (errs)
	{
		rtc = 1;
	}
	else
	{
		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;

		char *fname = database_fname(db_name, &ap);
		if (fname == NULL)
			rtc = 1;
		else
		{
			if (no_daemon_opt == 0)
			{
				char *p = strrchr(fname, '/');
				char const *what = NULL;
				if (what == NULL && daemon(p != NULL /* nochdir */, 0))
					what = "daemon";
				if (what == NULL && setsigs())
					what = "sigaction";
				if (p)
				{
					*p = 0;
					if (chdir(fname))
						what = "chdir";
					*p = '/';
				}
				openlog(err_prefix, LOG_PID, LOG_DAEMON);
				ap.mode = LOG_DAEMON;
				if (what)
				{
					syslog(LOG_CRIT, "cannot %s: %s - exiting\n",
						what, strerror(errno));
					rtc = 1;
				}
			}
		}

		while (rtc == 0)
		{
			DB_ENV *db_env = NULL;
			DB *db = NULL;
			TRACE(2, "Open database %s\n", fname);
			rtc = open_database(fname, &ap, &db_env, &db);

			if (rtc == 0)
			{
				// from nfqnl_test.c
				struct nfq_handle *h;

				rtc = 2;

				// printf("opening library handle\n");
				h = nfq_open();
				if (!h)
				{
					report_error(&ap, LOG_CRIT, "error during nfq_open()\n");
				}
				else
				{
					// printf("unbinding existing nf_queue handler for AF_INET (if any)\n");
					if (nfq_unbind_pf(h, AF_INET) < 0)
					{
						report_error(&ap, LOG_CRIT, "error during nfq_unbind_pf()\n");
					}

					// printf("binding nfnetlink_queue as nf_queue handler for AF_INET\n");
					else if (nfq_bind_pf(h, AF_INET) < 0)
					{
						report_error(&ap, LOG_CRIT, "error during nfq_bind_pf()\n");
					}

					else
					{
						rtc = daemon_loop(h, db, queue_num);
						TRACE(2, "Daemon loop exited with rtc=%d\n", rtc);
					}


	#ifdef INSANE
					/* normally, applications SHOULD NOT issue this command, since
					 * it detaches other programs/sockets from AF_INET, too ! */
					// printf("unbinding from AF_INET\n");
					nfq_unbind_pf(h, AF_INET);
	#endif

					// printf("closing library handle\n");
					nfq_close(h);
				}


				close_db(db);
				close_dbenv(db_env, !no_cleanup_opt);

				if (rtc >= 0)
					break;

				/*
				* when daemon_loop returns a negative number,
				* enter the loop again.
				*/
				TRACE(2, "Restarting loop in 2 secs...\n");
				rtc = 0;
				sleep(2);
			}
		}
	}

	return rtc;
}
