/*
* ibd-parse.c
* written by ale in milano on 28 sep 2008
* read from a socket and extract pcre patterns to add to block.db

Copyright (C) 2008 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

#define _GNU_SOURCE 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <math.h>
#include <errno.h>

#define __USE_BSD 1
// for u_int in db.h
#include <sys/types.h>

#include <sys/stat.h>
#include <fcntl.h>

// misc. ip stuff
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

// Berkeley DB v4.4
#include <db.h>

#include <popt.h>

#include <pcre.h>

// for inet_aton
#define _GNU_SOURCE 1
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

// format of data packet, defines
#include "config_names.h"
#include "dbstruct.h"

#include <assert.h>

static char *trim(char *s)
{
	char *end = s + strlen(s);
	while (s < end && isspace(*(unsigned char*)s))
		++s;
	while (s < end && isspace(*(unsigned char*)--end))
			*end = 0;
	return s;
}

/*
* return NULL if there are only spaces left
*/
static char *ltrim(char *s)
{
	int ch;
	while ((ch = *(unsigned char*)s) != 0 && isspace(ch))
		++s;
	return ch? s: NULL;
}

/*
* terminate previous field, possibly skipping the leading quote,
* and get next
*/
static char *next_field(char **ps)
{
	char *s = *ps;
	assert(!isspace(*(unsigned char*)s));

	int ch = *(unsigned char*)s;
	if (ch == '"' || ch == '\'')
	{
		s = strchr(*ps = s + 1, ch);
		if (s == NULL)
			return NULL; // in this case the previous field lasts til eol
	}
	else
	{
		while (ch != 0 && !isspace(ch))
			ch = *(unsigned char*)++s;
		if (ch == 0)
			return NULL;
	}

	*s = 0;
	return ltrim(s + 1);
}

static app_private ap;
static char const err_prefix[] = "ipqbdbd";
int caught_signal;

#if !defined NDEBUG
#define TRACE(x...) while (ap.mode == 0) { fprintf(stderr, ## x); break; }
#else
#define TRACE(x...)
#endif

static void sig_catcher(int sig)
{
#if !defined(NDEBUG)
	if (ap.mode == 0)
	{
		char buf[80];
		unsigned s = snprintf(buf, sizeof buf,
			"%s [%d]: received signal %s\n",
			err_prefix, (int)getpid(), strsignal(sig));
		if (s >= sizeof buf)
		{
			s = sizeof buf;
			buf[s - 1] = '\n';
		}
		write(2, buf, s);
	}
#endif
	if (sig == SIGABRT)
		syslog(LOG_CRIT, "aborting %s\n", err_prefix);
	caught_signal = sig;
}

static int setsigs(void)
{
	int rtc;
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = sig_catcher;

	rtc =
		sigaction(SIGPIPE, &act, NULL) ||
		sigaction(SIGILL, &act, NULL)  ||
		sigaction(SIGPWR,  &act, NULL) ||
		sigaction(SIGABRT, &act, NULL) ||
		sigaction(SIGHUP, &act, NULL)  ||
		sigaction(SIGUSR1, &act, NULL) ||
		sigaction(SIGINT, &act, NULL)  ||
		sigaction(SIGQUIT, &act, NULL) ||
		sigaction(SIGTERM, &act, NULL);

	act.sa_handler = SIG_IGN;
	rtc |=
		sigaction(SIGALRM, &act, NULL) ||
		sigaction(SIGIO,   &act, NULL) ||
		sigaction(SIGUSR2, &act, NULL);
//		sigaction(SIGEMT,  &act, NULL);
	return rtc;
}

static int initial_count = 5;
static double initial_decay = IPQBDB_INITIAL_DECAY;
static char *socket_name = IPQBDB_PARSE_SOCKET;
static char *db_block_name = IPQBDB_DATABASE_NAME;
static char *db_white_name = IPQBDB_WHITE_DATABASE_NAME;
static char *db_descr_name = IPQBDB_DESCR_DATABASE_NAME;
static char *pcre_name = IPQBDB_PCRE_FILE;
static int version_opt, help_opt, no_cleanup_opt, chartable_opt, no_daemon_opt;
static int verbose = -1;
static struct poptOption opttab[] =
{
	{"db-block", 'b', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_block_name, 0,
	"Default database of blocked IPv4 addresses", "filename"},
	{"db-white", 'w', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_white_name, 0,
	"The whitelist database.", "filename"},
	{"db-descr", 'd', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_descr_name, 0,
	"The reason description table.", "filename"},
	{"fifo-socket", 's', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &socket_name, 0,
	"The listening mkfifo socket.", "name"},
	{"pcre-file", 'f', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &pcre_name, 0,
	"The pcre action file", "fullpath"},
	{"initial-decay", 't', POPT_ARG_DOUBLE|POPT_ARGFLAG_SHOW_DEFAULT, &initial_decay, 0,
	"The default time for the block probability to halve (seconds)", "float"},
	{"initial-count", 'c', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &initial_count, 0,
	"Default initial full probability count", "integer"},
	{"no-daemon", 'D', POPT_ARG_NONE, &no_daemon_opt, 0,
	"Stay foreground and use stderr", NULL},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"pcre-use-locale", 'l', POPT_ARG_NONE, &chartable_opt, 0,
	"Build pcre character tables using current locale settings", NULL},
	{"no-db-cleanup", '\0', POPT_ARG_NONE, &no_cleanup_opt, 0,
	"On exit don't cleanup environment (__db.00? files) if not still busy", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	POPT_TABLEEND
};

typedef struct re_list_t
{
	double decay;
	struct re_list_t *next;
	pcre* code;
	pcre_extra *extra;
	int line;
	int db_ndx;
	int reason;
	int init_prob;
	int ip_sub; // 0 if named "IP"
	unsigned int match, errs; // count of matches, errors
} re_list_t;

typedef struct initialization_stuff
{
	int inited;

	DB_ENV* db_env;
	DB *db1, *db2;

	re_list_t *re_list;
	int max_sub;

	char *home_dir;
	int default_ini_prob;

	char **db_fnames;
	size_t fnames_alloc, fnames_count;

	int verbose_header_given;
} initialization_stuff;

typedef struct runtime_stuff
{
	char *home_dir;

	DB_ENV *db_env;
	DB **db, *db_white;

	re_list_t *re_list;

	int *ovector;
	size_t ovecsize; // number of entries in ovector

	size_t db_count; // number of entries in db

} runtime_stuff;

typedef struct socket_stuff
{
	char *name;
	char *buf;
	int fd;

	size_t len;      // length of string read from socket
	size_t buflen;   // bytes alloced in buf
	size_t in;       // amount read in buf when catching a signal
	size_t errs;     // consecutive errors reading pipe
} socket_stuff;

static unsigned char const *pcre_chartables = NULL;

static void clear_run(runtime_stuff *run)
{
	if (run->db)
		for (size_t i = 0; i < run->db_count; ++i)
			close_db(run->db[i]);

	close_db(run->db_white);
	close_dbenv(run->db_env, !no_cleanup_opt);
	free(run->db);
	free(run->home_dir);

	for (re_list_t *r = run->re_list; r != NULL;)
	{
		void *tmp = r;
		pcre_free(r->code);
		pcre_free(r->extra);
		r = r->next;
		free(tmp);
	}

	memset(run, 0, sizeof *run);
}

static void clear_sock(socket_stuff *sock)
{
	if (sock->fd >= 0)
		close(sock->fd);
	if (sock->name)
	{
		unlink(sock->name);
		free(sock->name);
	}
	free(sock->buf);
	memset(sock, 0, sizeof *sock);
	sock->fd = -1;
}

static void init_inidb(initialization_stuff *ini)
{
	char *descr_fname = database_fname(db_descr_name, &ap);

	if (descr_fname &&
		open_database(descr_fname, &ap, &ini->db_env, NULL) == 0 &&
		open_descrdb(descr_fname, ini->db_env, &ini->db1, &ini->db2) == 0)
			ini->inited = 1;
	caught_signal = 0; // this is tested by get_reason_id()
	free(descr_fname);
}

static int add_fname(initialization_stuff *ini, char *fname)
{
	char *dir_name = database_fname(fname, &ap);
	if (dir_name)
	{
		size_t i;
		char *const p = strrchr(dir_name, '/');
		char *const db_fname = p + 1;
		assert(p); // IPQBDB_DATABASE_PREFIX must contain a full path
		if (p == NULL)
			return -1;

		*p = 0;
		if (ini->home_dir)
		{
			// simpler if all db's share an env in their home directory
			if (strcmp(ini->home_dir, dir_name) != 0)
			{
				free(dir_name);
				return -4;
			}
		}
		else if ((ini->home_dir = strdup(dir_name)) == NULL)
			return -1;

		for (i = 0; i < ini->fnames_count; ++i)
			if (strcmp(db_fname, ini->db_fnames[i]) == 0)
			{
				free(dir_name);
				return i;
			}
		if (ini->fnames_alloc <= ini->fnames_count)
		{
			size_t alloc = 2 * ini->fnames_alloc + 4;
			char **db_fnames = (char**)calloc(alloc, sizeof(char*));
			if (db_fnames == NULL)
				return -1;

			if (ini->db_fnames)
			{
				if (ini->fnames_count)
					memcpy(db_fnames, ini->db_fnames,
						ini->fnames_count * sizeof (char*));
				free(ini->db_fnames);
			}
			ini->db_fnames = db_fnames;
			ini->fnames_alloc = alloc;
		}

		assert(i == ini->fnames_count);
		ini->fnames_count += 1;
		if ((ini->db_fnames[i] = strdup(db_fname)) != NULL)
		{
			free(dir_name);
			return i;
		}
	}
	return -1;
}

static int insert_pcre(initialization_stuff *ini, int line,
	char *re, char *ip, char *reason, char *count, char *fname, char *decay)
{
	int errs = 0, rc;
	char *t = NULL;
	char *malloced_re = NULL;
	const char *error;
	int err_off, tot_sub;
	unsigned long l;
	re_list_t r;

	memset(&r, 0, sizeof r);

	r.line = line;

	/*
	* The symbol '*' in ip indicates to replace <HOST>, before compiling...
	*/
	if (ip[0] == '*' && ip[1] == 0)
	{
		static const char host_repl[] = "<HOST>";
		static const char host_expr[] = "(?P<IP>([0-9]{1,3}\\.){3}[0-9]{1,3})";
		char *repl = strstr(re, host_repl);
		if (repl == NULL || strstr(repl + 1, host_repl) != NULL)
		{
			report_error(&ap, LOG_ERR,
				"replacement \"%s\" is missing or ambiguous at line %d in %s\n",
				host_repl, line, pcre_name);
			errs |= 8;
		}
		else
		{
			size_t off1 = repl - re, off2 = off1 + sizeof host_repl -1;
			malloced_re = (char*)malloc(strlen(re) + sizeof host_expr);
			if (malloced_re == NULL)
				return -1;
			memcpy(malloced_re, re, off1);
			memcpy(malloced_re + off1, host_expr, sizeof host_expr);
			strcat(malloced_re + off2, repl + sizeof host_repl -1);
			re = malloced_re;
			ip[0] = '-';
		}
	}

	/*
	* compile and study, giving verbose details if asked for
	*/
	r.code = pcre_compile(re, 0, &error, &err_off, pcre_chartables);
	if (r.code == NULL)
	{
		report_error(&ap, LOG_ERR,
			"%s \"%.*s<< ABOUT HERE>>%s\" at line %d in %s\n",
			error, err_off, re, re + err_off, line, pcre_name);
		errs |= 2;
		tot_sub = INT_MAX;
	}
	else
	{
		r.extra = pcre_study(r.code, 0, &error);
		if (error)
		{
			report_error(&ap, LOG_WARNING,
				"expression reported as %s at line %d in %s\n",
				error, line, pcre_name);
			errs |= 1;
		}

		rc = pcre_fullinfo(r.code, r.extra, PCRE_INFO_CAPTURECOUNT, &tot_sub);
		if (rc)
		{
			report_error(&ap, LOG_ERR, "info=%d\n", rc);
			errs |= 4;
		}
		else if (verbose && ap.mode == error_report_stderr)
		{
			int backrefmax, firstbyte, lastliteral;
			size_t size, studysize;
			int rc2 =
				pcre_fullinfo(r.code, r.extra, PCRE_INFO_BACKREFMAX, &backrefmax) |
				pcre_fullinfo(r.code, r.extra, PCRE_INFO_FIRSTBYTE, &firstbyte) |
				pcre_fullinfo(r.code, r.extra, PCRE_INFO_LASTLITERAL, &lastliteral) |
				pcre_fullinfo(r.code, r.extra, PCRE_INFO_SIZE, &size) |
				pcre_fullinfo(r.code, r.extra, PCRE_INFO_STUDYSIZE, &studysize);
			if (rc2 == 0)
			{
				char firststr[10];
				if (firstbyte >= 0)
					if (isgraph(firstbyte))
						sprintf(firststr, "'%c'", firstbyte);
					else
						sprintf(firststr, "%#x", firstbyte);
				else if (firstbyte == -1)
					strcpy(firststr, "no");
				else
					strcpy(firststr, "anchored");

				if (ini->verbose_header_given == 0)
				{
					printf("PCRE expressions from %s fullinfo\n"
						"%10.10s %10.10s %10.10s %10.10s %10.10s %10.10s %s\n",
						pcre_name,
						"backrefmax", "firstbyte", "lastliteral", "size",
						"studysize", "line no", "expression");
					ini->verbose_header_given = 1;
				}
				printf("%10d %10.10s %10d %10zu %10zu %10d %s\n",
					backrefmax, firststr, lastliteral, size,
					studysize, line, re);
			}
			else
			{
				report_error(&ap, LOG_WARNING,
					"unable to retrieve fullinfo, rc2=%d\n", rc);
			}
		}
	}

	/*
	* named subexpression or subexpression number. ambiguous references
	* will have to be resolved at runtime
	*/
	l = strtoul(ip, &t, 0);
	if (l > 0 && l <= (unsigned) tot_sub && *t == 0)
		r.ip_sub = (int)l;
	else if (ip[0] == '-' && ip[1] == 0)
	{
		int namecount, esize, count = 0, ndx = 0;
		char *nametable;

		if (pcre_fullinfo(r.code, r.extra, PCRE_INFO_NAMECOUNT, &namecount) ||
			pcre_fullinfo(r.code, r.extra, PCRE_INFO_NAMEENTRYSIZE, &esize) ||
			pcre_fullinfo(r.code, r.extra, PCRE_INFO_NAMETABLE, &nametable))
				report_error(&ap, LOG_ERR, "info failed\n");
		else
			while (namecount--> 0)
			{
				if (strcmp(nametable + 2, "IP") == 0)
				{
					ndx = 256 * nametable[0] + nametable[1];
					++count;
				}
				nametable += esize;
			}
		if (count == 0)
		{
			report_error(&ap, LOG_ERR,
				"missing (?P<IP>...) named sub expression at line %d in %s\n",
				line, pcre_name);
			errs |= 2;
		}
		else if (count == 1 && ndx > 0 && ndx < tot_sub)
			r.ip_sub = ndx;
		else
			r.ip_sub = 0;
	}
	else
	{
		report_error(&ap, LOG_ERR,
			"invalid ip sub num \"%s\" (pattern has %d subs) at line %d in %s\n",
			ip, tot_sub, line, pcre_name);
		errs |= 2;
	}

	/*
	* reason can be either a number or a literal
	*/
	l = strtoul(reason, &t, 0);
	if (l > 0 && l < INT_MAX && *t == 0)
		r.reason = (int)l;
	else
	{
		if (ini->inited == 0)
			init_inidb(ini);
		if (ini->inited > 0)
		{
			if (get_reason_id(&r.reason, reason, &ap,
				ini->db1, ini->db2, verbose) != 0)
					errs |= 2;
		}
		else
			errs |= 2;
	}

	/*
	* initial count
	*/
	if (count)
	{
		l = strtoul(count, &t, 0);
		if (!(*t == 0 && l < CHAR_BIT * sizeof(int) &&
			(r.init_prob = RAND_MAX >> (int)l) > 0))
		{
			r.init_prob = 1;
			report_error(&ap, *t == 0? LOG_NOTICE: LOG_ERR,
				"%s is too high: initial probability set to 1 at line %d in %s\n",
				count, line, pcre_name);
			if (*t)
				errs |= 2;
		}
	}
	else
		r.init_prob = ini->default_ini_prob;

	/*
	* database filename
	*/
	r.db_ndx = add_fname(ini, fname? fname: db_block_name);
	if (r.db_ndx < 0)
	{
		if (r.db_ndx == -4)
			report_error(&ap, LOG_ERR,
				"%s has a directory different than %s at line %d in %s\n",
				fname? fname: db_block_name, ini->home_dir, line, pcre_name);
		errs |= 2;
	}

	/*
	* decay
	*/
	if (decay)
	{
		r.decay = strtod(decay, &t);
		if (*t != 0 || !isnormal(r.decay) || r.decay <= 0)
		{
			report_error(&ap, LOG_ERR, "invalid decay %s at line %d in %s\n",
				decay, line, pcre_name);
			errs |= 2;
		}
	}
	else
		r.decay = initial_decay;

	/*
	* allocate new expression if no errors
	*/
	free(malloced_re);

	if (errs == 0)
	{
		re_list_t *new_re = (re_list_t*)malloc(sizeof(re_list_t));
		if (new_re == NULL)
			return -1;

		/*
		* insert in front: list will be reversed
		*/
		memcpy(new_re, &r, sizeof *new_re);
		new_re->next = ini->re_list;
		ini->re_list = new_re;

		if (ini->max_sub < tot_sub)
			ini->max_sub = tot_sub;
	}

	return errs;
}

static int parse_pcre(char *s, int line, initialization_stuff *ini)
{
	char *re, *ip, *reason, *count = NULL, *fname = NULL, *decay = NULL;
	int delim = s[0];

	if (delim == '#' || delim == 0)
		return 0;

	ip = strchr(re = s + 1, delim);
	if (ip == NULL)
	{
		report_error(&ap, LOG_ERR,
			"missing matching %c RE delimiter at line %d in %s\n",
			delim, line, pcre_name);
		return 1;
	}

	*ip = 0;
	ip = ltrim(ip + 1);
	if (ip == NULL)
	{
		report_error(&ap, LOG_ERR,
			"missing ip subexpression number at line %d in %s\n",
			line, pcre_name);
		return 1;
	}

	reason = next_field(&ip);
	if (reason == NULL)
	{
		report_error(&ap, LOG_ERR,
			"missing reason field at line %d in %s\n",
			line, pcre_name);
		return 1;
	}

	count = next_field(&reason);
	if (count)
	{
		fname = next_field(&count);
		if (fname)
		{
			decay = next_field(&fname);
			if (decay)
			{
				char *x = next_field(&decay); // terminate decay
				if (x)
					report_error(&ap, LOG_WARNING,
						"ignoring extra stuff \"%s\" at line %d in %s\n",
						x, line, pcre_name);
				if (decay[0] == '-' && decay[1] == 0)
					decay = NULL;
			}
			if (fname[0] == '-' && fname[1] == 0)
				fname = NULL;
		}
		if (count[0] == '-' && count[1] == 0)
			count = NULL;
	}

	return insert_pcre(ini, line, re, ip, reason, count, fname, decay);
}

enum { read_pcre_after_sighup = 0, read_pcre_daemonize_if_ok};
static int read_pcre(int daemonize_if_ok, runtime_stuff *run)
/*
* run is cleared before entry. This function initializes it.
* return 0 if no errors.
*/
{
	char buf[4096], *s;
	int line = 0;
	int errs = 0;
	FILE *fp = fopen(pcre_name, "r");

	initialization_stuff ini;
	memset(&ini, 0, sizeof ini);

	if (!((unsigned)initial_count < CHAR_BIT * sizeof(int) &&
		(ini.default_ini_prob = RAND_MAX >> initial_count) > 0))
			ini.default_ini_prob = 1;

	if (fp == NULL)
	{
		report_error(&ap, LOG_CRIT, "cannot open %s: %s\n",
			pcre_name, strerror(errno));
		return -1;
	}

	while ((s = fgets(buf, sizeof buf, fp)) != NULL)
		errs |= parse_pcre(trim(s), ++line, &ini);

	fclose(fp);
	if (ini.inited)
	{
		close_db(ini.db1);
		close_db(ini.db2);
	}

	if (errs ||
		ini.fnames_count <= 0 ||
		ini.db_fnames == NULL ||
		ini.home_dir == NULL)
	{
		report_error(&ap, LOG_CRIT,
			"Failed to initialize: %s\n",
			errs? "errors in pcre file":
					"no block db referenced");
		close_dbenv(ini.db_env, !no_cleanup_opt);
		return -1;
	}

	if (daemonize_if_ok)
	{
		char const *what = NULL;
		if (no_daemon_opt == 0 && daemon(1 /* nochdir */, 0))
			what = "daemon";
		else
		{
			if (no_daemon_opt == 0)
				openlog(err_prefix, LOG_PID, ap.mode = LOG_DAEMON);
			if (setsigs())
				what = "sigaction";
			else if (chdir(ini.home_dir))
				what = "chdir";
		}

		if (what)
		{
			report_error(&ap, LOG_CRIT, "cannot %s: %s - exiting\n",
				what, strerror(errno));
			close_dbenv(ini.db_env, !no_cleanup_opt);
			return -1;
		}
	}
	else if (chdir(ini.home_dir))
	{
		report_error(&ap, LOG_CRIT, "cannot chdir to %s: %s\n",
			ini.home_dir, strerror(errno));
		close_dbenv(ini.db_env, !no_cleanup_opt);
		return -1;
	}

	run->home_dir = ini.home_dir;
	run->db_count = ini.fnames_count;
	run->db_env = ini.db_env;
	run->db = (DB**)malloc(ini.fnames_count * sizeof(DB*));
	run->ovecsize = (ini.max_sub + 1) * 3;
	run->ovector = (int*)malloc(run->ovecsize * sizeof(int));

	char *white_fname = database_fname(db_white_name, &ap);

	if (run->db &&
		run->ovector &&
		white_fname &&
		open_database(white_fname, &ap, &run->db_env, &run->db_white) == 0)
	{
		size_t i;
		for (i = 0; i < ini.fnames_count; ++i)
			if (open_database(ini.db_fnames[i], &ap, &run->db_env, &run->db[i]))
				break;

		if (i < ini.fnames_count)
		{
			close_dbenv(ini.db_env, !no_cleanup_opt);
			return -1;
		}
	}

	free(white_fname);

	/*
	* reverse the list, thus obtaining original ordering
	*/
	while (ini.re_list)
	{
		re_list_t *r = ini.re_list;
		ini.re_list = r->next;
		r->next = run->re_list;
		run->re_list = r;
		assert(r->db_ndx >= 0 && (unsigned)r->db_ndx <= run->db_count);
		assert(r->ip_sub >= 0 && (unsigned)r->ip_sub * 2 + 1 < run->ovecsize);
		assert(r->decay > 0.0);
	}

	free(ini.db_fnames);
	return errs;
}

static int get_ip_address(uint32_t* ip, char *dest, size_t dsize,
	char const *src, size_t ssize, int ex_line)
{
	struct in_addr in;

	assert(src);
	assert(dest);
	assert(dsize > 0);

	if (ssize >= dsize)
	{
		report_error(&ap, LOG_WARNING,
			"captured subexp %.*s has length %zu >= %zu, from expr %d\n",
			(int)ssize, src, ssize, dsize, ex_line);
		*dest = 0;
		return 1;
	}

	memcpy(dest, src, ssize);
	dest[ssize] = 0;
	int rtc = inet_aton(dest, &in);
	if (rtc == 0) // not valid
	{
		report_error(&ap, LOG_WARNING,
			"invalid IP captured from expr %d: %s\n",
			ex_line, dest);
		return 1;
	}

	memcpy(ip, &in, sizeof *ip);
	return 0;
}

static double percent_prob(int p)
{
	return 100.0 * (double)p / ((double)RAND_MAX);
}

static void display_written_block(uint32_t ip_addr,
	ip_data_t *ip_data, ip_data_t *old_data, DB *is_null)
{
	(void)is_null; // is null

	struct in_addr in;
	memcpy(&in, &ip_addr, sizeof in);

	report_error(&ap, LOG_INFO,
		"%s record for %s, decay: %g, probability: %d=%.2f%%\n",
		old_data? "changed": "inserted",
		inet_ntoa(in),
		ip_data->decay,
		ip_data->probability,
		percent_prob(ip_data->probability));
}

static int block_matched(runtime_stuff *run, re_list_t const * const r,
	char const *str, size_t const len, int const rc)
{
	char dest32[32]; // captured IP
	uint32_t ip_addr;
	int no_ip_addr = 1;
	int err = 0;

	if (rc == 0)
		report_error(&ap, LOG_WARNING,
			"pcre_exec returned 0 for expr at line %d, ovecsize=%zd\n",
			r->line, run->ovecsize);

	if (r->ip_sub == 0) // not determined at compile time
	{
		char const *stringptr = NULL;
		int rc2 = pcre_get_named_substring(r->code, str, run->ovector,
				rc > 0? rc: (int)run->ovecsize/3, "IP", &stringptr);
		if (rc2 > 0 && stringptr != NULL)
			no_ip_addr = get_ip_address(&ip_addr, dest32, sizeof dest32,
				stringptr, rc2, r->line);
		else
			report_error(&ap, LOG_ERR,
				"get_substring rc=%d for expr at line %d, subject=%s"/* no\n */,
				rc2, r->line, str);
	}
	else
	{
		int const firstndx = r->ip_sub * 2;
		if (/* asserted: firstndx + 1 < run->ovecsize && */
			run->ovector[firstndx] >= 0 &&
			run->ovector[firstndx + 1] > 0 &&
			run->ovector[firstndx] < run->ovector[firstndx + 1] &&
			(unsigned)run->ovector[firstndx + 1] < len)
		{
			no_ip_addr = get_ip_address(&ip_addr, dest32, sizeof dest32,
				str + run->ovector[firstndx],
				run->ovector[firstndx + 1] - run->ovector[firstndx],
				r->line);
		}
		else
			report_error(&ap, LOG_ERR,
				"mismatch index: ip_sub=%d, first=%d, second=%d"
				" for expr at line %d, subject=%s"/* no\n */,
				r->ip_sub, run->ovector[firstndx], run->ovector[firstndx + 1],
				r->line, str);
	}

	if (no_ip_addr)
		err = 1;
	else
	{
		double decay;
		int rtc, dont_block;
		rtc = dont_block = check_whitelist(run->db_white, ip_addr, &decay);
		if (rtc == 1)
		{
			decay = r->decay;
			dont_block = 0;
		}
		else if (rtc == 0)
		{
			if (decay == 0.0)
				dont_block = 1;
			if (verbose || dont_block)
				report_error(&ap, LOG_NOTICE,
					"IP captured is whitelisted%s: %s\n",
					dont_block? " and won't be blocked": "", dest32);
		}
		else
			err = 1;

		if (!dont_block)
		{
			rtc = write_block(run->db[r->db_ndx], ip_addr, decay,
				r->reason, r->init_prob, 0,
				verbose? display_written_block: NULL, NULL);

			if (rtc < 0)
				err = 1;
		}
	}

	return err;
}

static int run_pcre(runtime_stuff *run, char const *str, size_t const len)
/*
* run all expression until one is found.
* if a match is found, set the record in the corresponding block db.
* return -1 if an unexpected error occurs.
*/
{
	re_list_t *r = run->re_list, *prev = NULL;
	while (r)
	{
		int const rc =
			pcre_exec(r->code, r->extra, str, len,
				0, 0, run->ovector, run->ovecsize);
		if (rc < 0) // failed
		{
			if (rc != PCRE_ERROR_NOMATCH)
				report_error(&ap, LOG_WARNING,
					"pcre_exec returned %d for expr at line %d, subject=%s"/* no\n */,
					rc, r->line, str);
			prev = r;
			r = r->next;
		}
		else
		{
			int err = block_matched(run, r, str, len, rc);

			++r->match;
			if (err)
				++r->errs;
			/*
			* move matched rule to front, if not already there
			*/
			else if (prev)
			{
				prev->next = r->next;
				r->next = run->re_list;
				run->re_list = r;
			}
			break;
		}
	}

	if (r == NULL && verbose > 1)
		report_error(&ap, LOG_DEBUG, "no match for %s"/* no\n */, str);

	return 0;
}

static void log_re_list(runtime_stuff *run)
{
	re_list_t *r = run->re_list;
	report_error(&ap, LOG_INFO,
		"%5.5s %5.5s %5.5s %5.5s %5.5s %19.19s %5.5s %5.5s\n",
		"LINE", "MATCH", "ERRS", "subex",
		"db", "ini. block prob.", "Rcode", "decay");
	while (r)
	{
		report_error(&ap, LOG_INFO,
			"%5d %5u %5u %5d %5d"
			" %10d %6.2f%% %5d %g\n",
			r->line, r->match, r->errs, r->ip_sub, r->db_ndx,
			r->init_prob, percent_prob(r->init_prob), r->reason, r->decay);
		r = r->next;
	}
}

static int open_socket(socket_stuff *sock, char const *home_dir)
/*
* sock is cleared before entry. This function initializes it.
* return 0 if no errors or interrupted system call.
*/
{
	char *name, *tmpname;
	size_t len = strlen(home_dir) + strlen(socket_name) + 5;
	size_t buflen = 8*1024;

	if (len + len >= buflen)
	{
		report_error(&ap, LOG_CRIT,
			"Failed to initialize: name too long: %s/%s\n",
			home_dir, socket_name);
		return -1;
	}

	sock->buf = (char*)malloc(sock->buflen = buflen);

	name = sock->buf;
	tmpname = sock->buf + len;

	strcpy(name, home_dir);
	strcat(strcat(name, "/"), socket_name);
	strcpy(tmpname, name);
	strcat(tmpname, ".tmp");

	unlink(tmpname);
	if (mkfifo(tmpname, 0660))
	{
		report_error(&ap, LOG_CRIT, "cannot mkfifo %s: %s\n",
			tmpname, strerror(errno));
		return -1;
	}

	if (rename(tmpname, name))
	{
		report_error(&ap, LOG_CRIT, "cannot rename %s as %s: %s\n",
			tmpname, name, strerror(errno));
		return -1;
	}

	sock->name = strdup(name);
	if (sock->name == NULL)
	{
		report_error(&ap, LOG_CRIT, "memory fault\n");
		return -1;
	}

	sock->fd = -1;
	return 0;
}

static int read_socket(socket_stuff *sock)
/*
* read into sock->buf and return 0 on newline.
* return 1 for recoverable error (EOF, signal).
* return -1 for unexpected error.
*/
{
	char *const ebuf = &sock->buf[sock->buflen - 1];
	char *sbuf = &sock->buf[sock->in];
	char *const bbuf = &sock->buf[0];
	int fd = sock->fd;

	if (fd < 0)
	{
		/*
		* this blocks until syslog opens it for writing
		*/
		errno = 0;
		fd = sock->fd = open(sock->name, O_RDONLY);
		if (fd < 0)
		{
			if (errno == EINTR)
				return 1;

			report_error(&ap, LOG_CRIT, "cannot open %s: %s\n",
				sock->name, strerror(errno));
			return -1;
		}
	}

	errno = 0;
	for (;;)
	{
		int p;

		p = read(fd, sbuf, 1);
		if (p > 0)
		{
			p = *sbuf++;
			if (p == '\n' || sbuf >= ebuf)
			{
				*sbuf = 0;
				sock->len = sbuf - bbuf;
				sock->in = sock->errs = 0;
				return 0;
			}
		}
		else if (p < 0)
		{
			switch (errno)
			{
				case EINTR:
					sock->in = sbuf - bbuf;
					return 1;

				case EAGAIN:
				default:
					/*
					* something wrong, try and recover
					* see if it works
					*/
					p = ++sock->errs > 4? -1: 1;
					report_error(&ap, p > 0? LOG_CRIT: LOG_ERR,
						"cannot read %s (%zd): %s\n",
						sock->name, sock->errs, strerror(errno));
					return p;
			}
		}
		else /* (p == 0): pipe EOF */
		{
			sock->in = sock->len = 0;
			close(sock->fd);
			sock->fd = -1;
			report_error(&ap, LOG_NOTICE, "found EOF on %s\n", sock->name);
			return 0;
		}
	}
}

int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s\n",
			err_prefix, poptStrerror(rtc));
		errs = 1;
	}
	else
	{
		if (poptPeekArg(opt) != NULL)
		{
			fprintf(stderr, "%s: unexpected argument: %s\n",
				err_prefix, poptGetArg(opt));
			errs = 1;
		}

		if (version_opt)
		{
			int utf8, unicode, newl;
			pcre_config(PCRE_CONFIG_UTF8, &utf8);
			pcre_config(PCRE_CONFIG_UNICODE_PROPERTIES, &unicode);
			pcre_config(PCRE_CONFIG_NEWLINE, &newl);
			fprintf(stdout,
				"%s: version %s\n"
				"pcre %s%s%s, with newline %s\n",
				err_prefix, PACKAGE_VERSION,
				pcre_version(),
					utf8? ", with utf8": "",
					unicode? ", with Unicode support": "",
					newl == 10? "LF": newl == 13? "CR": newl == 3338? "CRLF":
						newl == -2? "ANYCRLF": newl == -1? "ANY": "UNKNOWN");
			errs = 2;
		}

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			fputs("\nNon-comment lines in the pcre-file are formatted as\n\n"
			"\t/pcre/ N \"reason\" [initial-count] [filename] [decay]\n\n"
			"where N is the integer for the captured pcre subexpression. "
			"Slashes around\npcre can be replaced, quotes around the reason are "
			"optional (can specify the\ncorresponding integer) and the optional "
			"fields can be replaced by a hyphen (-).\nThe filename is the block "
			"database, subject to prefix/suffix arrangenments.\nAll DB filenames"
			" must end up in the same directory, which is also used for\nthe "
			"socket.\n", stdout);
			fputs_initial_count_help();
			errs = 2;
		}

		// popt sets verbose to 0 if no arg is given
		// otherwise it stays -1
		verbose += 1;

		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;

		if (!isnormal(initial_decay) || initial_decay <= 0)
		{
			fprintf(stderr, "%s: invalid initial decay %g\n",
				err_prefix, initial_decay);
			errs = 2;
		}
	}

	if (errs == 1)
		poptPrintUsage(opt, stdout, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (errs)
	{
		rtc = 1;
	}
	else
	{
		runtime_stuff run;
		socket_stuff sock;
		memset(&run, 0, sizeof run);
		memset(&sock, 0, sizeof sock);
		sock.fd = -1;

		if (chartable_opt)
			pcre_chartables = pcre_maketables();
		rtc = read_pcre(read_pcre_daemonize_if_ok, &run);
		if (rtc == 0)
		{
			rtc = open_socket(&sock, run.home_dir);
			while (rtc == 0)
			{
				rtc = read_socket(&sock);
				if (rtc == 0 && sock.len > 0)
				{
					if (run_pcre(&run, sock.buf, sock.len) < 0)
						break;
				}
				else if (rtc > 0) // interrupted by signal or reopen socket
				{
					int exit_rtc = 2;
					switch (caught_signal)
					{
						case 0: // temporary(?) error reading pipe
							clear_sock(&sock);
							rtc = open_socket(&sock, run.home_dir);
							break;

						case SIGHUP:
							clear_run(&run);
							rtc = read_pcre(read_pcre_after_sighup, &run);
							break;

						case SIGUSR1:
							log_re_list(&run);
							rtc = 0;
							break;

						case SIGINT:
						case SIGQUIT:
						case SIGTERM:
							exit_rtc = 1;
							// through
						default:
							report_error(&ap, exit_rtc > 1? LOG_ALERT: LOG_NOTICE,
								"exiting after signal %s\n",
								strsignal(caught_signal));
							rtc = exit_rtc;
							break;
					}
					caught_signal = 0;
				}
			}
			rtc -= 1;
			clear_sock(&sock);
		}
		clear_run(&run);
	}

	return rtc;
}
