/*
* Ipq Berkeley db Daemon  DEL - ibd-del
* written by ale in milano on 15 nov 2008
* delete and/or list records in a block.db

Copyright (C) 2008 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#define _GNU_SOURCE 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <limits.h>
#include <ctype.h>
#include <signal.h>
#include <syslog.h>

#define __USE_BSD 1
// for u_int in db.h
#include <sys/types.h>

// Berkeley DB v4.4
#include <db.h>

#include <popt.h>

// format of data packet
#include "config_names.h"
#include "dbstruct.h"

#include <time.h>

// for inet_aton
#define _GNU_SOURCE 1
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "utf8_util.h"


static app_private ap;
static char const err_prefix[] = "ibd-del";
int caught_signal;

#if !defined NDEBUG
#define TRACE(x...) while (ap.mode == 0) { fprintf(stderr, ## x); break; }
#else
#define TRACE(x...)
#endif

static void sig_catcher(int sig)
{
#if !defined(NDEBUG)
	if (ap.mode == 0)
	{
		char buf[80];
		unsigned s = snprintf(buf, sizeof buf,
			"%s [%d]: received signal %s\n",
			err_prefix, (int)getpid(), strsignal(sig));
		if (s >= sizeof buf)
		{
			s = sizeof buf;
			buf[s - 1] = '\n';
		}
		write(2, buf, s);
	}
#endif
	if (sig == SIGABRT)
		syslog(LOG_CRIT, "aborting %s\n", err_prefix);
	caught_signal = sig;
}

static int setsigs(void)
{
	int rtc;
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = sig_catcher;

	rtc =
		sigaction(SIGPIPE, &act, NULL) ||
		sigaction(SIGILL, &act, NULL)  ||
		sigaction(SIGHUP, &act, NULL)  ||
		sigaction(SIGINT, &act, NULL)  ||
		sigaction(SIGQUIT, &act, NULL) ||
		sigaction(SIGPWR,  &act, NULL) ||
		sigaction(SIGABRT, &act, NULL) ||
		sigaction(SIGTERM, &act, NULL);

	act.sa_handler = SIG_IGN;
	rtc |=
		sigaction(SIGALRM, &act, NULL) ||
		sigaction(SIGIO,   &act, NULL) ||
		sigaction(SIGUSR1, &act, NULL) ||
		sigaction(SIGUSR2, &act, NULL);
//		sigaction(SIGEMT,  &act, NULL);
	return rtc;
}

static char *db_block_name = IPQBDB_DATABASE_NAME;
static char *db_descr_name = IPQBDB_DESCR_DATABASE_NAME;
static char *ip_address = "0.0.0.0/0";
static char *min_time = "0";
static char *max_probability = "100%";
static int version_opt, help_opt, syslog_opt, cleanup_opt, verbose = -1;
static int list_opt, del_opt;
static struct poptOption opttab[] =
{
	{"db-block", 'b', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_block_name, 0,
	"The database of blocked IPv4 addresses", "filename"},
	{"db-descr", 'd', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_descr_name, 0,
	"The reason description table.", "filename"},
	{"min-time", 't', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &min_time, 0,
	"The minimum time elapsed since record was last updated", "int[D|H|M|S]"},
	{"max-probability", 'p', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &max_probability, 0,
	"The maximum probability a record may have when being deleted", "int[%]"},
	{"ip-addr", 'i', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &ip_address, 0,
	"The single address or range to delete or list, if any", "ip[-last|/cidr]"},
	{"ls", 'L', POPT_ARG_NONE, &list_opt, 0,
	"List selected records", NULL},
	{"del", 'D', POPT_ARG_NONE, &del_opt, 0,
	"Delete selected records", NULL},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"log-syslog", 'l', POPT_ARG_NONE, &syslog_opt, 0,
	"Log to syslog rather than std I/O", NULL},
	{"db-cleanup", '\0', POPT_ARG_NONE, &cleanup_opt, 0,
	"On exit cleanup environment (__db.00? files) if not still busy", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	POPT_TABLEEND
};

typedef struct ip4_range
{
	uint32_t first;
	uint32_t last;
} ip4_range;

static int get_ip_address(ip4_range *ip)
{
	int i, mode_flag = 0;
	union bytewise
	{
		uint32_t ip;
		unsigned char b[4];
	} uf, ul, *un = &uf;
	char *p = ip_address;

	memset(&ul, 0, sizeof ul);
	memset(&uf, 0, sizeof uf);

	/*
	* parse any of "1.2.3.4", "1.2.3.0-1.2.3.7", "1.2.3.0/29",
	* store first ip in uf, other stuff in ul
	*/
	for (i = 0;;)
	{
		char *t = NULL;
		int sep;
		unsigned long l = strtoul(p, &t, 10);
		int err = 0;
		if (l > UCHAR_MAX)
		{
			err = 1;
		}
		else
		{
			un->b[i] = (unsigned char)l;
			p = t;
			sep = *p++;
			if (sep == '.' && mode_flag != '/')
			{
				if (++i >= 4)
					err = 2;
			}
			else if (sep == 0)
				break;
			else if (mode_flag == 0 && (sep == '-' || sep == '/'))
			{
				mode_flag = sep;
				i = 0;
				un = &ul;
			}
			else
				err = 3;
		}

		if (err)
		{
			report_error(&ap, LOG_ERR, "%s: invalid ip_address %s (err=%d)\n",
				err_prefix, ip_address, err);
			return 1;
		}
	}

	/*
	* set first and last ip according to parsed values
	* (ip is untouched on error)
	*/
	if (mode_flag == '-')
	{
		if (memcmp(&uf, &ul, sizeof uf) > 0)
		{
			report_error(&ap, LOG_ERR, "%s: invalid ip range %s\n",
				err_prefix, ip_address);
			return 1;
		}
		ip->last = ul.ip;
	}
	else if (mode_flag == '/')
	{
		if (ul.b[0] > 32)
		{
			report_error(&ap, LOG_ERR, "%s: invalid ip CIDR %s\n",
				err_prefix, ip_address);
			return 1;
		}

		if (ul.b[0])
		{
			ip->last = uf.ip;
			if (ul.b[0] < 32)
				ip->last |= htonl((1U << (32 - ul.b[0])) - 1);
		}
		else
			ip->last = -1;
	}
	else
		ip->last = uf.ip;

	ip->first = uf.ip;

	return 0;
}

static int get_seconds(int *secs)
{
	char *t = NULL;
	unsigned long l = strtoul(min_time, &t, 10);
	if (t && *t)
	{
		switch (toupper(*(unsigned char*)t))
		{
			case 'D':
				l *= 24UL;
				// all thru...
			case 'H':
				l *= 60UL;
			case 'M':
				l *= 60UL;
			case 'S':
				++t;
			default:
				break;
		}
	}

	if (t == NULL || *t != 0 || l > INT_MAX)
	{
		report_error(&ap, LOG_ERR, "%s: invalid time %s\n",
			err_prefix, min_time);
		return 1;
	}

	*secs = (int)l;
	return 0;
}

static int get_probability(int *prob)
{
	char *t = NULL;
	unsigned long l = strtoul(max_probability, &t, 10);
	if (t && *t == '%')
	{
		l *= RAND_MAX;
		l /= 100;
		++t;
	}

	if (t == NULL || *t != 0 || l > RAND_MAX)
	{
		report_error(&ap, LOG_ERR, "%s: invalid probability %s\n",
			err_prefix, max_probability);
		return 1;
	}

	*prob = (int)l;
	return 0;
}

static double percent_prob(int p)
{
	return 100.0 * (double)p / ((double)RAND_MAX);
}

static const char *month_abbr(unsigned ndx)
{
	static const char *abbr[] =
	{"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	if (ndx >= sizeof abbr / sizeof abbr[0]) return "---";
	return abbr[ndx];
}

#define TIME_STRING_BUFSIZE 9
static char *
time_string(time_t datum, time_t now, char *dest)
{
	char buf[256]; // avoid overflows in case of error

	if (datum == 0) // never set: blank
		buf[0] = 0;
	else if (now - datum <= 900) // less than a quarter: NNNs ago
		sprintf(buf, "%ds ago", (int)(now - datum));
	else
	{
		struct tm tm;
		localtime_r(&datum, &tm);
		if (now - datum < 20L*3600) // less than 20 hour: hh:mm:ss
			sprintf(buf, "%02d:%02d:%02d", tm.tm_hour, tm.tm_min, tm.tm_sec);
		else if (now - datum < 20L*86400L) // less than 20 days: dd hh:mm
			sprintf(buf, "%d %02d:%02d", tm.tm_mday, tm.tm_hour, tm.tm_min);
		else // Mon YYYY
			sprintf(buf, "%s %d", month_abbr(tm.tm_mon), tm.tm_year + 1900);
	}

	buf[TIME_STRING_BUFSIZE - 1] = 0;
	memcpy(dest, buf, TIME_STRING_BUFSIZE);
	return dest;
}

static int
do_del(DB *db, DB*db_d, ip4_range* ip, int secs, int prob)
{
	DBC *curs;
	DBT key, data;
	uint32_t ip_last = ip->last;
	uint32_t cur_key = ip->first;
	int rtc, rtc2;
	unsigned total_found = 0, total_del = 0, total_delfail = 0;
	time_t const now = time(NULL), last_time = now - secs;

	char descr[IPQBDB_DESCR_MAX_LENGTH];

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);

	key.data = &cur_key;
	key.ulen = key.size = sizeof cur_key;
	key.flags = data.flags = DB_DBT_USERMEM;
	rtc = db->get_pagesize(db, &data.ulen);
	data.ulen *= 32;
	data.flags = DB_DBT_USERMEM;
	if (rtc ||
		(data.data = malloc(data.ulen)) == NULL)
	{
		report_error(&ap, LOG_ERR, "%s: memory fault\n", err_prefix);
		return -1;
	}

	/*
	* acquire a read cursor: this can possibly delete a record that has
	* been modified since we read it, even if it was requested a min
	* modified time (secs).
	*/
	rtc = db->cursor(db, NULL, &curs, 0);
	if (rtc)
	{
		db->err(db, rtc, "cannot create cursor");
		free(data.data);
		return rtc;
	}

	for (;;)
	{
		void *p, *retkey, *retdata;
		size_t retklen, retdlen;

		rtc = curs->c_get(curs, &key, &data, DB_MULTIPLE_KEY|DB_SET_RANGE);
		if (rtc)
		{
			if (rtc != DB_NOTFOUND)
				db->err(db, rtc, "cannot read cursor");
			break;
		}

		for (DB_MULTIPLE_INIT(p, &data);;)
		{
			DB_MULTIPLE_KEY_NEXT(p, &data, retkey, retklen, retdata, retdlen);
			if (p == NULL ||
				(rtc =
					memcmp(&ip_last, retkey, sizeof ip_last) < 0?
						DB_NOTFOUND: 0) != 0)
							break;

			ip_data_t *const ip_data = (ip_data_t*)retdata;
			if (retdlen < sizeof *ip_data || retklen < sizeof cur_key)
			{
				rtc = 1;
				break;
			}

			cur_key = *(uint32_t*)retkey;
			if (ip_data->last_update <= last_time &&
				ip_data->last_block <= last_time &&
				ip_data->probability <= prob)
			{
				if (del_opt)
				{
					rtc2 = db->del(db, NULL, &key, 0);
					if (rtc2)
					{
						total_delfail += 1;
						db->err(db, rtc2, "delete");
					}
					else
						total_del += 1;
				}

				if (list_opt)
				{
					char buf1[TIME_STRING_BUFSIZE],
						buf2[TIME_STRING_BUFSIZE],
						buf3[TIME_STRING_BUFSIZE];
					if (verbose && !total_found)
						printf("%15s %8s %15s %8s %10s %8s %10s %10s DESCRIPTION\n",
							"IP", "CREATED", "PROBABILITY", "BLOCKED", "PACKETS",
							"UPDATED", "DECAY", "THRESHOLD");

					get_descr(db_d, ip_data->reason_id, 0, descr, sizeof descr);
					printf("%15s %8s %10d=%3.0f%% %8s %10d %8s %10g %10d %s\n",
						inet_ntoa(*(struct in_addr*)&cur_key),
						time_string(ip_data->created, now, buf1),
						ip_data->probability,
						percent_prob(ip_data->probability),
						time_string(ip_data->last_block, now, buf2),
						ip_data->block_cnt,
						time_string(ip_data->last_update, now, buf3),
						ip_data->decay,
						ip_data->decay_add_cnt,
						descr);
				}

				total_found += 1;
			}
		}

		if (rtc)
			break;

		cur_key = htonl(ntohl(cur_key) + 1);
	}

	rtc2 = curs->c_close(curs);
	if (rtc2)
	{
		db->err(db, rtc2, "cannot close cursor");
		if (rtc == 0 || rtc == DB_NOTFOUND)
			rtc = rtc2;
	}

	free(data.data);

	if (verbose)
	{
		time_t const end = time(NULL);
		if (end != now)
			sprintf(descr, " in %d secs", (int)(end - now));
		else
			descr[0] = 0;
		printf("%u record(s) found, %u deleted, %u failed deletion(s)%s\n",
			total_found, total_del, total_delfail, descr);
	}

	return (rtc != 0 && rtc != DB_NOTFOUND) || total_delfail;
}

int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0, secs = 0, prob = 0;
	ip4_range ip;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s\n",
			err_prefix, poptStrerror(rtc));
		errs = 1;
	}
	else
	{
		if (poptPeekArg(opt) != NULL)
		{
			fprintf(stderr, "%s: unexpected argument: %s\n",
				err_prefix, poptGetArg(opt));
			errs = 1;
		}

		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n", err_prefix);
			errs = 2;
		}

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			errs = 2;
		}

		else if (!list_opt && !del_opt)
		{
			fprintf(stdout, "%s: at least one of -L or -D must be specified\n",
				err_prefix);
			errs = 1;
		}

		if (get_ip_address(&ip) || get_seconds(&secs) || get_probability(&prob))
			errs = 3;

		// popt sets verbose to 0 if no arg is given
		verbose += 1;

		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;
		if (syslog_opt)
		{
			openlog(err_prefix, LOG_PID, LOG_DAEMON);
			ap.mode = LOG_DAEMON;
		}
	}

	if (errs == 1)
		poptPrintUsage(opt, stdout, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (errs)
		rtc = 1;
	else
	{
		char *block_fname = database_fname(db_block_name, &ap);
		char *descr_fname = NULL;

		if (block_fname == NULL)
			rtc = 1;
		else if (list_opt)
		{
			descr_fname = database_fname(db_descr_name, &ap);
			if (descr_fname == NULL)
				rtc = 1;
		}


		if (rtc)
		{
			if (verbose)
				report_error(&ap, LOG_INFO, "Bailing out...\n");
		}
		else
		{
			DB_ENV *db_env = NULL;
			DB *db_block = NULL, *db_de1 = NULL, *db_de2 = NULL;

			setsigs();

			rtc = open_database(block_fname, &ap, &db_env, &db_block);

			if (rtc == 0 && descr_fname)
				rtc = open_descrdb(descr_fname, db_env, &db_de1, &db_de2);

			if (rtc == 0)
			{
				if (verbose)
				{
					printf("%s%s%s records in IP range %d.%d.%d.%d-%d.%d.%d.%d,\n"
						"\tmin age %d secs, max prob %d=%.2f%%.\n",
						del_opt? "delete": "",
						del_opt && list_opt? " and ": "",
						list_opt? "list": "",
						((unsigned char*)&ip.first)[0],
						((unsigned char*)&ip.first)[1],
						((unsigned char*)&ip.first)[2],
						((unsigned char*)&ip.first)[3],
						((unsigned char*)&ip.last)[0],
						((unsigned char*)&ip.last)[1],
						((unsigned char*)&ip.last)[2],
						((unsigned char*)&ip.last)[3],
						secs, prob, percent_prob(prob));
				}
				rtc = do_del(db_block, db_de1, &ip, secs, prob);
			}

			if (rtc)
				rtc = 2;

			close_db(db_de2);
			close_db(db_de1);
			close_db(db_block);

			close_dbenv(db_env, cleanup_opt);
		}
	}

	return rtc;
}
