/* uconfig.c -- manages the program configuration
   Copyright (C) 2004 Julio A. Becerra

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _POSIX_SOURCE
#define _ISOC99_SOURCE
#define _XOPEN_SOURCE		/* TODO we should not need this... */
#define _XOPEN_SOURCE_EXTENDED	/* ... bug in GNU libc6? */

#include <ctype.h>  
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include "user_iface.h"
#include "uconfig.h"
#include "common.h"
#include "misc.h"

#define IPCHAT_DIR ".ipchat"
#define CONFIG_FILE ".ipchat/config"
#define CONFIG_FILE_C ".ipchat/config_old"
#define MAJOR_VERSION 1
#define MINOR_VERSION 0
#define CONF_VERSION "1.0"
#define CONF_VERSION_SIZE 3
#define VERS_BUF_SIZE 8
#define PORT_BUF_SIZE 8

static int cf_load_1_0 (int fd);
static int cf_save_1_0 (int fd);

config_t cfg = {DEFAULT_NICK, DEFAULT_PORT};

int 
cf_load ()
{
	char *file;
	int fd;
	ssize_t ret;
	char vers [VERS_BUF_SIZE];
	struct passwd *pass;
	int r;
	
	pass = getpwuid (getuid());
	if (pass != NULL) 
		strncpy (cfg.nick, pass->pw_name, MAX_NICK);
	else	
		strncpy (cfg.nick, DEFAULT_NICK, MAX_NICK);
	cfg.nick [MAX_NICK-1] = '\0';
	cfg.listen_port=DEFAULT_PORT;
	
	file = home_path (CONFIG_FILE);	
	if (file == NULL)
		return ERR;
		
	fd = open (file, 0, 0);
	free (file);
	if (fd == -1) {
		if (errno == ENOENT) {
			ui_redraw_nick ();
			return OK;
		}
		if (errno == EACCES)
			ui_output_err ("Cannot access config file");
		return ERR;
	}

	ret = read_tag (fd, NULL, "version", vers, VERS_BUF_SIZE);
	if (ret == ERR) {
		close (fd);
		return ERR;
	}
	
	else if (ret != 0)
		ui_output_err ("Invalid config file");
		    
	else {
		if (strcmp (CONF_VERSION, vers) == 0) {
			lseek (fd, 0, SEEK_SET);
			r = cf_load_1_0 (fd);
			if (r == ERR) {
				close (fd);
				return ERR;
			}
			else if (r == ERR_C)
				ui_output_err ("Invalid config file");
		} 
		else 
 			ui_output_err ("Unknown config file version");	
	}
	
	ui_redraw_nick ();
	close (fd);
	return OK;
}

static int
cf_load_1_0 (int fd)
{
	int r;
	char port [PORT_BUF_SIZE];
	char nick [MAX_NICK];
	
	r = read_tag (fd, NULL, "nick", nick, MAX_NICK);
	if (r == 0) 
		strncpy (cfg.nick, nick, MAX_NICK);	
	
	lseek (fd, 0, SEEK_SET); 
	
	r = read_tag (fd, NULL, "listen_port", port, PORT_BUF_SIZE);
	if (r == 0)
		cfg.listen_port = (port_t) strtoul (port, NULL, 10);
	
	return OK;
}
	

int 
cf_save ()
{
	char *file;
	char *file2;
	int fd;
	char vers[VERS_BUF_SIZE];
	int r;
					
        file = home_path (CONFIG_FILE);
        if (file == NULL)
		return ERR;

	fd = open (file, O_CREAT | O_RDWR,  S_IRUSR | S_IWUSR);
	if (fd == -1) {
		if (errno == EACCES)
			ui_output_err ("Cannot access config file.");
		free (file);
		return ERR;
	}

	r = read_tag (fd, NULL, "version", vers, VERS_BUF_SIZE); 
	if (r == ERR) {
		free (file);
		close (fd);
		return ERR;
	}

		
	if (r == 0 && strncmp (vers, CONF_VERSION, CONF_VERSION_SIZE) != 0) {
		close (fd);
		file2 = home_path (CONFIG_FILE_C);
		if (file == NULL) {
			free (file);
			return ERR;
		}

		r = rename (file, file2);
		free (file2);
		if (r == -1) {
			free (file);
			return ERR;
		}
	
		fd = open (file, O_CREAT | O_RDWR, S_IRUSR | S_IWUSR);
		if (fd == -1) {
			if (errno == EACCES)
				ui_output_err ("Cannot access config file");
			free (file);

			return ERR;
		}
	}
	else
		lseek (fd, 0, SEEK_SET); 
         
	free (file);
		
	ftruncate (fd, 0);

	r = write_tag (fd, "version", CONF_VERSION, 0);
	if (r == ERR) {
		close (fd);
		return ERR;
	}

	r = cf_save_1_0 (fd);
	if (r != OK) {
		close (fd);
		return ERR;
	}

	close (fd);
	return OK;
}


static int
cf_save_1_0 (int fd)
{
	char b [PORT_BUF_SIZE];
	
	CHECK (write_tag (fd, "nick", cfg.nick, 0) != ERR);
	snprintf (b, PORT_BUF_SIZE, "%u", cfg.listen_port);
	CHECK (write_tag (fd, "listen_port", b, 0) != ERR);
	
	return OK;
error:
	return ERR;
}

