/*
   Copyright (C) 2011  Stephane Pion
   This file is part of Intifada.

    Intifada is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Intifada is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Intifada.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <prefs.h>
#include <epan/uat.h>
#include <epan/emem.h>
#include <epan/report_err.h>
#include <string.h>
#include <intifada_wrapper.h>
#include <macros.h>
#include <register_dissector.h>
#include <dissect_mac_asterix.h>
#include <dissect_udp_asterix.h>

static void user_asterix_update_cb(void* r _U_, const char** err _U_);

static void* user_asterix_copy_cb(void* dest, const void* orig, unsigned len _U_);

static void user_asterix_free_cb(void* r);

static void change_cb();

static void uat_update_vs();

static void uat_creation(module_t *m);

static module_t *module=NULL;

static gboolean dump_all=TRUE;
static gboolean debug=FALSE;
static const char *asterix_description = NULL; 
static uat_t* mac_uat=NULL;

static value_string *asterix_families = NULL;
UAT_VS_DEF(user_asterix, family, asterix_parameters_type, 0, "")
// network type choice
UAT_VS_DEF(user_asterix, layer, asterix_parameters_type, 0, "")
UAT_CSTRING_CB_DEF(user_asterix,addr,asterix_parameters_type)
UAT_VS_DEF(user_asterix, enabled, asterix_parameters_type, 0, "enabled")

static asterix_parameters_type* macline = NULL;
static guint num_macline = 0;


static value_string toggle_enabled_disabled[] = {
		{ ENABLED_FLAG        , "enabled" },
		{ DISABLED_FLAG        , "disabled" },
		{ 0x00        , NULL }
};
static value_string toggle_network_layer[] = {
		{ MACLLC_TYPE        , "MAC/LLC" },
		{ UDP_TYPE        , "UDP" },
		{ 0x00        , NULL }
};

static uat_field_t user_flds[] = {
		INTIFADA_UAT_FLD_VS(user_asterix,family,"Families",asterix_families,"The known asterix families"),
		INTIFADA_UAT_FLD_VS(user_asterix,layer,"Network layer",toggle_network_layer,"Network protocol"),
		INTIFADA_UAT_FLD_CSTRING_ISPRINT(user_asterix,addr,"network address","The network address.\n" "Mac address format : hh:hh:hh:hh:hh:hh \nUDP address format : PORT"),
		INTIFADA_UAT_FLD_VS(user_asterix,enabled,"Listen",toggle_enabled_disabled,"Enable Listen on"),
		UAT_END_FIELDS
};

void register_preferences(int proto)
{
	// handle preferences
	module=prefs_register_protocol(proto, change_cb);
	get_factory_description(&asterix_description);

	prefs_register_static_text_preference(module
			,"license"
			,"Copyright 2009,2011 .\n\
			Have fun"
			,"stephane.pion@free.fr");

	prefs_register_bool_preference(module,"dump_all","dump all items","dump all items even if they are 0 or empty string",&dump_all);
	prefs_register_bool_preference(module,"debug","debug asterix plugin","print to stdout debug informations",&debug);
	prefs_register_string_preference(module, "description","Asterix XML description","Name of the XML file which describes Asterix categories",&asterix_description);


	change_cb();

}

value_string* get_asterix_families()
						{
	return asterix_families;
						}

void get_listen_addresses(asterix_parameters_type** m,guint* l)
{
	*m=macline;
	*l=num_macline;
}

gboolean is_dump_all()
{
	return dump_all;
}

gboolean is_debug()
{
	return debug;
}

void user_asterix_update_cb(void* r _U_, const char** err _U_)
{
	if(is_debug()==TRUE)printf("user_asterix_update_cb\n");
	change_cb();
}

void* user_asterix_copy_cb(void* dest, const void* orig, unsigned len _U_)
{
	asterix_parameters_type* d = (asterix_parameters_type *)dest;
	const asterix_parameters_type* o = (const asterix_parameters_type *)orig;

	d->family = o->family;
	d->addr = g_strdup(o->addr);
	d->enabled = o->enabled;
	return dest;
}

void user_asterix_free_cb(void* r)
{
	asterix_parameters_type* h = (asterix_parameters_type*)r;
	if (h->addr) g_free(h->addr);
}

void change_cb()
{
	int err=0;
	if(is_debug()==TRUE)printf("change_cb call\n");
	// Called after prefs initialization and after each pref modification
	err = init_xml_asterix(asterix_description);
	if(err==0)
	{
		asterix_families_update(&asterix_families);
		uat_update_vs();
		//uat_destroy(mac_uat);
		uat_creation(module);

		// update dissectors load
		// first step, clean up and then, reload
		asterix_parameters_type* parameters;
		guint size;
		get_listen_addresses(&parameters,&size);
		update_dissectors(parameters,size,dissect_mac_asterix,dissect_udp_asterix);
	}
	else
	{
		printf("No description found\n");
	}

}

void uat_update_vs()
{
	if(debug==TRUE)printf("uat_update_vs:change families from 0x%p -> 0x%p\n",(void*)&user_flds,(void*)&asterix_families);
	user_flds[0].cbdata.chk=&asterix_families[0];
	user_flds[0].cbdata.set=&asterix_families[0];
	user_flds[0].cbdata.tostr=&asterix_families[0];
	user_flds[0].fld_data=&asterix_families[0];
}

void uat_creation(module_t *m)
{
	char *err=NULL;
	if(mac_uat==NULL)
	{
		mac_uat = intifada_uat_new("Asterix association Table",
				sizeof(asterix_parameters_type),
				"user_asterix",
				TRUE,
				(void**) &macline,
				&num_macline,
				UAT_CAT_GENERAL,
				"ChUserAsterixSection",
				user_asterix_copy_cb,
				user_asterix_update_cb,
				user_asterix_free_cb,
				NULL,
				user_flds );
		prefs_register_uat_preference(m,"mac_table","Addresses Table","A table that handle addresses which be tap",mac_uat);
		uat_load(mac_uat,&err);
		if(err!=NULL)
		{
			report_failure("Error loading table :%s (Suppress above line from user_asterix. Update it, you certainly change description file)",err);
			g_free(err);
		}
	}
}
